// ************************************************************************* //
//  File: avtLineoutFilter.C
// ************************************************************************* //

#include <avtLineoutFilter.h>

#include <vtkDataSet.h>
#include <vtkLineoutFilter.h>
#include <vtkPolyData.h>
#include <avtDatasetExaminer.h>
#include <avtExtents.h>
#include <avtMetaData.h>
#include <avtVector.h>
#include <avtIntervalTree.h>
#include <ImproperUseException.h>
#include <InvalidDimensionsException.h>
#include <UnexpectedValueException.h>
#include <DebugStream.h>

#include <vtkVisItCellLocator.h>
#include <vtkUnsignedIntArray.h>
#include <vtkDataSetRemoveGhostCells.h>
#include <vtkPoints.h>
#include <vtkIdList.h>
#include <vtkMath.h>
#include <vtkCellData.h>
#include <vtkPointData.h>
#include <vtkCellArray.h>
#include <map>
#include <set>

using std::map;
using std::set;

struct Point
{
    float x[3];
};

struct CellInfo
{
    int origCell;
    int origDomain;
    intVector currCell;
    vector<Point> isect;  
} ;

void ClosestPointOnLine(const float *, const float *, const float*, float *);

// ****************************************************************************
//  Method: avtLineoutFilter constructor
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Thu Apr 25 16:01:28 PST 2002
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 20 17:20:38 PDT 2004
//    Initialize useOriginalCells.
//
// ****************************************************************************

avtLineoutFilter::avtLineoutFilter()
{
    OverrideTrueSpatialExtents();
    useOriginalCells = false;
}


// ****************************************************************************
//  Method:  avtLineoutFilter::Create
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Thu Apr 25 16:01:28 PST 2002
//
// ****************************************************************************

avtFilter *
avtLineoutFilter::Create()
{
    return new avtLineoutFilter();
}


// ****************************************************************************
//  Method:      avtLineoutFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Thu Apr 25 16:01:28 PST 2002
//
// ****************************************************************************

void
avtLineoutFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const LineoutAttributes*)a;
}


// ****************************************************************************
//  Method: avtLineoutFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtLineoutFilter with the given
//      parameters would result in an equivalent avtLineoutFilter.
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Thu Apr 25 16:01:28 PST 2002
//
// ****************************************************************************

bool
avtLineoutFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(LineoutAttributes*)a);
}


// ****************************************************************************
//  Method: avtLineoutFilter::ExecuteData
//
//  Purpose:
//      Sends the specified input and output through the Lineout filter.
//
//  Arguments:
//      in_ds      The input dataset.
//      <unused>   The domain number.
//      <unused>   The label.
//
//  Returns:       The output dataset.
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Thu Apr 25 16:01:28 PST 2002
//
//  Modifications:
//    Kathleen Bonnell, Fri Jul 12 17:28:31 PDT 2002 
//    No longer send YScale to vtkLineoutFilter, it is not needed.
//
//    Hank Childs, Tue Sep 10 16:46:57 PDT 2002
//    Re-work memory management.
//
//    Kathleen Bonnell, Tue Dec 23 10:18:06 PST 2003 
//    Set vtkLineoutFilter's UpdateGhostLevel, so that ghost levels can be 
//    ignored.  Ensure output has points.
//
//    Kathleen Bonnell, Thu Jul 29 09:55:49 PDT 2004 
//    Moved code to Sampling method, added call to NoSampling. 
//
// ****************************************************************************

vtkDataSet *
avtLineoutFilter::ExecuteData(vtkDataSet *in_ds, int domain, std::string)
{
    if (!atts.GetSamplingOn())
        return NoSampling(in_ds, domain);
    else 
        return Sampling(in_ds, domain);
}


// ****************************************************************************
//  Method: avtLineoutFilter::RefashionDataObjectInfo
//
//  Purpose:
//      Allows the filter to change its output's data object information, which
//      is a description of the data object.
//
//  Programmer: Kathleen Bonnell 
//  Creation:   April 25, 202 
//
//  Modifications:
//    Kathleen Bonnell, Tue Mar 23 08:48:33 PST 2004
//    Set X and Y axis labels.
//
//    Brad Whitlock, Thu Jul 22 17:20:05 PST 2004
//    Set the Y units.
//
//    Kathleen Bonnell, Thu Jan  6 10:34:57 PST 2005 
//    Remove TRY-CATCH block in favor of testing for ValidActiveVariable. 
//
// ****************************************************************************

void
avtLineoutFilter::RefashionDataObjectInfo(void)
{
    GetOutput()->GetInfo().GetAttributes().SetTopologicalDimension(1);
    GetOutput()->GetInfo().GetAttributes().SetXLabel("Distance");
    GetOutput()->GetInfo().GetAttributes().SetYLabel("Value");
    GetOutput()->GetInfo().GetValidity().InvalidateZones();
    GetOutput()->GetInfo().GetValidity().InvalidateSpatialMetaData();

    if (GetInput()->GetInfo().GetAttributes().ValidActiveVariable())
    {
        std::string units(GetInput()->GetInfo().GetAttributes().GetVariableUnits());
        if(units != "")
            GetOutput()->GetInfo().GetAttributes().SetYUnits(units);
    }
}


// ****************************************************************************
//  Method: avtLineoutFilter::VerifyInput
//
//  Purpose:
//      Verifies that the input is 2D data, throws an exception if not.
//
//  Programmer: Kathleen Bonnell
//  Creation:   April 26, 2002
//
//  Modifications:
//    Kathleen Bonnell, Mon Dec 23 11:50:54 PST 2002  
//    Modified to test topological dimenions, so that lineouts of point-var
//    or lines data will be disallowed.  Lineouts of 3d now allowed.
//
// ****************************************************************************
 
void
avtLineoutFilter::VerifyInput(void)
{
    if  (GetInput()->GetInfo().GetAttributes().GetTopologicalDimension() < 2)
    {
        EXCEPTION2(InvalidDimensionsException, "Lineout", "2D or 3D");
    }
}




// ****************************************************************************
//  Method: avtLineoutFilter::PerformRestriction
//
//  Purpose:
//      Calculates the restriction on the meta-data and the line endpoints. 
//
//  Arguments:
//      spec    The current pipeline specification.
//
//  Returns:    The new specification.
//
//  Programmer: Kathleen Bonnell 
//  Creation:   December 19, 2003 
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 20 17:20:38 PDT 2004
//    Set useOriginalCells.
//
// ****************************************************************************


avtPipelineSpecification_p
avtLineoutFilter::PerformRestriction(avtPipelineSpecification_p spec)
{
    avtPipelineSpecification_p rv = new avtPipelineSpecification(spec);

    useOriginalCells = false;
    if (!GetInput()->GetInfo().GetValidity().GetZonesPreserved())
    {
        rv->GetDataSpecification()->TurnZoneNumbersOn();
        useOriginalCells = true;
        return rv;
    }

    //
    // Get the interval tree.
    //
    avtIntervalTree *it = GetMetaData()->GetSpatialExtents();
    if (it == NULL)
    {
        return rv;
    }

    double *dpt = atts.GetPoint1();
    float pt1[3] = {dpt[0], dpt[1], dpt[2]};
    dpt = atts.GetPoint2();
    float pt2[3] = {dpt[0], dpt[1], dpt[2]};
    float rayDir[3] = {pt2[0]-pt1[0], pt2[1]-pt1[1], pt2[2]-pt1[2]};   

    vector<int> domains;
    it->GetDomainsList(pt1, rayDir, domains);
    rv->GetDataSpecification()->GetRestriction()->RestrictDomains(domains);

    return rv;
}


// ****************************************************************************
//  Method: avtLineoutFilter::PostExecute
//
//  Purpose:
//      Cleans up after the execution.  This manages extents.
//
//  Programmer: Kathleen Bonnell 
//  Creation:   January 14, 2004
//
//  Modifications:
//
// ****************************************************************************

void
avtLineoutFilter::PostExecute(void)
{
    avtDataAttributes &outAtts = GetOutput()->GetInfo().GetAttributes();
    outAtts.GetTrueSpatialExtents()->Clear();
    outAtts.GetEffectiveSpatialExtents()->Clear();

    double bounds[6];
    avtDataset_p ds = GetTypedOutput();
    avtDatasetExaminer::GetSpatialExtents(ds, bounds);
    outAtts.GetCumulativeTrueSpatialExtents()->Set(bounds);
}


// ****************************************************************************
//  Method: avtLineoutFilter::CreatePolys
//
//  Purpose:
//    Creates a vtkPolyData (vertices) from  the passed information.
//    The x-coordinate is determined using the distance of each point (pts)
//    from the origin of the line (pt1).  The y-coordinate is determined by
//    the scalar value at each intersected cell (cells).  If the scalars
//    are point-centered, then they are averaged for each cell.
//
//  Arguments:
//    ds        The input dataset.
//    pt1       The origin of the line (used to calculate distances).
//    pt2       The endpoint of the line (used to calculate distances).
//    pts       Cell center points.
//    cells     A list of intersected cells. 
//
//  Returns:    The output poly data.
//
//  Programmer: Kathleen Bonnell 
//  Creation:   July 27, 2004 
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 20 17:20:38 PDT 2004
//    Added pt2 arg.  Pts being passed are now cell centers, so find
//    closest point along the line defined by pt1 and pt2, and use it to
//    calculate distance.
//
// ****************************************************************************

vtkPolyData *
avtLineoutFilter::CreatePolys(vtkDataSet *ds, float *pt1, float *pt2,
                              vtkPoints *pts, vtkIdList *cells)
{
    vtkPolyData *polys = vtkPolyData::New();
    bool pointData = true; 
    vtkDataArray *scalars = ds->GetPointData()->GetScalars();
    if (scalars == NULL)
    {
        pointData = false;
        scalars = ds->GetCellData()->GetScalars();
        if (scalars == NULL)
        {
            return polys;
        }
    }
    int npts = pts->GetNumberOfPoints();
    float currentPoint[3], closestPoint[3];
    float newPoint[3] = {0., 0., 0.};
    vtkPoints *outPts = vtkPoints::New();
    polys->SetPoints(outPts);
    outPts->Delete();
    vtkIdList *ptIds = vtkIdList::New();
    float sum = 0.;
    int i, j;
    float oldX = -1.;
    bool requiresSort = false;
    for (i = 0; i < npts; i++)
    {
        pts->GetPoint(i, currentPoint);
        ClosestPointOnLine(pt1, pt2, currentPoint, closestPoint);
        newPoint[0] = sqrt(vtkMath::Distance2BetweenPoints(pt1, closestPoint));
        if (newPoint[0] < oldX)
        {
            requiresSort = true;
        }
        oldX = newPoint[0];
        if (pointData)
        {
            sum = 0;
            ds->GetCellPoints(cells->GetId(i), ptIds);
            int numCellPts = ptIds->GetNumberOfIds();
            for (j = 0; j < numCellPts; j++)
                sum += scalars->GetTuple1(ptIds->GetId(j));
            if (numCellPts > 0)
               sum /= (float) numCellPts;

            newPoint[1] = sum;
        }
        else 
        {
            newPoint[1] = scalars->GetTuple1(cells->GetId(i));
        }
        outPts->InsertNextPoint(newPoint);
    }
    ptIds->Delete();

    vtkPoints *sortedPts;
    if (requiresSort)
    {
        sortedPts = vtkPoints::New();
        std::map <float, int> sortedIds;
        float x;
        for (i = 0; i < outPts->GetNumberOfPoints(); i++)
        {
            x = outPts->GetPoint(i)[0];
            sortedIds.insert(std::map < float, int> ::value_type(x, i));
        }
        std::map <float, int>::iterator it;
        for (it = sortedIds.begin(); it != sortedIds.end(); it++)
        {
            sortedPts->InsertNextPoint(outPts->GetPoint((*it).second));
        }
        polys->SetPoints(sortedPts);
        sortedPts->Delete();
    }
    else
    {
        sortedPts = outPts;
    }

    vtkCellArray *verts = vtkCellArray::New();
    polys->SetVerts(verts);
    verts->Delete();

    verts->InsertNextCell(sortedPts->GetNumberOfPoints());
    for (i = 0; i < sortedPts->GetNumberOfPoints(); i++)
    {
        verts->InsertCellPoint(i);
    }

    return polys;
}


// ****************************************************************************
//  Method: avtLineoutFilter::NoSampling
//
//  Purpose:
//    Peforms a lineout by intersecting cells.  
//
//  Arguments:
//    in_ds      The input dataset.
//    domain     The domain number.
//
//  Returns:       The output dataset.
//
//  Programmer: Kathleen Bonnell 
//  Creation:   July 27, 2004 
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 20 17:20:38 PDT 2004
//    Use different CreatePolys method if we must use Original Cell Numbers.
//    Tell the locator to ignore ghost zones.
//
// ****************************************************************************

vtkDataSet *
avtLineoutFilter::NoSampling(vtkDataSet *in_ds, int domain)
{
    double *dpt = atts.GetPoint1();

    float pt1[3] = {dpt[0], dpt[1], dpt[2]};
    dpt = atts.GetPoint2();
    float pt2[3] = {dpt[0], dpt[1], dpt[2]};

    vtkVisItCellLocator *locator = vtkVisItCellLocator::New();
    locator->SetDataSet(in_ds);
    locator->SetIgnoreGhosts(true);
    locator->BuildLocator();

    vtkPoints *pts = vtkPoints::New();
    vtkIdList *cells = vtkIdList::New();
    vtkDataSet *rv = NULL;
    int success = locator->IntersectWithLine(pt1, pt2, pts, cells);

    if (success)
    {
        if (!useOriginalCells)
        {
            rv = CreatePolys(in_ds, pt1, pt2, pts, cells);
        }
        else 
        {
            rv = CreatePolysFromOrigCells(in_ds, pt1, pt2, pts, cells);
        }
        if (rv->GetNumberOfCells() == 0 ||
            rv->GetNumberOfPoints() == 0)
        {
            debug5 << "vtkVisItCellLocator returned empty DS for domain " 
                   << domain << "." << endl;
            rv = NULL;
        }
    }
    else
    {
        debug5 << "vtkVisItCellLocator returned empty DS for domain " 
               << domain << "." << endl;
    }

    pts->Delete();
    cells->Delete();
    locator->Delete();

    ManageMemory(rv);
    return rv;
}


// ****************************************************************************
//  Method: avtLineoutFilter::Sampling
//
//  Purpose:
//    Executes the lineout using a sampling method. 
//
//  Arguments:
//    in_ds      The input dataset.
//    domain     The domain number.
//
//  Returns:       The output dataset.
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Thu Apr 25 16:01:28 PST 2002
//
//  Modifications:
//    Kathleen Bonnell, Fri Jul 12 17:28:31 PDT 2002 
//    No longer send YScale to vtkLineoutFilter, it is not needed.
//
//    Hank Childs, Tue Sep 10 16:46:57 PDT 2002
//    Re-work memory management.
//
//    Kathleen Bonnell, Tue Dec 23 10:18:06 PST 2003 
//    Set vtkLineoutFilter's UpdateGhostLevel, so that ghost levels can be 
//    ignored.  Ensure output has points.
//
//    Kathleen Bonnell, Tue Jul 27 10:18:14 PDT 2004
//    Moved from 'ExecueData' method. 
// 
//    Hank Childs, Wed Sep  8 19:57:21 PDT 2004
//    Remove ghost zones before doing a lineout.  This is because the
//    vtkLineoutFilter is a bit touchy about ghost zone values.  If you have
//    values > 1, then it can do interpolations to nodal data that can 
//    mistakenly identify real zones as ghost.
//
// ****************************************************************************

vtkDataSet *
avtLineoutFilter::Sampling(vtkDataSet *in_ds, int domain)
{
    vtkDataSetRemoveGhostCells *ghosts = vtkDataSetRemoveGhostCells::New();
    ghosts->SetInput(in_ds);

    vtkLineoutFilter *filter = vtkLineoutFilter::New();
    double *dpt = atts.GetPoint1();

    float pt1[3] = {dpt[0], dpt[1], dpt[2]};
    dpt = atts.GetPoint2();
    float pt2[3] = {dpt[0], dpt[1], dpt[2]};

    filter->SetInput(ghosts->GetOutput());
    filter->SetPoint1(pt1);
    filter->SetPoint2(pt2);
    filter->SetNumberOfSamplePoints(atts.GetNumberOfSamplePoints());
    filter->GetOutput()->SetUpdateGhostLevel(0);
    vtkPolyData *outPolys = filter->GetOutput();
    outPolys->Update();

    vtkDataSet *rv = outPolys;
    if (outPolys->GetNumberOfCells() == 0 ||
        outPolys->GetNumberOfPoints() == 0)
    {
        debug5 << "vtkLineoutFilter returned empty DS for domain " 
               << domain << "." << endl;
        rv = NULL;
    }

    ManageMemory(rv);
    filter->Delete();
    ghosts->Delete();

    return rv;
}

// ****************************************************************************
//  Method: avtLineoutFilter::CreatePolysFromOrigCells
//
//  Purpose:
//    Creates a vtkPolyData (vertices) from  the passed information.
//    The x-coordinate is determined using the distance of each point (pts)
//    from the origin of the line (pt1).  The y-coordinate is determined by
//    the scalar value at each intersected cell (cells).  If the scalars
//    are point-centered, then they are averaged for each cell.
//
//    This method uses the avtOriginalCellNumbers array to combine values
//    that come from cells that share the same original cell. (e.g. a Slice,
//    which will triangulate quads -- we only want 1 intersection point
//    for the quad, not 1 for each triangle.
//
//  Arguments:
//    ds        The input dataset.
//    pt1       The origin of the line (used to calculate distances).
//    pts       Intersection points along the line.
//    cells     A list of intersected cells. 
//
//  Returns:    The output poly data.
//
//  Programmer: Kathleen Bonnell 
//  Creation:   July 27, 2004 
//
//  Modifications:
//    Brad Whitlock, Wed Nov 3 10:16:32 PDT 2004
//    Fixed on win32.
//
// ****************************************************************************

vtkPolyData *
avtLineoutFilter::CreatePolysFromOrigCells(vtkDataSet *ds, float *pt1, float *pt2,
                              vtkPoints *pts, vtkIdList *cells)
{
    vtkPolyData *polys = vtkPolyData::New();
    bool pointData = true; 
    vtkDataArray *scalars = ds->GetPointData()->GetScalars();
    if (scalars == NULL)
    {
        pointData = false;
        scalars = ds->GetCellData()->GetScalars();
        if (scalars == NULL)
        {
            return polys;
        }
    }

    int i, j, k;
    int npts = pts->GetNumberOfPoints();
    vtkUnsignedIntArray *origCells = vtkUnsignedIntArray::SafeDownCast(
        ds->GetCellData()->GetArray("avtOriginalCellNumbers"));

    if (!origCells)
    {
       EXCEPTION2(UnexpectedValueException, "avtOriginalCellNumbers", 
                  "nada.  Internal error.");
    }
    int currentCell;
    int origCell;
    int origDomain;
    float center[3];
    vector<CellInfo> cellInfoList;
    bool dup = false;
    for (i = 0; i < npts; i++)
    {
        currentCell = cells->GetId(i); 
        pts->GetPoint(i, center);
        if (origCells)
        {
            origCell = (int)origCells->GetComponent(currentCell, 1); 
            origDomain = (int)origCells->GetComponent(currentCell, 0);
            for (j = 0, dup = false; j < cellInfoList.size() && !dup; j++)
            {
                if ((origCell == cellInfoList[j].origCell) &&
                    (origDomain == cellInfoList[j].origDomain))
                {
                    dup = true;
                    cellInfoList[j].currCell.push_back(currentCell);
                    Point p;
                    p.x[0]  = center[0];
                    p.x[1]  = center[1];
                    p.x[2]  = center[2];
                    cellInfoList[j].isect.push_back(p);
                }
            }
        }
        if (!dup)
        {
            CellInfo a;
            if (origCells)
            {
                a.origCell = origCell;
                a.origDomain = origDomain;
            }
            a.currCell.push_back(currentCell);
            Point p;
            p.x[0]  = center[0];
            p.x[1]  = center[1];
            p.x[2]  = center[2];
            a.isect.push_back(p);
            cellInfoList.push_back(a);
        }
    }

    float closestPoint[3];
    float newPoint[3] = {0., 0., 0.};
    vtkPoints *outPts = vtkPoints::New();
    polys->SetPoints(outPts);
    outPts->Delete();
    vtkIdList *ptIds = vtkIdList::New();
    float sum = 0.;
    float oldX = -1.;
    bool requiresSort = false;
    
    for (i = 0; i < cellInfoList.size(); i++)
    {
        int nDups = cellInfoList[i].currCell.size();
        if (nDups == 1)
        {
            ClosestPointOnLine(pt1, pt2, cellInfoList[i].isect[0].x, closestPoint);
            newPoint[0] = sqrt(vtkMath::Distance2BetweenPoints(pt1, closestPoint));
        }
        else
        {
            sum = 0.;
            for (j = 0; j < nDups; j++)
            {
                ClosestPointOnLine(pt1, pt2, cellInfoList[i].isect[j].x, closestPoint);
                sum += vtkMath::Distance2BetweenPoints(pt1, closestPoint);
            }
            newPoint[0] = sqrt(sum/(float)nDups);
        }
        if (newPoint[0] < oldX)
        {
            requiresSort = true;
        }
        oldX = newPoint[0];
        if (pointData)
        {
            if (nDups == 1)
            {
                sum = 0;
                ds->GetCellPoints(cellInfoList[i].currCell[0], ptIds);
                int numCellPts = ptIds->GetNumberOfIds();
                for (j = 0; j < numCellPts; j++)
                    sum += scalars->GetTuple1(ptIds->GetId(j));
                if (numCellPts > 0)
                   sum /= (float) numCellPts;
                newPoint[1] = sum;
            }
            else 
            {
                sum = 0;
                set<int> uniquePts;
                for (j = 0; j < nDups; j++)
                { 
                    ds->GetCellPoints(cellInfoList[i].currCell[j], ptIds); 
                    int numCellPts = ptIds->GetNumberOfIds();
                    for (k = 0; k < numCellPts; k++)
                    { 
                        if (uniquePts.count(ptIds->GetId(k)) == 0)
                        { 
                            uniquePts.insert(ptIds->GetId(k)); 
                            sum += scalars->GetTuple1(ptIds->GetId(j));
                        } 
                    } 
                } 
                if (uniquePts.size() > 0)
                   sum /= (float) uniquePts.size();
                newPoint[1] = sum;
            }
        }
        else 
        {
            if (nDups == 1)
            {
                newPoint[1] = scalars->GetTuple1(cellInfoList[i].currCell[0]);
            }
            else
            {
                newPoint[1] = 0.;
                for (j = 0; j < nDups; j++)
                    newPoint[1] += scalars->GetTuple1(cellInfoList[i].currCell[j]);
                newPoint[1] /= (float) cellInfoList[i].currCell.size();
            }
        }
        outPts->InsertNextPoint(newPoint);
    }
    ptIds->Delete();

    vtkPoints *sortedPts;
    if (requiresSort)
    {
        sortedPts = vtkPoints::New();
        map <float, int> sortedIds;
        float x;
        for (i = 0; i < outPts->GetNumberOfPoints(); i++)
        {
            x = outPts->GetPoint(i)[0];
            sortedIds.insert(std::map < float, int> ::value_type(x, i));
        }
        std::map <float, int>::iterator it;
        for (it = sortedIds.begin(); it != sortedIds.end(); it++)
        {
            sortedPts->InsertNextPoint(outPts->GetPoint((*it).second));
        }
        polys->SetPoints(sortedPts);
        sortedPts->Delete();
    }
    else
    {
        sortedPts = outPts;
    }

    vtkCellArray *verts = vtkCellArray::New();
    polys->SetVerts(verts);
    verts->Delete();

    verts->InsertNextCell(sortedPts->GetNumberOfPoints());
    for (i = 0; i < sortedPts->GetNumberOfPoints(); i++)
    {
        verts->InsertCellPoint(i);
    }

    return polys;
}


// ****************************************************************************
//  ClosestPointOnLine
//
//  Purpose:  Determines the point on a line closest to a given point not on
//            the line.
//
//  Arguments:
//    pt1     The origin of the line.
//    pt2     The endpoint of the line.
//    cpt     The point not-on-the-line.
//    npt     The closest point.
//
//  Programmer: Kathleen bonnell
//  Creation:   October 20, 2004 
//
// ****************************************************************************

void 
ClosestPointOnLine(const float *pt1, const float *pt2, const float *cpt, 
                   float *npt)
{
    // Take the points and convert them to vectors.
    avtVector p1(pt1);
    avtVector p2(pt2);
    avtVector cp(cpt);

    // Create direction vectors of the line, and from the point to the lineorigin 
    avtVector v(p2-p1);
    avtVector R(cp-p1);

    double t = (R*v)/(v*v);

    avtVector newPt(p1 + v*t);
    
    npt[0] = newPt.x;
    npt[1] = newPt.y;
    npt[2] = newPt.z;
}

