// ************************************************************************* //
//  File: avtIsovolumeFilter.C
// ************************************************************************* //

#include <avtIsovolumeFilter.h>

#include <float.h>

#include <vtkVisItClipper.h>
#include <vtkDataSet.h>
#include <vtkDataArray.h>
#include <vtkPointData.h>
#include <vtkCellData.h>
#include <vtkPolyData.h>
#include <vtkUnstructuredGrid.h>
#include <vtkCellDataToPointData.h>

#include <avtIntervalTree.h>
#include <avtMetaData.h>

#include <DebugStream.h>
#include <VisItException.h>


// ****************************************************************************
//  Method: avtIsovolumeFilter constructor
//
//  Programmer: meredith -- generated by xml2info
//  Creation:   Fri Jan 30 14:50:21 PST 2004
//
// ****************************************************************************

avtIsovolumeFilter::avtIsovolumeFilter()
{
}


// ****************************************************************************
//  Method: avtIsovolumeFilter destructor
//
//  Programmer: meredith -- generated by xml2info
//  Creation:   Fri Jan 30 14:50:21 PST 2004
//
//  Modifications:
//
// ****************************************************************************

avtIsovolumeFilter::~avtIsovolumeFilter()
{
}


// ****************************************************************************
//  Method:  avtIsovolumeFilter::Create
//
//  Programmer: meredith -- generated by xml2info
//  Creation:   Fri Jan 30 14:50:21 PST 2004
//
// ****************************************************************************

avtFilter *
avtIsovolumeFilter::Create()
{
    return new avtIsovolumeFilter();
}


// ****************************************************************************
//  Method:      avtIsovolumeFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: Jeremy Meredith
//  Creation:   January 30, 2004
//
// ****************************************************************************

void
avtIsovolumeFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const IsovolumeAttributes*)a;

    // We need to specify that we want a secondary variable as soon as 
    // possible.
    if (strcmp(atts.GetVariable().c_str(), "default") != 0)
    {
        SetActiveVariable(atts.GetVariable().c_str());
    }
}


// ****************************************************************************
//  Method: avtIsovolumeFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtIsovolumeFilter with the given
//      parameters would result in an equivalent avtIsovolumeFilter.
//
//  Programmer: meredith -- generated by xml2info
//  Creation:   Fri Jan 30 14:50:21 PST 2004
//
// ****************************************************************************

bool
avtIsovolumeFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(IsovolumeAttributes*)a);
}


// ****************************************************************************
//  Method:  avtIsovolumeFilter::ExecuteSingleClip
//
//  Purpose:
//    Clip against either a low or high scalar.  Convert cell to point
//    data as necessary.
//
//  Arguments:
//    in_ds      the data set to clip
//    val        the value to clip against
//    flip       false if val is a max, true if val is a min
//
//  Programmer:  Jeremy Meredith
//  Creation:    May  6, 2004
//
// ****************************************************************************
vtkDataSet *
avtIsovolumeFilter::ExecuteSingleClip(vtkDataSet *in_ds, float val, bool flip)
{
    vtkVisItClipper *clipper = vtkVisItClipper::New();
    clipper->SetInsideOut(flip);

    //
    // Get the scalar array we'll use for clipping; it must be nodal
    //
    vtkCellDataToPointData *cd2pd = NULL;
    if (in_ds->GetPointData()->GetScalars())
    {
        vtkDataArray *s = in_ds->GetPointData()->GetScalars();
        clipper->SetClipScalars((float*)(s->GetVoidPointer(0)), val);
    }
    else if (in_ds->GetCellData()->GetScalars())
    {
        //
        // Okay, our active variable was cell-centered.  Recenter it....
        //
        vtkDataSet *temp_ds = (vtkDataSet *) in_ds->NewInstance();
        temp_ds->CopyStructure(in_ds);
        temp_ds->GetCellData()->SetScalars(in_ds->GetCellData()->GetScalars());

        cd2pd = vtkCellDataToPointData::New();
        cd2pd->SetInput(temp_ds);
        cd2pd->Update();

        vtkDataSet *temporary = cd2pd->GetOutput();

        // Now tell the clipper about it....
        vtkDataArray *s = temporary->GetPointData()->GetScalars();
        clipper->SetClipScalars((float*)(s->GetVoidPointer(0)), val);

        // Wait until after the clipping is done to delete 'cd2pd' (which
        // will take 'temporary' with it)
        temp_ds->Delete();
    }
    else
    {
        debug1 << "Could not find any data for isovolume operation\n";
        EXCEPTION1(VisItException, "No variable was present for the Isovolume");
    }

    //
    // Do the clipping!
    //
    vtkDataSet *out_ds;
    clipper->SetInput(in_ds);
    out_ds = clipper->GetOutput();
    out_ds->Update();
    ManageMemory(out_ds);
    clipper->Delete();

    //
    // Free the temporary filter used to convert to point data
    //
    if (cd2pd)
        cd2pd->Delete();

    return out_ds;
}


// ****************************************************************************
//  Method: avtIsovolumeFilter::ExecuteData
//
//  Purpose:
//      Sends the specified input and output through the Isovolume filter.
//
//  Arguments:
//      in_ds      The input dataset.
//      <unused>   The domain number.
//      <unused>   The label.
//
//  Returns:       The output dataset.
//
//  Programmer: Jeremy Meredith
//  Creation:   January 30, 2004
//
//  Modifications:
//    Jeremy Meredith, Mon Feb  2 13:13:05 PST 2004
//    Fixed memory leak.
//
//    Jeremy Meredith, Wed May  5 14:56:35 PDT 2004
//    Removed the "3D" from the end of vtkVisItClipper because I made it
//    fully support 2D as well.  I also changed it to only take a single
//    cutoff so the math is more robust, and that required making a "min"
//    pass as well as a "max" pass, so that had to change in this function.
//
//    Jeremy Meredith, Thu May  6 11:38:21 PDT 2004
//    I neglected to create a new scalar array in the event that we do both
//    a min and max pass, so I split most of this routine into a new function
//    and called the new one (ExecuteSingleClip) twice.
//
//    Hank Childs, Wed Oct 20 16:16:15 PDT 2004
//    Basic optimizations -- only do min pass and max pass when the data
//    requires it.  Also pass the dataset through if it is wholly contained
//    within the range and return a NULL dataset if it is outside the range.
//
//    Hank Childs, Wed Nov 17 11:43:53 PST 2004
//    At the end, we try to convert poly-data input back to poly-data output.
//    There was an assumption that after processing the data, it would be
//    in unstructured grid form, which is not true.  Add a check here. ['5640]
//
// ****************************************************************************

vtkDataSet *
avtIsovolumeFilter::ExecuteData(vtkDataSet *in_ds, int, std::string)
{
    //
    // Start off by calculating the range of the dataset.
    //
    float *vals = NULL;
    int nvals = 0;
    if (in_ds->GetPointData()->GetScalars() != NULL)
    {
        vals = (float *) in_ds->GetPointData()->GetScalars()->GetVoidPointer(0);
        nvals = in_ds->GetNumberOfPoints();
    }
    else if (in_ds->GetCellData()->GetScalars() != NULL)
    {
        vals = (float *) in_ds->GetCellData()->GetScalars()->GetVoidPointer(0);
        nvals = in_ds->GetNumberOfCells();
    }

    if (vals == NULL)
        return in_ds;

    float min = +FLT_MAX;
    float max = -FLT_MAX;
    for (int i = 0 ; i < nvals ; i++)
    {
        min = (min < vals[i] ? min : vals[i]);
        max = (max > vals[i] ? max : vals[i]);
    }

    //
    // Check to see if our range is below the min or above the max.  If so,
    // we will have an empty intersection.
    //
    if (max < atts.GetLbound() || min > atts.GetUbound())
    {
        return NULL;
    }

    //
    // Determine if we need to do the min clip or max clip.  Because of the
    // above logic, we can assume that the dataset's max is bigger than
    // the isovolume's lbound and the min is less than the ubound.
    //
    bool doMinClip = false;
    if ((atts.GetLbound() > -1e37) && (min < atts.GetLbound()))
        doMinClip = true;
    bool doMaxClip = false;
    if ((atts.GetUbound() < 1e37) && (max > atts.GetUbound()))
        doMaxClip = true;

    //
    // Do the clipping!
    //
    vtkDataSet *out_ds = in_ds;
    if (doMinClip)
        out_ds = ExecuteSingleClip(out_ds, atts.GetLbound(), true);
    if (doMaxClip)
        out_ds = ExecuteSingleClip(out_ds, atts.GetUbound(), false);

    //
    // Make sure there's something there
    //
    if (out_ds->GetNumberOfCells() <= 0)
    {
        out_ds = NULL;
    }

    //
    // If we had poly data input, we want poly data output.  The VTK filter
    // only returns unstructured grids, so convert that now.  Note: we don't
    // necessarily have a ugrid, since it might be that we didn't process the
    // dataset.
    //
    bool shouldDelete = false;
    if (in_ds->GetDataObjectType() == VTK_POLY_DATA && out_ds != NULL &&
        out_ds->GetDataObjectType() == VTK_UNSTRUCTURED_GRID)
    {
        vtkUnstructuredGrid *ugrid = (vtkUnstructuredGrid *) out_ds;
        vtkPolyData *out_pd = vtkPolyData::New();
        out_pd->SetPoints(ugrid->GetPoints());
        out_pd->GetPointData()->ShallowCopy(ugrid->GetPointData());
        out_pd->GetCellData()->ShallowCopy(ugrid->GetCellData());
        int ncells = ugrid->GetNumberOfCells();
        out_pd->Allocate(ncells);
        for (int i = 0 ; i < ncells ; i++)
        {
            int celltype = ugrid->GetCellType(i);
            vtkIdType *pts;
            int npts;
            ugrid->GetCellPoints(i, npts, pts);
            out_pd->InsertNextCell(celltype, npts, pts);
        }
        out_ds = out_pd;
        shouldDelete = true;
    }

    ManageMemory(out_ds);
    if (shouldDelete)
        out_ds->Delete();

    return out_ds;
}


// ****************************************************************************
//  Method: avtIsovolumeFilter::RefashionDataObjectInfo
//
//  Purpose:
//      Indicates the zones no longer correspond to the original problem.
//
//  Programmer: Jeremy Meredith
//  Creation:   February 16, 2004
//
// ****************************************************************************

void
avtIsovolumeFilter::RefashionDataObjectInfo(void)
{
    GetOutput()->GetInfo().GetValidity().InvalidateZones();
}


// ****************************************************************************
//  Method: avtIsovolumeFilter::PerformRestriction
//
//  Purpose:
//      Tell the database that we will need ghost zones.
//
//  Programmer: Hank Childs
//  Creation:   August 11, 2004
//
//  Modifications:
//
//    Hank Childs, Wed Oct 20 17:04:52 PDT 2004
//    Use interval trees to only read in the domains we need.
//
// ****************************************************************************

avtPipelineSpecification_p
avtIsovolumeFilter::PerformRestriction(avtPipelineSpecification_p in_spec)
{
    avtPipelineSpecification_p spec = new avtPipelineSpecification(in_spec);

    const char *varname = NULL;
    if (atts.GetVariable() != "default")
        varname = atts.GetVariable().c_str();
    else
        varname = in_spec->GetDataSpecification()->GetVariable();

    //
    // We will need the ghost zones so that we can interpolate along domain
    // boundaries and get no cracks in our isosurface.
    //
    avtDataAttributes &in_atts = GetInput()->GetInfo().GetAttributes();
    bool skipGhost = false;
    if (in_atts.ValidVariable(varname) &&
        in_atts.GetCentering(varname) == AVT_NODECENT)
        skipGhost = true;
    if (!skipGhost)
        spec->GetDataSpecification()->SetDesiredGhostDataType(GHOST_ZONE_DATA);

    string iso_var = atts.GetVariable();;
    if (iso_var == "default")
        iso_var = in_spec->GetDataSpecification()->GetVariable();

    avtIntervalTree *it = GetMetaData()->GetDataExtents(iso_var.c_str());
    if (it != NULL)
    {
        float min = atts.GetLbound();
        float max = atts.GetUbound();
        vector<int> dl;
        it->GetDomainsListFromRange(&min, &max, dl);
        spec->GetDataSpecification()->GetRestriction()->RestrictDomains(dl);
    }

    return spec;
}


