// ************************************************************************* //
//  File: avtIndexSelectFilter.C
// ************************************************************************* //

#include <avtIndexSelectFilter.h>

#include <vtkCell.h>
#include <vtkCellData.h>
#include <vtkDataSetRemoveGhostCells.h>
#include <vtkExtractGrid.h>
#include <vtkExtractRectilinearGrid.h>
#include <vtkIntArray.h>
#include <vtkPointData.h>
#include <vtkPolyData.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnsignedIntArray.h>
#include <vtkUnstructuredGrid.h>
#include <vtkVisItUtility.h>

#include <avtCallback.h>
#include <avtSILRestrictionTraverser.h>
#include <avtLogicalSelection.h>

#include <DebugStream.h>
#include <ImproperUseException.h>
#include <snprintf.h>


// ****************************************************************************
//  Method: avtIndexSelectFilter constructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
//  Modifications:
//
//    Mark C. Miller, Tue Sep 28 19:57:42 PDT 2004
//    Added data selection id
//
// ****************************************************************************

avtIndexSelectFilter::avtIndexSelectFilter()
{
    curvilinearFilter = vtkExtractGrid::New();
    rectilinearFilter = vtkExtractRectilinearGrid::New();
    haveIssuedWarning = false;
    selID             = -1;
}


// ****************************************************************************
//  Method: avtIndexSelectFilter destructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
//  Modifications:
//
// ****************************************************************************

avtIndexSelectFilter::~avtIndexSelectFilter()
{
    if (curvilinearFilter != NULL)
    {
        curvilinearFilter->Delete();
        curvilinearFilter = NULL;
    }
    if (rectilinearFilter != NULL)
    {
        rectilinearFilter->Delete();
        rectilinearFilter = NULL;
    }
}


// ****************************************************************************
//  Method:  avtIndexSelectFilter::Create
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
// ****************************************************************************

avtFilter *
avtIndexSelectFilter::Create()
{
    return new avtIndexSelectFilter();
}


// ****************************************************************************
//  Method:      avtIndexSelectFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
// ****************************************************************************

void
avtIndexSelectFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const IndexSelectAttributes*)a;
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::PrepareFilters
//
//  Purpose:
//      Prepares the curvilinear and rectilinear filters for the current
//      dataset.  This can take into account block indices.
//
//  Programmer: Hank Childs
//  Creation:   June 25, 2002
//
//  Modifications:
//    Kathleen Bonnell, Wed Sep  8 09:36:30 PDT 2004
//    Always Set IncludeBoundary to true for filters, so they can handle
//    modulo prolbems (eg sample rate of 3, but dimension is 10).
//
// ****************************************************************************

void
avtIndexSelectFilter::PrepareFilters(int groupIndices[3])
{
    //
    // Only adjust the base index if we are index selecting by group.
    //
    int bi[3];
    if (atts.GetWhichData() == IndexSelectAttributes::OneGroup)
    {
        bi[0] = (groupIndices[0]-1 > 0 ? groupIndices[0]-1 : 0);
        bi[1] = (groupIndices[1]-1 > 0 ? groupIndices[1]-1 : 0);
        bi[2] = (groupIndices[2]-1 > 0 ? groupIndices[2]-1 : 0);
    }
    else
    {
        bi[0] = 0;
        bi[1] = 0;
        bi[2] = 0;
    }

    int voi[6];
    voi[0] = (atts.GetXMin() - bi[0] < 0 ? 0 : atts.GetXMin() - bi[0]);
    voi[1] = (atts.GetXMax() < 0 ? 1000000 : atts.GetXMax() - bi[0]);
    if (atts.GetDim() == IndexSelectAttributes::TwoD ||
        atts.GetDim() == IndexSelectAttributes::ThreeD)
    {
        voi[2] = (atts.GetYMin() - bi[1] < 0 ? 0 : atts.GetYMin() - bi[1]);
        voi[3] = (atts.GetYMax() < 0 ? 1000000 : atts.GetYMax() - bi[1]);
    }
    else
    {
        voi[2] = 0;
        voi[3] = 1000000;
    }
    if (atts.GetDim() == IndexSelectAttributes::ThreeD)
    {
        voi[4] = (atts.GetZMin() - bi[2] < 0 ? 0 : atts.GetZMin() - bi[2]);
        voi[5] = (atts.GetZMax() < 0 ? 1000000 : atts.GetZMax() - bi[2]);
    }
    else
    {
        voi[4] = 0;
        voi[5] = 1000000;
    }
    curvilinearFilter->SetVOI(voi);
    rectilinearFilter->SetVOI(voi);
    int sampleRate[3];
    sampleRate[0] = atts.GetXIncr();
    sampleRate[1] = atts.GetYIncr();
    sampleRate[2] = atts.GetZIncr();
    curvilinearFilter->SetSampleRate(sampleRate);
    rectilinearFilter->SetSampleRate(sampleRate);

    curvilinearFilter->SetIncludeBoundary(1);
    rectilinearFilter->SetIncludeBoundary(1);
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtIndexSelectFilter with the given
//      parameters would result in an equivalent avtIndexSelectFilter.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
// ****************************************************************************

bool
avtIndexSelectFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(IndexSelectAttributes*)a);
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::ExecuteData
//
//  Purpose:
//      Sends the specified input and output through the IndexSelect filter.
//
//  Arguments:
//      in_ds      The input dataset.
//      <unused>   The domain number.
//      <unused>   The label.
//
//  Returns:       The output dataset.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
//  Modifications:
//
//    Hank Childs, Wed Jun 19 09:47:01 PDT 2002
//    Fix stupid crash when applying to unstructured grids.
//
//    Hank Childs, Tue Jun 25 18:22:35 PDT 2002
//    Use group indices when calculating indices.
//
//    Hank Childs, Wed Oct  2 10:13:26 PDT 2002
//    Added support for meshes that have already been broken up.
//
//    Hank Childs, Mon Dec  9 08:53:15 PST 2002
//    Account for potentially changing format of avtOriginalCellNumbers.
//
//    Kathleen Bonnell, Fri Dec 13 16:41:12 PST 2002   
//    Use NewInstance instead of MakeObject in order to match vtk's new api. 
//    
//    Jeremy Meredith, Fri Jan 30 17:45:47 PST 2004
//    Added code to preserve dataset type if the input was polydata.
//
//    Hank Childs, Fri Aug 27 15:25:22 PDT 2004
//    Rename ghost data array.
//
//    Mark C. Miller, Tue Sep 28 19:57:42 PDT 2004
//    Added code to bypass the operator if the selection is applied by a
//    plugin
//
//    Kathleen Bonnell, Fri Feb 18 09:41:16 PST 2005 
//    Account for the fact the vtkExtractGrid and vtkRectilinearExtractGrid
//    may return 'empty' datasets, so we want to return a NULL dataset. 
//
// ****************************************************************************

vtkDataSet *
avtIndexSelectFilter::ExecuteData(vtkDataSet *in_ds, int, std::string)
{
    vtkDataSet *out_ds = NULL;

    //
    // If the selection this filter exists to create has already been handled,
    // then we can skip execution
    //
    if (GetInput()->GetInfo().GetAttributes().GetSelectionApplied(selID)) 
    {
        debug1 << "Bypassing IndexSelect operator because database plugin "
                  "cliams to have applied the selection already" << endl;
        successfullyExecuted = true;
        return in_ds;
    }
    else if (GetInput()->GetInfo().GetValidity().GetZonesPreserved())
    {
        //
        // We have the normal case -- a structured mesh that we are going to
        // index select.
        //
        vtkDataSet *ds = in_ds;

        //
        // All of our indices are incorrect if we leave the ghost zones in -- 
        // we can also have some weird phenomenon where boundaries are missing
        // between blocks.
        //
        vtkDataSetRemoveGhostCells *removeGhostCells = NULL;
        if (in_ds->GetCellData()->GetArray("avtGhostZones"))
        {
            removeGhostCells  = vtkDataSetRemoveGhostCells::New();
            removeGhostCells->SetInput(ds);
    
            //
            // There is something buggy about the extents when this filter is 
            // used for repeated executions.  Just force the execution now.
            //
            ds = removeGhostCells->GetOutput();
            ds->Update();
            ds->SetSource(NULL);
        }
    
        //
        // The indices should reflect the "base_index"'s, so dummy one up if
        // we don't have one.
        //
        vtkDataArray *arr = in_ds->GetFieldData()->GetArray("base_index");
        int ind[3] = { 0, 0, 0 };
        if (arr != NULL)
        {
            vtkIntArray *ar2 = (vtkIntArray *) arr;
            ind[0] = ar2->GetValue(0);
            ind[1] = ar2->GetValue(1);
            ind[2] = ar2->GetValue(2);
        }
        PrepareFilters(ind);
    
        vtkDataSet *rv = NULL;
        int dstype = ds->GetDataObjectType();
        if (dstype == VTK_STRUCTURED_GRID)
        {
            curvilinearFilter->SetInput((vtkStructuredGrid *) ds);
            curvilinearFilter->Update();
            rv = curvilinearFilter->GetOutput();
        }
        else if (dstype == VTK_RECTILINEAR_GRID)
        {
            rectilinearFilter->SetInput((vtkRectilinearGrid *) ds);
            rectilinearFilter->Update();
            rv = rectilinearFilter->GetOutput();
        }
        else
        {
            if (!haveIssuedWarning)
            {
                avtCallback::IssueWarning("The index select operator was "
                            "applied to a non-structured mesh.  It is not "
                            "being applied.");
                haveIssuedWarning = true;
            }
            return in_ds;
        }

        rv->Update();

        if (removeGhostCells != NULL)
        {
            removeGhostCells->Delete();
        }

        if (rv->GetNumberOfPoints() > 0 && rv->GetNumberOfCells() > 0)
        {
            out_ds = (vtkDataSet *) rv->NewInstance();
            out_ds->ShallowCopy(rv);
            ManageMemory(out_ds);
            out_ds->Delete();
        }
    }
    else
    {
        //
        // The dataset has been changed before it got here -- most likely it
        // was material selected.  We should have passed enough clues
        // downstream to figure out what happened.
        //
        vtkUnsignedIntArray *origZones = (vtkUnsignedIntArray *)
                                         in_ds->GetCellData()->
                                         GetArray("avtOriginalCellNumbers");
        if (origZones == NULL)
        {
            if (!haveIssuedWarning)
            {
                avtCallback::IssueWarning("An internal error occurred and the "
                                          "index select operator was not "
                                          "applied.");
                haveIssuedWarning = true;
            }
            return in_ds;
        }

        vtkUnsignedIntArray *dims = (vtkUnsignedIntArray *)
                                            in_ds->GetFieldData()->GetArray(
                                            "avtOriginalStructuredDimensions");
        if (dims == NULL)
        {
            if (!haveIssuedWarning)
            {
                avtCallback::IssueWarning("An internal error occurred and the "
                                          "index select operator was not "
                                          "applied.");
                haveIssuedWarning = true;
            }
            return in_ds;
        }
        int d[3];
        d[0] = dims->GetValue(0);
        d[1] = dims->GetValue(1);
        d[2] = dims->GetValue(2);

        int base[3] = { 0, 0, 0 };
        if (atts.GetWhichData() == IndexSelectAttributes::OneGroup)
        {
            vtkDataArray *arr = in_ds->GetFieldData()->GetArray("base_index");
            if (arr != NULL)
            {
                vtkIntArray *ar2 = (vtkIntArray *) arr;
                base[0] += ar2->GetValue(0);
                base[1] += ar2->GetValue(1);
                base[2] += ar2->GetValue(2);
            }
        }
        vtkUnsignedIntArray *real_dims = (vtkUnsignedIntArray *)
                                         in_ds->GetCellData()->
                                         GetArray("avtRealDims");
        if (real_dims != NULL)
        {
            base[0] -= real_dims->GetValue(0)-1;
            base[1] -= real_dims->GetValue(1)-1;
            base[2] -= real_dims->GetValue(2)-1;
        }

        //
        // We should have everything lined up now -- we know what the original
        // indexing of the structured mesh was and what the base index is.
        //
        vtkUnstructuredGrid *out_ug = vtkUnstructuredGrid::New();
        vtkPoints *p1 = vtkVisItUtility::GetPoints(in_ds);
        out_ug->SetPoints(p1);
        p1->Delete();
        out_ug->GetPointData()->PassData(in_ds->GetPointData());
        vtkCellData *out_cd = out_ug->GetCellData();
        vtkCellData *in_cd  = in_ds->GetCellData();
        out_cd->CopyAllocate(in_cd);
        int ncells = in_ds->GetNumberOfCells();
        out_ug->Allocate(ncells);

        int out_cell = 0;
        int xmin = atts.GetXMin();
        int xmax = atts.GetXMax();
        xmax = (xmax < 0 ? 10000000 : xmax);
        int ymin = atts.GetYMin();
        int ymax = atts.GetYMax();
        ymax = (ymax < 0 ? 10000000 : ymax);
        int zmin = atts.GetZMin();
        int zmax = atts.GetZMax();
        zmax = (zmax < 0 ? 10000000 : zmax);
        int ncomps = origZones->GetNumberOfComponents();
        int comp = ncomps-1;
        for (int i = 0 ; i < ncells ; i++)
        {
            int cell_id = (int) origZones->GetComponent(i, comp);
            int x = cell_id % (d[0]-1);
            int y = (cell_id / (d[0]-1)) % (d[1]-1);
            int z = cell_id / ((d[0]-1)*(d[1]-1));
            x += base[0];
            y += base[1];
            z += base[2];
            if (x < xmin || x >= xmax)
            {
                continue;
            }
            if (atts.GetDim() == IndexSelectAttributes::TwoD ||
                atts.GetDim() == IndexSelectAttributes::ThreeD)
            {
                if (y < ymin || y >= ymax)
                    continue;
            }
            if (atts.GetDim() == IndexSelectAttributes::ThreeD)
            {
                if (z < zmin || z >= zmax)
                    continue;
            }
            out_cd->CopyData(in_cd, i, out_cell++);
            vtkCell *cell = in_ds->GetCell(i);
            vtkIdList *list = cell->GetPointIds();
            out_ug->InsertNextCell(in_ds->GetCellType(i), list);
        }
        out_ds = out_ug;

        if (out_ds->GetNumberOfCells() <= 0)
        {
            out_ds = NULL;
        }

        //
        // If we had poly data input, we want poly data output.  The VTK filter
        // only returns unstructured grids, so convert that now.
        //
        bool shouldDelete = false;
        if (in_ds->GetDataObjectType() == VTK_POLY_DATA && out_ds != NULL)
        {
            vtkUnstructuredGrid *ugrid = (vtkUnstructuredGrid *) out_ds;
            vtkPolyData *out_pd = vtkPolyData::New();
            out_pd->SetPoints(ugrid->GetPoints());
            out_pd->GetPointData()->ShallowCopy(ugrid->GetPointData());
            out_pd->GetCellData()->ShallowCopy(ugrid->GetCellData());
            int ncells = ugrid->GetNumberOfCells();
            out_pd->Allocate(ncells);
            for (int i = 0 ; i < ncells ; i++)
            {
                int celltype = ugrid->GetCellType(i);
                vtkIdType *pts;
                int npts;
                ugrid->GetCellPoints(i, npts, pts);
                out_pd->InsertNextCell(celltype, npts, pts);
            }
            out_ds = out_pd;
            shouldDelete = true;
        }

        ManageMemory(out_ds);
        out_ug->Delete();
        if (shouldDelete)
            out_ds->Delete();
    }

    successfullyExecuted = true;
    return out_ds;
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::PerformRestriction
//
//  Purpose:
//      Restricts the SIL to the domains requested by the user.
//
//  Programmer: Hank Childs
//  Creation:   June 5, 2002
//
//  Modifications:
//
//    Hank Childs, Sun Jun 16 20:50:53 PDT 2002
//    Add support for non 0-origin blocks.
//
//    Hank Childs, Mon Sep 30 17:23:33 PDT 2002
//    Add support for index selecting after a destructive operation.
//
//    Hank Childs, Mon Dec  2 09:59:56 PST 2002
//    Account for changing interface for SIL restriction.
//
//    Hank Childs, Thu Aug 14 07:44:49 PDT 2003
//    Also request the structured indices if we are specifically told to do
//    interface reconstruction.
//
//    Mark C. Miller, Tue Sep 28 19:57:42 PDT 2004
//    Added code to build a data selection
//
//    Kathleen Bonnell, Tue Nov 16 16:13:08 PST 2004 
//    Gracefully handle domainIndex that is out-of-range, and issue warning. 
//    Also, use domainIndex when determining chunk for trav.GetMaterials, when
//    appropriate.
//
// ****************************************************************************

avtPipelineSpecification_p
avtIndexSelectFilter::PerformRestriction(avtPipelineSpecification_p spec)
{
    avtPipelineSpecification_p rv = new avtPipelineSpecification(spec);

    int chunk = 0;
    if (atts.GetWhichData() == IndexSelectAttributes::OneDomain)
    {
        vector<int> domains;
        int blockOrigin 
                      = GetInput()->GetInfo().GetAttributes().GetBlockOrigin();
        rv->GetDataSpecification()->GetSIL().GetDomainList(domains);
        int maxDomain = domains.size() - 1 + blockOrigin;
        domains.clear();
        if (atts.GetDomainIndex() < blockOrigin)
        {
            char warning[128];
            SNPRINTF(warning, 128, "\nThe selected block number (%d) is too "
                            "small for this data, using %d instead.", 
                            atts.GetDomainIndex(), blockOrigin);
            avtCallback::IssueWarning(warning);
            chunk = 0;
        }
        else if (atts.GetDomainIndex() > maxDomain)
        {
            char warning[128];
            SNPRINTF(warning, 128, "\nThe selected block number (%d) is too "
                            "large for this data, using %d instead.", 
                            atts.GetDomainIndex(), maxDomain);
            avtCallback::IssueWarning(warning);
            chunk = maxDomain -blockOrigin; 
        }
        else
        {
            chunk = atts.GetDomainIndex()-blockOrigin;
        }
        domains.push_back(chunk);
        rv->GetDataSpecification()->GetRestriction()->RestrictDomains(domains);
    }
    else if (atts.GetWhichData() == IndexSelectAttributes::OneGroup)
    {
        avtSILRestriction_p silr =rv->GetDataSpecification()->GetRestriction();
        int nc = silr->GetNumCollections();
        for (int i = 0 ; i < nc ; i++)
        {
             avtSILCollection_p coll = silr->GetSILCollection(i);
             if (coll->GetRole() == SIL_BLOCK)
             {
                 //
                 // So we are assuming that the first collection we run across
                 // is the right one -- there really is no other way to do this
                 // when we try to take a single text field from a GUI and
                 // translate into the context of a SIL -- the text field would
                 // have to have SIL qualifiers otherwise.
                 //
                 const vector<int> &els = coll->GetSubsetList();
                 int group = atts.GetGroupIndex();
                 if (group < els.size())  // Sanity check
                 {
                     //
                     // We think we found the group, so set up the new spec.
                     //
                     rv = new avtPipelineSpecification(spec);
                     silr = rv->GetDataSpecification()->GetRestriction();
                     silr->TurnOffAll();
                     silr->TurnOnSet(els[group]);
                     break;
                 }
             }
        }
    }
    if (!GetInput()->GetInfo().GetValidity().GetZonesPreserved())
    {
        rv->GetDataSpecification()->SetNeedStructuredIndices(true);
    }
    else if (rv->GetDataSpecification()->
                                       MustDoMaterialInterfaceReconstruction())
    {
        rv->GetDataSpecification()->SetNeedStructuredIndices(true);
    }
    else
    {
        bool hasMats;
        avtSILRestriction_p silr =rv->GetDataSpecification()->GetRestriction();
        avtSILRestrictionTraverser trav(silr);
        trav.GetMaterials(chunk, hasMats);
        if (hasMats)
        {
            rv->GetDataSpecification()->SetNeedStructuredIndices(true);
        }
    }

    //
    // Indicate this operator's data selection 
    //
    avtLogicalSelection *sel = new avtLogicalSelection;
    switch (atts.GetDim())
    {
        case IndexSelectAttributes::OneD:   sel->SetNDims(1); break;
        case IndexSelectAttributes::TwoD:   sel->SetNDims(2); break;
        case IndexSelectAttributes::ThreeD: sel->SetNDims(3); break;
    }
    int vec[3];
    vec[0] = atts.GetXMin();
    vec[1] = atts.GetYMin();
    vec[2] = atts.GetZMin();
    sel->SetStarts(vec);
    // avtLogicalSelection's stops are inclusive
    // also, we need to deal with using '-1' to mean 'max'
    if (atts.GetXMax() == -1)
        vec[0] = -1;
    else
        vec[0] = atts.GetXMax()-1;
    if (atts.GetYMax() == -1)
        vec[1] = -1;
    else
        vec[1] = atts.GetYMax()-1;
    if (atts.GetZMax() == -1)
        vec[2] = -1;
    else
        vec[2] = atts.GetZMax()-1;
    sel->SetStops(vec);
    vec[0] = atts.GetXIncr();
    vec[1] = atts.GetYIncr();
    vec[2] = atts.GetZIncr();
    sel->SetStrides(vec);
    selID = rv->GetDataSpecification()->AddDataSelection(sel);

    return rv;
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::PreExecute
//
//  Purpose:
//      Called before Execute, which in turn calls ExecuteData.
//
//  Programmer: Hank Childs
//  Creation:   June 29, 2002
//
// ****************************************************************************

void
avtIndexSelectFilter::PreExecute(void)
{
    avtPluginStreamer::PreExecute();
    successfullyExecuted = false;
    if (!GetInput()->GetInfo().GetValidity().GetZonesPreserved())
    {
        if (atts.GetXIncr()!=1 || atts.GetYIncr()!=1 || atts.GetZIncr()!=1)
        {
            avtCallback::IssueWarning("The data was already modified "
                           "before the index select operator was applied."
                           "  It is only possible to do increments of 1.");
        }
    }
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::PostExecute
//
//  Purpose:
//      Called after Execute (which in turn called ExecuteData).
//
//  Programmer: Hank Childs
//  Creation:   June 29, 2002
//
// ****************************************************************************

void
avtIndexSelectFilter::PostExecute(void)
{
    avtPluginStreamer::PostExecute();

    if (successfullyExecuted)
    {
        int numMatches = 0;
        switch (atts.GetDim())
        {
          case IndexSelectAttributes::ThreeD:
            if (atts.GetZMin() == atts.GetZMax())
            {
                numMatches++;
            }
    
            // FALLTHRU
    
          case IndexSelectAttributes::TwoD:
            if (atts.GetYMin() == atts.GetYMax())
            {
                numMatches++;
            }
    
            // FALLTHRU
    
          case IndexSelectAttributes::OneD:
            if (atts.GetXMin() == atts.GetXMax())
            {
                numMatches++;
            }
            break;
    
          default:
            EXCEPTION0(ImproperUseException);
        }

        int indim = GetInput()->GetInfo().GetAttributes().
                                                     GetTopologicalDimension();
        int newdim = indim - numMatches;
        newdim = (newdim < 0 ? 0 : newdim);
        GetOutput()->GetInfo().GetAttributes().SetTopologicalDimension(newdim);
    }
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::ReleaseData
//
//  Purpose:
//      Releases the problem size data associated with this filter.
//
//  Programmer: Hank Childs
//  Creation:   September 10, 2002
//
// ****************************************************************************

void
avtIndexSelectFilter::ReleaseData(void)
{
    avtPluginStreamer::ReleaseData();

    curvilinearFilter->SetInput(NULL);
    curvilinearFilter->SetOutput(NULL);
    rectilinearFilter->SetInput(NULL);
    rectilinearFilter->SetOutput(NULL);
}


