#include "QvisPreferencesWindow.h"

#include <GlobalAttributes.h>
#include <ViewerProxy.h>

#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qgroupbox.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qradiobutton.h>
#include <qspinbox.h>

// ****************************************************************************
// Method: QvisPreferencesWindow::QvisPreferencesWindow
//
// Purpose: 
//   Constructor
//
// Programmer: Eric Brugger
// Creation:   Thu Mar 13 11:15:53 PST 2003
//
// Modifications:
//   Brad Whitlock, Mon Oct 13 16:54:32 PST 2003
//   Added tsFormat and made the window stretch.
//
//   Brad Whitlock, Fri Jan 30 14:16:18 PST 2004
//   Added showSelFiles.
//
//   Brad Whitlock, Fri Apr 9 14:14:09 PST 2004
//   Added allowFileSelectionChangeToggle.
//
// ****************************************************************************

QvisPreferencesWindow::QvisPreferencesWindow(
                       GlobalAttributes *subj,
                       const char *caption,
                       const char *shortName,
                       QvisNotepadArea *notepad)
    : QvisPostableWindowObserver(subj, caption, shortName, notepad,
                       QvisPostableWindowObserver::ApplyButton, true),
    tsFormat()
{
    atts = subj;
    timeStateDisplayMode = 0;
    showSelFiles = true;
    selectedFilesToggle = 0;
    allowFileSelChange = true;
    allowFileSelectionChangeToggle = 0;
}


// ****************************************************************************
// Method: QvisPreferencesWindow::~QvisPreferencesWindow
//
// Purpose: 
//   Destructor
//
// Programmer: Eric Brugger
// Creation:   Thu Mar 13 11:15:53 PST 2003
//
// Modifications:
//   Brad Whitlock, Mon Oct 13 16:54:32 PST 2003
//   Added timeStateDisplayMode.
//
// ****************************************************************************

QvisPreferencesWindow::~QvisPreferencesWindow()
{
    delete timeStateDisplayMode;
}


// ****************************************************************************
// Method: QvisPreferencesWindow::CreateWindowContents
//
// Purpose: 
//   Creates the widgets for the window.
//
// Programmer: Eric Brugger
// Creation:   Thu Mar 13 11:15:53 PST 2003
//
// Modifications:
//   Brad Whitlock, Fri Sep 5 15:45:16 PST 2003
//   I added a toggle for posting windows when showing them.
//
//   Brad Whitlock, Mon Oct 13 16:54:32 PST 2003
//   Added radio buttons for changing the timestate display mode.
//
//   Brad Whitlock, Fri Jan 30 14:16:40 PST 2004
//   I added a toggle button for showing the selected files.
//
//   Brad Whitlock, Fri Apr 9 14:14:22 PST 2004
//   I added a toggle button for highlighting the selected files. I also
//   turned the group box from "Time formatting" to "File panel properties".
//
//   Brad Whitlock, Fri Aug 6 09:21:57 PDT 2004
//   I added toggles for makeDefaultConfirm and automaticallyApplyOperator.
//
// ****************************************************************************

void
QvisPreferencesWindow::CreateWindowContents()
{
    cloneWindowOnFirstRefToggle =
        new QCheckBox("Clone window on first reference",
                      central, "cloneWindowOnFirstRefToggle");
    connect(cloneWindowOnFirstRefToggle, SIGNAL(toggled(bool)),
            this, SLOT(cloneWindowOnFirstRefToggled(bool)));
    topLayout->addWidget(cloneWindowOnFirstRefToggle);

    postWindowsWhenShownToggle =
        new QCheckBox("Post windows when shown",
                      central, "postWindowsWhenShownToggle");
    connect(postWindowsWhenShownToggle, SIGNAL(toggled(bool)),
            this, SLOT(postWindowsWhenShownToggled(bool)));
    topLayout->addWidget(postWindowsWhenShownToggle);

    makeDefaultConfirmToggle =
        new QCheckBox("Prompt before setting default attributes",
                      central, "makeDefaultConfirmToggle");
    connect(makeDefaultConfirmToggle, SIGNAL(toggled(bool)),
            this, SLOT(makeDefaultConfirmToggled(bool)));
    topLayout->addWidget(makeDefaultConfirmToggle);

    automaticallyApplyOperatorToggle =
        new QCheckBox("Prompt before applying new operator",
                      central, "automaticallyApplyOperatorToggle");
    connect(automaticallyApplyOperatorToggle, SIGNAL(toggled(bool)),
            this, SLOT(automaticallyApplyOperatorToggled(bool)));
    topLayout->addWidget(automaticallyApplyOperatorToggle);

    //
    // Create group box for time controls.
    //
    QGroupBox *filePanelControlsGroup = new QGroupBox(central, "filePanelControlsGroup");
    filePanelControlsGroup->setTitle("File panel properties");
    topLayout->addWidget(filePanelControlsGroup, 5);
    QVBoxLayout *innerTopLayout = new QVBoxLayout(filePanelControlsGroup);
    innerTopLayout->setMargin(10);
    innerTopLayout->addSpacing(15);
    innerTopLayout->setSpacing(10);
    QGridLayout *tsModeLayout = new QGridLayout(innerTopLayout, 5, 3);
    tsModeLayout->setSpacing(5);

    //
    // Widgets that let you control the file panel.
    //
    selectedFilesToggle = new QCheckBox("Show selected files",
        filePanelControlsGroup, "selectedFilesToggle");
    selectedFilesToggle->setChecked(showSelFiles);
    connect(selectedFilesToggle, SIGNAL(toggled(bool)),
            this, SLOT(selectedFilesToggled(bool)));
    tsModeLayout->addMultiCellWidget(selectedFilesToggle, 0, 0, 0, 3);

    allowFileSelectionChangeToggle = new QCheckBox(
        "Automatically highlight open file", filePanelControlsGroup,
        "allowFileSelectionChangeToggle");
    allowFileSelectionChangeToggle->setChecked(allowFileSelChange);
    connect(allowFileSelectionChangeToggle, SIGNAL(toggled(bool)),
            this, SLOT(allowFileSelectionChangeToggled(bool)));
    tsModeLayout->addMultiCellWidget(allowFileSelectionChangeToggle,
        1, 1, 0, 3);

    //
    // Create radio button controls to let us change the timestate display mode.
    //
    tsModeLayout->addMultiCellWidget(new QLabel("Display time using:",
        filePanelControlsGroup), 2, 2, 0, 2);
    timeStateDisplayMode = new QButtonGroup(0, "timeStateDisplayMode");
    QRadioButton *rb = new QRadioButton("Cycles", filePanelControlsGroup);
    timeStateDisplayMode->insert(rb);
    tsModeLayout->addWidget(rb, 3, 0);
    rb = new QRadioButton("Times", filePanelControlsGroup);
    timeStateDisplayMode->insert(rb);
    tsModeLayout->addWidget(rb, 3, 1);
    rb = new QRadioButton("Cycles and times", filePanelControlsGroup);
    timeStateDisplayMode->insert(rb);
    tsModeLayout->addWidget(rb, 3, 2);
    timeStateDisplayMode->setButton(int(tsFormat.GetDisplayMode()));
    connect(timeStateDisplayMode, SIGNAL(clicked(int)),
            this, SLOT(handleTimeStateDisplayModeChange(int)));

    //
    // Create widgets that let you set the time format.
    //
    tsModeLayout->addMultiCellWidget(
        new QLabel("Number of significant digits", filePanelControlsGroup),
        4, 4, 0, 1);
    timeStateNDigits = new QSpinBox(1, 16, 1, filePanelControlsGroup, "timeStateNDigits");
    timeStateNDigits->setValue(tsFormat.GetPrecision());
    connect(timeStateNDigits, SIGNAL(valueChanged(int)),
            this, SLOT(timeStateNDigitsChanged(int)));
    tsModeLayout->addWidget(timeStateNDigits, 4, 2);

    topLayout->addStretch(100);
}

// ****************************************************************************
// Method: QvisPreferencesWindow::UpdateWindow
//
// Purpose: 
//   Updates the widgets in the window when the subject changes.
//
// Programmer: Eric Brugger
// Creation:   Thu Mar 13 11:15:53 PST 2003
//
// Modifications:
//   Brad Whitlock, Fri Sep 5 15:46:50 PST 2003
//   I added a toggle button for posting windows when they are shown.
//
//   Brad Whitlock, Fri Jan 30 14:19:11 PST 2004
//   I added a toggle for showing the selected files.
//
//   Brad Whitlock, Fri Apr 9 14:22:34 PST 2004
//   I added a toggle for automatically highlighting the open file.
//
//   Brad Whitlock, Fri Aug 6 09:21:57 PDT 2004
//   I added toggles for makeDefaultConfirm and automaticallyApplyOperator.
//
// ****************************************************************************

void
QvisPreferencesWindow::UpdateWindow(bool doAll)
{
    if (doAll || atts->IsSelected(9))
    {
        //
        // Prompt before making default attributes.
        //
        makeDefaultConfirmToggle->blockSignals(true);
        makeDefaultConfirmToggle->setChecked(
            atts->GetMakeDefaultConfirm());
        makeDefaultConfirmToggle->blockSignals(false);
    }

    if (doAll || atts->IsSelected(10))
    {
        //
        // Clone window on first reference
        //
        cloneWindowOnFirstRefToggle->blockSignals(true);
        cloneWindowOnFirstRefToggle->setChecked(
            atts->GetCloneWindowOnFirstRef());
        cloneWindowOnFirstRefToggle->blockSignals(false);
    }

    if (doAll || atts->IsSelected(13))
    {
        //
        // Automatically add operator.
        //
        automaticallyApplyOperatorToggle->blockSignals(true);
        automaticallyApplyOperatorToggle->setChecked(
            !atts->GetAutomaticallyAddOperator());
        automaticallyApplyOperatorToggle->blockSignals(false);
    }

    if(doAll)
    {
        postWindowsWhenShownToggle->blockSignals(true);
        postWindowsWhenShownToggle->setChecked(postWhenShown);
        postWindowsWhenShownToggle->blockSignals(false);

        selectedFilesToggle->blockSignals(true);
        selectedFilesToggle->setChecked(showSelFiles);
        selectedFilesToggle->blockSignals(false);

        selectedFilesToggle->blockSignals(true);
        selectedFilesToggle->setChecked(showSelFiles);
        selectedFilesToggle->blockSignals(false);

        allowFileSelectionChangeToggle->blockSignals(true);
        allowFileSelectionChangeToggle->setChecked(allowFileSelChange);
        allowFileSelectionChangeToggle->blockSignals(false);
    }
}

// ****************************************************************************
// Method: QvisPreferencesWindow::SetTimeStateDisplayMode
//
// Purpose: 
//   Sets the timeDisplayMode toggle.
//
// Arguments:
//   mode : The new display mode.
//
// Programmer: Brad Whitlock
// Creation:   Mon Oct 13 17:05:17 PST 2003
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::SetTimeStateFormat(const TimeFormat &fmt)
{
    tsFormat = fmt;
    if(timeStateDisplayMode != 0 && timeStateNDigits != 0)
    {
        timeStateDisplayMode->blockSignals(true);
        timeStateDisplayMode->setButton(int(tsFormat.GetDisplayMode()));
        timeStateDisplayMode->blockSignals(false);

        timeStateNDigits->blockSignals(true);
        timeStateNDigits->setValue(tsFormat.GetPrecision());
        timeStateNDigits->blockSignals(false);
    }
}

// ****************************************************************************
// Method: QvisPreferencesWindow::SetShowSelectedFiles
//
// Purpose: 
//   This method sets the toggle for the selected files.
//
// Programmer: Brad Whitlock
// Creation:   Fri Jan 30 14:40:03 PST 2004
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::SetShowSelectedFiles(bool val)
{
    showSelFiles = val;

    if(selectedFilesToggle != 0)
    {
        selectedFilesToggle->blockSignals(true);
        selectedFilesToggle->setChecked(showSelFiles);
        selectedFilesToggle->blockSignals(false);
    }
}

// ****************************************************************************
// Method: QvisPreferencesWindow::SetAllowFileSelectionChange
//
// Purpose: 
//   This method sets the toggle for the automatic file highlighting.
//
// Programmer: Brad Whitlock
// Creation:   Fri Apr 9 14:24:33 PST 2004
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::SetAllowFileSelectionChange(bool val)
{
    allowFileSelChange = val;

    if(allowFileSelectionChangeToggle != 0)
    {
        allowFileSelectionChangeToggle->blockSignals(true);
        allowFileSelectionChangeToggle->setChecked(allowFileSelChange);
        allowFileSelectionChangeToggle->blockSignals(false);
    }
}

// ****************************************************************************
// Method: QvisPreferencesWindow::Apply
//
// Purpose: 
//   Called to apply changes in the subject.
//
// Programmer: Eric Brugger
// Creation:   Thu Mar 13 11:15:53 PST 2003
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::Apply(bool ignore)
{
    if(AutoUpdate() || ignore)
    {
        atts->Notify();
    }
}


//
// Qt Slot functions
//


// ****************************************************************************
// Method: QvisPreferencesWindow::apply
//
// Purpose: 
//   Qt slot function called when apply button is clicked.
//
// Programmer: Eric Brugger
// Creation:   Thu Mar 13 11:15:53 PST 2003
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::apply()
{
    Apply(true);
}


void
QvisPreferencesWindow::cloneWindowOnFirstRefToggled(bool val)
{
    atts->SetCloneWindowOnFirstRef(val);
    Apply();
}

// ****************************************************************************
// Method: QvisPreferencesWindow::postWindowsWhenShownToggled
//
// Purpose: 
//   This is a Qt slot function that sets the flag that tells the GUI whether
//   or not windows should automatically post when they are shown.
//
// Arguments:
//   val : The new post value.
//
// Programmer: Brad Whitlock
// Creation:   Fri Sep 5 15:47:26 PST 2003
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::postWindowsWhenShownToggled(bool val)
{
    postWhenShown = val;
}

// ****************************************************************************
// Method: QvisPreferencesWindow::makeDefaultConfirmToggled
//
// Purpose: 
//   This is a Qt slot function that gets the flag that tells the GUI whether
//   it should prompt users before "make defaults".
//
// Arguments:
//   val : The new value.
//
// Programmer: Brad Whitlock
// Creation:   Fri Aug 6 09:28:57 PDT 2004
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::makeDefaultConfirmToggled(bool val)
{
    atts->SetMakeDefaultConfirm(val);
    Apply();
}

// ****************************************************************************
// Method: QvisPreferencesWindow::automaticallyApplyOperatorToggled
//
// Purpose: 
//   This is a Qt slot function that gets the flag that tells the GUI whether
//   it should prompt users before automatically adding an operator when
//   the user sets the operator attributes when none were applied.
//
// Arguments:
//   val : The new value.
//
// Programmer: Brad Whitlock
// Creation:   Fri Aug 6 09:28:57 PDT 2004
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::automaticallyApplyOperatorToggled(bool val)
{
    atts->SetAutomaticallyAddOperator(!val);
    SetUpdate(false);
    Apply();
}

// ****************************************************************************
// Method: QvisPreferencesWindow::handleTimeStateDisplayModeChange
//
// Purpose: 
//   This is a Qt slot function that is called when the display mode radio
//   buttons are clicked.
//
// Arguments:
//   val : The new timestate display mode.
//
// Programmer: Brad Whitlock
// Creation:   Tue Oct 14 09:57:55 PDT 2003
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::handleTimeStateDisplayModeChange(int val)
{
    tsFormat.SetDisplayMode((TimeFormat::DisplayMode)val);
    emit changeTimeFormat(tsFormat);
}

// ****************************************************************************
// Method: QvisPreferencesWindow::timeStateNDigitsChanged
//
// Purpose: 
//   This is a Qt slot function that is called when the time format spin box
//   is changed.
//
// Arguments:
//   val : The new number of digits in the time format.
//
// Programmer: Brad Whitlock
// Creation:   Tue Oct 14 13:34:11 PST 2003
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::timeStateNDigitsChanged(int val)
{
    tsFormat.SetPrecision(val);
    emit changeTimeFormat(tsFormat);
}

// ****************************************************************************
// Method: QvisPreferencesWindow::selectedFilesToggled
//
// Purpose: 
//   This is a Qt slot function that is called when the selectedFilesToggle
//   is clicked.
//
// Arguments:
//   val : Whether the selected files should show.
//
// Programmer: Brad Whitlock
// Creation:   Fri Jan 30 14:28:44 PST 2004
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::selectedFilesToggled(bool val)
{
    showSelFiles = val;
    emit showSelectedFiles(val);
}

// ****************************************************************************
// Method: QvisPreferencesWindow::allowFileSelectionChangeToggled
//
// Purpose: 
//   This is a Qt slot function that is called when the
//    allowFileSelectionChange toggle is clicked.
//
// Arguments:
//   val : Whether the selected files should show.
//
// Programmer: Brad Whitlock
// Creation:   Fri Jan 30 14:28:44 PST 2004
//
// Modifications:
//   
// ****************************************************************************

void
QvisPreferencesWindow::allowFileSelectionChangeToggled(bool val)
{
    allowFileSelChange = val;
    emit allowFileSelectionChange(val);
}
