// ************************************************************************* //
//                            avtDuneFileFormat.C                            //
// ************************************************************************* //

#include <avtDuneFileFormat.h>

#include <string>
using std::getline;

#include <vtkFloatArray.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnstructuredGrid.h>
#include <vtkGenericCell.h>

#include <avtDatabase.h>
#include <avtDatabaseMetaData.h>
#include <Expression.h>
#include <DebugStream.h>

#include <InvalidFilesException.h>
#include <InvalidVariableException.h>

#if defined(_WIN32) && defined(USING_MSVC6)
//
// Don't use std::getline because it requires the STL version of the input
// stream libraries, which we're not using with MSVC 6.0.
//
ifstream &
getline(ifstream &in, string &buffer)
{
    char tmp[4096];
    in.getline(tmp, 4096);
    buffer = tmp;
    return in;
}

void
READ_STRING(ifstream &in, std::string &buffer)
{
    bool keepReading = true;
    char tmp[4096];
    int index = 0;
    do
    {
        in.get(tmp[index]);
        if(index < 4096 && tmp[index] != ' ')
            ++index;
        else
            keepReading = false;
    } while (keepReading);
    tmp[index] = '\0';
    debug4 << "READ_STRING=" << buffer.c_str() << endl;
    buffer = tmp;
}

#else
//
// Regular case. Use the code as it was originally written.
//
using std::getline;
#define READ_STRING(I, S) (I) >> (S)
#endif

// ****************************************************************************
//  Method: avtDune constructor
//
//  Programmer: dslone -- generated by xml2info
//  Creation:   Thu Mar 11 09:14:32 PDT 2004
//  Modified:
//      Thu Jul 2 09:44:32 PDT 2004 
//          (DMS) - Reworked LOADER, RESTART for new syntax.
//
// ****************************************************************************

avtDuneFileFormat::avtDuneFileFormat(const char *filename)
    : avtMTSDFileFormat(&filename, 1)
{
    // INITIALIZE DATA MEMBERS

    ifile.open(filename);
    if (!ifile)
        EXCEPTION1(InvalidFilesException, filename);

    fname = filename;
    ftype = UNKNOWN;

    string title = "TITLE =";
    string domain = "Domain{";
    string fragments = "Fragments{";
    string materials = "Materials{";
    string nname = "NAME =";
    string sinkedFragments = "SinkedFragments{";
    string buffer;

    while (ftype == UNKNOWN && !ifile.eof()) {
        getline(ifile, buffer);
        if (buffer.find(sinkedFragments) != string::npos) {
            ftype = RESTART;
        }
    }
    ifile.clear();
    ifile.seekg((streampos) 0, ios::beg);

    while (ftype == UNKNOWN) {
        getline(ifile, buffer);
        if (buffer.find(title) != string::npos) {
            ftype = TECPLOT;
        }
        if (buffer.find(domain) != string::npos ||
            buffer.find(fragments) != string::npos ||
            buffer.find(materials) != string::npos) {
            ftype = LOADER;
        }
    }

    if (ftype == UNKNOWN) {
        EXCEPTION1(InvalidFilesException, filename);
    }

    string time = "ZONE T=\"";
    string npart = "\" I =";
    string field = "F =";
    int    np = 0;
    string x = "X(";
    string timeSink = "TimeSink(";
    string spart; 
    string rparen = ")";
    string massD = "MassDensity(";
    string name = "Name( \"";
    string quote = "\"";
    string matname = "Material( \"";
    string tag = "Tag( \"";
    string lastMatname;
    bool   needSpace = true;
    bool   noSpace = false;
    map<string, double> density;

    ntimes = 0;

    switch (ftype) {
        case TECPLOT:
            while (getline(ifile, buffer)) {
                if (buffer.find(time) != string::npos) {
                    cycles.push_back(ntimes);
                    ++ntimes;

                    fpos.push_back(ifile.tellg());

                    // we have a line of the form:
                    // ZONE T=" 0.200000000D-1" I =    557 F = "POINT"

                    spart = string_substr(buffer, time, npart, needSpace);
                    times.push_back(fortranDoubleToCDouble(spart));

                    spart = string_substr(buffer, npart, field, noSpace);      
                    num_particles.push_back(atoi(spart.c_str()));
                }
            }
        break;
        case LOADER:
            cycles.push_back(1);
            times.push_back(0.0);
            ntimes++;
            fpos.push_back(ifile.tellg());

            while (getline(ifile, buffer)) {
                if (buffer.find(x) != string::npos) {
                    ++np;
                }
                if (buffer.find(massD) != string::npos) {
                    spart = string_substr(buffer, massD, rparen, needSpace);
                    if (density.find(lastMatname) != density.end()) {
                        cerr << "already have a " << lastMatname.c_str() << endl;
                    }
                    density[lastMatname] = fortranDoubleToCDouble(spart);
                }
                if (buffer.find(name) != string::npos) {
                    spart = string_substr(buffer, name, quote, noSpace);
                    matnames.push_back(spart);
                    lastMatname = spart;
                }
                if (buffer.find(tag) != string::npos) {
                    spart = string_substr(buffer, tag, quote, noSpace);
                    if (buffer.find(matname) != string::npos) {
                        string name = string_substr(buffer, matname, quote, 
                                                    noSpace);
                        struct matInfoStruct matTemp;
                        matTemp.name = name;
                        if (density.find(matname) == density.end()) {
                            matTemp.massDensity = -1;
                        }
                        else {
                           matTemp.massDensity = density[matname];
                        }
                        matInfo[spart] = matTemp;
                    }
                }
            }

            num_particles.push_back(np);

            if (matInfo.empty()) {
                if (matnames.size() == 0) {
                    string defaultName = "material1";
                    struct matInfoStruct matTemp;
                    matTemp.name = defaultName;
                    matTemp.massDensity = 1.0;
                    matInfo[defaultName] = matTemp;
                }
                else {
                    for (int i = 0; i < matnames.size(); i++) {
                        struct matInfoStruct matTemp;
                        matTemp.name = matnames[i];
                        matTemp.massDensity = density[matnames[i]];
                        matInfo[matnames[i]] = matTemp;
                    }
                }
            }
        break;
        case RESTART:
            do {
                if (buffer.find(timeSink) != string::npos ||
                    (buffer.find(fragments) != string::npos &&
                     buffer.find(sinkedFragments) == string::npos)) {
                    cycles.push_back(ntimes);
                    fpos.push_back(ifile.tellg());
                    if(buffer.find(timeSink) != string::npos) {
                        spart = string_substr(buffer, timeSink, rparen, 
                                              needSpace);
                        times.push_back(fortranDoubleToCDouble(spart));
                    }
                    else {
                        times.push_back(ntimes*1.0);
                    }
                    ++ntimes;
                    if (ntimes > 1) {
                        num_particles.push_back(np);
                    }
                    np = 0;
                }
                if (buffer.find(x) != string::npos) {
                    ++np;
                }
                if (buffer.find(massD) != string::npos) {
                    spart = string_substr(buffer, massD, rparen, needSpace);
                    if (density.find(lastMatname) != density.end()) {
                        debug4 << "already have a " << lastMatname.c_str() << endl;
                    }
                    density[lastMatname] = fortranDoubleToCDouble(spart);
                }
                if (buffer.find(name) != string::npos) {
                    spart = string_substr(buffer, name, quote, noSpace);
                    matnames.push_back(spart);
                    lastMatname = spart;
                }
                if (buffer.find(tag) != string::npos) {
                    spart = string_substr(buffer, tag, quote, noSpace);
                    if (buffer.find(matname) != string::npos) {
                        string name = string_substr(buffer, matname, quote, 
                                                    noSpace);
                        struct matInfoStruct matTemp;
                        matTemp.name = name;
                        if (density.find(matname) == density.end()) {
                            matTemp.massDensity = -1;
                        }
                        else {
                           matTemp.massDensity = density[matname];
                        }
                        matInfo[spart] = matTemp;
                    }
                }
            } while (getline(ifile, buffer));

            num_particles.push_back(np);

            if (matInfo.empty()) {
                if (matnames.size() == 0) {
                    string defaultName = "material1";
                    struct matInfoStruct matTemp;
                    matTemp.name = defaultName;
                    matTemp.massDensity = 1.0;
                    matInfo[defaultName] = matTemp;
                }
                else {
                    for (int i = 0; i < matnames.size(); i++) {
                        struct matInfoStruct matTemp;
                        matTemp.name = matnames[i];
                        matTemp.massDensity = density[matnames[i]];
                        matInfo[matnames[i]] = matTemp;
                    }
                }
            }
            break;
        default:
            EXCEPTION1(InvalidFilesException, filename);
    }

    if (!matInfo.empty()) {
        for (std::map<string, struct matInfoStruct>::iterator cur = matInfo.begin();
             cur != matInfo.end();
             ++cur) {
            if ((*cur).second.massDensity == -1) {
                if (density.find((*cur).second.name) == density.end()) {
                    debug4 << "can't find a density for "
                           << (*cur).second.name.c_str() << endl;
                }
                else {
                    (*cur).second.massDensity = density[(*cur).second.name];
                }
            }
        }
    }

    ifile.clear();

    lastTimestate = -1;
    nparticles = -1;
}


// ****************************************************************************
//  Method: avtEMSTDFileFormat::GetNTimesteps
//
//  Purpose:
//      Tells the rest of the code how many timesteps there are in this file.
//
//  Programmer: dslone -- generated by xml2info
//  Creation:   Thu Mar 11 09:14:32 PDT 2004
//
// ****************************************************************************

int
avtDuneFileFormat::GetNTimesteps(void)
{
    //return YOU_MUST_DECIDE;
    return ntimes;
}


//****************************************************************************
//  Method: avtDuneFileFormat::GetTimes
//  
//  Purpose:
//      Populates an STL vector with the times.
//  
//  Programmer: Dale Slone
//  Creation:   March 9, 2004
//
//****************************************************************************

void
avtDuneFileFormat::GetTimes(vector<double> &out_times)
{
    out_times = times;
}


//****************************************************************************
//  Method: avtDuneFileFormat::GetCycles
//
//  Purpose:
//      Populates an STL vector with the cycles.
//
//  Programmer: Dale Slone
//  Creation:   March 9, 2004
//
//****************************************************************************

void
avtDuneFileFormat::GetCycles(vector<int> &out_cycles)
{
    out_cycles = cycles;
}


// ****************************************************************************
//  Method: avtDuneFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: dslone -- generated by xml2info
//  Creation:   Thu Mar 11 09:14:32 PDT 2004
//  Modified:
//      Thu Jun 24 09:14:32 PDT 2004 
//          (DMS) - Added impulsTtime, impulseVelocity, and totalVelocity vars.
//      Thu Jul 2 09:44:32 PDT 2004 
//          (DMS) - Added angularVelocity var.
//
// ****************************************************************************

void
avtDuneFileFormat::FreeUpResources(void)
{
    species.clear();
    radius.clear();
    impulseTime.clear();
    coordinates.clear();
    velocities.clear();
    impulseVelocities.clear();
    totalVelocities.clear();
    angularVelocities.clear();
    mass.clear();

    lastTimestate = -1;
    nparticles = -1;
}


// ****************************************************************************
//  Method: avtDuneFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: dslone -- generated by xml2info
//  Creation:   Thu Mar 11 09:14:32 PDT 2004
//  Modified:
//      Thu Jun 24 09:14:32 PDT 2004 
//          (DMS) - Added impulseTime, impulseVelocity, and totalVelocity vars.
//      Thu Jul 2 09:44:32 PDT 2004 
//          (DMS) - Added angularVelocity var.
//
// ****************************************************************************

void
avtDuneFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md)
{
    md->SetNumStates(ntimes);
    //
    // CODE TO ADD A MESH
    //
    string meshname = "particles";
    //
    // AVT_RECTILINEAR_MESH, AVT_CURVILINEAR_MESH, AVT_UNSTRUCTURED_MESH,
    // AVT_POINT_MESH, AVT_SURFACE_MESH, AVT_UNKNOWN_MESH
    avtMeshType mt = AVT_UNSTRUCTURED_MESH;
    //
    int nblocks = 1;         // <-- this must be 1 for MTSD
    int block_origin = 0;
    int spatial_dimension = 3;
    int topological_dimension = 0;
    float *extents = NULL;
    //
    // Here's the call that tells the meta-data object that we have a mesh:
    //
    AddMeshToMetaData(md, meshname, mt, extents, nblocks, block_origin,
                      spatial_dimension, topological_dimension);
    //

    //
    // CODE TO ADD A SCALAR VARIABLE
    //
    string mesh_for_this_var = meshname; // ??? -- could be multiple meshes
    vector<string> svarname;
    svarname.push_back("radius");
    svarname.push_back("mass");
    svarname.push_back("impulse_time");
    //
    // AVT_NODECENT, AVT_ZONECENT, AVT_UNKNOWN_CENT
    avtCentering cent = AVT_NODECENT;
    //
    //
    // Here's the call that tells the meta-data object that we have a var:
    //
    int i;
    for (i = 0; i < svarname.size(); i++) {
        AddScalarVarToMetaData(md, svarname[i], mesh_for_this_var, cent);
    }
    //

    //
    // CODE TO ADD A VECTOR VARIABLE
    //
    //string mesh_for_this_var = meshname; // ??? -- could be multiple meshes
    vector<string> vvarname;
    vvarname.push_back("velocity");
    vvarname.push_back("impulse_velocity");
    vvarname.push_back("angular_velocity");
    vvarname.push_back("total_velocity");

    int vector_dim = 3;
    //
    // AVT_NODECENT, AVT_ZONECENT, AVT_UNKNOWN_CENT
    // avtCentering cent = AVT_NODECENT;
    //
    //
    // Here's the call that tells the meta-data object that we have a var:
    //
    for (i = 0; i < vvarname.size(); i++) {
        AddVectorVarToMetaData(md, vvarname[i], mesh_for_this_var, cent,
                               vector_dim);
    }
    //

    //
    // CODE TO ADD A TENSOR VARIABLE
    //
    // string mesh_for_this_var = meshname; // ??? -- could be multiple meshes
    // string varname = ...
    // int tensor_dim = 9;
    //
    // AVT_NODECENT, AVT_ZONECENT, AVT_UNKNOWN_CENT
    // avtCentering cent = AVT_NODECENT;
    //
    //
    // Here's the call that tells the meta-data object that we have a var:
    //
    // AddTensorVarToMetaData(md, varname, mesh_for_this_var, cent,tensor_dim);
    //

    //
    // CODE TO ADD A MATERIAL
    //
    string mesh_for_mat = meshname; // ??? -- could be multiple meshes
    string matname = "species";
    if (matnames.size() == 0) {
        int nmats = 1;
        vector<string>mnames;
        for (int i = 0 ; i < nmats ; i++)
        {
            char str[32];
            sprintf(str, "mat%d", i);
        //     -- or -- 
        //    strcpy(str, "Unknown");
            mnames.push_back(str);
        }
        // 
        // Here's the call that tells the meta-data object that we have a mat:
        //
        AddMaterialToMetaData(md, matname, mesh_for_mat, nmats, mnames);
        //
    }
    else {
        // 
        // Here's the call that tells the meta-data object that we have a mat:
        //
        AddMaterialToMetaData(md, matname, mesh_for_mat, matnames.size(), 
                              matnames);
        //
    }

    Expression momentum_expr;
    momentum_expr.SetName("momentum");
    momentum_expr.SetDefinition("total_velocity*mass");
    momentum_expr.SetType(Expression::VectorMeshVar);
    md->AddExpression(&momentum_expr); 
}


// ****************************************************************************
//  Method: avtDuneFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: dslone -- generated by xml2info
//  Creation:   Thu Mar 11 09:14:32 PDT 2004
//
// ****************************************************************************

vtkDataSet *
avtDuneFileFormat::GetMesh(int timestate, const char *meshname)
{
    //YOU MUST IMPLEMENT THIS

    if (strcmp(meshname, "particles") != 0)
        EXCEPTION1(InvalidVariableException, meshname);
    
    ReadDuneData(timestate);
    
    vtkUnstructuredGrid *dataset = vtkUnstructuredGrid::New();
    dataset->Allocate(nparticles); 
    vtkPoints *vtkpoints = vtkPoints::New();
    vtkpoints->SetNumberOfPoints(nparticles);
    float *pts = (float *) vtkpoints->GetVoidPointer(0);
    
    int i;
    for (i = 0; i < coordinates.size(); i++) {
        pts[i] = coordinates[i];
    }

    for (i = 0; i < nparticles; i++) {
        vtkIdType part[3] = {i, i, i};
        dataset->InsertNextCell(VTK_VERTEX, 3, part);
    }
    
    dataset->SetPoints(vtkpoints);
    vtkpoints->Delete();
    
    return dataset;
}


// ****************************************************************************
//  Method: avtDuneFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: dslone -- generated by xml2info
//  Creation:   Thu Mar 11 09:14:32 PDT 2004
//  Modified:
//      Thu Jun 24 09:14:32 PDT 2004 
//          (DMS) - Added impulseTime.
//
// ****************************************************************************

vtkDataArray *
avtDuneFileFormat::GetVar(int timestate, const char *varname)
{
    //YOU MUST IMPLEMENT THIS

    //
    // If you have a file format where variables don't apply (for example a
    // strictly polygonal format like the STL (Stereo Lithography) format,
    // then uncomment the code below.
    //
    // EXCEPTION0(InvalidVariableException, varname);
    //

    ReadDuneData(timestate);

    //
    // If you do have a scalar variable, here is some code that may be helpful.
    //
    int ntuples = nparticles; // this is the number of entries in the variable.
    vtkFloatArray *rv = vtkFloatArray::New();
    rv->SetNumberOfTuples(ntuples);
    if (strcmp(varname, "radius") == 0) {
        for (int i = 0 ; i < ntuples ; i++)
        {
            rv->SetTuple1(i, radius[i]);  // you must determine value for ith entry.
        }
    }
    else if (strcmp(varname, "mass") == 0) {
        for (int i = 0; i < ntuples; i++) {
            rv->SetTuple1(i, mass[i]);
        }              
    }
    else if (strcmp(varname, "impulse_time") == 0) {
        for (int i = 0; i < ntuples; i++) {
            rv->SetTuple1(i, impulseTime[i]);
        }              
    }
    else {
        EXCEPTION1(InvalidVariableException, varname);
    }

    return rv;

}


// ****************************************************************************
//  Method: avtDuneFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: dslone -- generated by xml2info
//  Creation:   Thu Mar 11 09:14:32 PDT 2004
//  Modified:
//      Thu Jun 24 09:14:32 PDT 2004 
//          (DMS) - Added impulseVelocity, and totalVelocity vars.
//      Thu Jul 2 09:44:32 PDT 2004 
//          (DMS) - Added angularVelocity var.
//
// ****************************************************************************

vtkDataArray *
avtDuneFileFormat::GetVectorVar(int timestate, const char *varname)
{
    //YOU MUST IMPLEMENT THIS

    ReadDuneData(timestate);

    //
    // If you have a file format where variables don't apply (for example a
    // strictly polygonal format like the STL (Stereo Lithography) format,
    // then uncomment the code below.
    //
    // EXCEPTION0(InvalidVariableException, varname);
    //

    //
    // If you do have a vector variable, here is some code that may be helpful.
    //
    int ncomps = 3;        // This is the rank of the vector - typically 2 or 3.
    int ntuples = nparticles; // this is the number of entries in the variable.
    vtkFloatArray *rv = vtkFloatArray::New();
    int ucomps = (ncomps == 2 ? 3 : ncomps);
    rv->SetNumberOfComponents(ucomps);
    rv->SetNumberOfTuples(ntuples);
    float *one_entry = new float[ucomps];

    vector<double> data;
    if (strcmp(varname, "velocity") == 0) {
        data = velocities;
    }
    else if (strcmp(varname, "impulse_velocity") == 0) {
        data = impulseVelocities;
    }
    else if (strcmp(varname, "angular_velocity") == 0) {
        data = angularVelocities;
    }
    else if (strcmp(varname, "total_velocity") == 0) {
        data = totalVelocities;
    }
    else {
        EXCEPTION1(InvalidVariableException, varname);
    }

    for (int i = 0 ; i < ntuples ; i++)
    {
         int j;
         for (j = 0 ; j < ncomps ; j++)
              one_entry[j] = data[i*3 + j];
         for (j = ncomps ; j < ucomps ; j++)
              one_entry[j] = 0.;
         rv->SetTuple(i, one_entry); 
    }

    delete [] one_entry;
    return rv;
}


// ****************************************************************************
//  Method: avtDuneFileFormat::ReadDuneData
//
//  Purpose:
//      Opens the data file and reads the information for the current timestep,
//      filling the appropriate data members required everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//
//  Programmer: dslone -- 
//  Creation:   Thu Mar 4 11:59:59 PDT 2004
//  Modified:
//      Thu Jun 24 09:14:32 PDT 2004 
//          (DMS) - Added impulseTime, impulseVelocity, and totalVelocity vars.
//      Thu Jul 2 09:44:32 PDT 2004 
//          (DMS) - Added angularVelocity var.
//                  Redid LOADER, RESTART for new syntax.
//
// ****************************************************************************

void
avtDuneFileFormat::ReadDuneData(const int timestate) {
    //if (timestate == lastTimestate) {
    //    return;
    //}

    if (ifile.bad()) {
        ifile.open(fname.c_str());
        if (!ifile) {
            EXCEPTION1(InvalidFilesException, fname.c_str());
        }
    }
    else {
        if (ifile.eof()) {
            ifile.clear();
        }
    }

    if (nparticles != num_particles[timestate]) {
        nparticles = num_particles[timestate];
        species.resize(nparticles);
        radius.resize(nparticles);
        impulseTime.resize(nparticles);
        coordinates.resize(3 * nparticles);
        velocities.resize(3 * nparticles);
        impulseVelocities.resize(3 * nparticles);
        angularVelocities.resize(3 * nparticles);
        totalVelocities.resize(3 * nparticles);
        mass.resize(nparticles);
    }

    ifile.seekg(fpos[timestate], ios::beg);

    nparticles = num_particles[timestate];

    int    np = 0;
    size_t bcp;
    size_t ecp = (size_t) 0;
    string dexp = "D";
    string exp = "e";
    string spart;
    string buffer;
    string x = "X(";
    string v = "V(";
    string w = "W(";
    string rparen = ")";
    string comma = ",";
    string r = "R(";
    double pi = 3.141569;
    string tag = "Tag( \"";
    double rho;
    string quote = "\"";
    string time = "Time(";
    string impulse = "Impulse[";
    string vi = "DV(";

    const int fields = 11;
    string    fragment[fields];
    int i,j;

    switch (ftype) {
        case TECPLOT:
            for (i = 0; i < nparticles; i++) {
                for (j = 0; j < fields; j++) {
                    READ_STRING(ifile, fragment[j]);

                    if (fragment[j].length()) {
                         bool more = true;
                         do {
                             string dexp = "D";
                             size_t pos = fragment[j].find(dexp);
                             string exp = "e";

                             if (pos != string::npos) {
                                 fragment[j].replace(pos, 1, exp);
                             }
                             else {
                                 more = false;
                             }
                        } while (more);
                    }
                }

               species[i] = atoi(fragment[0].c_str());
               radius[i] = atof(fragment[1].c_str());
               // Dune not currently outputing impulses
               //impulseTime[i] = atof(fragment[14].c_str());
               impulseTime[i] = 0.0;
               mass[i] = 4./3. * pi * pow(radius[i], 3);
               for (j = 0; j < 3; j++) {
                    coordinates[i*3 + j] = atof(fragment[2 + j].c_str());
                    velocities[i*3 + j] = atof(fragment[5 + j].c_str());
                    angularVelocities[i*3 + j] = atof(fragment[8 + j].c_str());
                    //impulseVelocities[i*3 + j] = atof(fragment[11 + j].c_str());
                    impulseVelocities[i*3 + j] = 0.0;
                }
            }
            break;
        case LOADER:
        case RESTART:
            rho = 1.0;
            if (matInfo.empty()) {
                EXCEPTION1(InvalidVariableException, "matInfo");
            }

            while (np < nparticles) {
                bool needSpace = true;
                bool noSpace = false;
                bool rFound = false;
                bool tagFound = false;

                getline(ifile, buffer);
                if (ifile.eof()) {
                    EXCEPTION1(InvalidFilesException, fname.c_str());
                }

                if (buffer.find(r) != string::npos) {
                    spart = string_substr(buffer, r, rparen, needSpace);
                    radius[np] = fortranDoubleToCDouble(spart);
                    rFound = true;
                }
                if (buffer.find(tag) != string::npos) {
                   spart = string_substr(buffer, tag, quote, noSpace);
                    struct matInfoStruct matTemp;
                    std::map<string, struct matInfoStruct>::iterator cur = matInfo.find(spart);
                    if (cur != matInfo.end()) {
                        rho = (*cur).second.massDensity;
                    }
                    tagFound = true;
                }
                if (rFound && tagFound) {
                   mass[np] = rho * 4.0/3.0 * pi * pow(radius[np], 3.0);
                }
                if (buffer.find(x) != string::npos) {
                    for (i = 0; i < 3; i++) {
                        if (i == 0) {
                            bcp = buffer.find(x) + x.length() + 1;
                        }
                        else {
                            bcp = ecp + 1;
                        }
                        if (i == 2) {
                            ecp = buffer.find(rparen);
                        }
                        else {
                            ecp = buffer.find(comma, bcp);
                        }
                        spart = buffer.substr(bcp, ecp-bcp);
                        coordinates[np*3 + i] = fortranDoubleToCDouble(spart);
                    }
                }
                if (buffer.find(v) != string::npos &&
                    buffer.find(vi) == string::npos) {
                    for (i = 0; i < 3; i++) {
                        if (i == 0) {
                            bcp = buffer.find(v) + v.length() + 1;
                        }
                        else {
                            bcp = ecp + 1;
                        }
                        if (i == 2) {
                            ecp = buffer.find(rparen);
                        }
                        else {
                            ecp = buffer.find(comma, bcp);
                        }
                        spart = buffer.substr(bcp, ecp-bcp);
                        velocities[np*3 + i] = fortranDoubleToCDouble(spart);
                    }
                }
                if (buffer.find(w) != string::npos) {
                    for (i = 0; i < 3; i++) {
                        if (i == 0) {
                            bcp = buffer.find(w) + w.length() + 1;
                        }
                        else {
                            bcp = ecp + 1;
                        }
                        if (i == 2) {
                            ecp = buffer.find(rparen);
                        }
                        else {
                            ecp = buffer.find(comma, bcp);
                        }
                        spart = buffer.substr(bcp, ecp-bcp);
                        angularVelocities[np*3 + i] = fortranDoubleToCDouble(spart);
                    }
                }
                if (buffer.find(impulse) != string::npos) {
                    if (timestate == 0) {
                        getline(ifile, buffer);
                    }
                    if (buffer.find(time) != string::npos) {
                        spart = string_substr(buffer, time, rparen, needSpace);
                        impulseTime[np] = fortranDoubleToCDouble(spart);
                    }
                    if (timestate == 0) {
                        getline(ifile, buffer);
                    }
                    if (buffer.find(vi) != string::npos) {
                        for (i = 0; i < 3; i++) {
                            if (i == 0) {
                                bcp = buffer.find(vi) + vi.length() + 1;
                            }
                            else {
                                bcp = ecp + 1;
                            }
                            if (i == 2) {
                                ecp = buffer.find(rparen);
                            }
                            else {
                                ecp = buffer.find(comma, bcp);
                            }
                            spart = buffer.substr(bcp, ecp-bcp);
                            impulseVelocities[np*3 + i] = fortranDoubleToCDouble(spart);
                        }
                    }
                    np++;
                }
            }
            break;
        default:
            EXCEPTION1(InvalidFilesException, fname.c_str());
    }

    for (i = 0; i < nparticles; i++) {
        for (j = 0; j < 3; j++) {
            totalVelocities[i*3 + j] = velocities[i*3 + j] + 
                                       angularVelocities[i*3 + j] +
                                       impulseVelocities[i*3 + j];
        }
    }

    lastTimestate = timestate;
}


// ****************************************************************************
//  Method: avtDuneFileFormat::string_substr
//
//  Purpose:
//      Return a substring from a the given string buffer. The substring starts 
//      immedieately after the token string pre and terminates before the token
//      string post.
//
//  Arguments:
//      buffer        input source string
//      pre           token string preceding the desired substring
//      post          token string following the desired substring
//      addOneToPre   boolean flag indicating substring follows a blank char
//
//  Programmer: dslone -- 
//  Creation:   Thu Mar 4 11:59:59 PDT 2004
//
// ****************************************************************************

string avtDuneFileFormat::string_substr(const string buffer, 
                                        const string pre, 
                                        const string post, 
                                        const bool   addOneToPre) {
    size_t bcp = buffer.find(pre) + pre.length();
    if (addOneToPre) {
        ++bcp;
    }
    size_t ecp = buffer.find(post, bcp);
    string spart = buffer.substr(bcp, ecp-bcp);

    return spart;
}


// ****************************************************************************
//  Method: avtDuneFileFormat::fortranDoubleToCDouble
//
//  Purpose:
//      convert a Fortran style double var (.123D+01) to a C double (.123e+01).
//      filling the appropriate data members required everywhere through VisIt.
//
//  Arguments:
//      arg        string containing Fortran double var
//
//  Programmer: dslone -- 
//  Creation:   Thu Mar 4 11:59:59 PDT 2004
//
// ****************************************************************************

double avtDuneFileFormat::fortranDoubleToCDouble(const string arg) {
    string dexp = "D";
    string exp = "e";
    string spart = arg;
    size_t pos = spart.find(dexp);

    if (pos != string::npos) {
        spart.replace(pos, 1, exp);
    }

    return atof(spart.c_str());
}

