// ************************************************************************* //
//                        avtCurveConstructorFilter.C                        //
// ************************************************************************* //

#include <avtCurveConstructorFilter.h>
#include <avtDataTree.h>

#include <vtkCellArray.h>
#include <vtkDataSet.h>
#include <vtkPolyData.h>
#include <InvalidDimensionsException.h>
#include <NoInputException.h>

#include <DebugStream.h>

#ifdef PARALLEL
#include <mpi.h>
#include <avtParallel.h>
#include <vtkPolyDataReader.h>
#include <vtkDataSetWriter.h>
#include <vtkCharArray.h>
#endif


#include <map>
using std::map;

// ****************************************************************************
//  Method: avtCurveConstructorFilter constructor
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Sat Apr 20 13:01:58 PST 2002
//
// ****************************************************************************

avtCurveConstructorFilter::avtCurveConstructorFilter()
{
}


// ****************************************************************************
//  Method: avtCurveConstructorFilter destructor
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Sat Apr 20 13:01:58 PST 2002
//
// ****************************************************************************

avtCurveConstructorFilter::~avtCurveConstructorFilter()
{
}


// ****************************************************************************
//  Method: avtCurveConstructorFilter::ExecuteData
//
//  Purpose:
//      Does the actual VTK code to modify the dataset.
//
//  Arguments:
//      inDS      The input dataset.
//      <unused>  The domain number.
//
//  Returns:      The output dataset.
//
//  Programmer: kbonnell -- generated by xml2info
//  Creation:   Sat Apr 20 13:01:58 PST 2002
//
//  Modifications:
//
//    Hank Childs, Tue May 28 11:41:21 PDT 2002
//    Use the variable name as the label for our output data tree.
//
//    Kathleen Bonnell, Fri Jul 12 16:53:11 PDT 2002
//    Removed vtk filters associated with label-creation.  Now handled by
//    the plot.
//
//    Kathleen Bonnell, Tue Dec 23 10:18:06 PST 2003 
//    Added logic to handle point-sorting when necessary. Add vertex cells,
//    so they can be displayed upon user request.
//    
//    Mark C. Miller, Wed Jun  9 21:50:12 PDT 2004
//    Eliminated use of MPI_ANY_TAG and modified to use GetUniqueMessageTags
//
// ****************************************************************************

void avtCurveConstructorFilter::Execute()
{
    avtDataTree_p inTree = GetInputDataTree();

#ifdef PARALLEL
    //
    //  Gather all data onto one processor, so that 
    //  there will be no discontinuities in the curve.
    //
    int myRank, numProcs;

    MPI_Comm_size(MPI_COMM_WORLD, &numProcs);
    MPI_Comm_rank(MPI_COMM_WORLD, &myRank);
    int mpiNdsTag = GetUniqueMessageTag();
    int mpiSizeTag = GetUniqueMessageTag();
    int mpiDataTag = GetUniqueMessageTag();

    if (myRank == 0)
    {
        int i , j;
        for (i = 1; i < numProcs; i++)
        {
           MPI_Status stat;
           MPI_Status stat2;
           int nds = 0, size = 0;
           MPI_Recv(&nds, 1, MPI_INT, MPI_ANY_SOURCE, mpiNdsTag,
                    MPI_COMM_WORLD, &stat);
           for (j = 0; j < nds; j++)
           {
               vtkPolyDataReader *reader = vtkPolyDataReader::New(); 
               reader->ReadFromInputStringOn();
               MPI_Recv(&size, 1, MPI_INT, stat.MPI_SOURCE, mpiSizeTag,
                         MPI_COMM_WORLD, &stat2);
               char *str = new char[size];
               MPI_Recv(str, size, MPI_CHAR, stat.MPI_SOURCE, mpiDataTag,
                        MPI_COMM_WORLD, &stat2);
               vtkCharArray *charArray = vtkCharArray::New();
               charArray->SetArray((char*)str, size, 1);
               reader->SetInputArray(charArray);
               reader->Update();
      
               inTree->Merge(new avtDataTree(reader->GetOutput(), -1));
               delete [] str;
               reader->Delete(); 
               charArray->Delete(); 
           }
        }
    }
    else
    {
        SetOutputDataTree(new avtDataTree());
        char *str = NULL;;
        int i = 0, size = 0, nleaves = 0;
        if (inTree->IsEmpty())
        {
            MPI_Send(&nleaves, 1, MPI_INT, 0, mpiNdsTag, MPI_COMM_WORLD);
            return;
        }

        vtkDataSet **ds = inTree->GetAllLeaves(nleaves);
        MPI_Send(&nleaves, 1, MPI_INT, 0, mpiNdsTag, MPI_COMM_WORLD);
 
        vtkDataSetWriter *writer = vtkDataSetWriter::New();
        writer->WriteToOutputStringOn();
        writer->SetFileTypeToBinary();

        for (i = 0; i < nleaves; i++)
        {
            writer->SetInput(ds[i]);
            writer->Write();
            size =  writer->GetOutputStringLength();
            str  =  writer->RegisterAndGetOutputString();

            MPI_Send(&size, 1, MPI_INT, 0, mpiSizeTag, MPI_COMM_WORLD);
            MPI_Send(str, size, MPI_CHAR, 0, mpiDataTag, MPI_COMM_WORLD);
            delete [] str; //allocated by writer
        }
        writer->Delete(); 
        delete [] ds;
        return;
    }
#endif

    if (inTree->IsEmpty())
    {
        SetOutputDataTree(inTree);
        return; 
    }

    //
    //  This filter doesn't do much right now.  Basically just a 
    //  "connect-the-dots" between the vertices.  
    //
    int nleaves, j, k;
    float x;
    avtDataTree_p outTree;
    vtkDataSet **ds;
    ds = inTree->GetAllLeaves(nleaves);

    map <float, int> minX;

    //
    //  Cannot assume that there are no overlaps of points
    //  between datasets, so keep track of first and last
    //  x values, to determine if sorting is required. 
    //  Now must order the datasets so that the points will
    //  be accessed in proper order, to avoid discontinuities
    //  between domains.
    //  Store in map with x value as the key.  
    //  (from first point in each ds).
    //
    float *mm = new float[nleaves*2];
    int npts;

    for (j = 0; j < nleaves; j++)
    {
        npts = ((vtkPolyData*)ds[j])->GetNumberOfPoints();
        x = ((vtkPolyData*)ds[j])->GetPoint(0)[0]; 
        minX.insert(std::map <float, int> ::value_type(x, j));
        mm[j*2] = x;
        mm[j*2+1] = ((vtkPolyData*)ds[j])->GetPoint(npts-1)[0]; 
    }

    bool requiresSort = false;
    for (j = 0; j < nleaves && !requiresSort; j++)
    {
        for (k = 0; k < nleaves && !requiresSort; k++)
        {
            if (k != j && mm[k*2] > mm[j*2] && mm[k*2] < mm[j*2+1])
                requiresSort = true;
        }
    }
    delete [] mm;
    vtkPolyData *outPolys = vtkPolyData::New();
    
    vtkPoints *inPts; 
    vtkPoints *outPts  = vtkPoints::New();
    outPolys->SetPoints(outPts);
    outPts->Delete();

    vtkCellArray *lines   = vtkCellArray::New();
    outPolys->SetLines(lines);
    lines->Delete();

    vtkCellArray *verts   = vtkCellArray::New();
    outPolys->SetVerts(verts);
    verts->Delete();

    int nPoints; 
    float pt[3];
    vtkIdType i, ptIds[2];

    //
    //  Ensure that the output is 2d by setting z-component to zero.
    //
    std::map <float, int>::iterator it;
    for (it = minX.begin(); it != minX.end(); it++)
    {
        inPts = ((vtkPolyData*)ds[(*it).second])->GetPoints();
        nPoints = inPts->GetNumberOfPoints();
        for (i = 0; i < nPoints; i++)
        {
            inPts->GetPoint(i, pt);
            pt[2] = 0; 
            outPts->InsertNextPoint(pt);
        }
    }

    // 
    // Sort if necessary
    // 
    vtkPoints *sortedPts;
    if (requiresSort)
    {
        sortedPts = vtkPoints::New();
        map <float, int> sortedIds;
        nPoints = outPts->GetNumberOfPoints();
        for (i = 0; i < nPoints; i++)
        {
            x = outPts->GetPoint(i)[0]; 
            sortedIds.insert(std::map <float, int> ::value_type(x, i));
        }
        std::map <float, int>::iterator it;
        for (it = sortedIds.begin(); it != sortedIds.end(); it++)
        {
            sortedPts->InsertNextPoint(outPts->GetPoint((*it).second));
        }
        outPolys->SetPoints(sortedPts);
        sortedPts->Delete();
    }
    else
    {
        sortedPts = outPts;
    }
    nPoints = sortedPts->GetNumberOfPoints();
    for (i = 0; i < nPoints-1; i++)
    {
        ptIds[0] = i; 
        ptIds[1] = i+1; 
        lines->InsertNextCell(2, ptIds);        
        verts->InsertNextCell(1, ptIds);        
    } 
    // need to add one last vertex:
    ptIds[0] = ptIds[1];
    verts->InsertNextCell(1, ptIds);        

    const char *varname = (pipelineVariable != NULL ? pipelineVariable : "");

    avtDataRepresentation dr(outPolys, -1, varname);
    outTree = new avtDataTree(dr);
    outPolys->Delete();

    SetOutputDataTree(outTree);

    //
    //  Clean up.
    // 
    delete [] ds;
}

// ****************************************************************************
//  Method: avtCurveConstructorFilter::VerifyInput
//
//  Purpose:
//      Verifies that the input is 2D data, throws an exception if not.
//
//  Programmer: Kathleen Bonnell
//  Creation:   April 26, 2002 
//
// ****************************************************************************
 
void
avtCurveConstructorFilter::VerifyInput(void)
{
    if  (GetInput()->GetInfo().GetAttributes().GetTopologicalDimension() != 1)
    {
        EXCEPTION2(InvalidDimensionsException, "Curve", " Lines ");
    }
}

// ****************************************************************************
//  Method: avtCurveConstructorFilter::PerformRestriction
//
//  Purpose:
//    Indicates that we cannot do dynamic load balancing with this filter.  
//
//  Programmer: Kathleen Bonnell
//  Creation:   April 26, 2002 
//
// ****************************************************************************

avtPipelineSpecification_p
avtCurveConstructorFilter::PerformRestriction(avtPipelineSpecification_p spec)
{
    spec->NoDynamicLoadBalancing();
    return spec;
}

// ****************************************************************************
//  Method: avtCurveConstructorFilter::PerformRestriction
//
//  Purpose:
//      Allows the filter to change its output's data object information, which
//      is a description of the data object.
//
//  Programmer: Kathleen Bonnell
//  Creation:   December 23, 2002 
//
// ****************************************************************************

void
avtCurveConstructorFilter::RefashionDataObjectInfo(void)
{
    GetOutput()->GetInfo().GetAttributes().SetSpatialDimension(2);
}

