// ************************************************************************* //
//  File: avtCoordSystemConvert.C
// ************************************************************************* //

#include <avtCoordSystemConvert.h>

#include <vtkCellData.h>
#include <vtkDataSet.h>
#include <vtkFloatArray.h>
#include <vtkMath.h>
#include <vtkPointData.h>
#include <vtkPoints.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnstructuredGrid.h>

#include <vtkVisItUtility.h>

#include <avtExtents.h>


static vtkDataSet *CreateNewDataset(vtkDataSet *in_ds, vtkPoints *newPts);
static vtkDataSet *SphericalToCartesian(vtkDataSet *in_ds);
static vtkDataSet *CylindricalToSpherical(vtkDataSet *in_ds);
static vtkDataSet *CartesianToCylindrical(vtkDataSet *in_ds);
static vtkDataSet *FixWraparounds(vtkDataSet *in_ds, int comp_idx);

// ****************************************************************************
//  Method: avtCoordSystemConvert constructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Fri Jun 27 16:41:32 PST 2003
//
// ****************************************************************************

avtCoordSystemConvert::avtCoordSystemConvert()
{
    inputSys  = CARTESIAN;
    outputSys = CARTESIAN;
}


// ****************************************************************************
//  Method: avtCoordSystemConvert destructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Fri Jun 27 16:41:32 PST 2003
//
//  Modifications:
//
// ****************************************************************************

avtCoordSystemConvert::~avtCoordSystemConvert()
{
}


// ****************************************************************************
//  Method: avtCoordSystemConvert::ExecuteData
//
//  Purpose:
//      Sends the specified input and output through the CoordConvert filter.
//
//  Arguments:
//      in_ds      The input dataset.
//      <unused>   The domain number.
//      <unused>   The label.
//
//  Returns:       The output dataset.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Fri Jun 27 16:41:32 PST 2003
//
// ****************************************************************************

vtkDataSet *
avtCoordSystemConvert::ExecuteData(vtkDataSet *in_ds, int, std::string)
{
    vector<vtkDataSet *> deleteList;

    CoordSystem ct_current = inputSys;

    vtkDataSet *cur_ds = in_ds;
    while (ct_current != outputSys)
    {
        switch (ct_current)
        {
          case CARTESIAN:
          {
            vtkDataSet *new_ds = CartesianToCylindrical(cur_ds);
            deleteList.push_back(new_ds);
            cur_ds = new_ds;
            ct_current = CYLINDRICAL;
            break;
          }
          case CYLINDRICAL:
          {
            vtkDataSet *new_ds = CylindricalToSpherical(cur_ds);
            deleteList.push_back(new_ds);
            cur_ds = new_ds;
            ct_current = SPHERICAL;
            break;
          }
          case SPHERICAL:
          {
            vtkDataSet *new_ds = SphericalToCartesian(cur_ds);
            deleteList.push_back(new_ds);
            cur_ds = new_ds;
            ct_current = CARTESIAN;
            break;
          }
        }
    }

    if (outputSys == SPHERICAL)
    {
        cur_ds = FixWraparounds(cur_ds, 0);
        deleteList.push_back(cur_ds);
        cur_ds = FixWraparounds(cur_ds, 1);
        deleteList.push_back(cur_ds);
    }
    else if (outputSys == CYLINDRICAL)
    {
        cur_ds = FixWraparounds(cur_ds, 0);
        deleteList.push_back(cur_ds);
    }

    ManageMemory(cur_ds);

    for (int i = 0 ; i < deleteList.size() ; i++)
    {
         deleteList[i]->Delete();
    }

    return cur_ds;
}


// ****************************************************************************
//  Method: avtCoordSystemConvert::PostExecute
//
//  Purpose:
//      This is called to set up the output extents.
//
//  Programmer: Hank Childs
//  Creation:   June 30, 2003
//
// ****************************************************************************
 
void
avtCoordSystemConvert::PostExecute()
{
    avtDataAttributes &inAtts  = GetInput()->GetInfo().GetAttributes();
    avtDataAttributes &outAtts = GetOutput()->GetInfo().GetAttributes();

/*
    double b[6];
    THIS HAS BEEN MORE TROUBLE THAN ITS WORTH.  LEAVING THE CODE HERE IN
    CASE THIS EVER BECOMES IMPORTANT TO ANYONE.  THE ISSUE IS THAT IT JUST
    DOESN'T COME CLOSE ENOUGH ON THE REAL EXTENTS.
    if (inAtts.GetTrueSpatialExtents()->HasExtents())
    {
        inAtts.GetTrueSpatialExtents()->CopyTo(b);
        TransformExtents(b);
        outAtts.GetTrueSpatialExtents()->Set(b);
    }

    if (inAtts.GetCumulativeTrueSpatialExtents()->HasExtents())
    {
        inAtts.GetCumulativeTrueSpatialExtents()->CopyTo(b);
        TransformExtents(b);
        outAtts.GetCumulativeTrueSpatialExtents()->Set(b);
    }

    if (inAtts.GetEffectiveSpatialExtents()->HasExtents())
    {
        inAtts.GetEffectiveSpatialExtents()->CopyTo(b);
        TransformExtents(b);
        outAtts.GetEffectiveSpatialExtents()->Set(b);
    }

    if (inAtts.GetCurrentSpatialExtents()->HasExtents())
    {
        inAtts.GetCurrentSpatialExtents()->CopyTo(b);
        TransformExtents(b);
        outAtts.GetCurrentSpatialExtents()->Set(b);
    }

    if (inAtts.GetCumulativeCurrentSpatialExtents()->HasExtents())
    {
        inAtts.GetCumulativeCurrentSpatialExtents()->CopyTo(b);
        TransformExtents(b);
        outAtts.GetCumulativeCurrentSpatialExtents()->Set(b);
    }
*/
}


// ****************************************************************************
//  Method: avtCoordSystemConvert::TransformExtents
//
//  Purpose:
//      Transforms a bounding box to get the new extents.
//
//  Programmer: Hank Childs
//  Creation:   June 30, 2003
//
// ****************************************************************************

void
avtCoordSystemConvert::TransformExtents(double *extents)
{
    //
    // Set up a one cell-ed rectilinear grid based on the bounding box.
    //
    vtkFloatArray *x = vtkFloatArray::New();
    x->SetNumberOfTuples(10);
    int i;
    for (i = 0 ; i < 10 ; i++)
        x->SetTuple1(i, (extents[1]-extents[0]) * ((float)i)/10. + extents[0]);
 
    vtkFloatArray *y = vtkFloatArray::New();
    y->SetNumberOfTuples(10);
    for (i = 0 ; i < 10 ; i++)
        y->SetTuple1(i, (extents[3]-extents[2]) * ((float)i)/10. + extents[2]);
 
    vtkFloatArray *z = vtkFloatArray::New();
    z->SetNumberOfTuples(10);
    for (i = 0 ; i < 10 ; i++)
        z->SetTuple1(i, (extents[5]-extents[4]) * ((float)i)/10. + extents[4]);
 
    vtkRectilinearGrid *rgrid = vtkRectilinearGrid::New();
    rgrid->SetDimensions(10, 10, 10);
    rgrid->SetXCoordinates(x);
    rgrid->SetYCoordinates(y);
    rgrid->SetZCoordinates(z);

    vtkDataSet *rv = ExecuteData(rgrid, -1, "");
    float new_extents[6];
    rv->GetBounds(new_extents);

    extents[0] = new_extents[0];
    extents[1] = new_extents[1];
    extents[2] = new_extents[2];
    extents[3] = new_extents[3];
    extents[4] = new_extents[4];
    extents[5] = new_extents[5];

    x->Delete();
    y->Delete();
    z->Delete();
    rgrid->Delete();
    //rv does not need to be deleted.
    //rv->Delete();
}


// ****************************************************************************
//  Function: CreateNewDataset
//
//  Purpose:
//      Creates a dataset just like the input, except with the new set of
//      points.
//
//  Programmer: Hank Childs
//  Creation:   June 30, 2003
//
// ****************************************************************************

static vtkDataSet *
CreateNewDataset(vtkDataSet *in_ds, vtkPoints *newPts)
{
    vtkDataSet *rv = NULL;

    int dstype = in_ds->GetDataObjectType();
    if (dstype == VTK_STRUCTURED_GRID || dstype == VTK_POLY_DATA ||
        dstype == VTK_UNSTRUCTURED_GRID)
    {
        vtkPointSet *rv2 = (vtkPointSet *) in_ds->NewInstance();
        rv2->ShallowCopy(in_ds);
        rv2->SetPoints(newPts);

        rv = rv2;
    }
    else if (dstype == VTK_RECTILINEAR_GRID)
    {
        int dims[3];
        vtkRectilinearGrid *rg = (vtkRectilinearGrid *) in_ds;
        rg->GetDimensions(dims);

        vtkStructuredGrid *rv2 = vtkStructuredGrid::New();
        rv2->SetDimensions(dims);
        rv2->GetPointData()->ShallowCopy(in_ds->GetPointData());
        rv2->GetCellData()->ShallowCopy(in_ds->GetCellData());
        rv2->SetPoints(newPts);

        rv = rv2;
    }

    return rv;
}


// ****************************************************************************
//  Function: SphericalToCartesian
//
//  Purpose:
//      Converts spherical coordinates to cartesian coordinates.
//
//      More info at: 
//      http://www.geom.uiuc.edu/docs/reference/CRC-formulas/node42.html
//      (For the record, I did this myself and only *confirmed* the formulas at
//      this site.)
//
//  Programmer: Hank Childs
//  Creation:   June 28, 2003
//
// ****************************************************************************

static vtkDataSet *
SphericalToCartesian(vtkDataSet *in_ds)
{
    vtkPoints *pts = vtkVisItUtility::GetPoints(in_ds);
    int npts = pts->GetNumberOfPoints();

    vtkPoints *newPts = vtkPoints::New();
    newPts->SetNumberOfPoints(npts);
    for (int i = 0 ; i < npts ; i++)
    {
        float pt[3];
        pts->GetPoint(i, pt);
        float newpt[3];
        newpt[0] = pt[2]*cos(pt[0])*sin(pt[1]);
        newpt[1] = pt[2]*sin(pt[0])*sin(pt[1]);
        newpt[2] = pt[2]*cos(pt[1]);
        newPts->SetPoint(i, newpt);
    }

    vtkDataSet *rv = CreateNewDataset(in_ds, newPts);
    pts->Delete();
    newPts->Delete();

    return rv;
}


// ****************************************************************************
//  Function: CylindricalToSpherical
//
//  Purpose:
//      Converts spherical coordinates to cartesian coordinates.
//
//      More info at: 
//      http://www.geom.uiuc.edu/docs/reference/CRC-formulas/node42.html
//      (For the record, I did this myself and only *confirmed* the formulas at
//      this site.)
//
//  Programmer: Hank Childs
//  Creation:   June 28, 2003
//
// ****************************************************************************

static vtkDataSet *
CylindricalToSpherical(vtkDataSet *in_ds)
{
    vtkPoints *pts = vtkVisItUtility::GetPoints(in_ds);
    int npts = pts->GetNumberOfPoints();

    vtkPoints *newPts = vtkPoints::New();
    newPts->SetNumberOfPoints(npts);
    for (int i = 0 ; i < npts ; i++)
    {
        float pt[3];
        pts->GetPoint(i, pt);
        float newpt[3];
        newpt[0] = pt[0];
        newpt[1] = atan2(pt[2], pt[1]);
        if (newpt[1] < 0.)
            newpt[1] = 2*vtkMath::Pi() + newpt[1];
        newpt[2] = sqrt(pt[1]*pt[1] + pt[2]*pt[2]);
        newPts->SetPoint(i, newpt);
    }

    vtkDataSet *rv = CreateNewDataset(in_ds, newPts);
    pts->Delete();
    newPts->Delete();

    return rv;
}


// ****************************************************************************
//  Function: CartesianToCylindrical
//
//  Purpose:
//      Converts spherical coordinates to cartesian coordinates.
//
//      More info at: 
//      http://www.geom.uiuc.edu/docs/reference/CRC-formulas/node42.html
//      (For the record, I did this myself and only *confirmed* the formulas at
//      this site.)
//
//  Programmer: Hank Childs
//  Creation:   June 28, 2003
//
// ****************************************************************************

static vtkDataSet *
CartesianToCylindrical(vtkDataSet *in_ds)
{
    vtkPoints *pts = vtkVisItUtility::GetPoints(in_ds);
    int npts = pts->GetNumberOfPoints();

    vtkPoints *newPts = vtkPoints::New();
    newPts->SetNumberOfPoints(npts);
    for (int i = 0 ; i < npts ; i++)
    {
        float pt[3];
        pts->GetPoint(i, pt);
        float newpt[3];
        newpt[0] = atan2(pt[1], pt[0]);
        if (newpt[0] < 0.)
            newpt[0] = 2*vtkMath::Pi() + newpt[0];
        newpt[1] = pt[2];
        newpt[2] = sqrt(pt[0]*pt[0] + pt[1]*pt[1]);
        newPts->SetPoint(i, newpt);
    }

    vtkDataSet *rv = CreateNewDataset(in_ds, newPts);
    pts->Delete();
    newPts->Delete();

    return rv;
}


// ****************************************************************************
//  Function: FixWraparounds
//
//  Purpose:
//      Locates cells where the dataset has been wrapped-around from 0 radians
//      to 2*pi radians.
//
//  Programmer: Hank Childs
//  Creation:   July 7, 2003
//
// ****************************************************************************

static vtkDataSet *
FixWraparounds(vtkDataSet *in_ds, int comp_idx)
{
    int   i, j;

    if (in_ds->GetDataObjectType() != VTK_UNSTRUCTURED_GRID)
    {
        in_ds->Register(NULL);
        return in_ds;
    }

    vtkUnstructuredGrid *ugrid = (vtkUnstructuredGrid *) in_ds;
    vtkPoints *pts = ugrid->GetPoints();
    int npts = pts->GetNumberOfPoints();

    vtkPoints *new_pts = vtkPoints::New();
    new_pts->SetNumberOfPoints(2*npts);
    vtkUnstructuredGrid *new_grid = vtkUnstructuredGrid::New();
    new_grid->SetPoints(new_pts);
    vtkPointData *out_pd = new_grid->GetPointData();
    vtkPointData *in_pd  = in_ds->GetPointData();
    out_pd->CopyAllocate(in_pd, 2*npts);
   
    for (i = 0 ; i < npts ; i++)
    {
        float pt[3];
        pts->GetPoint(i, pt);
        new_pts->SetPoint(2*i, pt);
        if (pt[comp_idx] > vtkMath::Pi())
            pt[comp_idx] -= 2*vtkMath::Pi();
        else
            pt[comp_idx] += 2*vtkMath::Pi();
        new_pts->SetPoint(2*i+1, pt);
        out_pd->CopyData(in_pd, i, 2*i);
        out_pd->CopyData(in_pd, i, 2*i+1);
    }

    int ncells = ugrid->GetNumberOfCells();
    new_grid->Allocate(2*ncells*8);
    vtkCellData *out_cd = new_grid->GetCellData();
    vtkCellData *in_cd  = in_ds->GetCellData();
    out_cd->CopyAllocate(in_cd, 2*ncells);
   
    float pi = vtkMath::Pi();
    float twoPiCutoff = 2*vtkMath::Pi()*0.95;
    float zeroPiCutoff = vtkMath::Pi()*0.1;
    int cellCnt = 0;
    for (i = 0 ; i < ncells ; i++)
    {
        vtkIdType *ids;
        int cellNPts;
        ugrid->GetCellPoints(i, cellNPts, ids);
        bool closeToZero = false;
        bool closeToTwoPi = false;
        bool closeToLow[8];
        for (j = 0 ; j < cellNPts ; j++)
        {
            float pt[3];
            pts->GetPoint(ids[j], pt);
            if (pt[comp_idx] > twoPiCutoff)
                closeToTwoPi = true;
            if (pt[comp_idx] < zeroPiCutoff)
                closeToZero  = true;
            closeToLow[j] = (pt[comp_idx] < pi ? false : true);
        }
        if (closeToTwoPi && closeToZero)
        {
            // Make two cells -- start with the one close to 0 radians.
            vtkIdType low_ids[8];
            for (j = 0 ; j < cellNPts ; j++)
                low_ids[j] = (closeToLow[j] ? 2*ids[j] : 2*ids[j]+1);
            new_grid->InsertNextCell(ugrid->GetCellType(i), cellNPts, low_ids);
            out_cd->CopyData(in_cd, i, cellCnt++);
            
            vtkIdType hi_ids[8];
            for (j = 0 ; j < cellNPts ; j++)
                hi_ids[j] = (!closeToLow[j] ? 2*ids[j] : 2*ids[j]+1);
            new_grid->InsertNextCell(ugrid->GetCellType(i), cellNPts, hi_ids);
            out_cd->CopyData(in_cd, i, cellCnt++);
        }
        else
        {
            vtkIdType new_ids[8];
            for (j = 0 ; j < cellNPts ; j++)
                new_ids[j] = 2*ids[j];
            new_grid->InsertNextCell(ugrid->GetCellType(i), cellNPts, new_ids);
            out_cd->CopyData(in_cd, i, cellCnt++);
        }
    }
    new_grid->Squeeze();
    new_pts->Delete();

    return new_grid;
}


