// ************************************************************************* //
//                          avtBinaryMultiplyFilter.C                        //
// ************************************************************************* //

#include <avtBinaryMultiplyFilter.h>

#include <vtkDataArray.h>
#include <vtkDataArray.h>

#include <ExpressionException.h>


// ****************************************************************************
//  Method: avtBinaryMultiplyFilter constructor
//
//  Purpose:
//      Defines the constructor.  Note: this should not be inlined in the
//      header because it causes problems for certain compilers.
//
//  Programmer: Hank Childs
//  Creation:   February 5, 2004
//
// ****************************************************************************

avtBinaryMultiplyFilter::avtBinaryMultiplyFilter()
{
    ;
}


// ****************************************************************************
//  Method: avtBinaryMultiplyFilter destructor
//
//  Purpose:
//      Defines the destructor.  Note: this should not be inlined in the header
//      because it causes problems for certain compilers.
//
//  Programmer: Hank Childs
//  Creation:   February 5, 2004
//
// ****************************************************************************

avtBinaryMultiplyFilter::~avtBinaryMultiplyFilter()
{
    ;
}


// ****************************************************************************
//  Method: avtBinaryMultiplyFilter::DoOperation
//
//  Purpose:
//      Multiplies two arrays into a third array.
//
//  Arguments:
//      in1           The first input data array.
//      in2           The second input data array.
//      out           The output data array.
//      ncomponents   The number of components ('1' for scalar, '2' or '3' for
//                    vectors, etc.)
//      ntuples       The number of tuples (ie 'npoints' or 'ncells')
//
//  Programmer: Sean Ahern          <Header added by Hank Childs>
//  Creation:   November 18, 2002   <Header creation date>
//
//  Modifications:
//
//    Hank Childs, Mon Nov 18 07:35:07 PST 2002
//    Added support for vectors and arbitrary data types.
//
//    Hank Childs, Thu Aug 14 13:37:27 PDT 2003
//    Added support for mixing scalars and vectors.
//
//    Hank Childs, Mon Sep 22 17:06:01 PDT 2003
//    Added support for tensors.
//
// ****************************************************************************
 
void
avtBinaryMultiplyFilter::DoOperation(vtkDataArray *in1, vtkDataArray *in2,
                                     vtkDataArray *out, int ncomponents,
                                     int ntuples)
{
    int in1ncomps = in1->GetNumberOfComponents();
    int in2ncomps = in2->GetNumberOfComponents();

    if ((in1ncomps == 9) && (in2ncomps == 9))
    {
        float vals[9];
        for (int i = 0 ; i < ntuples ; i++)
        {
            vals[0] = in1->GetComponent(i, 0) * in2->GetComponent(i, 0) + 
                      in1->GetComponent(i, 1) * in2->GetComponent(i, 3) + 
                      in1->GetComponent(i, 2) * in2->GetComponent(i, 6); 
            vals[1] = in1->GetComponent(i, 0) * in2->GetComponent(i, 1) + 
                      in1->GetComponent(i, 1) * in2->GetComponent(i, 4) + 
                      in1->GetComponent(i, 2) * in2->GetComponent(i, 7); 
            vals[2] = in1->GetComponent(i, 0) * in2->GetComponent(i, 2) + 
                      in1->GetComponent(i, 1) * in2->GetComponent(i, 5) + 
                      in1->GetComponent(i, 2) * in2->GetComponent(i, 8); 
            vals[3] = in1->GetComponent(i, 3) * in2->GetComponent(i, 0) + 
                      in1->GetComponent(i, 4) * in2->GetComponent(i, 3) + 
                      in1->GetComponent(i, 5) * in2->GetComponent(i, 6); 
            vals[4] = in1->GetComponent(i, 3) * in2->GetComponent(i, 1) + 
                      in1->GetComponent(i, 4) * in2->GetComponent(i, 4) + 
                      in1->GetComponent(i, 5) * in2->GetComponent(i, 7); 
            vals[5] = in1->GetComponent(i, 3) * in2->GetComponent(i, 2) + 
                      in1->GetComponent(i, 4) * in2->GetComponent(i, 5) + 
                      in1->GetComponent(i, 5) * in2->GetComponent(i, 8); 
            vals[6] = in1->GetComponent(i, 6) * in2->GetComponent(i, 0) + 
                      in1->GetComponent(i, 7) * in2->GetComponent(i, 3) + 
                      in1->GetComponent(i, 8) * in2->GetComponent(i, 6); 
            vals[7] = in1->GetComponent(i, 6) * in2->GetComponent(i, 1) + 
                      in1->GetComponent(i, 7) * in2->GetComponent(i, 4) + 
                      in1->GetComponent(i, 8) * in2->GetComponent(i, 7); 
            vals[8] = in1->GetComponent(i, 6) * in2->GetComponent(i, 2) + 
                      in1->GetComponent(i, 7) * in2->GetComponent(i, 5) + 
                      in1->GetComponent(i, 8) * in2->GetComponent(i, 8); 
            out->SetTuple(i, vals);
        }
    }
    else if ((in1ncomps == 3) && (in2ncomps == 9))
    {
        float vals[3];
        for (int i = 0 ; i < ntuples ; i++)
        {
            vals[0] = in1->GetComponent(i, 0) * in2->GetComponent(i, 0) + 
                      in1->GetComponent(i, 1) * in2->GetComponent(i, 3) + 
                      in1->GetComponent(i, 2) * in2->GetComponent(i, 6); 
            vals[1] = in1->GetComponent(i, 0) * in2->GetComponent(i, 1) + 
                      in1->GetComponent(i, 1) * in2->GetComponent(i, 4) + 
                      in1->GetComponent(i, 2) * in2->GetComponent(i, 7); 
            vals[2] = in1->GetComponent(i, 0) * in2->GetComponent(i, 2) + 
                      in1->GetComponent(i, 1) * in2->GetComponent(i, 5) + 
                      in1->GetComponent(i, 2) * in2->GetComponent(i, 8); 
            out->SetTuple(i, vals);
        }
    }
    else if ((in1ncomps == 9) && (in2ncomps == 3))
    {
        float vals[3];
        for (int i = 0 ; i < ntuples ; i++)
        {
            vals[0] = in1->GetComponent(i, 0) * in2->GetComponent(i, 0) + 
                      in1->GetComponent(i, 1) * in2->GetComponent(i, 1) + 
                      in1->GetComponent(i, 2) * in2->GetComponent(i, 2); 
            vals[1] = in1->GetComponent(i, 3) * in2->GetComponent(i, 0) + 
                      in1->GetComponent(i, 4) * in2->GetComponent(i, 1) + 
                      in1->GetComponent(i, 5) * in2->GetComponent(i, 2); 
            vals[2] = in1->GetComponent(i, 6) * in2->GetComponent(i, 0) + 
                      in1->GetComponent(i, 7) * in2->GetComponent(i, 1) + 
                      in1->GetComponent(i, 8) * in2->GetComponent(i, 2);
            out->SetTuple(i, vals);
        }
    }
    else if (in1ncomps == in2ncomps)
    {
        for (int i = 0 ; i < ntuples ; i++)
        {
            float dot = 0.;
            for (int j = 0 ; j < in1ncomps ; j++)
            {
                float val1 = in1->GetComponent(i, j);
                float val2 = in2->GetComponent(i, j);
                dot += val1*val2;
            }
            out->SetTuple1(i, dot);
        }
    }
    else if (in1ncomps > 1 && in2ncomps == 1)
    {
        for (int i = 0 ; i < ntuples ; i++)
        {
            float val2 = in2->GetTuple1(i);
            for (int j = 0 ; j < in1ncomps ; j++)
            {
                float val1 = in1->GetComponent(i, j);
                out->SetComponent(i, j, val1 * val2);
            }
        }
    }
    else if (in1ncomps == 1 && in2ncomps > 1)
    {
        for (int i = 0 ; i < ntuples ; i++)
        {
            float val1 = in1->GetTuple1(i);
            for (int j = 0 ; j < in2ncomps ; j++)
            {
                float val2 = in2->GetComponent(i, j);
                out->SetComponent(i, j, val1 * val2);
            }
        }
    }
    else
    {
        EXCEPTION1(ExpressionException, "Don't know how to multiply vectors "
                                        "of differing dimensions.");
    }
}


// ****************************************************************************
//  Method: avtBinaryMultiplyFilter::GetNumberOfComponentsInOutput
//
//  Purpose:
//      Sets the number of components in the output.
//
//  Programmer: Hank Childs
//  Creation:   August 14, 2003
//
//  Modifications:
//
//    Hank Childs, Mon Sep 22 17:06:01 PDT 2003
//    Add matrix/tensor cases.
//
// ****************************************************************************

int
avtBinaryMultiplyFilter::GetNumberOfComponentsInOutput(int in1, int in2)
{
    if ((in1 == 3) && (in2 == 3))
        return 1;  // We will do a dot product.

    if ((in1 == 9) && (in2 == 9))
        return 9;  // Matrix multiply

    if ((in1 == 3) && (in2 == 9))
        return 3;  // Vector x Matrix = Vector

    if ((in1 == 9) && (in2 == 3))
        return 3;  // Matrix x Vector = Vector

    return (in1 > in2 ? in1 : in2);
}


