/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmGetTargetsCommand.h"

#include <cassert>
#include <sstream>

#include "cmExportSet.h"
#include "cmGlobalGenerator.h"
#include "cmMakefile.h"
#include "cmSystemTools.h"
#include "cmTargetExport.h"

bool cmGetTargetsCommand::InitialPass(args_t const& args, cmExecutionStatus&)
{
  if (args.empty()) {
    this->SetError("missing name of output variable");
    return false;
  }

  // REVIEWER: Who handles the check that the variable name is legal?

  if (args.size() == 1) {
    args_t const default_args{
      args[0],
      "DIRECTORY",
      this->Makefile->GetCurrentSourceDirectory(),
    };
    return GetTargetsFromDirectory(default_args);
  }

  std::string const& mode = args[1];
  if (mode == "DIRECTORY") {
    return GetTargetsFromDirectory(args);
  }
  if (mode == "EXPORT") {
    return GetTargetsFromExportSet(args);
  }

  std::ostringstream e;
  // REVIEWER: Is there no convention for quoting user-supplied values (the
  // mode in this case) in error messages?
  e << "given invalid scope " << mode << ".  "
    << "Valid scopes are DIRECTORY and EXPORT.";
  this->SetError(e.str());
  return false;
}

bool cmGetTargetsCommand::GetTargetsFromDirectory(args_t const& args)
{
  // Caller is responsible for checking the first two arguments only.
  assert(args.size() >= 2);
  assert(args[1] == "DIRECTORY");

  std::string dir = args[2];
  if (!cmSystemTools::FileIsFullPath(dir)) {
    dir = this->Makefile->GetCurrentSourceDirectory() + "/" + dir;
  }

  dir = cmSystemTools::CollapseFullPath(dir);
  cmMakefile* mf = this->Makefile->GetGlobalGenerator()->FindMakefile(dir);
  if (!mf) {
    std::ostringstream e;
    e << "directory not found: " << args[2] << "; has it been processed yet?";
    this->SetError(e.str());
    return false;
  }

  std::vector<std::string> targets;
  std::vector<std::string> subdirs{ dir };

  // REVIEWER: Do we want FIFO instead? Does it matter?
  // My guess is that users will expect FIFO.
  while (!subdirs.empty()) {
    std::string subdir = subdirs.back();
    subdirs.pop_back();

    subdir = cmSystemTools::CollapseFullPath(subdir);
    // REVIEWER: There is no way this can return nullptr right?
    cmMakefile const* submf =
      this->Makefile->GetGlobalGenerator()->FindMakefile(subdir);

    // REVIEWER: Is there a faster way to get this property like there is with
    // `CMAKE_CURRENT_SOURCE_DIR` / `GetCurrentSourceDirectory()`?
    char const* strSubdirs = submf->GetProperty("SUBDIRECTORIES");
    // REVIEWER: When can this be nullptr? When can it be empty string?
    if (strSubdirs && strSubdirs[0] != '\0') {
      cmSystemTools::ExpandListArgument(strSubdirs, subdirs);
    }

    char const* strTargets = submf->GetProperty("BUILDSYSTEM_TARGETS");
    // REVIEWER: When can this be nullptr? When can it be empty string?
    if (strTargets && strTargets[0] != '\0') {
      targets.push_back(strTargets);
    }
  }

  std::string const& targetListString = cmJoin(cmMakeRange(targets), ";");
  this->Makefile->AddDefinition(args[0], targetListString.c_str());
  return true;
}

bool cmGetTargetsCommand::GetTargetsFromExportSet(args_t const& args)
{
  // REVIEWER: Can I use asserts for sanity checks?
  // Caller is responsible for checking the first two arguments only.
  assert(args.size() >= 2);
  assert(args[1] == "EXPORT");

  if (args.size() < 3) {
    this->SetError("missing name of export set");
    return false;
  }

  std::string const& exportName = args[2];
  auto* exportSet =
    this->Makefile->GetGlobalGenerator()->GetExportSets()[exportName];
  if (!exportSet) {
    std::ostringstream e;
    e << "no export set named " << exportName;
    this->SetError(e.str());
    return false;
  }

  auto* targets = exportSet->GetTargetExports();
  // Perfect opportunity for ranges...
  std::string const& targetListString =
    cmJoin(cmMakeRange(*targets).transform(
             [](cmTargetExport const* target) { return target->TargetName; }),
           ";");

  this->Makefile->AddDefinition(args[0], targetListString.c_str());
  return true;
}
