/*=========================================================================

  Program:   Visualization Toolkit
  Module:    TestAbortExecute.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkmLookupTable.h"

#include "vtkFloatArray.h"
#include "vtkIntArray.h"
#include "vtkNew.h"
#include "vtkSmartPointer.h"
#include "vtkUnsignedCharArray.h"

#include <algorithm>
#include <iostream>
#include <limits>
#include <random>
#include <type_traits>

namespace
{

//------------------------------------------------------------------------------
class TestError
{
public:
  TestError(const std::string& message, int line)
    : Message(message)
    , Line(line)
  {
  }

  void PrintMessage(std::ostream& out) const
  {
    out << "Error at line " << this->Line << ": " << this->Message << "\n";
  }

private:
  std::string Message;
  int Line;
};

#define RAISE_TEST_ERROR(msg) throw TestError((msg), __LINE__)

#define TEST_VERIFY(cond, msg)                                                                     \
  if (!(cond))                                                                                     \
  RAISE_TEST_ERROR((msg))

//------------------------------------------------------------------------------
class RNG
{
public:
  static void Init(std::default_random_engine::result_type seed = std::random_device{}())
  {
    std::cout << "using seed: " << seed << "\n";
    RNG::Engine.seed(seed);
  }

  template <typename T>
  struct UniformRandomValueGenerator
  {
    typename std::conditional<std::numeric_limits<T>::is_integer,
                              std::uniform_int_distribution<T>,
                              std::uniform_real_distribution<T>>::type Distribution;

    UniformRandomValueGenerator(T min, T max) : Distribution(min, max)
    {
    }

    T operator()()
    {
      return this->Distribution(RNG::Engine);
    }
  };

  template <typename T>
  static T GetRandomValue(const T& min, const T& max)
  {
    return UniformRandomValueGenerator<T>(min, max)();
  }

private:
  static std::default_random_engine Engine;
};

std::default_random_engine RNG::Engine;

//------------------------------------------------------------------------------
static constexpr int ArraySize = 64;

template <typename DataArrayType>
void GenerateRandomArray(
  DataArrayType* array,
  int numComps,
  typename DataArrayType::ValueType min,
  typename DataArrayType::ValueType max)
{
  auto numVals = ArraySize * numComps;

  array->SetNumberOfComponents(numComps);
  array->SetNumberOfTuples(ArraySize);
  std::generate(
    array->GetPointer(0),
    array->GetPointer(numVals),
    RNG::UniformRandomValueGenerator<typename DataArrayType::ValueType>(min, max));
  array->Modified();
}

template <typename T>
T MakePrintable(T val)
{
  return val;
}

int MakePrintable(unsigned char val)
{
  return static_cast<int>(val);
}

template <typename T>
void printArray(const T* array, vtkIdType numTuples, int numComponents)
{
  std::cout << "Number of tuples: " << numTuples << "\n";
  std::cout << "Number of components: " << numComponents << "\n";
  std::cout << "Values: ";
  for (vtkIdType j = 0, idx = 0; j < numTuples; ++j)
  {
    std::cout << "[" << MakePrintable(array[idx]);
    ++idx;
    for (int i = 1; i < numComponents; ++i, ++idx)
    {
      std::cout << ", " << MakePrintable(array[idx]);
    }
    std::cout << "] ";
  }
  std::cout << "\n";
}

void CompareResults(vtkUnsignedCharArray* r1, vtkUnsignedCharArray* r2)
try
{
  TEST_VERIFY(r1->GetNumberOfTuples() == r2->GetNumberOfTuples(), "result sizes don't match");
  TEST_VERIFY(r1->GetNumberOfComponents() == r2->GetNumberOfComponents(), "result number of components don't match");

  for (vtkIdType j = 0; j < r1->GetNumberOfTuples(); ++j)
  {
    for (int i = 0; i < r1->GetNumberOfComponents(); ++i)
    {
      TEST_VERIFY(r1->GetTypedComponent(j, i) == r2->GetTypedComponent(j, i), "result values don't match");
    }
  }
}
catch (const TestError&)
{
  std::cout << "Failed\n";
  std::cout << "vtkm result: \n";
  printArray(r1->GetPointer(0), r1->GetNumberOfTuples(), r1->GetNumberOfComponents());
  std::cout << "vtk result: \n";
  printArray(r2->GetPointer(0), r2->GetNumberOfTuples(), r2->GetNumberOfComponents());
  throw;
}

struct TestParameters
{
  int ColorMode = VTK_COLOR_MODE_DEFAULT;
  int Component = 0;
  int VectorMode = vtkScalarsToColors::COMPONENT;
  int OutputFormat = VTK_RGB;
  int Scale = VTK_SCALE_LINEAR;
};

void PrintConfiguration(const TestParameters& tp, vtkLookupTable* lut)
{
  std::cout << "ColorMode: ";
  switch (tp.ColorMode)
  {
    case VTK_COLOR_MODE_DIRECT_SCALARS:
      std::cout << "VTK_COLOR_MODE_DIRECT_SCALARS\n";
      break;
    case VTK_COLOR_MODE_DEFAULT:
      std::cout << "VTK_COLOR_MODE_DEFAULT\n";
      break;
    default:
      break;
  }

  if (tp.ColorMode != VTK_COLOR_MODE_DIRECT_SCALARS)
  {
    if (tp.Component != -1)
    {
      std::cout << "Component: " << tp.Component << "\n";
      std::cout << "Range: " << lut->GetTableRange()[0] << ", " << lut->GetTableRange()[1] << "\n";
    }
    else
    {
      std::cout << "VectorMode: ";
      switch (tp.VectorMode)
      {
        case vtkScalarsToColors::COMPONENT:
          std::cout << "COMPONENT\n";
          std::cout << "Range: " << lut->GetTableRange()[0] << ", " << lut->GetTableRange()[1] << "\n";
          break;
        case vtkScalarsToColors::MAGNITUDE:
          std::cout << "MAGNITUDE\n";
          std::cout << "Range: " << lut->GetTableRange()[0] << ", " << lut->GetTableRange()[1] << "\n";
          break;
        case vtkScalarsToColors::RGBCOLORS:
          std::cout << "RGBCOLORS\n";
          break;
        default:
          break;
      }
      std::cout << "VectorComponent: " << lut->GetVectorComponent() << "\n";
      std::cout << "VectorSize: " << lut->GetVectorSize() << "\n";
    }

    std::cout << "Scale: "
              << ((tp.Scale == VTK_SCALE_LOG10) ? "VTK_SCALE_LOG10" : "VTK_SCALE_LINEAR")
              << "\n";

    std::cout << "OutputFormat: ";
    switch (tp.OutputFormat)
    {
      case VTK_RGB:
        std::cout << "VTK_RGB\n";
        break;
      case VTK_RGBA:
        std::cout << "VTK_RGBA\n";
        break;
      default:
        break;
    }
  }

  if ((tp.ColorMode == VTK_COLOR_MODE_DIRECT_SCALARS) || (tp.OutputFormat == VTK_RGBA))
  {
    std::cout << "Alpha: " << lut->GetAlpha() << "\n";
  }
}

void RunMainTest(vtkDataArray* array, TestParameters tp)
{
  vtkNew<vtkLookupTable> vtkLUT;

  double range[2] = { 0.0, 10.0 };
  if (tp.Component == -1)
  {
    vtkLUT->SetVectorMode(tp.VectorMode);

    if (tp.VectorMode == vtkScalarsToColors::COMPONENT)
    {
      vtkLUT->SetVectorComponent(RNG::GetRandomValue(0, array->GetNumberOfComponents() - 1));
      array->GetRange(range, vtkLUT->GetVectorComponent());
    }
    else if (tp.VectorMode == vtkScalarsToColors::MAGNITUDE)
    {
      vtkLUT->SetVectorSize(RNG::GetRandomValue(1, array->GetNumberOfComponents()));
      array->GetRange(range, -1);
    }
    else
    {
      vtkLUT->SetVectorSize(RNG::GetRandomValue(1, array->GetNumberOfComponents()));
    }
  }
  else
  {
    array->GetRange(range, tp.Component);
  }

  if ((tp.ColorMode == VTK_COLOR_MODE_DIRECT_SCALARS) || (tp.OutputFormat == VTK_RGBA))
  {
    vtkLUT->SetAlpha(RNG::GetRandomValue(0.0, 1.0));
  }

  vtkLUT->SetTableRange(range[0], range[1]);
  vtkLUT->SetScale(tp.Scale);
  vtkLUT->Build();

  std::cout << std::string(70, '-') << "\n";
  std::cout << "Testing using configuration: \n";
  PrintConfiguration(tp, vtkLUT);

  vtkNew<vtkmLookupTable> vtkmLUT;
  vtkmLUT->DeepCopy(vtkLUT);
  vtkmLUT->Build();

  vtkSmartPointer<vtkUnsignedCharArray> vtkmResult;
  vtkmResult = vtkmLUT->MapScalars(array, tp.ColorMode, tp.Component, tp.OutputFormat);

  vtkSmartPointer<vtkUnsignedCharArray> vtkResult;
  vtkResult = vtkLUT->MapScalars(array, tp.ColorMode, tp.Component, tp.OutputFormat);

  try
  {
    CompareResults(vtkmResult, vtkResult);
  }
  catch (const TestError& e)
  {
    std::cout << "Input: \n";
    switch (array->GetDataType())
    {
      vtkTemplateMacro(
        printArray(static_cast<VTK_TT*>(array->GetVoidPointer(0)), array->GetNumberOfTuples(), array->GetNumberOfComponents());
      );
    }
    std::cout << "vtkmLUT:\n";
    vtkmLUT->Print(std::cout);
    std::cout << "vtkLUT:\n";
    vtkLUT->Print(std::cout);
    throw;
  }
  std::cout << "Passed\n";
}

void TestOutputFormats(vtkDataArray* array, TestParameters tp)
{
  tp.OutputFormat = VTK_RGB;
  RunMainTest(array, tp);

  tp.OutputFormat = VTK_RGBA;
  RunMainTest(array, tp);
}

void TestBasicOptions(vtkDataArray* array)
{
  TestParameters tp;

  tp.Component = RNG::GetRandomValue(0, array->GetNumberOfComponents() - 1);
  TestOutputFormats(array, tp);

  tp.Component = -1;
  tp.VectorMode = vtkScalarsToColors::COMPONENT;
  TestOutputFormats(array, tp);

  tp.VectorMode = vtkScalarsToColors::MAGNITUDE;
  TestOutputFormats(array, tp);

  tp.VectorMode = vtkScalarsToColors::RGBCOLORS;
  TestOutputFormats(array, tp);
}

void TestLogScale(vtkDataArray* array)
{
  TestParameters tp;
  tp.Scale = VTK_SCALE_LOG10;

  tp.Component = RNG::GetRandomValue(0, array->GetNumberOfComponents() - 1);
  TestOutputFormats(array, tp);

  tp.Component = -1;
  tp.VectorMode = vtkScalarsToColors::COMPONENT;
  TestOutputFormats(array, tp);

  tp.VectorMode = vtkScalarsToColors::MAGNITUDE;
  TestOutputFormats(array, tp);
}

void TestColorModeDirectScalars(vtkDataArray* array)
{
  TestParameters tp;
  tp.ColorMode = VTK_COLOR_MODE_DIRECT_SCALARS;
  RunMainTest(array, tp);
}

static std::string breaker(70, '=');

void TestUnsignedCharArray()
{
  vtkNew<vtkUnsignedCharArray> array;
  for (int c = 1; c <= 4; ++c)
  {
    std::cout << breaker << "\n";
    std::cout << "Testing unsigned char array with " << c << " component\n";
    GenerateRandomArray(array.GetPointer(), c, 0, 255);
    TestBasicOptions(array);
    TestColorModeDirectScalars(array);
    GenerateRandomArray(array.GetPointer(), c, 1, 255);
    TestLogScale(array);
    std::cout << breaker << "\n\n";
  }
}

void TestIntArray()
{
  vtkNew<vtkIntArray> array;
  for (int c = 1; c <= 4; ++c)
  {
    std::cout << breaker << "\n";
    std::cout << "Testing int array with " << c << " component\n";
    GenerateRandomArray(array.GetPointer(), c, -1000, 1000);
    TestBasicOptions(array);
    GenerateRandomArray(array.GetPointer(), c, 0, 255);
    TestColorModeDirectScalars(array);
    GenerateRandomArray(array.GetPointer(), c, 1, 10000);
    TestLogScale(array);
    std::cout << breaker << "\n\n";
  }
}

void TestFloatArray()
{
  vtkNew<vtkFloatArray> array;
  for (int c = 1; c <= 4; ++c)
  {
    std::cout << breaker << "\n";
    std::cout << "Testing float array with " << c << " component\n";
    GenerateRandomArray(array.GetPointer(), c, -100.0f, 100.0f);
    TestBasicOptions(array);
    GenerateRandomArray(array.GetPointer(), c, 0.0f, 1.0f);
    TestColorModeDirectScalars(array);
    GenerateRandomArray(array.GetPointer(), c, 1e-5f, 10000.0f);
    TestLogScale(array);
    std::cout << breaker << "\n\n";
  }
}

} // namespace

int TestVTKMLookupTable(int, char*[])
{
  RNG::Init();

  try
  {
    TestUnsignedCharArray();
    TestIntArray();
    TestFloatArray();
  }
  catch(const TestError& e)
  {
    e.PrintMessage(std::cout);
    return 1;
  }

  return EXIT_SUCCESS;
}
