/*=========================================================================

   Library: iMSTK

   Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
   & Imaging in Medicine, Rensselaer Polytechnic Institute.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0.txt

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

   =========================================================================*/

#include "imstkVTKTetrahedralMeshRenderDelegate.h"

#include "imstkTetrahedralMesh.h"

#include <vtkUnstructuredGrid.h>
#include <vtkDataSetMapper.h>
#include <vtkPoints.h>
#include <vtkDoubleArray.h>
#include <vtkCellArray.h>
#include <vtkProperty.h>
#include <vtkPolyData.h>
#include <vtkPointData.h>
#include <vtkFloatArray.h>
#include <vtkScalarBarActor.h>

#include "g3log/g3log.hpp"

#include <iostream>
namespace imstk
{

VTKTetrahedralMeshRenderDelegate::VTKTetrahedralMeshRenderDelegate(std::shared_ptr<TetrahedralMesh> tetrahedralMesh) :
    m_geometry(tetrahedralMesh)
{
    // Map vertices
    StdVectorOfVec3d& vertices = m_geometry->getVerticesPositionsNotConst();
    double* vertData = reinterpret_cast<double*>(vertices.data());
    m_mappedVertexArray = vtkSmartPointer<vtkDoubleArray>::New();
    m_mappedVertexArray->SetNumberOfComponents(3);
    m_mappedVertexArray->SetArray(vertData, vertices.size()*3, 1);

    // Map nodal vonMises stress values (scalars)
    auto & nodalData = m_geometry->getScalarAttribute();
    double *nodalDataPtr = reinterpret_cast<double*>(nodalData.data());
    this->m_mappedDataArray = vtkSmartPointer<vtkDoubleArray>::New();
    auto numPts = m_geometry->getNumVertices();
    this->m_mappedDataArray->SetName("vms");
    this->m_mappedDataArray->SetArray(nodalDataPtr, numPts, 1);

    // Create points
    auto points = vtkSmartPointer<vtkPoints>::New();
    points->SetNumberOfPoints(m_geometry->getNumVertices());
    points->SetData(m_mappedVertexArray);

    // Copy cells
    auto cells = vtkSmartPointer<vtkCellArray>::New();
    vtkIdType cell[4];
    for(const auto &t : m_geometry->getTetrahedraVertices())
    {
        for(size_t i = 0; i < 4; ++i)
        {
            cell[i] = t[i];
        }
        cells->InsertNextCell(4,cell);
    }

    // Create Unstructured Grid
    m_mesh = vtkSmartPointer<vtkUnstructuredGrid>::New();
    m_mesh->SetPoints(points);
    m_mesh->SetCells(VTK_TETRA, cells);
    m_mesh->GetPointData()->SetScalars(m_mappedDataArray);

    // Mapper
    auto mapper = vtkSmartPointer<vtkDataSetMapper>::New();
    mapper->SetInputData(m_mesh);

    // Actor
    //m_actor->SetMapper(mapper);
    //m_actor->GetProperty()->EdgeVisibilityOn();
    //mapper->SetScalarModeToUsePointData();
    mapper->SetColorModeToMapScalars();

    //Create a loopup table to share between mapper and the scalarbar
    m_hueLut = vtkSmartPointer<vtkLookupTable>::New();
    m_hueLut->SetTableRange(*std::min_element(nodalData.begin(), nodalData.end()), *std::max_element(nodalData.begin(), nodalData.end()));
    m_hueLut->SetHueRange(0.667, 0.0);
    m_hueLut->SetSaturationRange(1, 1);
    m_hueLut->Build();
    mapper->SetLookupTable(m_hueLut);

    //Create Scalar Bar
    m_scalarBar = vtkSmartPointer<vtkScalarBarActor>::New();
    m_scalarBar->SetLookupTable(mapper->GetLookupTable());
    m_scalarBar->SetTitle("Von-Mises Stress(Pa)");
    m_scalarBar->SetNumberOfLabels(4);
    m_scalarBar->DrawTickLabelsOn();
    m_scalarBar->SetBarRatio(0.15);
    this->addProp(m_scalarBar);

    this->m_actor->SetMapper(mapper);
    this->m_actor->GetProperty()->EdgeVisibilityOn();
    this->m_actor->GetProperty()->SetOpacity(1);
    this->m_actor->GetProperty()->SetColor(128 / 255.0, 174.0 / 255.0, 128.0 / 255.0); //tissue
    //this->m_actor->GetProperty()->SetColor(221 / 255.0, 130.0 / 255.0, 101.0 / 255.0); //artery

    // Transform
    this->updateActorTransform();
}

void
VTKTetrahedralMeshRenderDelegate::update()
{
    // Base class update
    VTKRenderDelegate::update();

    auto & nodalData = m_geometry->getScalarAttribute();
    auto dataMin = *std::min_element(nodalData.begin(), nodalData.end());
    auto dataMax = *std::max_element(nodalData.begin(), nodalData.end());
    this->m_actor->GetMapper()->SetUseLookupTableScalarRange(1);
    this->m_hueLut->SetTableRange(dataMin, dataMax / 5.0);
    //this->m_hueLut->SetTableRange(dataMin, dataMax);
    //this->m_hueLut->SetTableRange(0, 50000); // for artery
    this->m_hueLut->SetHueRange(0.667, 0.0);
    this->m_hueLut->SetSaturationRange(1, 1);
    this->m_actor->GetMapper()->SetLookupTable(m_hueLut);

    // Update scalar bar
    this->m_scalarBar->SetLookupTable(m_actor->GetMapper()->GetLookupTable());
    this->m_scalarBar->SetTitle("Von Mises Stress (Pa)");
    this->m_scalarBar->SetNumberOfLabels(4);
    this->addProp(m_scalarBar);

    //m_mappedVertexArray->Modified(); // TODO: only modify if vertices change
    m_mappedDataArray->Modified();

    // Copy cells
    auto & maskTets = std::dynamic_pointer_cast<Mesh>(m_geometry)->getRemovedElems().second;
    auto cells = vtkSmartPointer<vtkCellArray>::New();
    vtkIdType cell[4];
    size_t cellId = 0;
    bool isMasked;
    for (const auto &t : m_geometry->getTetrahedraVertices())
    {
        isMasked = false;
        for (auto i : maskTets)
        {
            if (i == cellId)
            {
                isMasked = true;
                break;
            }
        }
        if (!isMasked)
        {
            for (size_t i = 0; i < 4; ++i)
            {
                cell[i] = t[i];
            }
            cells->InsertNextCell(4, cell);
        }

        ++cellId;
    }
    m_mesh->SetCells(VTK_TETRA, cells);
}


std::shared_ptr<Geometry>
VTKTetrahedralMeshRenderDelegate::getGeometry() const
{
    return m_geometry;
}

} // imstk
