/*=========================================================================

   Library: iMSTK

   Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
   & Imaging in Medicine, Rensselaer Polytechnic Institute.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0.txt

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

   =========================================================================*/

#ifndef imstkHyperElasticFEMForceModel_h
#define imstkHyperElasticFEMForceModel_h

#include <g3log/g3log.hpp>

//imstk
#include "imstkInternalForceModel.h"
#include "imstkForceModelConfig.h"
#include "imstkIsotropicHyperelasticFEM.h"

//vega
#include "isotropicHyperelasticFEM.h"
#include "StVKIsotropicMaterial.h"
#include "neoHookeanIsotropicMaterial.h"
#include "MooneyRivlinIsotropicMaterial.h"

namespace imstk
{

///
/// \class IsotropicHyperelasticFEForceModel
///
/// \brief Force model for the isotropic hyperelastic material
///
class IsotropicHyperelasticFEForceModel : public InternalForceModel
{

public:
    ///
    /// \brief Constructor
    ///
    IsotropicHyperelasticFEForceModel(const HyperElasticMaterialType materialType,
        std::shared_ptr<vega::VolumetricMesh> mesh,
        const double inversionThreshold,
        const bool withGravity = false,
        const double gravity = 10.0) : InternalForceModel()
    {
        auto tetMesh = std::dynamic_pointer_cast<vega::TetMesh>(mesh);

        const int enableCompressionResistance = 1;
        const double compressionResistance = 500;
        switch (materialType)
        {
            case HyperElasticMaterialType::StVK:
                m_isotropicMaterial = std::make_shared<vega::StVKIsotropicMaterial>(
                    tetMesh.get(),
                    enableCompressionResistance,
                    compressionResistance);
                break;

            case HyperElasticMaterialType::NeoHookean:
                m_isotropicMaterial = std::make_shared<vega::NeoHookeanIsotropicMaterial>(
                    tetMesh.get(),
                    enableCompressionResistance,
                    compressionResistance);
                break;

            case HyperElasticMaterialType::MooneyRivlin:
                m_isotropicMaterial = std::make_shared<vega::MooneyRivlinIsotropicMaterial>(
                    tetMesh.get(),
                    enableCompressionResistance,
                    compressionResistance);
                break;

            default:
                LOG(WARNING) << "Error: Invalid hyperelastic material type.";
        }

        m_isotropicHyperelasticFEM = std::make_shared<vega::IsotropicHyperelasticFEM>(
            tetMesh.get(),
            m_isotropicMaterial.get(),
            inversionThreshold,
            withGravity,
            gravity);
    }

    ///
    /// \brief Constructor type that is not allowed
    ///
    IsotropicHyperelasticFEForceModel() = delete;

    ///
    /// \brief Destructor
    ///
    virtual ~IsotropicHyperelasticFEForceModel() = default;

    ///
    /// \brief Get the internal force
    ///
    inline void getInternalForce(const Vectord& u, Vectord& internalForce) override
    {
        double *data = const_cast<double*>(u.data());
        m_isotropicHyperelasticFEM->ComputeForces(data, internalForce.data());
    }

    ///
    /// \brief Get the modified internalForce (to simulate cutting)
    ///
    void getInternalForce(const Vectord& u, Vectord& internalForce, const std::vector<size_t>& removeElemsIDs)
    {
        const auto computation_mode = vega::IsotropicHyperelasticFEM::COMPUTE_INTERNALFORCES;
        double *data = const_cast<double*>(u.data());

        // implementation based on the assumption that the removeTets vector, and tetIDs are in sorted form, and moreover in an ascending order.
        const auto numElem = m_isotropicHyperelasticFEM->GetTetMesh()->getNumElements();

        std::vector< std::vector<size_t> > tetIDsRange;
        InternalForceModel::calculateRanges(tetIDsRange, removeElemsIDs, numElem);
        if (!tetIDsRange.empty())
        {
            //internalForce.setZero();
            m_isotropicHyperelasticFEM->GetEnergyAndForceAndTangentStiffnessMatrixHelperPrologue(data, NULL, internalForce.data(), NULL, computation_mode); // resets the energy, internal forces and/or tangent stiffness matrix to zero
            for (size_t i = 0; i < tetIDsRange.size(); ++i)
            {
                // let tetIDsRange.at(i)[0]=n1, tetIDsRange.at(i)[1]=n2, then the internal force vector is calculated for elements (i=n1;i<n2;++i) ;
                int code = m_isotropicHyperelasticFEM->GetEnergyAndForceAndTangentStiffnessMatrixHelperWorkhorse(tetIDsRange.at(i)[0], tetIDsRange.at(i)[1], data, NULL, internalForce.data(), NULL, computation_mode);
            }
        }
        else
        {
            LOG(WARNING) << "class IsotropicHyperelasticFEForceModel::getModifiedInternalForce error: element range empty";
            return;
        }
    }

    ///
    /// \brief Get the tangent stiffness matrix
    ///
    inline void getTangentStiffnessMatrix(const Vectord& u, SparseMatrixd& tangentStiffnessMatrix) override
    {
        double *data = const_cast<double*>(u.data());
        m_isotropicHyperelasticFEM->GetTangentStiffnessMatrix(data, m_vegaTangentStiffnessMatrix.get());
        InternalForceModel::modifyValuesFromMatrix(m_vegaTangentStiffnessMatrix, tangentStiffnessMatrix.valuePtr(),InternalForceModel::operationType::Replace);
    }

    ///
    /// \brief Get the modified tangent stiffness matrix (to simulate cutting)
    ///
    void getTangentStiffnessMatrix(const Vectord& u, SparseMatrixd& tangentStiffnessMatrix, const std::vector<size_t>& removeElemsIDs)
    {
        const auto computation_mode = vega::IsotropicHyperelasticFEM::COMPUTE_TANGENTSTIFFNESSMATRIX;
        double *data = const_cast<double*>(u.data());
        // implmentation based on the assumption that the removeTets vector, and tetIDs are in sorted form, and moreover in an ascending order.
        const auto numElem = m_isotropicHyperelasticFEM->GetTetMesh()->getNumElements();

        std::vector< std::vector<size_t> > tetIDsRange;
        InternalForceModel::calculateRanges(tetIDsRange, removeElemsIDs, numElem);
        if (!tetIDsRange.empty())
        {
            m_isotropicHyperelasticFEM->GetEnergyAndForceAndTangentStiffnessMatrixHelperPrologue(data, NULL, NULL, m_vegaTangentStiffnessMatrix.get(), computation_mode);
            InternalForceModel::modifyValuesFromMatrix(m_vegaTangentStiffnessMatrix, tangentStiffnessMatrix.valuePtr(), InternalForceModel::operationType::Replace);
            for (size_t i = 0; i < tetIDsRange.size(); ++i)
            {
                // let tetIDRanges.at(i)[0]=n1, tetIDRanges.at(i)[1]=n2, then the stiffness is calculated for elements (i=n1;i<n2;++i) ;
                int code = m_isotropicHyperelasticFEM->GetEnergyAndForceAndTangentStiffnessMatrixHelperWorkhorse(tetIDsRange.at(i)[0], tetIDsRange.at(i)[1], data, NULL, NULL, m_vegaTangentStiffnessMatrix.get(), computation_mode);
            }
            InternalForceModel::modifyValuesFromMatrix(m_vegaTangentStiffnessMatrix, tangentStiffnessMatrix.valuePtr(), InternalForceModel::operationType::Replace);
        }
        else
        {
            LOG(WARNING) << "class IsotropicHyperelasticFEForceModel::getModifiedTangentStiffnessMatrix error: element range empty";
            return;
        }
    }

    ///
    /// \brief Get the tangent stiffness matrix topology
    ///
    inline void getTangentStiffnessMatrixTopology(vega::SparseMatrix** tangentStiffnessMatrix) override
    {
        m_isotropicHyperelasticFEM->GetStiffnessMatrixTopology(tangentStiffnessMatrix);
    }

    ///
    /// \brief Get the tangent stiffness matrix and internal force
    ///
    inline void getForceAndMatrix(const Vectord& u, Vectord& internalForce, SparseMatrixd& tangentStiffnessMatrix) override
    {
        double *data = const_cast<double*>(u.data());
        m_isotropicHyperelasticFEM->GetForceAndTangentStiffnessMatrix(data, internalForce.data(), m_vegaTangentStiffnessMatrix.get());
        InternalForceModel::modifyValuesFromMatrix(m_vegaTangentStiffnessMatrix, tangentStiffnessMatrix.valuePtr(),InternalForceModel::operationType::Replace);
    }

    ///
    /// \brief Set the tangent stiffness matrix
    ///
    inline void setTangentStiffness(std::shared_ptr<vega::SparseMatrix> K) override
    {
        m_vegaTangentStiffnessMatrix = K;
    }

    ///
    /// \brief Initializes (Set size and capacity) vonMisesStressArray data member of the imstkIsotropicHyperelasticFEM class
    ///
    void initVonMisesStressArray()
    {
        m_isotropicHyperelasticFEM->InitializeVonMisesStressArray(m_isotropicHyperelasticFEM->GetTetMesh()->getNumVertices());
    }

    ///
    /// \brief Sets vonMisesStressArray to zero
    ///
    void setVonMisesStressArrayToZero()
    {
        m_isotropicHyperelasticFEM->setZeroVonMisesStressArray();
    }

    ///
    /// \brief Get vonMisesStressArray
    ///
    void getVonMisesStressArray(std::vector<double> &vmsArray)
    {
        vmsArray = m_isotropicHyperelasticFEM->getVonMisesStress();
    }

protected:
    std::shared_ptr<vega::IsotropicHyperelasticFEM> m_isotropicHyperelasticFEM; ///>
    std::shared_ptr<vega::IsotropicMaterial> m_isotropicMaterial;               ///>
    std::shared_ptr<vega::SparseMatrix> m_vegaTangentStiffnessMatrix;           ///>
};

} // imstk

#endif // imstkHyperElasticFEMForceModel_h
