/*=========================================================================

   Library: iMSTK

   Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
   & Imaging in Medicine, Rensselaer Polytechnic Institute.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0.txt

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

   =========================================================================*/

#ifndef imstkInternalForceModel_h
#define imstkInternalForceModel_h

#include "g3log/g3log.hpp"

#include "imstkMath.h"

//vega
#include "forceModel.h"

namespace imstk
{

///
/// \class InternalForceModel
///
/// \brief Base class for internal force model within the following context:
///
///
class InternalForceModel
{

public:
    ///
    /// \brief Types of operation on a double data array based on the entries in a vega matrix.
    ///
    enum class operationType
    {
        Replace,
        Add,
        Subrtract,
        PruneAllNonZero
    };

    ///
    /// \brief Constructor
    ///
    InternalForceModel(){};

    ///
    /// \brief Destructor
    ///
    virtual ~InternalForceModel() = default;

    ///
    /// \brief Get the internal force given the present state
    ///
    virtual void getInternalForce(const Vectord& u, Vectord& internalForce) = 0;
    virtual void getInternalForce(const Vectord& u, Vectord& internalForce, const std::vector<size_t> & removeTets) {};

    ///
    /// \brief Return the tangent stiffness matrix the present state
    ///
    virtual void getTangentStiffnessMatrix(const Vectord& u, SparseMatrixd& tangentStiffnessMatrix) = 0;
    virtual void getTangentStiffnessMatrix(const Vectord& u, SparseMatrixd& tangentStiffnessMatrix, const std::vector<size_t> & removeTets) {};

    ///
    /// \brief Get the tangent stiffness matrix
    ///
    void getModifiedTangentStiffnessMatrix(const Vectord& u, SparseMatrixd& tangentStiffnessMatrix, const std::vector<size_t> & removeTets);

    ///
    /// \brief Return the tangent stiffness matrix the present state
    ///
    virtual void getTangentStiffnessMatrixTopology(vega::SparseMatrix** tangentStiffnessMatrix) = 0;

    ///
    /// \brief Return both internal force and tangent stiffness matrix given the present state
    ///
    virtual void getForceAndMatrix(const Vectord& u, Vectord& internalForce, SparseMatrixd& tangentStiffnessMatrix) = 0;

    ///
    /// \brief Given the linearized array of data from the Vega matrix, updates the values of the Eigen sparse matrix, based on the operationType.
    ///        "values" represents non-const pointer to the array of values of a row-major dynamic size (compressed) sparse matrix of doubles,
    ///         with same sparsity pattern as "vegaMatrix".
    static void modifyValuesFromMatrix(std::shared_ptr<vega::SparseMatrix> vegaMatrix, double *values, const InternalForceModel::operationType & opType);

    ///
    /// \brief Sets tangent stiffness matrix
    ///
    virtual void setTangentStiffness(std::shared_ptr<vega::SparseMatrix> K) = 0;

    ///
    /// \brief Calculate ranges
    ///
    static void calculateRanges(std::vector< std::vector<size_t> > & tetIDRanges, const std::vector<size_t>& removeElemsIDs, const int elemCount);

    ///
    /// \brief Initialize, set zero, and get method for von Mises array
    ///
    virtual void initVonMisesStressArray() {};
    virtual void setVonMisesStressArrayToZero() {};
    virtual void getVonMisesStressArray(std::vector<double> & vmsArray) {};

};

} //imstk

#endif // ifndef imstkTimeIntegrator_h
