/*=========================================================================

Library: iMSTK

Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
& Imaging in Medicine, Rensselaer Polytechnic Institute.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0.txt

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

=========================================================================*/

#include "imstkInternalForceModel.h"

namespace imstk
{

void
InternalForceModel::modifyValuesFromMatrix(std::shared_ptr<vega::SparseMatrix> vegaMatrix, double *values, const InternalForceModel::operationType & opType)
{
    auto rowLengths = vegaMatrix->GetRowLengths();
    auto nonZeroValues = vegaMatrix->GetEntries();

    // Flatten the internal non-zeros value array and store it in "values array".
    int offset = 0;

    for (int row = 0, end = vegaMatrix->GetNumRows(); row < end; ++row)
    {
        for (int j = 0, end_j = rowLengths[row]; j < end_j; ++j)
        {
            if (opType == InternalForceModel::operationType::Replace)
            {
                values[j + offset] = nonZeroValues[row][j];
            }
            else if (opType == InternalForceModel::operationType::Add)
            {
                values[j + offset] += nonZeroValues[row][j];
            }
            else if (opType == InternalForceModel::operationType::Subrtract)
            {
                values[j + offset] -= nonZeroValues[row][j];
            }
            else if (opType == InternalForceModel::operationType::PruneAllNonZero)
            {
                values[j + offset] = 0.0;
            }
            else
            {
                LOG(WARNING) << "InternalForceModel::modifyValuesFromMatrix error: operation type not defined/implemented!";
                return;
            }
        }

        offset += rowLengths[row];
    }
}

void
InternalForceModel::calculateRanges(std::vector< std::vector<size_t> > & tetIDsRange, const std::vector<size_t>& removedElemsIDs, const int elemCount)
{
    tetIDsRange.clear();// start a fresh !

    // generate ranges to be supplied for the calculation of the stiffness matrix
    size_t begRange = 0;
    size_t endRange;
    std::vector<size_t> iRange;
    iRange.push_back(0); // Assuming Id of the first element is 0 in the tetrahedral mesh
    //Exmaple:
    //let totalElem =10 then
    //  the array from which we calculate the ranges is [0,1,..,totalElem]
    // removeElemsIDs is the ID of the elements to be removed
    //if removeElemsIDs = { 0 };    output: (1, 10));
    //if removeElemsIDs = { 9 };    output: (0, 9), (10);
    //if removeElemsIDs = { 2 };    output: (0, 2), (3, 10));
    //if removeElemsIDs = { 3, 5 }; output: (0, 3), (4, 5), (6, 10));
    //if removeElemsIDs = {};       output: (0, 10);
    //if removeElemsIDs = {0,..,9}; output: (10);

    auto removedElems = removedElemsIDs; // make a copy
    std::sort(removedElems.begin(), removedElems.end());// sort the elements
    if (elemCount != removedElems.size())
    {
        for (const auto & iElem : removedElems)
        {
            endRange = iElem;
            if (endRange != begRange)
            {
                iRange.push_back(endRange);
                tetIDsRange.push_back(iRange);
            }
            begRange = endRange + 1;
            iRange.clear();
            iRange.push_back(begRange);
        }
        if (begRange != elemCount)
        {
            iRange.push_back(elemCount);
            tetIDsRange.push_back(iRange);
        }
    }
    else
    {
        LOG(WARNING) << "InternalForceModel::calculateRanges error: illegal arguments to the method";
        return;
    }
}

} // imstk
