/*=========================================================================

   Library: iMSTK

   Copyright (c) Kitware, Inc. & Center for Modeling, Simulation,
   & Imaging in Medicine, Rensselaer Polytechnic Institute.

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0.txt

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

   =========================================================================*/

#ifndef imstkPickingCH_h
#define imstkPenaltyCH_h

// std library
#include <memory>

// imstk
#include "imstkCollisionHandling.h"
#include "imstkNonlinearSystem.h"

namespace imstk
{

class CollidingObject;
class DeformableObject;
class CollisionData;

///
/// \class PickingCH
///
/// \brief Implements nodal picking
///
class PickingCH : public CollisionHandling
{
public:

    ///
    /// \brief Constructor
    ///
    PickingCH(const Side& side,
              const CollisionData& colData,
              std::shared_ptr<CollidingObject> obj,
              std::shared_ptr<NonLinearSystem> nlSystem) :
        CollisionHandling(Type::NodalPicking, side, colData),
        m_nlSystem(nlSystem),
        m_object(obj){}

    PickingCH() = delete;

    ///
    /// \brief Destructor
    ///
    ~PickingCH() = default;

    ///
    /// \brief Compute forces based on collision data
    ///
    void handleCollision() override
    {
        if (auto deformableObj = std::dynamic_pointer_cast<DeformableObject>(m_object))
        {
            this->addPickConstraints(deformableObj);
        }
        else
        {
            LOG(WARNING) << "PickingCH::handleCollision error: "
                << "no picking collision handling available for " << m_object->getName();
        }
    }

    ///
    /// \brief Add LPC constraints for the node that is picked
    ///
    void addPickConstraints(std::shared_ptr<DeformableObject> deformableObj)
    {
        if (m_colData.NodePickData.empty())
        {
            return;
        }

        if (deformableObj == nullptr)
        {
            LOG(WARNING) << "PenaltyRigidCH::addPickConstraints error: "
                << m_object->getName() << " is not a deformable object.";
            return;
        }

        auto lpcConstraints = m_nlSystem->getDynamicLinearProjectors();
        lpcConstraints->clear();

        // If collision data, append LPC constraints
        for (const auto& CD : m_colData.NodePickData)
        {
            auto s = LinearProjectionConstraint(CD., false);
            s.setProjectorToDirichlet(nodeId);
            s.setValue(Vec3d(0.001, 0, 0));
            projList.push_back(s);

            lpcConstraints->push_back(s);
        }
    }

private:

    std::shared_ptr<CollidingObject> m_object;      ///> Deformable object
    std::shared_ptr<NonLinearSystem> m_nlSystem;    ///> Nonlinear system

};

}

#endif // ifndef imstkPenaltyCH_h