// This file is part of the SimMedTK project.
// Copyright (c) Center for Modeling, Simulation, and Imaging in Medicine,
//                        Rensselaer Polytechnic Institute
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//---------------------------------------------------------------------------
//
// Authors:
//
// Contact:
//---------------------------------------------------------------------------

#ifndef SM_IMPLICIT_EULER_LCP
#define SM_IMPLICIT_EULER_LCP

#include "implicitNewmarkSparse.h"

// Eigen includes
#include "Eigen/Geometry"

///
/// \brief A class to timestep large sparse dynamics using implicit backward Euler.
///
class ImplicitBackwardEulerSparseLcp : public ImplicitNewmarkSparse
{
public:

    ///
    /// \brief constrainedDOFs is an integer array of degrees of freedom that are to be
    /// fixed to zero (e.g., to permanently fix a vertex in a deformable simulation)
    /// constrainedDOFs are 0-indexed (separate DOFs for x,y,z), and must be pre-sorted
    /// (ascending) numThreads applies only to the PARDISO and SPOOLES solvers;
    /// if numThreads > 0, the sparse linear solves are multi-threaded;
    /// default: 0 (use single-threading)
    ///
    ImplicitBackwardEulerSparseLcp(int r,
                                   double timestep,
                                   SparseMatrix * massMatrix,
                                   ForceModel * forceModel,
                                   int positiveDefiniteSolver = 0,
                                   int numConstrainedDOFs = 0,
                                   int * constrainedDOFs = NULL,
                                   double dampingMassCoef = 0.0,
                                   double dampingStiffnessCoef = 0.0,
                                   int maxIterations = 1,
                                   double epsilon = 1E-6,
                                   int numSolverThreads = 0);

    ///
    /// \brief destructor
    ///
    virtual ~ImplicitBackwardEulerSparseLcp();

	///
	/// \brief sets q, and (optionally) qvel
	/// returns 0
    ///
    virtual int SetState(double * q, double * qvel=NULL);

    ///
    /// \brief Assemble effective matrices and solve the resulting system
    ///
    virtual int DoTimestep();

	///
	/// \brief
	///
    void assignSolution(std::vector<double>& v);

    int getColIndex(int i, int j);
    ///
    /// \brief Assemble effective matrices
    ///
    int computeEffectiveSystemMatrixAndForce();

    ///
    /// \brief Returns effective force (rhs) of the linear system
    ///
    void getEffectiveForce(std::vector<double>& v);

    ///
    /// \brief Returns the system matrix with diagonal 3x3 sub-matrices removed
    ///
    void getOffDiagonalMatrix(SparseMatrix& m, std::vector<double>& BDarray);
};

#endif //SM_IMPLICIT_EULER_LCP