// This file is part of the SimMedTK project.
// Copyright (c) Center for Modeling, Simulation, and Imaging in Medicine,
//                        Rensselaer Polytechnic Institute
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//
//---------------------------------------------------------------------------
//
// Authors:
//
// Contact:
//---------------------------------------------------------------------------

#include <iostream>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "matrixIO.h"
#include "performanceCounter.h"
#include "insertRows.h"
#include "implicitBackwardEulerSparseLcp.h"
#include "linearSolver.h"

ImplicitBackwardEulerSparseLcp::ImplicitBackwardEulerSparseLcp(int r,
    double timestep,
    SparseMatrix * massMatrix_,
    ForceModel * forceModel_,
    int positiveDefiniteSolver_,
    int numConstrainedDOFs_,
    int * constrainedDOFs_,
    double dampingMassCoef,
    double dampingStiffnessCoef,
    int maxIterations,
    double epsilon,
    int numSolverThreads_) : ImplicitNewmarkSparse(r,
    timestep,
    massMatrix_,
    forceModel_,
    positiveDefiniteSolver_,
    numConstrainedDOFs_,
    constrainedDOFs_,
    dampingMassCoef,
    dampingStiffnessCoef,
    maxIterations,
    epsilon,
    0.25,
    0.5,
    numSolverThreads_)
{
}

ImplicitBackwardEulerSparseLcp::~ImplicitBackwardEulerSparseLcp()
{
}

int ImplicitBackwardEulerSparseLcp::SetState(double * q_, double * qvel_)
{
    memcpy(q, q_, sizeof(double)*r);

    if (qvel_ != NULL)
        memcpy(qvel, qvel_, sizeof(double)*r);

    return 0;
}

int ImplicitBackwardEulerSparseLcp::DoTimestep()
{
    int numIter = 0;

    double error0 = 0; // error after the first step
    double errorQuotient;

    // store current amplitudes and set initial guesses for qaccel, qvel
    for (int i = 0; i < r; i++)
    {
        qaccel_1[i] = qaccel[i] = 0; // acceleration is actually not used in this integrator
        q_1[i] = q[i];
        qvel_1[i] = qvel[i];
    }

    do
    {
        forceModel->GetForceAndMatrix(q, internalForces, tangentStiffnessMatrix);

        // scale internal forces
        for (int i = 0; i < r; i++)
            internalForces[i] *= internalForceScalingFactor;

        // scale stiffness matrix
        *tangentStiffnessMatrix *= internalForceScalingFactor;

        memset(qresidual, 0, sizeof(double) * r);

        if (useStaticSolver)
        {
            // fint + K * qdelta = fext

            // add externalForces, internalForces
            for (int i = 0; i < r; i++)
            {
                qresidual[i] = externalForces[i] - internalForces[i];
                qdelta[i] = qresidual[i];
            }
        }
        else
        {
            tangentStiffnessMatrix->ScalarMultiply(dampingStiffnessCoef, rayleighDampingMatrix);
            rayleighDampingMatrix->AddSubMatrix(dampingMassCoef, *massMatrix);

            // build effective stiffness:
            // Keff = M + h D + h^2 * K
            // compute force residual, store it into aux variable qresidual
            // qresidual = h * (-D qdot - fint + fext - h * K * qdot)) // this is semi-implicit Euler
            // qresidual = M (qvel_1 - qvel) + h * (-D qdot - fint + fext - K * (q_1 - q + h qdot) )) // for fully implicit Euler

            if (numIter != 0) // can skip on first iteration (zero contribution)
            {
                // add K * (q_1 - q) to qresidual (will multiply by -h later)
                for (int i = 0; i < r; i++)
                    buffer[i] = q_1[i] - q[i];
                tangentStiffnessMatrix->MultiplyVectorAdd(buffer, qresidual);
            }

            //add mass matrix and damping matrix to tangentStiffnessMatrix
            *tangentStiffnessMatrix *= timestep;

            *tangentStiffnessMatrix += *rayleighDampingMatrix;
            tangentStiffnessMatrix->AddSubMatrix(1.0, *dampingMatrix, 1); // at this point, tangentStiffnessMatrix = h * K + D
            tangentStiffnessMatrix->MultiplyVectorAdd(qvel, qresidual);
            *tangentStiffnessMatrix *= timestep;
            tangentStiffnessMatrix->AddSubMatrix(1.0, *massMatrix);

            // add externalForces, internalForces
            for (int i = 0; i < r; i++)
            {
                qresidual[i] += internalForces[i] - externalForces[i];
                qresidual[i] *= -timestep;
            }

            if (numIter != 0) // can skip on first iteration (zero contribution)
            {
                // add M * (qvel_1 - qvel) to qresidual
                for (int i = 0; i < r; i++)
                    buffer[i] = qvel_1[i] - qvel[i];

                massMatrix->MultiplyVectorAdd(buffer, qresidual);
            }

            tangentStiffnessMatrix->MultiplyVectorAdd(qvel, qresidual);

            for (int i = 0; i < r; i++)
                qdelta[i] = qresidual[i];

        }

        // remove rows corresponding to fixed vertices from qdelta,
        // and store the result in bufferConstrained
        RemoveRows(r, bufferConstrained, qdelta, numConstrainedDOFs, constrainedDOFs);

        double error = 0.0;
        for (int i = 0; i < r - numConstrainedDOFs; i++)
            error += bufferConstrained[i] * bufferConstrained[i];

        // on the first iteration, compute initial error
        if (numIter == 0)
        {
            error0 = error;
            errorQuotient = 1.0;
        }
        else
        {
            // error divided by the initial error, before performing this iteration
            errorQuotient = error / error0;
        }

        if (errorQuotient < epsilon * epsilon)
            break;

        systemMatrix->AssignSuperMatrix(tangentStiffnessMatrix);

        // solve: systemMatrix * buffer = bufferConstrained

        memset(buffer, 0, sizeof(double) * r);

        if (!this->linearSolver)
        {
            std::cerr << "Error: The linear solver is not set. Attach a solver \
                                     before calling this method." << std::endl;
            return 1;
        }

        int info = this->linearSolver->SolveLinearSystem(buffer, bufferConstrained);
        if (info != 0)
        {
            printf("Error: %s sparse solver returned non-zero exit status %d.\n",
                this->linearSolver->getSolverType().c_str(), (int)info);
            exit(-1);
            return 1;
        }

        InsertRows(r, buffer, qdelta, numConstrainedDOFs, constrainedDOFs);

        // update state
        if (useStaticSolver)
        {
            for (int i = 0; i < r; i++)
            {
                q[i] += qdelta[i];
                qvel[i] = (q[i] - q_1[i]) / timestep;
            }
        }
        else
        {
            /*for(int i=0; i<r; i++)
            {
            qvel[i] += qdelta[i];
            q[i] += q_1[i] - q[i] + timestep * qvel[i];
            }*/

            for (int i = 0; i < r; i++)
            {
                qvel[i] = qdelta[i];
                q[i] += q_1[i] - q[i] + timestep * qvel[i];
            }
        }

        for (int i = 0; i < numConstrainedDOFs; i++){
            q[constrainedDOFs[i]] =
            qvel[constrainedDOFs[i]] =
            qaccel[constrainedDOFs[i]] = 0.0;
        }

        numIter++;
    } while (numIter < maxIterations);

    return 0;
}

int ImplicitBackwardEulerSparseLcp::computeEffectiveSystemMatrixAndForce()
{
    int numIter = 0;

    double error0 = 0; // error after the first step
    double errorQuotient;

    // store current amplitudes and set initial guesses for qaccel, qvel
    for (int i = 0; i < r; i++)
    {
        qaccel_1[i] = qaccel[i] = 0; // acceleration is actually not used in this integrator
        q_1[i] = q[i];
        qvel_1[i] = qvel[i];
    }

    {
        forceModel->GetForceAndMatrix(q, internalForces, tangentStiffnessMatrix);

        // scale internal forces
        for (int i = 0; i < r; i++)
            internalForces[i] *= internalForceScalingFactor;

        // scale stiffness matrix
        *tangentStiffnessMatrix *= internalForceScalingFactor;

        memset(qresidual, 0, sizeof(double) * r);

        if (useStaticSolver)
        {
            // fint + K * qdelta = fext

            // add externalForces, internalForces
            for (int i = 0; i < r; i++)
            {
                qresidual[i] = externalForces[i] - internalForces[i];
                qdelta[i] = qresidual[i];
            }
        }
        else
        {
            tangentStiffnessMatrix->ScalarMultiply(dampingStiffnessCoef, rayleighDampingMatrix);
            rayleighDampingMatrix->AddSubMatrix(dampingMassCoef, *massMatrix);

            // build effective stiffness:
            // Keff = M + h D + h^2 * K
            // compute force residual, store it into aux variable qresidual
            // qresidual = h * (-D qdot - fint + fext - h * K * qdot)) // this is semi-implicit Euler
            // qresidual = M (qvel_1 - qvel) + h * (-D qdot - fint + fext - K * (q_1 - q + h qdot) )) // for fully implicit Euler

            if (numIter != 0) // can skip on first iteration (zero contribution)
            {
                // add K * (q_1 - q) to qresidual (will multiply by -h later)
                for (int i = 0; i < r; i++)
                    buffer[i] = q_1[i] - q[i];
                tangentStiffnessMatrix->MultiplyVectorAdd(buffer, qresidual);
            }

            //add mass matrix and damping matrix to tangentStiffnessMatrix
            *tangentStiffnessMatrix *= timestep;

            *tangentStiffnessMatrix += *rayleighDampingMatrix;
            tangentStiffnessMatrix->AddSubMatrix(1.0, *dampingMatrix, 1); // at this point, tangentStiffnessMatrix = h * K + D
            tangentStiffnessMatrix->MultiplyVectorAdd(qvel, qresidual);
            *tangentStiffnessMatrix *= timestep;
            tangentStiffnessMatrix->AddSubMatrix(1.0, *massMatrix);

            // add externalForces, internalForces
            for (int i = 0; i < r; i++)
            {
                qresidual[i] += internalForces[i] - externalForces[i];
                qresidual[i] *= -timestep;
            }

            if (numIter != 0) // can skip on first iteration (zero contribution)
            {
                // add M * (qvel_1 - qvel) to qresidual
                for (int i = 0; i < r; i++)
                    buffer[i] = qvel_1[i] - qvel[i];

                massMatrix->MultiplyVectorAdd(buffer, qresidual);
            }

            tangentStiffnessMatrix->MultiplyVectorAdd(qvel, qresidual);

            for (int i = 0; i < r; i++)
                qdelta[i] = qresidual[i];

        }

        // remove rows corresponding to fixed vertices from qdelta,
        // and store the result in bufferConstrained
        RemoveRows(r, bufferConstrained, qdelta, numConstrainedDOFs, constrainedDOFs);

        double error = 0.0;
        for (int i = 0; i < r - numConstrainedDOFs; i++)
            error += bufferConstrained[i] * bufferConstrained[i];

        // on the first iteration, compute initial error
        if (numIter == 0)
        {
            error0 = error;
            errorQuotient = 1.0;
        }
        else
        {
            // error divided by the initial error, before performing this iteration
            errorQuotient = error / error0;
        }

        /*if (errorQuotient < epsilon * epsilon)
            break;*/

        systemMatrix->AssignSuperMatrix(tangentStiffnessMatrix);

        // solve: systemMatrix * buffer = bufferConstrained

        memset(buffer, 0, sizeof(double) * r);

    }

    return 0;
}

void ImplicitBackwardEulerSparseLcp::getEffectiveForce(std::vector<double>& v)
{
    assert(v.size()==r);

    for (int i = 0; i < r; i++)
    {
        v[i] = bufferConstrained[i];
    }
}

void ImplicitBackwardEulerSparseLcp::assignSolution(std::vector<double>& v)
{
    for (int i = 0; i < r; i++)
    {
        buffer[i] = v[i];
        qdelta[i] = v[i];
    }

    //InsertRows(r, buffer, qdelta, numConstrainedDOFs, constrainedDOFs);

    for (int i = 0; i < r; i++)
    {
        qvel[i] = qdelta[i];
        q[i] += q_1[i] - q[i] + timestep * qvel[i];
    }

    for (int i = 0; i < numConstrainedDOFs; i++)
    {
        q[constrainedDOFs[i]] =
            qvel[constrainedDOFs[i]] =
            qaccel[constrainedDOFs[i]] = 0.0;
    }
}

int ImplicitBackwardEulerSparseLcp::getColIndex(int row, int col)
{
    for (int i = 0; i < systemMatrix->GetRowLength(row); i++)
    {
        if (systemMatrix->GetColumnIndex(row,i)==col)
        {
            return i;
        }
    }
    std::cout << "Error -1:" << row<< "\t" <<col<< "\n";
    return -1;
}

void ImplicitBackwardEulerSparseLcp::getOffDiagonalMatrix(SparseMatrix& m,
                                                          std::vector<double>& BDarray)
{
    int j;
    Eigen::Matrix3d aii;

    m = *systemMatrix;

    int size = m.Getn();

    for (int i = 0; i < size; i++)
    {

        if (i % 3 == 0)
        {
            // compute and store diagonal block in a linear array
            aii(0, 0) = systemMatrix->GetEntry(i, getColIndex(i, i));
            aii(0, 1) = systemMatrix->GetEntry(i, getColIndex(i, i+1));
            aii(0, 2) = systemMatrix->GetEntry(i, getColIndex(i, i+2));

            aii(1, 0) = systemMatrix->GetEntry(i + 1, getColIndex(i+1, i));
            aii(1, 1) = systemMatrix->GetEntry(i + 1, getColIndex(i+1, i+1));
            aii(1, 2) = systemMatrix->GetEntry(i + 1, getColIndex(i+2, i+2));

            aii(2, 0) = systemMatrix->GetEntry(i + 2, getColIndex(i+2, i));
            aii(2, 1) = systemMatrix->GetEntry(i + 2, getColIndex(i+2, i+1));
            aii(2, 2) = systemMatrix->GetEntry(i + 2, getColIndex(i+2, i+2));

            aii = aii.inverse();

            j = 3 * i;

            BDarray.at(j + 0) = aii(0, 0);
            BDarray.at(j + 1) = aii(0, 1);
            BDarray.at(j + 2) = aii(0, 2);

            BDarray.at(j + 3) = aii(1, 0);
            BDarray.at(j + 4) = aii(1, 1);
            BDarray.at(j + 5) = aii(1, 2);

            BDarray.at(j + 6) = aii(2, 0);
            BDarray.at(j + 7) = aii(2, 1);
            BDarray.at(j + 8) = aii(2, 2);

            // remove diagonal block
            /*m.SetEntry(i, getColIndex(i, i), 0.0);
            m.SetEntry(i, getColIndex(i, i + 1), 0.0);
            m.SetEntry(i, getColIndex(i, i + 2), 0.0);

            m.SetEntry(i + 1, getColIndex(i + 1, i), 0.0);
            m.SetEntry(i + 1, getColIndex(i + 1, i + 1), 0.0);
            m.SetEntry(i + 1, getColIndex(i + 1, i + 2), 0.0);

            m.SetEntry(i + 2, getColIndex(i + 2, i), 0.0);
            m.SetEntry(i + 2, getColIndex(i + 2, i + 1), 0.0);
            m.SetEntry(i + 2, getColIndex(i + 2, i + 2), 0.0);*/

        }
    }
}
