import sys
import vtk
from vtk.test import Testing

class PatchInterpolation(Testing.vtkTest):
    """Test interpolation of values on various-dimensional patches."""

    def verifyRationalCurve(self, ctrlPts, f, prange, ns, name):
        """Evaluate points along a curve.

        ctrlPts  is a list of 4-tuple control points defining a rational
                 Bezier curve (the 4-th coordinate is the weight).
        f        is an implicit function that takes a point in and returns
                 zero if it lies on the curve.
        prange   is list containing minimum and maximum parameter values.
        ns       is the number of points to sample between the min/max.
        """
        import vtk
        npts = len(ctrlPts)
        degree = [npts - 1,0,0]
        self.assertGreater(degree[0], 0,
            'Need at least 2 control points, got {n}'.format(n=npts))

        cpt = vtk.vtkDoubleArray()
        cpt.SetNumberOfComponents(len(ctrlPts[0]))
        cpt.SetNumberOfTuples(npts)
        [cpt.SetTuple(i, ctrlPts[i]) for i in range(npts)]
        erp = vtk.vtkPatchInterpolation()
        pts = vtk.vtkPoints()
        pd = vtk.vtkPolyData()
        ln = vtk.vtkCellArray()
        pd.SetPoints(pts)
        pd.SetLines(ln)
        params = [0., 0., 0.]
        delta = (prange[1] - prange[0]) / (ns - 1.)
        for r in [prange[0] + delta * x for x in range(ns)]:
            params[0] = r
            erp.InterpolateOnPatch(pts.GetData(), 1, cpt, degree, params)

            # Test that f(xyz) = 0 to within 6 decimal places. We use 6 places
            # because vtkPoints stores values as 'float' by default and floats
            # are at places only accurate to 7 or 8 decimal digits.
            xyz = pts.GetPoint(pts.GetNumberOfPoints() - 1)
            self.assertAlmostEqual(f(xyz), 0., 6,
                'Point does not satisfy implicit function; f{p} = {f} != 0'.format(p=xyz, f=f(xyz)))

        # For debugging, uncomment the lines below. They will
        # write a VTK polydata file containing the output points.
        ##   Insert line segments connecting the points and
        ##   write out the resulting polydata to a file:
        #[ln.InsertNextCell(2, [i, i+1]) for i in range(ns - 1)]
        #wri = vtk.vtkPolyDataWriter()
        #wri.SetInputDataObject(pd)
        #wri.SetFileName('bezier-{shape}.vtk'.format(shape=name))
        #wri.Write()

    def testRationalLine(self):
        self.verifyRationalCurve(
            [[0,0,0,1], [1,1,1,1]],
            lambda x: x[0] - x[1],
            [0, 1], 3, 'line')

    def testRationalCircle(self):
        self.verifyRationalCurve(
            [[1,0,0,1], [1,1,0,1], [0,2,0,2]],
            lambda x: 1. - x[0]**2 - x[1]**2,
            [0, 1], 21, 'circle')

if __name__ == '__main__':
    Testing.main([(PatchInterpolation,'test'),])
