/*=========================================================================

  Program:   Visualization Toolkit
  Module:    FTensors.hh
  Language:  C++
  Date:      $Date$
  Version:   $Revision$

This file is part of the Visualization Toolkit. No part of this file
or its contents may be copied, reproduced or altered in any way
without the express written consent of the authors.

Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen 1993, 1994 

=========================================================================*/
// .NAME vtkFloatTensors - floating point representation of tensor data
// .SECTION Description
// vtkFloatTensors is a concrete implementation of vtkTensors. Tensor values
// are represented using float values.

#ifndef __vtkFloatTensors_h
#define __vtkFloatTensors_h

#include "Tensors.hh"
#include "FArray.hh"

class vtkFloatTensors : public vtkTensors
{
public:
  vtkFloatTensors() {};
  vtkFloatTensors(const vtkFloatTensors& ft);
  vtkFloatTensors(int sz, int d=3, int ext=1000);
  ~vtkFloatTensors() {};
  int Allocate(const int sz, const int dim=3, const int ext=1000);
  void Initialize() {this->T.Initialize();};
  char *GetClassName() {return "vtkFloatTensors";};

  // vtkTensors interface
  vtkTensors *MakeObject(int sze, int d=3, int ext=1000);
  char *GetDataType() {return "float";};
  int GetNumberOfTensors();
  void Squeeze() {this->T.Squeeze();};
  vtkTensor &GetTensor(int i);
  void GetTensor(int i,vtkTensor &t) {this->vtkTensors::GetTensor(i,t);};
  void SetTensor(int i, vtkTensor &t);
  void InsertTensor(int i, vtkTensor &t);
  int InsertNextTensor(vtkTensor &t);

  // miscellaneous
  float *GetPtr(const int id);
  float *WritePtr(const int id, const int number);
  void WrotePtr();
  vtkFloatTensors &operator=(const vtkFloatTensors& ft);
  void operator+=(const vtkFloatTensors& ft) {this->T += ft.T;};
  void Reset() {this->T.Reset();};

protected:
  vtkFloatArray T;
};

// Description:
// Get pointer to array of data starting at data position "id".
inline float *vtkFloatTensors::GetPtr(const int id)
{
  return this->T.GetPtr(id);
}

// Description:
// Get pointer to data array. Useful for direct writes of data. MaxId is 
// bumped by number (and memory allocated if necessary). Id is the 
// location you wish to write into; number is the number of tensors to 
// write. Use the method WrotePtr() to mark completion of write.
// Make sure the dimension of the tensor is set prior to issuing this call.
inline float *vtkFloatTensors::WritePtr(const int id, const int number)
{
  return this->T.WritePtr(id,this->Dimension*this->Dimension*number);
}

// Description:
// Terminate direct write of data. Although dummy routine now, reserved for
// future use.
inline void vtkFloatTensors::WrotePtr() {}


inline vtkFloatTensors::vtkFloatTensors(const vtkFloatTensors& ft) 
{
  this->T = ft.T;this->Dimension = ft.Dimension;
}

inline vtkFloatTensors::vtkFloatTensors(int sz, int d, int ext):
T(d*d*sz,d*d*ext) 
{
  this->Dimension=d;
}

inline int vtkFloatTensors::Allocate(const int sz, const int dim,const int ext) 
{
  return this->T.Allocate(dim*dim*sz,dim*dim*ext);
}

inline int vtkFloatTensors::GetNumberOfTensors() 
{
  return (this->T.GetMaxId()+1)/(this->Dimension*this->Dimension);
}

inline void vtkFloatTensors::SetTensor(int id, vtkTensor &t) 
{
  id *= this->Dimension*this->Dimension; 
  
  for (int j=0; j < this->Dimension; j++) 
    for (int i=0; i < this->Dimension; i++) 
      this->T[id+i+t.GetDimension()*j] = t.GetComponent(i,j);
}

inline void vtkFloatTensors::InsertTensor(int id, vtkTensor &t) 
{
  id *= this->Dimension*this->Dimension; 
  
  for (int j=0; j < this->Dimension; j++) 
    for (int i=0; i < this->Dimension; i++) 
      this->T.InsertValue(id+i+t.GetDimension()*j,t.GetComponent(i,j));
}

inline int vtkFloatTensors::InsertNextTensor(vtkTensor &t) 
{
  int id = this->GetNumberOfTensors() + 1;
  for (int j=0; j < this->Dimension; j++) 
    for (int i=0; i < this->Dimension; i++) 
      this->T.InsertNextValue(t.GetComponent(i,j));

  return id;
}

#endif
