/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkAGraymap.hh
  Language:  C++
  Date:      $Date$
  Version:   $Revision$


Copyright (c) 1993-1995 Ken Martin, Will Schroeder, Bill Lorensen.

This software is copyrighted by Ken Martin, Will Schroeder and Bill Lorensen.
The following terms apply to all files associated with the software unless
explicitly disclaimed in individual files. This copyright specifically does
not apply to the related textbook "The Visualization Toolkit" ISBN
013199837-4 published by Prentice Hall which is covered by its own copyright.

The authors hereby grant permission to use, copy, and distribute this
software and its documentation for any purpose, provided that existing
copyright notices are retained in all copies and that this notice is included
verbatim in any distributions. Additionally, the authors grant permission to
modify this software and its documentation for any purpose, provided that
such modifications are not distributed without the explicit consent of the
authors and that existing copyright notices are retained in all copies. Some
of the algorithms implemented by this software are patented, observe all
applicable patent law.

IN NO EVENT SHALL THE AUTHORS OR DISTRIBUTORS BE LIABLE TO ANY PARTY FOR
DIRECT, INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT
OF THE USE OF THIS SOFTWARE, ITS DOCUMENTATION, OR ANY DERIVATIVES THEREOF,
EVEN IF THE AUTHORS HAVE BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

THE AUTHORS AND DISTRIBUTORS SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE, AND NON-INFRINGEMENT.  THIS SOFTWARE IS PROVIDED ON AN
"AS IS" BASIS, AND THE AUTHORS AND DISTRIBUTORS HAVE NO OBLIGATION TO PROVIDE
MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS, OR MODIFICATIONS.


=========================================================================*/
// .NAME vtkAGraymap - scalar data in intensity + alpha (grayscale  + texture) form
// .SECTION Description
// vtkAGraymap is a concrete implementation of vtkScalars. vtkAGraymap 
// represents scalars using using one value for intensity (grayscale) and
// one value for alpha (transparency). The intensity and alpha values range 
// between (0,255) (i.e., an unsigned char value).

#ifndef __vtkAGraymap_h
#define __vtkAGraymap_h

#include "vtkColorScalars.hh"
#include "vtkUnsignedCharArray.hh"

class vtkAGraymap : public vtkColorScalars 
{
public:
  vtkAGraymap() {};
  ~vtkAGraymap() {};
  vtkAGraymap(const vtkAGraymap& fs) {this->S = fs.S;};
  vtkAGraymap(const int sz, const int ext=1000):S(2*sz,2*ext){};
  int Allocate(const int sz, const int ext=1000) {return this->S.Allocate(2*sz,2*ext);};
  void Initialize() {this->S.Initialize();};
  char *GetClassName() {return "vtkAGraymap";};

  // vtkScalar interface
  vtkScalars *MakeObject(int sze, int ext=1000);
  int GetNumberOfScalars() {return (this->S.GetMaxId()+1)/2;};
  void Squeeze() {this->S.Squeeze();};
  int GetNumberOfValuesPerScalar() {return 2;};

  // miscellaneous
  vtkAGraymap &operator=(const vtkAGraymap& fs);
  void operator+=(const vtkAGraymap& fs) {this->S += fs.S;};
  void Reset() {this->S.Reset();};
  unsigned char *GetPtr(const int id);
  unsigned char *WritePtr(const int id, const int number);
  void WrotePtr();


  // vtkColorScalar interface.
  unsigned char *GetColor(int id);
  void GetColor(int id, unsigned char rgba[4]);
  void SetColor(int id, unsigned char rgba[4]);
  void InsertColor(int id, unsigned char rgba[4]);
  int InsertNextColor(unsigned char rgba[4]);

protected:
  vtkUnsignedCharArray S;
};

// Description:
// Set a rgba color value at a particular array location. Does not do 
// range checking.
inline void vtkAGraymap::SetColor(int i, unsigned char rgba[4]) 
{
  i *= 2; 
  this->S[i] = (rgba[0] > rgba[1] ? (rgba[0] > rgba[2] ? rgba[0] : rgba[2]) :
                                    (rgba[1] > rgba[2] ? rgba[1] : rgba[2]));
  this->S[i+1] = rgba[3]; 
}

// Description:
// Insert a rgba color value at a particular array location. Does range 
// checking and will allocate additional memory if necessary.
inline void vtkAGraymap::InsertColor(int i, unsigned char rgba[4]) 
{
  this->S.InsertValue(2*i+1, rgba[3]);
  this->S[2*i] = (rgba[0] > rgba[1] ? (rgba[0] > rgba[2] ? rgba[0] : rgba[2]) :
                                      (rgba[1] > rgba[2] ? rgba[1] : rgba[2]));
}

// Description:
// Insert a rgba color value at the next available slot in the array. Will
// allocate memory if necessary.
inline int vtkAGraymap::InsertNextColor(unsigned char rgba[4]) 
{
  int id = this->S.GetMaxId() + 1;
  this->S.InsertValue(id,rgba[3]);
  this->S[id-1] = (rgba[0] > rgba[1] ? (rgba[0] > rgba[2] ? rgba[0] : rgba[2]) :
                                       (rgba[1] > rgba[2] ? rgba[1] : rgba[2]));

  return id/2;
}

// Description:
// Get pointer to array of data starting at data position "id".
inline unsigned char *vtkAGraymap::GetPtr(const int id)
{
  return this->S.GetPtr(2*id);
}

// Description:
// Get pointer to data array. Useful for direct writes of data. MaxId is 
// bumped by number (and memory allocated if necessary). Id is the 
// location you wish to write into; number is the number of scalars to 
// write. Use the method WrotePtr() to mark completion of write.
inline unsigned char *vtkAGraymap::WritePtr(const int id, const int number)
{
  return this->S.WritePtr(2*id,2*number);
}

// Description:
// Terminate direct write of data. Although dummy routine now, reserved for
// future use.
inline void vtkAGraymap::WrotePtr() {}


#endif
