/*=========================================================================

  Program:   ParaView
  Module:    TestPVFilters.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkCleanUnstructuredGrid.h"
#include "vtkClipDataSet.h"
#include "vtkContourFilter.h"
#include "vtkDataSetMapper.h"
#include "vtkGlyphSource2D.h"
#include "vtkMergeArrays.h"
#include "vtkPVGeometryFilter.h"
#include "vtkPVLODActor.h"
#include "vtkPVLinearExtrusionFilter.h"
#include "vtkPlane.h"
#include "vtkPolyData.h"
#include "vtkPolyDataMapper.h"
#include "vtkRenderWindow.h"
#include "vtkRenderer.h"
#include "vtkRibbonFilter.h"
#include "vtkTestUtilities.h"
#include "vtkTesting.h"
#include "vtkThreshold.h"
#include "vtkUnstructuredGrid.h"
#include "vtkUnstructuredGridReader.h"
#include "vtkWarpScalar.h"

int TestPVFilters(int argc, char* argv[])
{
  char* fname = vtkTestUtilities::ExpandDataFileName(argc, argv, "Testing/Data/blow.vtk");

  auto reader = vtkSmartPointer<vtkUnstructuredGridReader>::New();
  reader->SetFileName(fname);
  reader->SetScalarsName("thickness9");
  reader->SetVectorsName("displacement9");

  delete[] fname;

  auto clean = vtkSmartPointer<vtkCleanUnstructuredGrid>::New();
  clean->SetInputConnection(reader->GetOutputPort());

  auto gs = vtkSmartPointer<vtkGlyphSource2D>::New();
  gs->SetGlyphTypeToThickArrow();
  gs->SetScale(1);
  gs->FilledOff();
  gs->CrossOff();

  auto contour = vtkSmartPointer<vtkContourFilter>::New();
  contour->SetInputConnection(clean->GetOutputPort());
  contour->SetValue(0, 0.5);

  auto plane = vtkSmartPointer<vtkPlane>::New();
  plane->SetOrigin(0.25, 0, 0);
  plane->SetNormal(-1, -1, 0);

  auto clip = vtkSmartPointer<vtkClipDataSet>::New();
  clip->SetInputConnection(clean->GetOutputPort());
  clip->SetClipFunction(plane);
  clip->GenerateClipScalarsOn();
  clip->SetValue(0.5);

  auto geometry = vtkSmartPointer<vtkPVGeometryFilter>::New();
  geometry->SetInputConnection(contour->GetOutputPort());

  auto mapper = vtkSmartPointer<vtkPolyDataMapper>::New();
  mapper->SetInputConnection(geometry->GetOutputPort());

  auto actor = vtkSmartPointer<vtkPVLODActor>::New();
  actor->SetMapper(mapper);

  // Now for the PVPolyData part:
  auto ribbon = vtkSmartPointer<vtkRibbonFilter>::New();
  ribbon->SetInputConnection(contour->GetOutputPort());
  ribbon->SetWidth(0.1);
  ribbon->SetWidthFactor(5);

  auto threshold = vtkSmartPointer<vtkThreshold>::New();
  threshold->SetInputConnection(ribbon->GetOutputPort());
  threshold->SetThresholdFunction(vtkThreshold::THRESHOLD_BETWEEN);
  threshold->SetLowerThreshold(0.25);
  threshold->SetUpperThreshold(0.75);

  auto warp = vtkSmartPointer<vtkWarpScalar>::New();
  warp->SetInputConnection(threshold->GetOutputPort());
  warp->XYPlaneOn();
  warp->SetScaleFactor(0.5);

  auto merge = vtkSmartPointer<vtkMergeArrays>::New();
  merge->AddInputConnection(warp->GetOutputPort());
  merge->AddInputConnection(clip->GetOutputPort());
  merge->Update(); // discard

  auto warpMapper = vtkSmartPointer<vtkDataSetMapper>::New();
  warpMapper->SetInputConnection(warp->GetOutputPort());

  auto warpActor = vtkSmartPointer<vtkActor>::New();
  warpActor->SetMapper(warpMapper);

  auto ren = vtkSmartPointer<vtkRenderer>::New();
  ren->AddActor(actor);
  ren->AddActor(warpActor);

  auto renWin = vtkSmartPointer<vtkRenderWindow>::New();
  renWin->AddRenderer(ren);

  renWin->Render();

  return 0;
}
