/*=========================================================================

  Program:   ParaView
  Module:    TestIceTShadowMapPass.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// Test of vtkDistributedDataFilter and supporting classes, covering as much
// code as possible.  This test requires 4 MPI processes.
//
// To cover ghost cell creation, use vtkDataSetSurfaceFilter.
//
// To cover clipping code:  SetBoundaryModeToSplitBoundaryCells()
//
// To run fast redistribution: SetUseMinimalMemoryOff() (Default)
// To run memory conserving code instead: SetUseMinimalMemoryOn()

#include "vtkActor.h"
#include "vtkCamera.h"
#include "vtkCameraPass.h"
#include "vtkClearZPass.h"
#include "vtkClientServerCompositePass.h"
#include "vtkCompositeRenderManager.h"
#include "vtkCompositeZPass.h"
#include "vtkConeSource.h"
#include "vtkCubeSource.h"
#include "vtkDataSetReader.h"
#include "vtkDataSetSurfaceFilter.h"
#include "vtkDepthPeelingPass.h"
#include "vtkDistributedDataFilter.h"
#include "vtkIceTCompositePass.h"
#include "vtkImageRenderManager.h"
#include "vtkInformation.h"
#include "vtkLight.h"
#include "vtkLightActor.h"
#include "vtkLightCollection.h"
#include "vtkLightsPass.h"
#include "vtkMPIController.h"
#include "vtkObjectFactory.h"
#include "vtkOpaquePass.h"
#include "vtkOpenGLRenderWindow.h"
#include "vtkOpenGLRenderer.h"
#include "vtkOverlayPass.h"
#include "vtkPKdTree.h"
#include "vtkPieceScalars.h"
#include "vtkPlaneSource.h"
#include "vtkPointData.h"
#include "vtkPolyDataMapper.h"
#include "vtkPolyDataNormals.h"
#include "vtkProperty.h"
#include "vtkRegressionTestImage.h"
#include "vtkRenderPassCollection.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkRenderer.h"
#include "vtkSequencePass.h"
#include "vtkShadowMapBakerPass.h"
#include "vtkShadowMapPass.h"
#include "vtkSmartPointer.h"
#include "vtkSocketController.h"
#include "vtkSphereSource.h"
#include "vtkSynchronizedRenderWindows.h"
#include "vtkSynchronizedRenderers.h"
#include "vtkTestUtilities.h"
#include "vtkTranslucentPass.h"
#include "vtkUnstructuredGrid.h"
#include "vtkVolumetricPass.h"

#include <cassert>

/*
** This test only builds if MPI is in use
*/
#include "vtkMPICommunicator.h"

#include "vtkProcess.h"
#include "vtk_mpi.h"

#include <vtksys/CommandLineArguments.hxx>

// Defined in TestLightActor.cxx
// For each spotlight, add a light frustum wireframe representation and a cone
// wireframe representation, colored with the light color.
void AddLightActors(vtkRenderer* r);

class MyProcess : public vtkProcess
{
  vtkSmartPointer<vtkPKdTree> KdTree;
  bool UseOrderedCompositing;
  bool UseDepthPeeling;

public:
  static MyProcess* New();
  vtkTypeMacro(MyProcess, vtkProcess);

  vtkSetVector2Macro(TileDimensions, int);
  vtkGetVector2Macro(TileDimensions, int);

  vtkSetMacro(ImageReductionFactor, int);
  vtkSetMacro(UseOrderedCompositing, bool);
  vtkSetMacro(UseDepthPeeling, bool);
  vtkSetMacro(ServerMode, bool);
  vtkSetObjectMacro(SocketController, vtkMultiProcessController);

  void SetArgs(int anArgc, char* anArgv[]);

  virtual void Execute();

private:
  // Creates the visualization pipeline and adds it to the renderer.
  void CreatePipeline(vtkRenderer* renderer);

  // Setups render passes.
  void SetupRenderPasses(vtkRenderer* renderer);

protected:
  MyProcess();
  ~MyProcess();

  vtkMultiProcessController* SocketController;
  int Argc;
  char** Argv;

  int ImageReductionFactor;
  int TileDimensions[2];
  bool ServerMode;
};

vtkStandardNewMacro(MyProcess);

//-----------------------------------------------------------------------------
MyProcess::MyProcess()
{
  this->Argc = 0;
  this->Argv = 0;
  this->ImageReductionFactor = 1;
  this->TileDimensions[0] = this->TileDimensions[1];
  this->UseOrderedCompositing = false;
  this->UseDepthPeeling = false;
  this->ServerMode = false;
  this->SocketController = 0;
}

//-----------------------------------------------------------------------------
MyProcess::~MyProcess()
{
  this->SetSocketController(nullptr);
}

//-----------------------------------------------------------------------------
void MyProcess::SetArgs(int anArgc, char* anArgv[])
{
  this->Argc = anArgc;
  this->Argv = anArgv;
}

//-----------------------------------------------------------------------------
void MyProcess::CreatePipeline(vtkRenderer* renderer)
{
  int numProcs = this->Controller->GetNumberOfProcesses();
  int me = this->Controller->GetLocalProcessId();

  auto rectangleSource = vtkSmartPointer<vtkPlaneSource>::New();
  rectangleSource->SetOrigin(-5.0, 0.0, 5.0);
  rectangleSource->SetPoint1(5.0, 0.0, 5.0);
  rectangleSource->SetPoint2(-5.0, 0.0, -5.0);
  rectangleSource->SetResolution(100, 100);

  auto rectangleMapper = vtkSmartPointer<vtkPolyDataMapper>::New();
  rectangleMapper->SetInputConnection(rectangleSource->GetOutputPort());
  rectangleMapper->SetScalarVisibility(0);

  auto rectangleActor = vtkSmartPointer<vtkActor>::New();
  auto rectangleKeyProperties = vtkSmartPointer<vtkInformation>::New();
  // rectangleKeyProperties->Set(vtkShadowMapBakerPass::OCCLUDER(), 0); // dummy val.
  // rectangleKeyProperties->Set(vtkShadowMapBakerPass::RECEIVER(), 0); // dummy val.
  rectangleActor->SetPropertyKeys(rectangleKeyProperties);
  rectangleActor->SetMapper(rectangleMapper);
  rectangleActor->SetVisibility(1);
  rectangleActor->GetProperty()->SetColor(1.0, 1.0, 1.0);

  auto boxSource = vtkSmartPointer<vtkCubeSource>::New();
  boxSource->SetXLength(2.0);
  auto boxNormals = vtkSmartPointer<vtkPolyDataNormals>::New();
  boxNormals->SetInputConnection(boxSource->GetOutputPort());
  boxNormals->SetComputePointNormals(0);
  boxNormals->SetComputeCellNormals(1);
  boxNormals->Update();
  boxNormals->GetOutput()->GetPointData()->SetNormals(0);

  auto boxMapper = vtkSmartPointer<vtkPolyDataMapper>::New();
  boxMapper->SetInputConnection(boxNormals->GetOutputPort());
  boxMapper->SetScalarVisibility(0);

  auto boxActor = vtkSmartPointer<vtkActor>::New();
  auto boxKeyProperties = vtkSmartPointer<vtkInformation>::New();
  // boxKeyProperties->Set(vtkShadowMapBakerPass::OCCLUDER(), 0); // dummy val.
  // boxKeyProperties->Set(vtkShadowMapBakerPass::RECEIVER(), 0); // dummy val.
  boxActor->SetPropertyKeys(boxKeyProperties);

  boxActor->SetMapper(boxMapper);
  boxActor->SetVisibility(1);
  boxActor->SetPosition(-2.0, 2.0, 0.0);
  boxActor->GetProperty()->SetColor(1.0, 0.0, 0.0);

  auto coneSource = vtkSmartPointer<vtkConeSource>::New();
  coneSource->SetResolution(24);
  coneSource->SetDirection(1.0, 1.0, 1.0);
  auto coneMapper = vtkSmartPointer<vtkPolyDataMapper>::New();
  coneMapper->SetInputConnection(coneSource->GetOutputPort());
  coneMapper->SetScalarVisibility(0);

  auto coneActor = vtkSmartPointer<vtkActor>::New();
  auto coneKeyProperties = vtkSmartPointer<vtkInformation>::New();
  // coneKeyProperties->Set(vtkShadowMapBakerPass::OCCLUDER(), 0); // dummy val.
  // coneKeyProperties->Set(vtkShadowMapBakerPass::RECEIVER(), 0); // dummy val.
  coneActor->SetPropertyKeys(coneKeyProperties);
  coneActor->SetMapper(coneMapper);
  coneActor->SetVisibility(1);
  coneActor->SetPosition(0.0, 1.0, 1.0);
  coneActor->GetProperty()->SetColor(0.0, 0.0, 1.0);
  //  coneActor->GetProperty()->SetLighting(false);

  auto sphereSource = vtkSmartPointer<vtkSphereSource>::New();
  sphereSource->SetThetaResolution(32);
  sphereSource->SetPhiResolution(32);
  auto sphereMapper = vtkSmartPointer<vtkPolyDataMapper>::New();
  sphereMapper->SetInputConnection(sphereSource->GetOutputPort());
  sphereMapper->SetScalarVisibility(0);

  auto sphereActor = vtkSmartPointer<vtkActor>::New();
  auto sphereKeyProperties = vtkSmartPointer<vtkInformation>::New();
  // sphereKeyProperties->Set(vtkShadowMapBakerPass::OCCLUDER(), 0); // dummy val.
  // sphereKeyProperties->Set(vtkShadowMapBakerPass::RECEIVER(), 0); // dummy val.
  sphereActor->SetPropertyKeys(sphereKeyProperties);
  sphereActor->SetMapper(sphereMapper);
  sphereActor->SetVisibility(1);
  sphereActor->SetPosition(2.0, 2.0, -1.0);
  sphereActor->GetProperty()->SetColor(1.0, 1.0, 0.0);

  renderer->AddViewProp(rectangleActor);
  renderer->AddViewProp(boxActor);
  renderer->AddViewProp(coneActor);
  renderer->AddViewProp(sphereActor);

  // Spotlights.

  // lighting the box.
  auto l1 = vtkSmartPointer<vtkLight>::New();
  l1->SetPosition(-4.0, 4.0, -1.0);
  l1->SetFocalPoint(boxActor->GetPosition());
  l1->SetColor(1.0, 1.0, 1.0);
  l1->SetPositional(1);
  renderer->AddLight(l1);
  l1->SetSwitch(1);

  // lighting the sphere
  auto l2 = vtkSmartPointer<vtkLight>::New();
  l2->SetPosition(4.0, 5.0, 1.0);
  l2->SetFocalPoint(sphereActor->GetPosition());
  l2->SetColor(1.0, 0.0, 1.0);
  //  l2->SetColor(1.0,1.0,1.0);
  l2->SetPositional(1);
  renderer->AddLight(l2);
  l2->SetSwitch(1);

  AddLightActors(renderer);

  // Tell the pipeline which piece we want to update.
  sphereMapper->SetNumberOfPieces(numProcs);
  sphereMapper->SetPiece(me);
  coneMapper->SetNumberOfPieces(numProcs);
  coneMapper->SetPiece(me);
  rectangleMapper->SetNumberOfPieces(numProcs);
  rectangleMapper->SetPiece(me);
  boxMapper->SetNumberOfPieces(numProcs);
  boxMapper->SetPiece(me);
}

//-----------------------------------------------------------------------------
void MyProcess::SetupRenderPasses(vtkRenderer* renderer)
{
  // the rendering passes
  auto cameraP = vtkSmartPointer<vtkCameraPass>::New();

  auto opaque = vtkSmartPointer<vtkOpaquePass>::New();

  auto peeling = vtkSmartPointer<vtkDepthPeelingPass>::New();
  peeling->SetMaximumNumberOfPeels(200);
  peeling->SetOcclusionRatio(0.1);

  auto translucent = vtkSmartPointer<vtkTranslucentPass>::New();
  peeling->SetTranslucentPass(translucent);

  auto volume = vtkSmartPointer<vtkVolumetricPass>::New();
  auto overlay = vtkSmartPointer<vtkOverlayPass>::New();

  auto lights = vtkSmartPointer<vtkLightsPass>::New();

  auto opaqueSequence = vtkSmartPointer<vtkSequencePass>::New();

  auto passes2 = vtkSmartPointer<vtkRenderPassCollection>::New();
  passes2->AddItem(lights);
  passes2->AddItem(opaque);
  opaqueSequence->SetPasses(passes2);

  auto opaqueCameraPass = vtkSmartPointer<vtkCameraPass>::New();
  opaqueCameraPass->SetDelegatePass(opaqueSequence);

  auto shadowsBaker = vtkSmartPointer<vtkShadowMapBakerPass>::New();
  // shadowsBaker->SetOpaquePass(opaqueCameraPass);
  opaqueCameraPass->Delete();
  shadowsBaker->SetResolution(1024);
  // To cancel self-shadowing.
  // shadowsBaker->SetPolygonOffsetFactor(3.1f);
  // shadowsBaker->SetPolygonOffsetUnits(10.0f);

  auto shadows = vtkSmartPointer<vtkShadowMapPass>::New();
  shadows->SetShadowMapBakerPass(shadowsBaker);
  // shadows->SetOpaquePass(opaqueSequence);

  auto compositeZPass = vtkSmartPointer<vtkCompositeZPass>::New();
  compositeZPass->SetController(this->Controller);
  shadowsBaker->SetCompositeZPass(compositeZPass);

  auto seq = vtkSmartPointer<vtkSequencePass>::New();
  auto passes = vtkSmartPointer<vtkRenderPassCollection>::New();
  passes->AddItem(shadowsBaker);
  passes->AddItem(shadows);
  passes->AddItem(lights);

  if (this->UseDepthPeeling && this->UseOrderedCompositing)
  {
    passes->AddItem(peeling);
  }
  else
  {
    passes->AddItem(translucent);
  }
  passes->AddItem(volume);
  passes->AddItem(overlay);
  seq->SetPasses(passes);

  auto iceTPass = vtkSmartPointer<vtkIceTCompositePass>::New();
  iceTPass->SetController(this->Controller);
  iceTPass->SetUseOrderedCompositing(this->UseOrderedCompositing);
  // iceTPass->SetKdTree(this->KdTree);
  iceTPass->SetRenderPass(seq);
  iceTPass->SetTileDimensions(this->TileDimensions);
  iceTPass->SetImageReductionFactor(this->ImageReductionFactor);

  if (this->ServerMode && this->Controller->GetLocalProcessId() == 0)
  {
    auto csPass = vtkSmartPointer<vtkClientServerCompositePass>::New();
    csPass->SetRenderPass(iceTPass);
    csPass->SetProcessIsServer(true);
    csPass->ServerSideRenderingOn();
    csPass->SetController(this->SocketController);
    cameraP->SetDelegatePass(csPass);
  }
  else
  {
    cameraP->SetDelegatePass(iceTPass);
  }

  cameraP->SetAspectRatioOverride((double)this->TileDimensions[0] / this->TileDimensions[1]);

  vtkOpenGLRenderer* glrenderer = vtkOpenGLRenderer::SafeDownCast(renderer);
  glrenderer->SetPass(cameraP);

  // setting viewport doesn't work in tile-display mode correctly yet.
  // renderer->SetViewport(0, 0, 0.75, 1);
}

//-----------------------------------------------------------------------------
void MyProcess::Execute()
{
  int myId = this->Controller->GetLocalProcessId();

  auto renWin = vtkSmartPointer<vtkRenderWindow>::New();

  int y = myId / this->TileDimensions[0];
  int x = myId % this->TileDimensions[0];

  const int width = 400;
  const int height = 400;

  // 400x400 windows, +24 in y to avoid gnome menu bar.
  renWin->SetPosition(x * (width + 6), y * (height + 20) + 24);

  // enable alpha bit-planes.
  renWin->AlphaBitPlanesOn();

  // use double bufferring.
  renWin->DoubleBufferOn();

  // don't waste time swapping buffers unless needed.
  renWin->SwapBuffersOff();

  // Depth-peeling requires that multi-samples ==0.
  renWin->SetMultiSamples(0);

  auto renderer = vtkSmartPointer<vtkRenderer>::New();
  renWin->AddRenderer(renderer);

  renderer->SetBackground(0.66, 0.66, 0.66);
  renderer->SetBackground2(157.0 / 255.0 * 0.66, 186 / 255.0 * 0.66, 192.0 / 255.0 * 0.66);
  renderer->SetGradientBackground(true);
  renWin->SetSize(width, height);

  auto syncWindows = vtkSmartPointer<vtkSynchronizedRenderWindows>::New();
  syncWindows->SetRenderWindow(renWin);
  syncWindows->SetParallelController(this->Controller);
  syncWindows->SetIdentifier(1);

  auto syncRenderers = vtkSmartPointer<vtkSynchronizedRenderers>::New();
  syncRenderers->SetRenderer(renderer);
  syncRenderers->SetParallelController(this->Controller);

  this->CreatePipeline(renderer);
  this->SetupRenderPasses(renderer);

  int retVal = vtkTesting::FAILED;
  if (myId == 0)
  {
    // root node
    auto iren = vtkSmartPointer<vtkRenderWindowInteractor>::New();
    iren->SetRenderWindow(renWin);
    renWin->SwapBuffersOn();

    if (this->ServerMode)
    {
      auto syncWindows2 = vtkSmartPointer<vtkSynchronizedRenderWindows>::New();
      syncWindows2->SetRenderWindow(renWin);
      syncWindows2->SetParallelController(this->SocketController);
      syncWindows2->SetIdentifier(2);
      syncWindows2->SetRootProcessId(1);

      auto syncRenderers2 = vtkSmartPointer<vtkSynchronizedRenderers>::New();
      syncRenderers2->SetRenderer(renderer);
      syncRenderers2->SetParallelController(this->SocketController);
      syncRenderers2->SetRootProcessId(1);

      this->SocketController->ProcessRMIs();
    }
    else
    {
      renWin->Render();
      retVal = vtkTesting::Test(this->Argc, this->Argv, renWin, 10);
      if (retVal == vtkRegressionTester::DO_INTERACTOR)
      {
        iren->Start();
      }
    }

    this->Controller->TriggerBreakRMIs();
    this->Controller->Barrier();
  }
  else
  {
    // satellite nodes
    if (this->TileDimensions[0] > 1 || this->TileDimensions[1] > 1)
    {
      renWin->SwapBuffersOn();
    }
    this->Controller->ProcessRMIs();
    this->Controller->Barrier();
    retVal = vtkTesting::PASSED;
  }

  this->ReturnValue = retVal;
}

//-----------------------------------------------------------------------------
int TestIceTShadowMapPass(int argc, char** argv)
{
  int retVal = 1;

  // This is here to avoid false leak messages from vtkDebugLeaks when
  // using mpich. It appears that the root process which spawns all the
  // main processes waits in MPI_Init() and calls exit() when
  // the others are done, causing apparent memory leaks for any objects
  // created before MPI_Init().
  MPI_Init(&argc, &argv);

  auto contr = vtkSmartPointer<vtkMPIController>::New();
  contr->Initialize(&argc, &argv, 1);

  int tile_dimensions[2] = { 1, 1 };
  int image_reduction_factor = 1;
  int use_ordered_compositing = 0;
  int use_depth_peeling = 0;
  int act_as_server = 0;
  int interactive = 0;
  std::string data;
  std::string temp;
  std::string baseline;

  vtksys::CommandLineArguments args;
  args.Initialize(argc, argv);
  args.StoreUnusedArguments(false);
  args.AddArgument("-D", vtksys::CommandLineArguments::SPACE_ARGUMENT, &data, "");
  args.AddArgument("-T", vtksys::CommandLineArguments::SPACE_ARGUMENT, &temp, "");
  args.AddArgument("-V", vtksys::CommandLineArguments::SPACE_ARGUMENT, &baseline, "");
  args.AddArgument("-I", vtksys::CommandLineArguments::NO_ARGUMENT, &interactive, "");
  args.AddArgument("--tdx", vtksys::CommandLineArguments::SPACE_ARGUMENT, &tile_dimensions[0], "");
  args.AddArgument("--tdy", vtksys::CommandLineArguments::SPACE_ARGUMENT, &tile_dimensions[1], "");
  args.AddArgument("--image-reduction-factor", vtksys::CommandLineArguments::SPACE_ARGUMENT,
    &image_reduction_factor, "Image reduction factor");
  args.AddArgument("--irf", vtksys::CommandLineArguments::SPACE_ARGUMENT, &image_reduction_factor,
    "Image reduction factor");
  args.AddArgument("--use-ordered-compositing", vtksys::CommandLineArguments::NO_ARGUMENT,
    &use_ordered_compositing, "Use ordered compositing");
  args.AddArgument("--use-depth-peeling", vtksys::CommandLineArguments::NO_ARGUMENT,
    &use_depth_peeling, "Use depth peeling."
                        "This works only when --use-ordered-compositing is true.");
  args.AddArgument("--server", vtksys::CommandLineArguments::NO_ARGUMENT, &act_as_server,
    "When present, the root process acts as a server process for a client.");

  if (!args.Parse())
  {
    if (contr->GetLocalProcessId() == 0)
    {
      cout << args.GetHelp() << endl;
    }
    contr->Finalize();
    return 1;
  }

  tile_dimensions[0] = tile_dimensions[0] < 1 ? 1 : tile_dimensions[0];
  tile_dimensions[1] = tile_dimensions[1] < 1 ? 1 : tile_dimensions[1];

  int numProcs = contr->GetNumberOfProcesses();
  if (tile_dimensions[0] > 1 || tile_dimensions[1] > 1)
  {
    if (numProcs != tile_dimensions[0] * tile_dimensions[1])
    {
      cerr << "When running in tile-display mode, number of processes must "
              "match number of tiles"
           << endl;
      return 1;
    }
    cout << "Rendering as a " << tile_dimensions[0] << "x" << tile_dimensions[1] << " tile display"
         << endl;
  }

  vtkMultiProcessController::SetGlobalController(contr);

  auto p = vtkSmartPointer<MyProcess>::New();
  p->SetArgs(argc, argv);
  p->SetTileDimensions(tile_dimensions);
  p->SetImageReductionFactor(image_reduction_factor);
  p->SetUseOrderedCompositing(use_ordered_compositing == 1);
  p->SetUseDepthPeeling(use_depth_peeling == 1);
  p->SetServerMode(act_as_server != 0);

  if (contr->GetLocalProcessId() == 0 && act_as_server)
  {
    auto socket_contr = vtkSmartPointer<vtkSocketController>::New();
    socket_contr->Initialize(&argc, &argv);
    cout << "Waiting for client on 11111" << endl;
    socket_contr->WaitForConnection(11111);
    p->SetSocketController(socket_contr);
  }

  contr->SetSingleProcessObject(p);
  contr->SingleMethodExecute();

  retVal = p->GetReturnValue();

  vtkMultiProcessController::SetGlobalController(0);
  contr = 0;
  return !retVal;
}

// DUPLICATE for VTK/Rendering/Testing/Cxx/TestLightActor.cxx

// For each spotlight, add a light frustum wireframe representation and a cone
// wireframe representation, colored with the light color.
void AddLightActors(vtkRenderer* r)
{
  assert("pre: r_exists" && r != nullptr);

  vtkLightCollection* lights = r->GetLights();

  lights->InitTraversal();
  vtkLight* l = lights->GetNextItem();
  while (l != nullptr)
  {
    double angle = l->GetConeAngle();
    if (l->LightTypeIsSceneLight() && l->GetPositional() && angle < 90.0) // spotlight
    {
      auto la = vtkSmartPointer<vtkLightActor>::New();
      la->SetLight(l);
      r->AddViewProp(la);
    }
    l = lights->GetNextItem();
  }
}
