// Qt includes
#include <QAction>
#include <QDebug>
#include <QMimeData>
#include <QStringList>
#include <QSharedPointer>
#include <QVector>
#include <QMap>

// qMRML includes
#include "qMRMLItemHelper.h"
#include "qMRMLSceneModel.h"
#include "qMRMLUtils.h"

// MRML includes 
#include <vtkMRMLScene.h>

// VTK includes 
#include <vtkSmartPointer.h>
#include <vtkCallbackCommand.h>
#include <vtkVariantArray.h>

// STD includes
#include <iostream>
#include <fstream>

//------------------------------------------------------------------------------
class QMRML_WIDGETS_EXPORT qMRMLFlatSceneItemHelper : public qMRMLAbstractSceneItemHelper
{
public:
  virtual qMRMLAbstractItemHelper* child(int row, int column) const;
  virtual int childCount() const;
  virtual bool hasChildren() const;
  virtual qMRMLAbstractItemHelper* parent()const;
  
protected:
  friend class qMRMLSceneModelItemHelperFactory;
  qMRMLFlatSceneItemHelper(vtkMRMLScene* scene, int column, const qMRMLAbstractItemHelperFactory* factory);
  /// here we know for sure that child is a child of this.
  virtual int childIndex(const qMRMLAbstractItemHelper* child)const;
};

//------------------------------------------------------------------------------
class QMRML_WIDGETS_EXPORT qMRMLFlatNodeItemHelper : public qMRMLAbstractNodeItemHelper
{
public:
  virtual qMRMLAbstractItemHelper* parent() const;
protected:
  friend class qMRMLSceneModelItemHelperFactory;
  qMRMLFlatNodeItemHelper(vtkMRMLNode* node, int column, const qMRMLAbstractItemHelperFactory* factory);
};

//------------------------------------------------------------------------------
class qMRMLSceneModelItemHelperFactoryPrivate: public ctkPrivate<qMRMLSceneModelItemHelperFactory>
{
public:
  vtkSmartPointer<vtkCollection> PreItems;
  vtkSmartPointer<vtkCollection> PostItems;
};

//------------------------------------------------------------------------------
qMRMLSceneModelItemHelperFactory::qMRMLSceneModelItemHelperFactory()
{
  CTK_INIT_PRIVATE(qMRMLSceneModelItemHelperFactory);
}

//------------------------------------------------------------------------------
qMRMLAbstractItemHelper* qMRMLSceneModelItemHelperFactory::createItem(vtkObject* object, int column)const
{
  if (!object)
    {
    Q_ASSERT(object);
    return 0;
    }
  if (object->IsA("vtkMRMLScene"))
    {
    qMRMLAbstractItemHelper* source = new qMRMLFlatSceneItemHelper(vtkMRMLScene::SafeDownCast(object), column, this);
    return new qMRMLExtraItemsHelper(this->preItems(), this->postItems(), source);
    }
  else if (object->IsA("vtkMRMLNode"))
    {
    return new qMRMLFlatNodeItemHelper(vtkMRMLNode::SafeDownCast(object), column, this);
    }
  else if (object->IsA("vtkVariantArray"))
    {
    return new qMRMLVariantArrayItemHelper(vtkVariantArray::SafeDownCast(object), column, this);    
    }
  else 
    {
    qDebug() << object->GetClassName();
    Q_ASSERT(false);
    }
  return 0;
}

//------------------------------------------------------------------------------
void qMRMLSceneModelItemHelperFactory::setPostItems(vtkCollection* itemCollection)
{
  CTK_D(qMRMLSceneModelItemHelperFactory);
  d->PostItems = itemCollection;
}

//------------------------------------------------------------------------------
vtkCollection* qMRMLSceneModelItemHelperFactory::postItems()const
{
  CTK_D(const qMRMLSceneModelItemHelperFactory);
  return d->PostItems.GetPointer();
}

//------------------------------------------------------------------------------
void qMRMLSceneModelItemHelperFactory::setPreItems(vtkCollection* itemCollection)
{
  CTK_D(qMRMLSceneModelItemHelperFactory);
  d->PreItems = itemCollection;
}

//------------------------------------------------------------------------------
vtkCollection* qMRMLSceneModelItemHelperFactory::preItems()const
{
  CTK_D(const qMRMLSceneModelItemHelperFactory);
  return d->PreItems.GetPointer();
}

// qMRMLFlatSceneItemHelper

//------------------------------------------------------------------------------
qMRMLFlatSceneItemHelper::qMRMLFlatSceneItemHelper(vtkMRMLScene* scene, int _column, 
                                                   const qMRMLAbstractItemHelperFactory* itemFactory)
  :qMRMLAbstractSceneItemHelper(scene, _column, itemFactory)
{
}

//------------------------------------------------------------------------------
qMRMLAbstractItemHelper* qMRMLFlatSceneItemHelper::child(int _row, int _column) const
{
  vtkMRMLNode* childNode = this->mrmlScene()->GetNthNode(_row);
  return this->factory()->createItem(childNode, _column);
  //if (childNode == 0)
  //  {
  //  return 0;
  //  }
  //qMRMLAbstractNodeItemHelper* _child = 
  //  new qMRMLFlatNodeItemHelper(childNode, _column);
  //return _child;
  //return childNode;
}

//------------------------------------------------------------------------------
int qMRMLFlatSceneItemHelper::childCount() const
{ 
  if (this->column() != 0)
    {
    return 0;
    }
  return this->mrmlScene() ? this->mrmlScene()->GetNumberOfNodes() : 0;; 
}

//------------------------------------------------------------------------------
int qMRMLFlatSceneItemHelper::childIndex(const qMRMLAbstractItemHelper* _child) const
{
  const qMRMLAbstractNodeItemHelper* nodeItemHelper = 
    dynamic_cast<const qMRMLAbstractNodeItemHelper*>(_child);
  //Q_ASSERT(nodeItemHelper);
  if (nodeItemHelper == 0)
    {
    return -1;
    }
  // Check what kind of value IsNodePresent(0) returns. If a node is not found: 0 or -1?
  // if it's 0, then we should decrease the index as we work in 0-based arrays.
  Q_ASSERT(!this->mrmlScene() || this->mrmlScene()->IsNodePresent(0) == 0);
  return this->mrmlScene() ? this->mrmlScene()->IsNodePresent(nodeItemHelper->mrmlNode()) - 1 : -1;
}

//------------------------------------------------------------------------------
bool qMRMLFlatSceneItemHelper::hasChildren() const
{
  if (this->column() != 0)
    {
    return 0;
    }
  return this->mrmlScene() ? this->mrmlScene()->GetNumberOfNodes() > 0 : false;
}

//------------------------------------------------------------------------------
qMRMLAbstractItemHelper* qMRMLFlatSceneItemHelper::parent() const
{
  return this->factory()->createRootItem(this->mrmlScene());
}

//------------------------------------------------------------------------------
// qMRMLFlatNodeItemHelper

//------------------------------------------------------------------------------
qMRMLFlatNodeItemHelper::qMRMLFlatNodeItemHelper(vtkMRMLNode* node, int _column, 
                                                 const qMRMLAbstractItemHelperFactory* itemFactory)
  :qMRMLAbstractNodeItemHelper(node, _column, itemFactory)
{
}

//------------------------------------------------------------------------------
qMRMLAbstractItemHelper* qMRMLFlatNodeItemHelper::parent() const
{
  //return new qMRMLFlatSceneItemHelper(this->mrmlNode()->GetScene(), 0);
  //return this->mrmlNode()->GetScene();
  return this->factory()->createItem(this->mrmlNode()->GetScene(), 0);
}

//------------------------------------------------------------------------------
// qMRMLSceneModelPrivate
//------------------------------------------------------------------------------
class qMRMLSceneModelPrivate: public ctkPrivate<qMRMLSceneModel>
{
public:
  CTK_DECLARE_PUBLIC(qMRMLSceneModel);
  qMRMLSceneModelPrivate();
  ~qMRMLSceneModelPrivate();
  void init(qMRMLSceneModelItemHelperFactory*);
  
  vtkObject* object(const QModelIndex &index)const;

  //qMRMLAbstractItemHelper* itemFromObject(vtkObject* object, int column)const;
  qMRMLAbstractItemHelper* itemFromIndex(const QModelIndex &index)const;
  QModelIndex indexFromItem(const qMRMLAbstractItemHelper* itemHelper)const;

  void listenNodeModifiedEvent();

  vtkSmartPointer<vtkCallbackCommand> CallBack;
  bool ListenNodeModifiedEvent;
  qMRMLSceneModelItemHelperFactory* ItemFactory;
  vtkMRMLScene* MRMLScene;
  vtkMRMLNode*  MRMLNodeToBe;
  QList<QAction*> Actions;
};

//------------------------------------------------------------------------------
qMRMLSceneModelPrivate::qMRMLSceneModelPrivate()
{
  this->CallBack = vtkSmartPointer<vtkCallbackCommand>::New();
  this->ListenNodeModifiedEvent = false;
  this->ItemFactory = 0;
  this->MRMLScene = 0;
  this->MRMLNodeToBe = 0;
}

//------------------------------------------------------------------------------
qMRMLSceneModelPrivate::~qMRMLSceneModelPrivate()
{
  if (this->MRMLScene)
    {
    this->MRMLScene->RemoveObserver(this->CallBack);
    }
  delete this->ItemFactory;
}

//------------------------------------------------------------------------------
void qMRMLSceneModelPrivate::init(qMRMLSceneModelItemHelperFactory* factory)
{
  CTK_P(qMRMLSceneModel);
  this->CallBack->SetClientData(p);
  this->CallBack->SetCallback(qMRMLSceneModel::DoCallback);
  this->ItemFactory = factory;
}

//------------------------------------------------------------------------------
vtkObject* qMRMLSceneModelPrivate::object(const QModelIndex &index)const
{
  return reinterpret_cast<vtkObject*>(index.internalPointer());
}

//------------------------------------------------------------------------------
qMRMLAbstractItemHelper* qMRMLSceneModelPrivate::itemFromIndex(const QModelIndex &index)const
{
  CTK_P(const qMRMLSceneModel);
  if ((index.row() < 0) || (index.column() < 0) || (index.model() != p))
    {
    //return new qMRMLFlatRootItemHelper(this->MRMLScene);
    return this->ItemFactory->createRootItem(this->MRMLScene);
    }
  //return this->itemFromObject(reinterpret_cast<vtkObject*>(index.internalPointer()), index.column());
  return this->ItemFactory->createItem(this->object(index), index.column());
}

//------------------------------------------------------------------------------
QModelIndex qMRMLSceneModelPrivate::indexFromItem(const qMRMLAbstractItemHelper* item)const
{
  CTK_P(const qMRMLSceneModel);
  if (item == 0 || item->object() == 0)
    {
    return QModelIndex();
    }
  if (dynamic_cast<const qMRMLRootItemHelper*>(item) != 0)
    {
    return QModelIndex();
    }
  return p->createIndex(item->row(), item->column(), 
                        reinterpret_cast<void*>(item->object()));
}

//------------------------------------------------------------------------------
void qMRMLSceneModelPrivate::listenNodeModifiedEvent()
{
  CTK_P(qMRMLSceneModel);
  p->qvtkDisconnect(0, vtkCommand::ModifiedEvent, p, SLOT(onMRMLNodeModified(vtkObject*)));
  if (!this->ListenNodeModifiedEvent)
    {
    return;
    }
  QModelIndex sceneIndex = p->mrmlSceneIndex();
  const int count = p->rowCount(sceneIndex);
  for (int i = 0; i < count; ++i)
    {
    p->qvtkConnect(p->mrmlNode(sceneIndex.child(i,0)),vtkCommand::ModifiedEvent,
                   p, SLOT(onMRMLNodeModified(vtkObject*)));
    }
}

//------------------------------------------------------------------------------
// qMRMLSceneModel
//------------------------------------------------------------------------------
qMRMLSceneModel::qMRMLSceneModel(QObject *_parent)
  :QAbstractItemModel(_parent)
{
  CTK_INIT_PRIVATE(qMRMLSceneModel);
  ctk_d()->init(new qMRMLSceneModelItemHelperFactory);
}

//------------------------------------------------------------------------------
qMRMLSceneModel::qMRMLSceneModel(qMRMLSceneModelItemHelperFactory* factory, QObject *parentObject)
  :QAbstractItemModel(parentObject)
{
  CTK_INIT_PRIVATE(qMRMLSceneModel);
  ctk_d()->init(factory);
}

//------------------------------------------------------------------------------
qMRMLSceneModel::~qMRMLSceneModel()
{
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::setPreItems(vtkObject* itemParent, const QStringList& extraItems)
{
  CTK_D(qMRMLSceneModel);
  vtkCollection* collection = vtkCollection::New();
  vtkVariantArray* variantArray = 0;

  foreach(const QString& extraItem, extraItems)
    {
    variantArray = vtkVariantArray::New();
    qMRMLVariantArrayItemHelper::createProperties(*variantArray, 
      itemParent, 
      extraItem.toStdString(), 
      Qt::ItemIsEnabled | Qt::ItemIsSelectable);
    collection->AddItem(variantArray);
    variantArray->Delete();
    }
  d->ItemFactory->setPreItems(collection);
  collection->Delete();
  // FIXME should probably do a row insertion thingy
  this->reset();
}

//------------------------------------------------------------------------------
QStringList qMRMLSceneModel::preItems(vtkObject* itemParent)const 
{
  CTK_D(const qMRMLSceneModel);
  QStringList res; 

  vtkCollection* collection = d->ItemFactory->preItems();
  if (collection == 0)
    {
    return res;
    }
  // FIXME: not thread safe
  collection->InitTraversal();
  vtkObject* item = 0;
  for(item = collection->GetNextItemAsObject(); item; item = collection->GetNextItemAsObject())
    {
    res.append(vtkVariantArray::SafeDownCast(item)->GetValue(1).ToString().c_str());
    }
  return res;
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::setPostItems(vtkObject* itemParent, const QStringList& extraItems)
{
  CTK_D(qMRMLSceneModel);
  vtkCollection* collection = vtkCollection::New();
  foreach(const QString& extraItem, extraItems)
    {
    vtkVariantArray* variantArray = vtkVariantArray::New();
    qMRMLVariantArrayItemHelper::createProperties(*variantArray, itemParent, extraItem.toStdString());
    collection->AddItem(variantArray);
    variantArray->Delete();
    }

  // FIXME should probably do a row insertion thingy
  //this->reset();
  this->beginResetModel();
  d->ItemFactory->setPostItems(collection);
  this->endResetModel();
  collection->Delete();
}

//------------------------------------------------------------------------------
QStringList qMRMLSceneModel::postItems(vtkObject* itemParent)const
{
  CTK_D(const qMRMLSceneModel);
  QStringList res; 

  vtkCollection* collection = d->ItemFactory->postItems();
  if (collection == 0)
    {
    return res;
    }

  // FIXME: not thread safe
  collection->InitTraversal();
  vtkObject* item = 0;
  for(item = collection->GetNextItemAsObject(); item; item = collection->GetNextItemAsObject())
    {
    res.append(vtkVariantArray::SafeDownCast(item)->GetValue(1).ToString().c_str());
    }
  return res;
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::setMRMLScene(vtkMRMLScene* scene)
{
  CTK_D(qMRMLSceneModel);
  /*
  // onMRMLSceneNodeAboutToBeAdded must be call as late as possible after 
  // vtkMRMLScene::NodeAboutToBeAddedEvent is fired. This fix the pb when a node
  // is created in the callback of the NodeAboutToBeAddedEvent.
  this->qvtkReconnect(d->MRMLScene, scene, vtkMRMLScene::NodeAboutToBeAddedEvent,
                      this, SLOT(onMRMLSceneNodeAboutToBeAdded(vtkObject*, vtkObject*)),
                      -10.);
  // onMRMLSceneNodeAdded must be call as soon as possible after 
  // vtkMRMLScene::NodeAddedEvent is fired. This fix the pb when a node is created in the 
  // callback of the NodeAddedEvent.
  this->qvtkReconnect(d->MRMLScene, scene, vtkMRMLScene::NodeAddedEvent,
                      this, SLOT(onMRMLSceneNodeAdded(vtkObject*, vtkObject*)), 
                      10.);
  this->qvtkReconnect(d->MRMLScene, scene, vtkMRMLScene::NodeAboutToBeRemovedEvent,
                      this, SLOT(onMRMLSceneNodeAboutToBeRemoved(vtkObject*, vtkObject*)),
                      -10.);
  this->qvtkReconnect(d->MRMLScene, scene, vtkMRMLScene::NodeRemovedEvent,
                      this, SLOT(onMRMLSceneNodeRemoved(vtkObject*, vtkObject*)), 10.);
  this->qvtkReconnect(d->MRMLScene, scene, vtkCommand::DeleteEvent,
                      this, SLOT(onMRMLSceneDeleted(vtkObject*)));
  */
  if (d->MRMLScene)
    {
    d->MRMLScene->RemoveObserver(d->CallBack);
    }
  if (scene)
    {
    scene->AddObserver(vtkMRMLScene::NodeAboutToBeAddedEvent, d->CallBack, -10.);
    scene->AddObserver(vtkMRMLScene::NodeAddedEvent, d->CallBack, 10.);
    scene->AddObserver(vtkMRMLScene::NodeAboutToBeRemovedEvent, d->CallBack, -10.);
    scene->AddObserver(vtkMRMLScene::NodeRemovedEvent, d->CallBack, 10.);
    scene->AddObserver(vtkCommand::DeleteEvent, d->CallBack);
    }
  this->beginResetModel();
  d->MRMLScene = scene;
  this->endResetModel();
  d->listenNodeModifiedEvent();
}

//------------------------------------------------------------------------------
vtkMRMLScene* qMRMLSceneModel::mrmlScene()const
{
  return ctk_d()->MRMLScene;
}

//------------------------------------------------------------------------------
QModelIndex qMRMLSceneModel::mrmlSceneIndex()const
{
  CTK_D(const qMRMLSceneModel);
  if (d->MRMLScene == 0)
    {
    return QModelIndex();
    }
  QSharedPointer<qMRMLAbstractItemHelper> sceneItem =
    QSharedPointer<qMRMLAbstractItemHelper>(d->ItemFactory->createItem(d->MRMLScene, 0));
  return this->indexFromItem(sceneItem.data());
}

//------------------------------------------------------------------------------
vtkMRMLNode* qMRMLSceneModel::mrmlNode(const QModelIndex &nodeIndex)const
{
  CTK_D(const qMRMLSceneModel);
  return vtkMRMLNode::SafeDownCast(d->object(nodeIndex));
}

//------------------------------------------------------------------------------
QModelIndexList qMRMLSceneModel::indexes(vtkMRMLNode* node)const
{
  CTK_D(const qMRMLSceneModel);
  QModelIndexList nodeIndexList;
  if (node == 0)
    {
    return nodeIndexList;
    }
  QSharedPointer<qMRMLAbstractItemHelper> firstNodeItem =
    QSharedPointer<qMRMLAbstractItemHelper>(d->ItemFactory->createItem(node, 0));
  Q_ASSERT(!firstNodeItem.isNull());
  QModelIndex firstNodeIndex = this->indexFromItem(firstNodeItem.data());
  const int nodeRow = firstNodeIndex.row();
  QModelIndex nodeParent = firstNodeIndex.parent();
  const int columns = this->columnCount(nodeParent);
  for (int i = 0; i < columns; ++i)
    {
    nodeIndexList.push_back(nodeParent.child(nodeRow, i));
    }
  return nodeIndexList;
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::setListenNodeModifiedEvent(bool listen)
{
  CTK_D(qMRMLSceneModel);
  if (d->ListenNodeModifiedEvent == listen)
    {
    return;
    }
  d->ListenNodeModifiedEvent = listen;
  d->listenNodeModifiedEvent();
}

//------------------------------------------------------------------------------
bool qMRMLSceneModel::listenNodeModifiedEvent()const
{
  CTK_D(const qMRMLSceneModel);
  return d->ListenNodeModifiedEvent;
}

//------------------------------------------------------------------------------
int qMRMLSceneModel::columnCount(const QModelIndex &_parent)const
{
  Q_UNUSED(_parent);
  return 2;
}

//------------------------------------------------------------------------------
QVariant qMRMLSceneModel::data(const QModelIndex &_index, int role)const
{
  CTK_D(const qMRMLSceneModel);
  if (!_index.isValid())
    {
    return QVariant();
    }
  Q_ASSERT(d->MRMLScene);

  QSharedPointer<qMRMLAbstractItemHelper> item = 
    QSharedPointer<qMRMLAbstractItemHelper>(d->itemFromIndex(_index));

  Q_ASSERT(!item.isNull());
  //qDebug() << "qMRMLSceneModel::data: " << item << item->data(role);
  return item->data(role);
}

//------------------------------------------------------------------------------
Qt::ItemFlags qMRMLSceneModel::flags(const QModelIndex &_index)const
{
  CTK_D(const qMRMLSceneModel);
  if (!_index.isValid())
    {
    return Qt::NoItemFlags;
    }

  QSharedPointer<qMRMLAbstractItemHelper> item = 
    QSharedPointer<qMRMLAbstractItemHelper>(d->itemFromIndex(_index));

  Q_ASSERT(!item.isNull());
  return item->flags() & ~Qt::ItemIsDropEnabled;
}

// Has to be reimplemented for speed issues
//------------------------------------------------------------------------------
bool qMRMLSceneModel::hasChildren(const QModelIndex &_parent)const
{
  CTK_D(const qMRMLSceneModel);

  QSharedPointer<qMRMLAbstractItemHelper> item = 
    QSharedPointer<qMRMLAbstractItemHelper>(d->itemFromIndex(_parent));
  Q_ASSERT(!item.isNull());
  if (dynamic_cast<qMRMLAbstractNodeItemHelper*>(item.data()))
    {
    return 0;
    }
  return item->hasChildren();
}

//------------------------------------------------------------------------------
QVariant qMRMLSceneModel::headerData(int section, Qt::Orientation orientation, int role) const
{
  if (orientation == Qt::Vertical || role != Qt::DisplayRole)
    {
    return QVariant();
    }
  switch (section)
    {
    case 0:
      return tr("Name");
      break;
    case 1:
      return tr("Id");
      break;
    default:
      break;
    };

  return QVariant();
}

//------------------------------------------------------------------------------
QModelIndex qMRMLSceneModel::index(int row, int column, const QModelIndex &_parent)const
{
  CTK_D(const qMRMLSceneModel);
  if (d->MRMLScene == 0 || row < 0 || column < 0)
    {
    return QModelIndex();
    }

  QSharedPointer<qMRMLAbstractItemHelper> parentItem = 
    QSharedPointer<qMRMLAbstractItemHelper>(d->itemFromIndex(_parent));
  Q_ASSERT(!parentItem.isNull() || parentItem->object());
  if (dynamic_cast<qMRMLAbstractNodeItemHelper*>(parentItem.data()))
    {
    return QModelIndex();
    }
  QSharedPointer<qMRMLAbstractItemHelper> item = 
    QSharedPointer<qMRMLAbstractItemHelper>(
      parentItem->child(row, column));
  Q_ASSERT(item.isNull() || item->object());

  return !item.isNull() ? this->createIndex(row, column, item->object()) : QModelIndex();
}

//------------------------------------------------------------------------------
//bool qMRMLSceneModel::insertRows(int row, int count, const QModelIndex &parent)
//{
//}

//------------------------------------------------------------------------------
qMRMLAbstractItemHelper* qMRMLSceneModel::itemFromIndex(const QModelIndex &modelIndex)const
{
  CTK_D(const qMRMLSceneModel);
  return d->itemFromIndex(modelIndex);
}

//------------------------------------------------------------------------------
qMRMLAbstractItemHelper* qMRMLSceneModel::itemFromObject(vtkObject* object, int column)const
{
  CTK_D(const qMRMLSceneModel);
  return d->ItemFactory->createItem(object, column);
}

//------------------------------------------------------------------------------
QModelIndex qMRMLSceneModel::indexFromItem(const qMRMLAbstractItemHelper* item)const
{
  CTK_D(const qMRMLSceneModel);
  return d->indexFromItem(item);
}

//------------------------------------------------------------------------------
qMRMLAbstractItemHelperFactory* qMRMLSceneModel::itemFactory()const
{
  CTK_D(const qMRMLSceneModel);
  return d->ItemFactory;
}

//------------------------------------------------------------------------------
QMap<int, QVariant> qMRMLSceneModel::itemData(const QModelIndex &_index)const
{
  QMap<int, QVariant> roles = this->QAbstractItemModel::itemData(_index);
  // In order to have the drag/drop working without defining our own MIME type
  // we need to add some way of uniquely identify an item. Here it's done
  // by adding a UID role to each item.
  QVariant mrmlIdData = this->data(_index, qMRML::UIDRole);
  if (mrmlIdData.type() != QVariant::Invalid)
    {
    roles.insert(qMRML::UIDRole, mrmlIdData);
    }
  return roles;
}

//-----------------------------------------------------------------------------
void qMRMLSceneModel::DoCallback(vtkObject* vtk_obj, unsigned long event,
                                 void* client_data, void* call_data)
{
  vtkMRMLScene* scene = reinterpret_cast<vtkMRMLScene*>(vtk_obj);
  qMRMLSceneModel* sceneModel = reinterpret_cast<qMRMLSceneModel*>(client_data);
  vtkMRMLNode* node = reinterpret_cast<vtkMRMLNode*>(call_data);
  Q_ASSERT(scene);
  Q_ASSERT(sceneModel);
  switch(event)
    {
    case vtkMRMLScene::NodeAboutToBeAddedEvent:
      Q_ASSERT(node);
      sceneModel->onMRMLSceneNodeAboutToBeAdded(scene, node);
      break;
    case vtkMRMLScene::NodeAddedEvent:
      Q_ASSERT(node);
      sceneModel->onMRMLSceneNodeAdded(scene, node);
      break;
    case vtkMRMLScene::NodeAboutToBeRemovedEvent:
      Q_ASSERT(node);
      sceneModel->onMRMLSceneNodeAboutToBeRemoved(scene, node);
      break;
    case vtkMRMLScene::NodeRemovedEvent:
      Q_ASSERT(node);
      sceneModel->onMRMLSceneNodeRemoved(scene, node);
      break;
    case vtkCommand::DeleteEvent:
      sceneModel->onMRMLSceneDeleted(scene);
      break;
    }
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::onMRMLSceneNodeAboutToBeAdded(vtkObject* scene, vtkObject* node)
{
  Q_UNUSED(scene);
  CTK_D(qMRMLSceneModel);
  Q_ASSERT(scene != 0);
  Q_ASSERT(scene == d->MRMLScene);
  
  Q_ASSERT(d->MRMLNodeToBe == 0);
  d->MRMLNodeToBe = vtkMRMLNode::SafeDownCast(node);
  Q_ASSERT(d->MRMLNodeToBe);
  QSharedPointer<qMRMLAbstractItemHelper> sceneItem = 
    QSharedPointer<qMRMLAbstractItemHelper>(d->ItemFactory->createItem(d->MRMLScene, 0));
  // vtkMRMLScene adds nodes at the end of its collection (but before the extra Items
  // FIXME: handle cases where extraItems are not set to the mrmlscene but to other items
  // (root, mrmlnode?)
  // Warning, if you change the next 2 lines, make sure you do it also in onMRMLSceneNodeAdded
  int insertLocation = sceneItem->childCount() - (d->ItemFactory->postItems()?d->ItemFactory->postItems()->GetNumberOfItems():0);
  this->beginInsertRows(this->indexFromItem(sceneItem.data()), insertLocation, insertLocation);
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::onMRMLSceneNodeAdded(vtkObject* scene, vtkObject* node)
{
  CTK_D(qMRMLSceneModel);
  Q_ASSERT(scene == d->MRMLScene);
  Q_ASSERT(vtkMRMLNode::SafeDownCast(node));
  if (d->MRMLNodeToBe == 0)
    {
    // it's kind of ugly to call just NodeAddedEvent without NodeAboutToBeAddedEvent, 
    // but we can handle that case...
    //qDebug() << "Warning, vtkMRMLScene::NodeAddedEvent has been fired without"
    //  " vtkMRMLScene::NodeAboutToBeAddedEvent.";
    //this->onMRMLSceneNodeAboutToBeAdded(scene, node);
    d->MRMLNodeToBe = vtkMRMLNode::SafeDownCast(node);
    Q_ASSERT(d->MRMLNodeToBe);
    Q_ASSERT(d->MRMLScene->IsNodePresent(d->MRMLNodeToBe));
    QSharedPointer<qMRMLAbstractItemHelper> sceneItem = 
      QSharedPointer<qMRMLAbstractItemHelper>(d->ItemFactory->createItem(d->MRMLScene, 0));
    int insertLocation = sceneItem->childCount() -
      (d->ItemFactory->postItems()?d->ItemFactory->postItems()->GetNumberOfItems():0);
    // the node has already been added, offset the position
    insertLocation -= 1;
    this->beginInsertRows(this->indexFromItem(sceneItem.data()), insertLocation , insertLocation);
    }
  Q_ASSERT(vtkMRMLNode::SafeDownCast(node) == d->MRMLNodeToBe);
  d->MRMLNodeToBe = 0;
  // endInsertRows fires the Qt signals
  this->endInsertRows();

  if (d->ListenNodeModifiedEvent)
    {
    qvtkConnect(node, vtkCommand::ModifiedEvent,
                this, SLOT(onMRMLNodeModified(vtkObject*)));
    }
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::onMRMLSceneNodeAboutToBeRemoved(vtkObject* scene, vtkObject* node)
{
  Q_UNUSED(scene);
  CTK_D(qMRMLSceneModel);
  Q_ASSERT(scene == d->MRMLScene);
  Q_ASSERT(d->MRMLNodeToBe == 0);
  d->MRMLNodeToBe = vtkMRMLNode::SafeDownCast(node);
  Q_ASSERT(d->MRMLNodeToBe);

  qvtkDisconnect(node, vtkCommand::ModifiedEvent,
                this, SLOT(onMRMLNodeModified(vtkObject*)));

  QSharedPointer<qMRMLAbstractItemHelper> item = 
    QSharedPointer<qMRMLAbstractItemHelper>(d->ItemFactory->createItem(node, 0));
  QSharedPointer<qMRMLAbstractItemHelper> _parent = 
    QSharedPointer<qMRMLAbstractItemHelper>(
      item->parent());
  this->beginRemoveRows(this->indexFromItem(_parent.data()), item->row(), item->row());
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::onMRMLSceneNodeRemoved(vtkObject* scene, vtkObject* node)
{
  Q_UNUSED(scene);
  Q_UNUSED(node);
  CTK_D(qMRMLSceneModel);
  Q_ASSERT(scene == d->MRMLScene);
  Q_ASSERT(vtkMRMLNode::SafeDownCast(node) == d->MRMLNodeToBe);
  //qDebug() << "onMRMLSceneNodeRemoved: " << d->MRMLNodeToBe->GetName();
  d->MRMLNodeToBe = 0;
  this->endRemoveRows();
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::onMRMLSceneDeleted(vtkObject* scene)
{
  Q_UNUSED(scene);
  Q_ASSERT(scene == ctk_d()->MRMLScene);
  qDebug() << "onMRMLSceneDeleted";
  this->setMRMLScene(0);
}

//------------------------------------------------------------------------------
void qMRMLSceneModel::onMRMLNodeModified(vtkObject* node)
{
  vtkMRMLNode* modifiedNode = vtkMRMLNode::SafeDownCast(node);
  Q_ASSERT(modifiedNode && modifiedNode->GetScene());
  Q_ASSERT(modifiedNode->GetScene()->IsNodePresent(modifiedNode));
  QModelIndexList nodeIndexes = this->indexes(modifiedNode);
  emit dataChanged(nodeIndexes.first(), nodeIndexes.last());
}

//------------------------------------------------------------------------------
//QMimeData *qMRMLSceneModel::mimeData(const QModelIndexList &indexes)const
//{
//}

//------------------------------------------------------------------------------
//QStringList qMRMLSceneModel::mimeTypes()const
//{
//}

//------------------------------------------------------------------------------
QModelIndex qMRMLSceneModel::parent(const QModelIndex &_index)const
{
  CTK_D(const qMRMLSceneModel);
  if (!_index.isValid())
    {
    return QModelIndex();
    }
  QSharedPointer<const qMRMLAbstractItemHelper> item = 
    QSharedPointer<const qMRMLAbstractItemHelper>(d->itemFromIndex(_index));
  Q_ASSERT(!item.isNull());
  if (item.isNull())
    {
    return QModelIndex();
    }
  // let polymorphism plays its role here...
  QSharedPointer<const qMRMLAbstractItemHelper> parentItem =
    QSharedPointer<const qMRMLAbstractItemHelper>(item->parent());
  if (parentItem.isNull())
    {
    return QModelIndex();
    }
  QModelIndex _parent = this->indexFromItem(parentItem.data());
  return _parent;
}

//------------------------------------------------------------------------------
//bool qMRMLSceneModel::removeColumns(int column, int count, const QModelIndex &parent=QModelIndex())
//{
//}

//------------------------------------------------------------------------------
//bool qMRMLSceneModel::removeRows(int row, int count, const QModelIndex &parent)
//{
//}

//------------------------------------------------------------------------------
int qMRMLSceneModel::rowCount(const QModelIndex &_parent) const
{
  CTK_D(const qMRMLSceneModel);
  QSharedPointer<qMRMLAbstractItemHelper> item = 
    QSharedPointer<qMRMLAbstractItemHelper>(d->itemFromIndex(_parent));
  Q_ASSERT(!item.isNull());
  if (dynamic_cast<qMRMLAbstractNodeItemHelper*>(item.data()))
    {
    return 0;
    }
  /*std::ofstream rowCountDebugFile("rowCount.txt", std::ios_base::app);
  rowCountDebugFile << _parent.row() << " " << _parent.column()
                    << " " << _parent.parent().row() << " " << _parent.parent().column() 
                    << " : " << item->childCount() << std::endl;
  rowCountDebugFile.close();*/
  return !item.isNull() ? item->childCount() : 0;
}

//------------------------------------------------------------------------------
bool qMRMLSceneModel::setData(const QModelIndex &modelIndex, const QVariant &value, int role)
{
  CTK_D(const qMRMLSceneModel);
  if (!modelIndex.isValid())
    {
    return false;
    }
  Q_ASSERT(ctk_d()->MRMLScene);
  QSharedPointer<qMRMLAbstractItemHelper> item = 
    QSharedPointer<qMRMLAbstractItemHelper>(d->itemFromIndex(modelIndex));
  Q_ASSERT(!item.isNull());
  bool changed = !item.isNull() ? item->setData(value, role) : false;
  if (changed)
    {
    emit dataChanged(modelIndex, modelIndex);
    }
  return changed;
}

//------------------------------------------------------------------------------
//bool qMRMLSceneModel::setItemData(const QModelIndex &index, const QMap<int, QVariant> &roles)
//{
//}

//------------------------------------------------------------------------------
Qt::DropActions qMRMLSceneModel::supportedDropActions()const
{
  return Qt::IgnoreAction;
}
