/*=========================================================================

   Program: ParaView
   Module:    $RCSfile$

   Copyright (c) 2005,2006 Sandia Corporation, Kitware Inc.
   All rights reserved.

   ParaView is a free software; you can redistribute it and/or modify it
   under the terms of the ParaView license version 1.2.

   See License_v1.2.txt for the full ParaView license.
   A copy of this license can be obtained by contacting
   Kitware Inc.
   28 Corporate Drive
   Clifton Park, NY 12065
   USA

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

========================================================================*/
#include "pqViewStreamingBehavior.h"

#include "pqActiveObjects.h"
#include "pqApplicationCore.h"
#include "pqServer.h"
#include "pqServerManagerModel.h"
#include "pqView.h"
#include "vtkCamera.h"
#include "vtkClientServerID.h"
#include "vtkClientServerStream.h"
#include "vtkCommand.h"
#include "vtkNew.h"
#include "vtkPVStreamingPiecesInformation.h"
#include "vtkPVView.h"
#include "vtkRenderer.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkSMRenderViewProxy.h"
#include "vtkSMSession.h"

#define PV_DEBUG_STREAMING
#include "vtkPVStreamingMacros.h"

static const int PQ_STREAMING_INTERVAL=1000;

//-----------------------------------------------------------------------------
pqViewStreamingBehavior::pqViewStreamingBehavior(QObject* parentObject)
  : Superclass(parentObject), Pass(0), DelayUpdate(false), DisableAutomaticUpdates(false)
{
  pqServerManagerModel* smmodel =
    pqApplicationCore::instance()->getServerManagerModel();
  QObject::connect(smmodel, SIGNAL(viewAdded(pqView*)),
    this, SLOT(onViewAdded(pqView*)));

  QObject::connect(&this->Timer, SIGNAL(timeout()),
    this, SLOT(onTimeout()));
  this->Timer.setSingleShot(true);

  foreach (pqView* view, smmodel->findItems<pqView*>())
    {
    this->onViewAdded(view);
    }
  this->Thread.start();
  this->Notifier.moveToThread(&this->Thread);
  QObject::connect(&this->Notifier,SIGNAL(streamedDataReady(StreamingUpdateData)),
                   this, SLOT(onStreamedDataReady(StreamingUpdateData)));
  qRegisterMetaType<StreamingUpdateInformation>();
  qRegisterMetaType<StreamingUpdateData>();
}

//-----------------------------------------------------------------------------
pqViewStreamingBehavior::~pqViewStreamingBehavior()
{
  this->Thread.quit();
  this->Thread.wait();
}

//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::onViewAdded(pqView* view)
{
  vtkSMRenderViewProxy* rvProxy =
    vtkSMRenderViewProxy::SafeDownCast(view->getProxy());
  if (rvProxy)
    {
    rvProxy->AddObserver(vtkCommand::UpdateDataEvent,
      this, &pqViewStreamingBehavior::onViewUpdated);
    rvProxy->GetInteractor()->AddObserver(
      vtkCommand::StartInteractionEvent,
      this, &pqViewStreamingBehavior::onStartInteractionEvent);
    rvProxy->GetInteractor()->AddObserver(
      vtkCommand::EndInteractionEvent,
      this, &pqViewStreamingBehavior::onEndInteractionEvent);
    }
}
//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::onViewUpdated(
  vtkObject* vtkNotUsed(caller), unsigned long, void*)
{
  // every time the view "updates", we may have to stream new data and hence we
  // restart the streaming loop.
  if (vtkPVView::GetEnableStreaming())
    {
    vtkStreamingStatusMacro("View updated. Restarting streaming loop.");
    this->Pass = 0;
    if (!this->DisableAutomaticUpdates)
      {
      this->Timer.start(PQ_STREAMING_INTERVAL);
      }
    }
}

//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::onStartInteractionEvent()
{
  this->DelayUpdate = true;
  if (this->Timer.isActive())
    {
    vtkStreamingStatusMacro("Pausing updates while interacting.");
    }
}

//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::onEndInteractionEvent()
{
  this->DelayUpdate = false;
  if (this->Timer.isActive())
    {
    vtkStreamingStatusMacro("Resuming updates since done interacting.");
    }
  else
    {
    if (vtkPVView::GetEnableStreaming())
      {
      this->Timer.start(PQ_STREAMING_INTERVAL);
      vtkStreamingStatusMacro("View interaction changed. Restart streaming loop.");
      }
    }
}

//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::onTimeout()
{
  pqView* view = pqActiveObjects::instance().activeView();
  if (view)
    {
    vtkSMRenderViewProxy* rvProxy = vtkSMRenderViewProxy::SafeDownCast(
      view->getProxy());
    if(rvProxy == NULL)
      {
      // Not the valid active view. Then do nothing
      vtkStreamingStatusMacro("no valid view.");
      return;
      }

    if (!rvProxy)
      {
      return;
      }

    if (rvProxy->GetSession()->GetPendingProgress() ||
      view->getServer()->isProcessingPending() || this->DelayUpdate)
      {
      if (this->DelayUpdate) vtkStreamingStatusMacro("deferring update....");
      if (view->getServer()->isProcessingPending()) vtkStreamingStatusMacro("server is busy....");
      if (rvProxy->GetSession()->GetPendingProgress()) vtkStreamingStatusMacro("session is busy....");
      this->Timer.start(PQ_STREAMING_INTERVAL);
      }
    else
      {
      vtkStreamingStatusMacro("Update Pass: " << this->Pass);
      rvProxy->StreamingUpdate();
//      QTimer::singleShot(0,&this->Notifier,SLOT(waitForStreamedData(vtkSMSession*,double[],vtkIdType));
      StreamingUpdateInformation info;
      info.session = rvProxy->GetSession();
      vtkRenderer* ren = rvProxy->GetRenderer();
      ren->GetActiveCamera()->GetFrustumPlanes(ren->GetTiledAspectRatio(), info.planes);
      info.id = rvProxy->GetGlobalID();
      QMetaObject::invokeMethod(&this->Notifier,"waitForStreamedData",
                                Q_ARG(StreamingUpdateInformation, info));
      }
    }
}

//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::onStreamedDataReady(StreamingUpdateData data)
{
  pqView* view = pqActiveObjects::instance().activeView();
  vtkSMRenderViewProxy* rvProxy = vtkSMRenderViewProxy::SafeDownCast(
                                    view->getProxy());
  std::vector<unsigned int> keys_to_deliver;
  data.info->GetKeys(keys_to_deliver);
  if (keys_to_deliver.size() > 0)
    {
    vtkSMSession* session = rvProxy->GetSession();
    vtkClientServerStream stream;
    stream << vtkClientServerStream::Invoke
           << VTKOBJECT(rvProxy)
           << "DeliverStreamedPieces"
           << static_cast<unsigned int>(keys_to_deliver.size())
           << vtkClientServerStream::InsertArray(
             &keys_to_deliver[0], static_cast<int>(keys_to_deliver.size()))
           << vtkClientServerStream::End;
    session->ExecuteStream(rvProxy->GetLocation(), stream, false);

    rvProxy->StillRender();

    this->Pass++;
    if (this->DisableAutomaticUpdates)
      {
      vtkStreamingStatusMacro("Pausing, since automatic updates are disabled.");
      }
    else
      {
      this->Timer.start(0);
      }
    }
  else
    {
    vtkStreamingStatusMacro("Finished. Stopping loop.");
    }
  data.info->Delete();
}

//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::stopAutoUpdates()
{
  vtkStreamingStatusMacro("Pausing automatic updates.");
  this->Timer.stop();
  this->DisableAutomaticUpdates = true;
}

//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::resumeAutoUpdates()
{
  vtkStreamingStatusMacro("Resuming automatic updates.");
  this->Timer.start(PQ_STREAMING_INTERVAL);
  this->DisableAutomaticUpdates = false;
}

//-----------------------------------------------------------------------------
void pqViewStreamingBehavior::triggerSingleUpdate()
{
  vtkStreamingStatusMacro("Trigger single automatic updates.");
  this->onTimeout();
}

void pqViewStreamingNotifier::waitForStreamedData(StreamingUpdateInformation info)
{
  vtkClientServerStream stream, substream;
  substream << vtkClientServerStream::Invoke
            << vtkClientServerID(1)
            << "GetVTKObject"
            << info.id
            << vtkClientServerStream::End;
  stream << vtkClientServerStream::Invoke
         << substream
         << "StreamingUpdate"
         << vtkClientServerStream::InsertArray(info.planes,24)
         << vtkClientServerStream::End;

  info.session->ExecuteStream(vtkPVSession::SERVERS, stream, false);

  vtkStreamingStatusMacro("Waiting for streamed data.");
  StreamingUpdateData data;
  data.info = vtkPVStreamingPiecesInformation::New();
  info.session->GatherInformation(vtkPVSession::SERVERS, data.info, info.id);
  emit streamedDataReady(data);
}
