// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkSynchronizableOpenGLAvatars.h"
#include "vtkCullerCollection.h"
#include "vtkObjectFactory.h"
#include "vtkOpenGLAvatar.h"
#include "vtkProperty.h"
#include "vtkRenderer.h"

#include "vtkTextProperty.h"

VTK_ABI_NAMESPACE_BEGIN

class vtkSynchronizableOpenGLAvatars::vtkInternals
{
public:
  struct AvatarInfo
  {
    double HeadPos[3];
    double HeadOri[3];
    double LeftHandPos[3];
    double LeftHandOri[3];
    double RightHandPos[3];
    double RightHandOri[3];
    double UpVector[3];
    double Scale[3];
    double Color[3];
    bool UseLeftHand;
    bool UseRightHand;
    bool ShowHandsOnly;
    std::string Label;

    // Save/restore the struct to/from a stream.
    void SaveAvatar(vtkMultiProcessStream& stream)
    {
      stream << this->HeadPos[0] << this->HeadPos[1] << this->HeadPos[2]
             << this->HeadOri[0] << this->HeadOri[1] << this->HeadOri[2]
             << this->LeftHandPos[0] << this->LeftHandPos[1] << this->LeftHandPos[2]
             << this->LeftHandOri[0] << this->LeftHandOri[1] << this->LeftHandOri[2]
             << this->RightHandPos[0] << this->RightHandPos[1] << this->RightHandPos[2]
             << this->RightHandOri[0] << this->RightHandOri[1] << this->RightHandOri[2]
             << this->UpVector[0] << this->UpVector[1] << this->UpVector[2]
             << this->Scale[0] << this->Scale[1] << this->Scale[2]
             << this->Color[0] << this->Color[1] << this->Color[2]
             << this->UseLeftHand << this->UseRightHand << this->ShowHandsOnly
             << this->Label;
    }

    void RestoreAvatar(vtkMultiProcessStream& stream)
    {
      stream >> this->HeadPos[0] >> this->HeadPos[1] >> this->HeadPos[2]
             >> this->HeadOri[0] >> this->HeadOri[1] >> this->HeadOri[2]
             >> this->LeftHandPos[0] >> this->LeftHandPos[1] >> this->LeftHandPos[2]
             >> this->LeftHandOri[0] >> this->LeftHandOri[1] >> this->LeftHandOri[2]
             >> this->RightHandPos[0] >> this->RightHandPos[1] >> this->RightHandPos[2]
             >> this->RightHandOri[0] >> this->RightHandOri[1] >> this->RightHandOri[2]
             >> this->UpVector[0] >> this->UpVector[1] >> this->UpVector[2]
             >> this->Scale[0] >> this->Scale[1] >> this->Scale[2]
             >> this->Color[0] >> this->Color[1] >> this->Color[2]
             >> this->UseLeftHand >> this->UseRightHand >> this->ShowHandsOnly
             >> this->Label;
    }

    void CopyFrom(vtkOpenGLAvatar* avatar)
    {
      avatar->GetHeadPosition(this->HeadPos);
      avatar->GetHeadOrientation(this->HeadOri);
      avatar->GetLeftHandPosition(this->LeftHandPos);
      avatar->GetLeftHandOrientation(this->LeftHandOri);
      avatar->GetRightHandPosition(this->RightHandPos);
      avatar->GetRightHandOrientation(this->RightHandOri);
      avatar->GetUpVector(this->UpVector);
      avatar->GetScale(this->Scale);
      avatar->GetProperty()->GetColor(this->Color);
      this->UseLeftHand = avatar->GetUseLeftHand();
      this->UseRightHand = avatar->GetUseRightHand();
      this->ShowHandsOnly = avatar->GetShowHandsOnly();

      const char* label = static_cast<const char*>(avatar->GetLabel());
      if (label)
      {
        this->Label = label;
      }
    }

    void CopyTo(vtkOpenGLAvatar* avatar)
    {
      avatar->SetHeadPosition(this->HeadPos);
      avatar->SetHeadOrientation(this->HeadOri);
      avatar->SetLeftHandPosition(this->LeftHandPos);
      avatar->SetLeftHandOrientation(this->LeftHandOri);
      avatar->SetRightHandPosition(this->RightHandPos);
      avatar->SetRightHandOrientation(this->RightHandOri);
      avatar->SetUpVector(this->UpVector);
      avatar->SetScale(this->Scale);
      avatar->GetProperty()->SetColor(this->Color);
      avatar->GetLabelTextProperty()->SetColor(this->Color);
      avatar->SetUseLeftHand(this->UseLeftHand);
      avatar->SetUseRightHand(this->UseRightHand);
      avatar->SetShowHandsOnly(this->ShowHandsOnly);

      if (this->Label.size() > 0)
      {
        avatar->SetLabel(this->Label.c_str());
      }
    }
  };

  // Iterate over actors in the renderer and serialize them to the stream
  void SaveCollection(vtkMultiProcessStream& stream, vtkRenderer* renderer)
  {
    std::vector<vtkOpenGLAvatar*> avatars;
    this->GetOpenGLAvatars(renderer, avatars);

    stream << 2906 << static_cast<unsigned int>(avatars.size());

    AvatarInfo aInfo;

    for (size_t i = 0; i < avatars.size(); ++i)
    {
      vtkOpenGLAvatar* avatar = avatars[i];
      aInfo.CopyFrom(avatar);
      aInfo.SaveAvatar(stream);
    }
  }

  // Read from the stream and update actors in the renderer
  bool RestoreCollection(vtkMultiProcessStream& stream, vtkRenderer* renderer)
  {
    std::vector<vtkOpenGLAvatar*> localAvatars;
    this->GetOpenGLAvatars(renderer, localAvatars);

    int tag;
    stream >> tag;
    if (tag != 2906)
    {
      return false;
    }

    unsigned int numRemoteAvatars;

    stream >> numRemoteAvatars;

    // 1) If we don't have enough avatars locally, make new ones
    // 2) Otherwise we have exactly enough or too many locally, so:
    //     a) go through the stream updating local avatar properties
    //     b) any remaining local avatars at the end are unnecessary, remove them

    AvatarInfo aInfo;
    size_t i = 0;

    for (; i < numRemoteAvatars; ++i)
    {
      vtkOpenGLAvatar* avatar;

      if (i >= localAvatars.size()) {
        // There are more remote avatars than local ones, so from here on, we need to make
        // new local avatars and add them to the renderer before updating.
        vtkSmartPointer<vtkOpenGLAvatar> newOne = vtkSmartPointer<vtkOpenGLAvatar>::New();
        avatar = newOne.GetPointer();
        renderer->AddActor(avatar);
      }
      else {
        // Already have a local one to update
        avatar = localAvatars[i];
      }

      aInfo.RestoreAvatar(stream);
      aInfo.CopyTo(avatar);
    }

    // There are more local avatars than remote ones, so remove any extras from
    // the renderer now.
    for (; i < localAvatars.size(); ++i)
    {
      renderer->RemoveActor(localAvatars[i]);
    }

    return true;
  }

  void GetOpenGLAvatars(vtkRenderer* renderer, std::vector<vtkOpenGLAvatar*>& avatarList)
  {
    vtkCollectionSimpleIterator pit;
    vtkPropCollection* props = renderer->GetViewProps();
    vtkProp* aProp;

    for (props->InitTraversal(pit); (aProp = props->GetNextProp(pit));)
    {
     vtkOpenGLAvatar* oglAvatar = vtkOpenGLAvatar::SafeDownCast(aProp);
      if (oglAvatar)
      {
        avatarList.push_back(oglAvatar);
      }
    }
  }

  // should we store the renderer we're given in Initialize() and only ever
  // operate on that one?
  // vtkRenderer* myRenderer;
};

vtkStandardNewMacro(vtkSynchronizableOpenGLAvatars);

//------------------------------------------------------------------------------
vtkSynchronizableOpenGLAvatars::vtkSynchronizableOpenGLAvatars()
  : Internal(new vtkSynchronizableOpenGLAvatars::vtkInternals()) {}

//------------------------------------------------------------------------------
vtkSynchronizableOpenGLAvatars::~vtkSynchronizableOpenGLAvatars()
{

}

//------------------------------------------------------------------------------
void vtkSynchronizableOpenGLAvatars::InitializeRenderer(vtkRenderer* ren)
{
  // if (this->Internal->myRenderer != ren)
  // {
  //   return;
  // }

  ren->GetCullers()->RemoveAllItems();
}

//------------------------------------------------------------------------------
void vtkSynchronizableOpenGLAvatars::CleanUpRenderer(vtkRenderer* ren)
{
  // if (this->Internal->myRenderer != ren)
  // {
  //   return;
  // }

  std::vector<vtkOpenGLAvatar*> localAvatars;
  this->Internal->GetOpenGLAvatars(ren, localAvatars);
  for (size_t i = 0; i < localAvatars.size(); ++i)
  {
    ren->RemoveActor(localAvatars[i]);
  }
}

//------------------------------------------------------------------------------
void vtkSynchronizableOpenGLAvatars::SaveToStream(vtkMultiProcessStream& stream, vtkRenderer* ren)
{
  // Save vtkOpenGLAvatars to the stream

  this->Internal->SaveCollection(stream, ren);
}

//------------------------------------------------------------------------------
void vtkSynchronizableOpenGLAvatars::RestoreFromStream(vtkMultiProcessStream& stream, vtkRenderer* ren)
{
  // Restore vtkOpenGLAvatars from the stream

  this->Internal->RestoreCollection(stream, ren);
}

//------------------------------------------------------------------------------
void vtkSynchronizableOpenGLAvatars::PrintSelf(ostream& os, vtkIndent indent) {}

VTK_ABI_NAMESPACE_END
