// SPDX-FileCopyrightText: Copyright (c) Kitware Inc.
// SPDX-FileCopyrightText: Copyright (c) Sandia Corporation
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkOpenGLCAVECamera.h"

#include "vtkMatrix3x3.h"
#include "vtkMatrix4x4.h"
#include "vtkNew.h"
#include "vtkObjectFactory.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"

VTK_ABI_NAMESPACE_BEGIN
class vtkOpenGLCAVECamera::vtkInternal
{
public:
  vtkNew<vtkMatrix4x4> WorldToPhysicalMatrix;
};

// ----------------------------------------------------------------------------
vtkStandardNewMacro(vtkOpenGLCAVECamera);

// ----------------------------------------------------------------------------
// Constructor method
vtkOpenGLCAVECamera::vtkOpenGLCAVECamera() : Internal(new vtkOpenGLCAVECamera::vtkInternal())
{

}

// ----------------------------------------------------------------------------
vtkOpenGLCAVECamera::~vtkOpenGLCAVECamera() = default;

// ----------------------------------------------------------------------------
void vtkOpenGLCAVECamera::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

// ----------------------------------------------------------------------------
void vtkOpenGLCAVECamera::GetKeyMatrices(vtkRenderer* ren, vtkMatrix4x4*& wcvc, vtkMatrix3x3*& normMat,
  vtkMatrix4x4*& vcdc, vtkMatrix4x4*& wcdc)
{
  vtkRenderWindow* win = ren->GetRenderWindow();

  // has anything that matters camera changed?
  if (ren != this->LastRenderer || this->MTime > this->KeyMatrixTime ||
    ren->GetMTime() > this->KeyMatrixTime || win->GetMTime() > this->KeyMatrixTime)
  {
    // std::cout << "vtkOpenGLCAVECamera::GetKeyMatrices" << std::endl;
    win->GetPhysicalToWorldMatrix(this->Internal->WorldToPhysicalMatrix);
    this->Internal->WorldToPhysicalMatrix->Invert();

    this->WCVCMatrix->DeepCopy(this->GetModelViewTransformMatrix());

    vtkMatrix4x4::Multiply4x4(this->WCVCMatrix, this->Internal->WorldToPhysicalMatrix, this->WCVCMatrix);

    for (int i = 0; i < 3; ++i)
    {
      for (int j = 0; j < 3; ++j)
      {
        this->NormalMatrix->SetElement(i, j, this->WCVCMatrix->GetElement(i, j));
      }
    }
    this->NormalMatrix->Invert();

    this->WCVCMatrix->Transpose();

    this->VCDCMatrix->DeepCopy(
      this->GetProjectionTransformMatrix(ren->GetTiledAspectRatio(), -1, 1));
    this->VCDCMatrix->Transpose();

    vtkMatrix4x4::Multiply4x4(this->WCVCMatrix, this->VCDCMatrix, this->WCDCMatrix);

    this->KeyMatrixTime.Modified();
    this->LastRenderer = ren;
  }

  wcvc = this->WCVCMatrix;
  normMat = this->NormalMatrix;
  vcdc = this->VCDCMatrix;
  wcdc = this->WCDCMatrix;
}
VTK_ABI_NAMESPACE_END
