/*=========================================================================

   Program: ParaView
   Module:    pqBasicWidgetEventPlayer.cxx

   Copyright (c) 2005-2008 Sandia Corporation, Kitware Inc.
   All rights reserved.

   ParaView is a free software; you can redistribute it and/or modify it
   under the terms of the ParaView license version 1.2.

   See License_v1.2.txt for the full ParaView license.
   A copy of this license can be obtained by contacting
   Kitware Inc.
   28 Corporate Drive
   Clifton Park, NY 12065
   USA

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "pqBasicWidgetEventPlayer.h"

#include "pqEventTypes.h"

#include <QApplication>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QKeyEvent>
#include <QRegularExpression>
#include <QWidget>
#include <QtDebug>

#include <stdexcept>

pqBasicWidgetEventPlayer::pqBasicWidgetEventPlayer(QObject* p)
  : pqWidgetEventPlayer(p)
{
}

bool pqBasicWidgetEventPlayer::playEvent(
  QObject* object, const QString& command, const QString& arguments, int eventType, bool& error)
{
  QWidget* widget = qobject_cast<QWidget*>(object);
  if (widget)
  {
    if (eventType == pqEventTypes::ACTION_EVENT)
    {
      {
        if (command == "key")
        {
          QKeyEvent kd(QEvent::KeyPress, arguments.toInt(), Qt::NoModifier);
          QKeyEvent ku(QEvent::KeyRelease, arguments.toInt(), Qt::NoModifier);
          qApp->notify(widget, &kd);
          qApp->notify(widget, &ku);
          return true;
        }
        else if (command == "keyEvent")
        {
          QStringList data = arguments.split(':');
          QKeyEvent ke(static_cast<QEvent::Type>(data[0].toInt()), data[1].toInt(),
            static_cast<Qt::KeyboardModifiers>(data[2].toInt()), data[3], !!data[4].toInt(),
            data[5].toInt());
          qApp->notify(widget, &ke);
          return true;
        }
        else if (command.startsWith("mouse"))
        {
          QStringList args = arguments.split(',');
          if (args.size() == 5)
          { // Old format
            Qt::MouseButtons buttons = static_cast<Qt::MouseButtons>(args[1].toInt());
            Qt::KeyboardModifiers keym = static_cast<Qt::KeyboardModifier>(args[2].toInt());
            int x = args[3].toInt();
            int y = args[4].toInt();
            QPoint pt(x, y);
            if (command == "mouseWheel")
            {
              int delta = args[0].toInt();
              QWheelEvent we(QPoint(x, y), delta, buttons, keym);
              QCoreApplication::sendEvent(object, &we);
              return true;
            }
            Qt::MouseButton button = static_cast<Qt::MouseButton>(args[0].toInt());
            QEvent::Type type = QEvent::MouseButtonPress;
            type = command == "mouseMove" ? QEvent::MouseMove : type;
            type = command == "mouseRelease" ? QEvent::MouseButtonRelease : type;
            type = command == "mouseDblClick" ? QEvent::MouseButtonDblClick : type;
            if (type == QEvent::MouseMove && button != Qt::NoButton)
            {
              // We have not been setting mouse move correctly.
              buttons = button;
              button = Qt::NoButton;
            }
            QMouseEvent e(type, pt, button, buttons, keym);
            qApp->notify(widget, &e);
            return true;
          }
          else
          { // New, verbose format

            QJsonParseError parseError;
            QJsonDocument jsonDocument(QJsonDocument::fromJson(arguments.toUtf8(), &parseError));
            if (jsonDocument.isNull())
            {
              qCritical() << parseError.errorString();
              error = true;
              return true;
            }

            QJsonObject json = jsonDocument.object();

            auto parseInt = [&json, &arguments](const QString& key) {
              if (!json.contains(key))
              {
                throw std::invalid_argument(
                  "Expected " + key.toStdString() + " in " + arguments.toStdString());
              }
              if (!json[key].isDouble())
              {
                throw std::invalid_argument("Expected " + key.toStdString() + " in " +
                  arguments.toStdString() + " to be an integer value");
              }
              return json[key].toInt();
            };

            auto parseQPoint = [&json, &arguments](const QString& key) {
              if (!json.contains(key))
              {
                throw std::invalid_argument(
                  "Expected " + key.toStdString() + " in " + arguments.toStdString());
              }
              if (!json[key].isArray())
              {
                throw std::invalid_argument("Expected " + key.toStdString() + " in " +
                  arguments.toStdString() + " to be an array of size 2");
              }
              QJsonArray array = json[key].toArray();
              if (array.size() != 2)
              {
                throw std::invalid_argument("Expected " + key.toStdString() + " in " +
                  arguments.toStdString() + " to be an array of size 2");
              }
              return QPoint(array.at(0).toInt(), array.at(1).toInt());
            };

            // clang-format off
            const QEvent::Type type = command == "mousePress" ? QEvent::MouseButtonPress
              : command == "mouseMove" ? QEvent::MouseMove
              : command == "mouseRelease" ? QEvent::MouseButtonRelease
              : command == "mouseDblClick" ? QEvent::MouseButtonDblClick
              : QEvent::None;
            // clang-format on

            if (type == QEvent::None)
            {
              qCritical() << "Unknown command " << command;
              error = true;
              return true;
            }

            try
            {
              const QPoint localPos = parseQPoint("local_pos");
              QMouseEvent e(type, localPos, parseQPoint("window_pos"),
                widget->mapToGlobal(localPos),
                static_cast<Qt::MouseButton>(parseInt("trigger_button")),
                static_cast<Qt::MouseButtons>(parseInt("pressed_buttons")),
                static_cast<Qt::KeyboardModifiers>(parseInt("key_modifiers")),
                Qt::MouseEventSynthesizedByApplication);
              qApp->notify(widget, &e);
            }
            catch (const std::exception& e)
            {
              qCritical() << e.what();
              error = true;
            }
            return true;
          }
        }
      }
    }
    else if (eventType == pqEventTypes::CHECK_EVENT)
    {
      // Recover QProperty
      QVariant propertyValue = object->property(command.toUtf8().data());

      // Check it is valid
      if (!propertyValue.isValid())
      {
        QString errorMessage = object->objectName() + " has no valid property named:" + command;
        qCritical() << errorMessage.toUtf8().data();
        error = true;
        return true;
      }

      // Check property value
      if (propertyValue.toString().replace("\t", " ") != arguments)
      {
        QString errorMessage = object->objectName() + " property value is: " +
          propertyValue.toString() + ". Expecting: " + arguments + ".";
        qCritical() << errorMessage.toUtf8().data();
        error = true;
      }
      return true;
    }
  }
  return this->Superclass::playEvent(object, command, arguments, error);
}
