﻿using ImstkUnity;
using System.Collections.Generic;
using System.IO;
using UnityEditor;
using UnityEngine;
using UnityEngine.UIElements;

namespace ImstkEditor
{
    /// <summary>
    /// Defines the gui to fill ImstkSettings
    /// </summary>
    class ImstkSettingsProvider : SettingsProvider
    {
        private ImstkSettings settings;

        public ImstkSettingsProvider(string path, SettingsScope scope = SettingsScope.User) : base(path, scope) { }

        public static bool IsSettingsAvailable() { return File.Exists(ImstkSettings.settingsPath); }

        // Read existing settings or create settings if they don't exist when user clicks them
        public override void OnActivate(string searchContext, VisualElement rootElement) { settings = ImstkSettings.Instance(); }

        public override void OnGUI(string searchContext)
        {
            EditorGUI.BeginChangeCheck();
            //bool useDebug = EditorGUILayout.Toggle("Use Debug", settings.useDebug);
            bool useLogger = EditorGUILayout.Toggle("Use Logger", settings.useLogger);
            bool useOptimalNumberOfThreads = EditorGUILayout.Toggle("Use Optimal # Threads", settings.useOptimalNumberOfThreads);
            int numThreads = settings.numThreads;
           
            if (!useOptimalNumberOfThreads)
                numThreads = EditorGUILayout.IntField("# Threads", numThreads);

            EditorGUILayout.HelpBox(
              "When developer mode is on you will be prompted to install imstk from its repo on startup" +
              " this is done on startup as editor will hook dll's preventing you from installing imstk" +
              " while the editor is running.", MessageType.Warning);
            bool useDevMode = EditorGUILayout.Toggle("Use Developer Mode", settings.useDeveloperMode);

            string installPath = settings.installSourcePath;
            if (useDevMode)
            {
                GUILayout.BeginVertical(EditorStyles.helpBox);
                GUILayout.BeginHorizontal();
                GUILayout.Label("Imstk Install Directory");
                installPath = EditorGUILayout.TextField(installPath);

                if (GUILayout.Button("Open Directory"))
                {
                    installPath = EditorUtility.OpenFolderPanel("Imstk Install Directory (Development Use)", settings.installSourcePath, "");
                }
                GUILayout.EndHorizontal();

                EditorGUILayout.HelpBox("Force install will perform install now. This may not work if " +
                    "a library is already hooked. Depends if any scripts that use native imstk have run yet.",
                    MessageType.Warning);

                if (GUILayout.Button("Force Install"))
                {
                    EditorUtils.installImstk(installPath);
                }
                GUILayout.EndVertical();
            }

            if (EditorGUI.EndChangeCheck())
            {
                Undo.RegisterCompleteObjectUndo(settings, "Change of Settings");
                settings.useDeveloperMode = useDevMode;
                settings.useLogger = useLogger;
                settings.useOptimalNumberOfThreads = useOptimalNumberOfThreads;
                settings.numThreads = numThreads;
                settings.installSourcePath = installPath;

                //string defines = PlayerSettings.GetScriptingDefineSymbolsForGroup(BuildTargetGroup.Standalone);
                //defines = defines.Replace("IMSTK_RELEASE;", "");
                //defines = defines.Replace("IMSTK_RELEASE", "");
                //defines = defines.Replace("IMSTK_DEBUG;", "");
                //defines = defines.Replace("IMSTK_DEBUG", "");
                //if (defines.Length != 0 && defines[defines.Length - 1] != ';')
                //    defines += ';';
                //if (useDebug)
                //    defines += "IMSTK_DEBUG;";
                //else
                //    defines += "IMSTK_RELEASE;";
                //PlayerSettings.SetScriptingDefineSymbolsForGroup(BuildTargetGroup.Standalone, defines);

                EditorUtility.SetDirty(settings);
                AssetDatabase.SaveAssets();
            }
        }

        /// <summary>
        /// Register the SettingsProvider so it can be found in Editor->Project Settings
        /// </summary>
        [SettingsProvider]
        public static SettingsProvider CreateMyCustomSettingsProvider()
        {
            var provider = new ImstkSettingsProvider("Project/Imstk Settings", SettingsScope.Project);
            provider.keywords = new HashSet<string>(new string[] { "Debug", "Logger", "Threads", "Imstk" });
            return provider;
        }
    }
}
