﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.InteropServices;
using UnityEngine;
using UnityEngine.SceneManagement;

namespace Imstk
{
    public enum ModuleStatus
    {
        starting,
        running,
        pausing,
        paused,
        terminating,
        inactive
    };

    [AddComponentMenu("Imstk/SimulationManager")]
    [DefaultExecutionOrder(-99)]
    public class SimulationManager : MonoBehaviour
    {
        public static SimulationManager simManager;

        public delegate void PostUpdateCallbackHandler(EventArgs e);
        public static event PostUpdateCallbackHandler PostUpdateCallback;

        public delegate void PreUpdateCallbackHandler(EventArgs e);
        public static event PreUpdateCallbackHandler PreUpdateCallback;


        protected static void PostUpdateInvoke() { PostUpdateCallback?.Invoke(new EventArgs()); }
        protected static void PreUpdateInvoke() { PreUpdateCallback?.Invoke(new EventArgs()); }

        /// <summary>
        /// Returns all the ImstkBehaviours
        /// </summary>
        /// <returns></returns>
        public List<ImstkBehaviour> getAllBehaviours()
        {
            List<ImstkBehaviour> behaviours = new List<ImstkBehaviour>();
            List<GameObject> objects = FindObjectsOfType<GameObject>().ToList();
            foreach (GameObject obj in objects)
            {
                behaviours.AddRange(obj.GetComponents<ImstkBehaviour>());
            }
            return behaviours;
        }

        // Sets the collision graph on the active scene
        public void setCollisionGraph(CollisionGraph graph)
        {
            // The collision graph uses instance ids, but the imported graph uses names
            // so lookup instance ids from names
            List<GameObject> objects = FindObjectsOfType<GameObject>().ToList();
            foreach (InteractionPair pair in graph.pairs)
            {
                GameObject obj1 = objects.Find(x => { return x.name == pair.object1Name; });
                GameObject obj2 = objects.Find(x => { return x.name == pair.object2Name; });
                if (obj1 == null || obj2 == null)
                    continue;
                DynamicalModel model1 = obj1.GetComponent<DynamicalModel>();
                DynamicalModel model2 = obj2.GetComponent<DynamicalModel>();
                if (model1 == null || model2 == null)
                    continue;
                addInteractionPair(model1.GetHandle(), model2.GetHandle(), (int)pair.interactionType, (int)pair.collisionDetectionType);
            }
        }


        private void Awake()
        {
            simManager = this;

            // Get the settings
            ImstkSettings settings = ImstkSettings.Instance();
            if (settings.useOptimalNumberOfThreads)
                settings.numThreads = 0;

            // Register reverse pinvoke function
            Logger.Register();

            // Start logger
            if (settings.useLogger)
                Logger.genLogger();

            setSimManagerPostUpdateCallback(Marshal.GetFunctionPointerForDelegate(new Action(PostUpdateInvoke)));
            setSimManagerPreUpdateCallback(Marshal.GetFunctionPointerForDelegate(new Action(PreUpdateInvoke)));

            // Create the simulation manager
            genSimManager(settings.numThreads);

            // Init order
            {
                List<ImstkBehaviour> behaviours = getAllBehaviours();
                foreach (ImstkBehaviour behaviour in behaviours)
                {
                    behaviour.ImstkInit();
                }
            }

            // After objects are added, but before init, add interactions if a collision graph is present
            CollisionGraphData data = CollisionGraphData.Read(SceneManager.GetActiveScene().name + "_CollisionGraphData");
            if (data != null)
                setCollisionGraph(data.ToCollisionGraph());

            initSimManager();

            setWriteTaskGraph(settings.writeTaskGraph);
        }

        private void Start()
        {
            // Start order
            {
                List<ImstkBehaviour> behaviours = getAllBehaviours();
                foreach (ImstkBehaviour behaviour in behaviours)
                {
                    behaviour.ImstkStart();
                }
            }

            startSimManager();
        }


        public void OnApplicationQuit()
        {
            // Destroy order
            {
                List<ImstkBehaviour> behaviours = getAllBehaviours();
                foreach (ImstkBehaviour behaviour in behaviours)
                {
                    behaviour.ImstkDestroy();
                }
            }

            stopSimManager();
            deleteSimManager();
            Logger.deleteLogger();
        }


        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void genSimManager(int numThreads);

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void initSimManager();

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void setSimManagerPostUpdateCallback(IntPtr funcPtr);
        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void setSimManagerPreUpdateCallback(IntPtr funcPtr);

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void setWriteTaskGraph(bool writeTaskGraph);

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void addInteractionPair(IntPtr objAInstanceId, IntPtr objBInstanceId, int interactionType, int collisionDetectionType);

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern void startSimManager();

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern void advanceFrame(double dt);

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern double getActiveSceneDt();

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern void pauseSimManager();

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern void resumeSimManager();

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern ModuleStatus getSimManagerStatus();

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern IntPtr getTaskGraph();

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern void stopSimManager();

        [DllImport(PInvoke.ImstkUnityLibName)]
        public static extern void deleteSimManager();
    }
}