﻿using System;
using UnityEngine;

namespace Imstk
{
    /// <summary>
    /// This is the base class of Imstk scripts. It exists to provide
    /// different init functions for Imstk classes. This is so that we
    /// can control initialization order in the SimulationManager
    /// </summary>
    public abstract class ImstkBehaviour : MonoBehaviour
    {
        bool isPreUpdating = false;
        bool isPostUpdating = false;

        private void ImstkPreUpdate(EventArgs e) { OnImstkPreUpdate(); }
        private void ImstkPostUpdate(EventArgs e) { OnImstkPostUpdate(); }

        /// <summary>
        /// Subclasses should call this function if they want ImstkPreUpdate to be called on this behaviour
        /// </summary>
        public void RegisterPreUpdate()
        {
            if (!isPreUpdating)
            {
                SimulationManager.PreUpdateCallback += ImstkPreUpdate;
                isPreUpdating = true;
            }
        }
        /// <summary>
        /// Subclasses should call this function if they want ImstkPostUpdate to be called on this behaviour
        /// </summary>
        public void RegisterPostUpdate()
        {
            if (!isPostUpdating)
            {
                SimulationManager.PostUpdateCallback += ImstkPostUpdate;
                isPostUpdating = true;
            }
        }

        public void ImstkDestroy()
        {
            if (isPreUpdating)
            {
                SimulationManager.PreUpdateCallback -= ImstkPreUpdate;
                isPreUpdating = false;
            }
            if (isPostUpdating)
            {
                SimulationManager.PostUpdateCallback -= ImstkPostUpdate;
                isPostUpdating = false;
            }
            OnImstkDestroy();
        }

        public void ImstkInit() { OnImstkInit(); }

        public void ImstkStart() { OnImstkStart(); }


        /// <summary>
        /// Called before initializing the scene
        /// </summary>
        protected virtual void OnImstkInit() { }

        /// <summary>
        /// Called after scene has been initialized
        /// </summary>
        protected virtual void OnImstkStart() { }

        /// <summary>
        /// Called before the Imstk scene advances
        /// </summary>
        protected virtual void OnImstkPreUpdate() { }

        /// <summary>
        /// Called after the Imstk scene advances
        /// </summary>
        protected virtual void OnImstkPostUpdate() { }


        /// <summary>
        /// Called when done
        /// </summary>
        protected virtual void OnImstkDestroy() { }
    }
}
