﻿using System;
using System.Collections.Generic;
using System.Runtime.InteropServices;
using UnityEngine;

namespace Imstk
{
    [RequireComponent(typeof(Transform))]
    [RequireComponent(typeof(MeshFilter))]
    public abstract class DynamicalModel : ImstkBehaviour
    {
        // Components
        //protected SceneObject sceneObjectComp = null;
        protected MeshFilter meshFilter = null;
        protected PhysicsGeometry physicsGeometryComp = null;
        protected CollisionGeometry collisionGeometryComp = null;
        protected Transform transformComp = null;
        public Matrix4x4 initTransform;

        protected Geometry physicsGeometry = null;
        protected IntPtr physicsGeometryHandle = IntPtr.Zero;
        protected Geometry visualGeometry = null;
        protected IntPtr visualGeometryHandle = IntPtr.Zero;
        protected Geometry collisionGeometry = null;
        protected IntPtr collisionGeometryHandle = IntPtr.Zero;

        private IntPtr handle = IntPtr.Zero;

        /// <summary>
        /// Get the pointer to the object in c (not available until after initialize)
        /// </summary>
        /// <returns></returns>
        public IntPtr GetHandle() { return handle; }

        /// <summary>
        /// Generates the object on the c side, subclasses define these constructors
        /// </summary>
        /// <returns></returns>
        protected abstract IntPtr genObject();

        protected override void OnImstkInit()
        {
            // Get dependencies
            //sceneObjectComp = gameObject.GetComponentOrCreate<SceneObject>();
            meshFilter = gameObject.GetComponentFatal<MeshFilter>();
            physicsGeometryComp = gameObject.GetComponentOrCreate<PhysicsGeometry>();
            collisionGeometryComp = gameObject.GetComponentOrCreate<CollisionGeometry>();
            transformComp = gameObject.GetComponentFatal<Transform>();

            // Make sure mesh filter is read/writable
            meshFilter.mesh.MarkDynamic();
            if (!meshFilter.mesh.isReadable)
            {
                Debug.LogWarning(gameObject.name + "'s MeshFilter Mesh must be readable (check the meshes import settings)");
                return;
            }

            // Initialize the object
            handle = genObject();
            if (handle == IntPtr.Zero)
                Debug.Log("Failed to create dynamic object, handle already exists");

            InitObject();
            InitGeometry();
            ProcessBoundaryConditions(gameObject.GetComponents<BoundaryCondition>());
            Configure();

            // Reset the transform (so transform isn't applied twice)
            // Unfortunately this means any component that needs the transform will get bad values
            // Instead initTransform is provided here as the transform will be reset
            initTransform = transformComp.localToWorldMatrix;
            transformComp.localPosition = Vector3.zero;
            transformComp.localScale = Vector3.one;
            transformComp.localRotation = Quaternion.identity;
        }

        protected abstract void InitObject();

        protected abstract void Configure();

        /// <summary>
        /// Each subclassed model may *apply* boundary conditions differently
        /// </summary>
        /// <param name="conditions">All the conditions to be processed</param>
        protected virtual void ProcessBoundaryConditions(BoundaryCondition[] conditions) { }

        protected void InitGeometry()
        {
            // Setup the visual geometry
            {
                visualGeometry = Geometry.MeshToGeometry(meshFilter.mesh, transform.localToWorldMatrix);
                // Create geometry obj in c api
                visualGeometryHandle = Geometry.ExportGeometry(visualGeometry);
                setDynamicObjectVisualGeometry(this.GetHandle(), visualGeometryHandle);
            }

            // Setup the collision geometry
            {
                // If user wants collision = visual
                if (collisionGeometryComp.usage == CollisionGeometryUsage.USE_VISUAL_GEOMETRY)
                {
                    collisionGeometry = visualGeometry;
                    setDynamicObjectCollisionGeometry(this.GetHandle(), visualGeometryHandle);
                    collisionGeometryHandle = visualGeometryHandle;
                }
                // If user wants collision = physics
                else if (collisionGeometryComp.usage == CollisionGeometryUsage.USE_PHYSICS_GEOMETRY)
                    Debug.LogError("Collision geometry cannot = physics geometry yet");
                // If use own geometry
                else
                {
                    // Make a transformed copy
                    collisionGeometry = ScriptableObject.CreateInstance<Geometry>();
                    if (collisionGeometryComp.IsImstkMesh)
                    {
                        (collisionGeometryComp.mesh as Geometry).CopyTo(collisionGeometry);
                        collisionGeometry.Transform(transformComp.localToWorldMatrix);
                    }
                    else
                        collisionGeometry = Geometry.MeshToGeometry(collisionGeometryComp.mesh as Mesh, transformComp.localToWorldMatrix);

                    // Create geometry in the c api
                    collisionGeometryHandle = Geometry.ExportGeometry(collisionGeometry);
                    setDynamicObjectCollisionGeometry(this.GetHandle(), collisionGeometryHandle);
                }
            }

            // Setup the physics geometry
            {
                // If user wants physics = visual
                if (physicsGeometryComp.usage == PhysicsGeometryUsage.USE_VISUAL_GEOMETRY)
                {
                    physicsGeometry = visualGeometry;
                    setDynamicObjectPhysicsGeometry(this.GetHandle(), visualGeometryHandle);
                    physicsGeometryHandle = visualGeometryHandle;
                }
                // If use own geometry
                else
                {
                    // Make a transformed copy
                    physicsGeometry = ScriptableObject.CreateInstance<Geometry>();
                    if (physicsGeometryComp.IsImstkMesh)
                    {
                        (physicsGeometryComp.mesh as Geometry).CopyTo(physicsGeometry);
                        physicsGeometry.Transform(transformComp.localToWorldMatrix);
                    }
                    else
                        physicsGeometry = Geometry.MeshToGeometry(physicsGeometryComp.mesh as Mesh, transformComp.localToWorldMatrix);

                    // Create geometry in the c api
                    physicsGeometryHandle = Geometry.ExportGeometry(physicsGeometry);
                    setDynamicObjectPhysicsGeometry(this.GetHandle(), physicsGeometryHandle);
                }
            }

            // Map the physics to collision geometry
            if (collisionGeometryComp.usage == CollisionGeometryUsage.USE_OWN_GEOMETRY)
                addDynamicObjectPhysicsToCollidingMap(this.GetHandle());

            // Map the physics to visual geometry
            if (physicsGeometryComp.usage == PhysicsGeometryUsage.USE_OWN_GEOMETRY)
                addDynamicObjectPhysicsToVisualMap(this.GetHandle());
        }


        public void Update()
        {
            // We don't import the entire geometry (we don't need to copy the whole thing)
            // just the vertices. In later implementations we'll import/export and ideally couple data arrays
            visualGeometry.PinVertices();
            updateDynamicObjectVisualGeometry(this.GetHandle(), visualGeometry.gcVertices.AddrOfPinnedObject());
            visualGeometry.FreeVertices();

            // Copy geometry to GPU
            meshFilter.mesh.vertices = visualGeometry.vertices;
            meshFilter.mesh.RecalculateNormals();
            //meshFilter.mesh.RecalculateBounds();
        }


        protected override void OnImstkDestroy()
        {
            // Delete non duplicate handles
            HashSet<IntPtr> geomHandles = new HashSet<IntPtr>();
            geomHandles.Add(visualGeometryHandle);
            geomHandles.Add(physicsGeometryHandle);
            geomHandles.Add(collisionGeometryHandle);
            foreach (IntPtr val in geomHandles)
            {
                if (val != IntPtr.Zero)
                    Geometry.deleteGeometry(val);
            }

            deleteDynamicObject(this.GetHandle());
        }

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void addDynamicObjectCollidingToVisualMap(IntPtr objectHandle);

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void addDynamicObjectPhysicsToCollidingMap(IntPtr objectHandle);

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void addDynamicObjectPhysicsToVisualMap(IntPtr objectHandle);


        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void setDynamicObjectPhysicsGeometry(IntPtr objectHandle, IntPtr geometryHandle);

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void setDynamicObjectVisualGeometry(IntPtr objectHandle, IntPtr geometryHandle);

        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void setDynamicObjectCollisionGeometry(IntPtr objectHandle, IntPtr geometryHandle);


        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void updateDynamicObjectVisualGeometry(IntPtr objectHandle, IntPtr vertexBufferPtr);


        [DllImport(PInvoke.ImstkUnityLibName)]
        protected static extern void deleteDynamicObject(IntPtr objectHandle);
    }
}