/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkWebGPUDevice.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

  This software is distributed WITHOUT ANY WARRANTY; without even
  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class vtkWebGPUDevice
 * @brief WebGPU device manager
 *
 */

#ifndef vtkWebGPUDevice_h
#define vtkWebGPUDevice_h

// vtk includes
#include "vtkCommand.h" // for custom events
#include "vtkObject.h"
#include "vtkRenderingWebGPUModule.h" // for export macro
#include "vtk_wgpu.h"                 // for webgpu

// STL includes
#include <map>
#include <string>

VTK_ABI_NAMESPACE_BEGIN
// Forward declarations
class vtkWebGPUCommandEncoder;
class vtkWebGPUPipeline;

class VTKRENDERINGWEBGPU_EXPORT vtkWebGPUDevice : public vtkObject
{
public:
  /**
   * Instantiate the class.
   */
  static vtkWebGPUDevice* New();

  ///@{
  /**
   * Standard methods for the VTK class.
   */
  vtkTypeMacro(vtkWebGPUDevice, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) override;
  ///@}

  ///@{
  /**
   * Create/Destroy an instance of the webgpu context with an adapter
   * Ideally, this should be called only once by vtkWebGPUInstance::Create.
   */
  void Create(WGPUAdapter a);
  void Destroy();
  ///@}

  ///@{
  /**
   * Set/Get a handle to the webgpu device
   */
  void SetHandle(WGPUDevice d);
  WGPUDevice GetHandle();
  ///@}

  ///@{
  /**
   * Set/Get the device label
   */
  vtkSetStringMacro(Label);
  vtkGetStringMacro(Label);
  ///@}

  enum vtkCustomEvents
  {
    DeviceRequestedEvent = vtkCommand::UserEvent + 101,
    DeviceErrorEvent,
    DeviceLostEvent,
    QueueWorkDoneEvent
  };

  /**
   * Report capabilities of the webgpu context
   */
  const char* ReportCapabilities();

  /**
   * Get the comamnd encoder
   */
  virtual vtkWebGPUCommandEncoder* GetCommandEncoder();

  /**
   * Initialize and cache a pipeline with the device
   */
  virtual void CreatePipeline(std::string pHash, vtkWebGPUPipeline*);

  /**
   * Given the pipeline hash, get the associated pipeline.
   * Retuns nullptr if none found.
   */
  vtkWebGPUPipeline* GetPipeline(std::string pipelineHash);

protected:
  vtkWebGPUDevice();
  ~vtkWebGPUDevice();

  // Helper members
  WGPUDevice Device = nullptr;
  char* Label = nullptr;
  char* Capabilities = nullptr;
  vtkWebGPUCommandEncoder* CommandEncoder = nullptr;
  std::map<std::string, vtkWebGPUPipeline*> Pipelines;

  // Callbacks
  static void OnDeviceRequested(
    WGPURequestDeviceStatus status, WGPUDevice device, const char* message, void* self);
  static void OnDeviceError(WGPUErrorType e, const char* message, void* self);
  static void OnDeviceLost(WGPUDeviceLostReason e, const char* message, void* self);
  static void OnSubmittedWorkDoneEvent(WGPUQueueWorkDoneStatus status, void* self);

private:
  vtkWebGPUDevice(const vtkWebGPUDevice&) = delete;
  void operator=(const vtkWebGPUDevice) = delete;
};

VTK_ABI_NAMESPACE_END
#endif // vtkWebGPUDevice_h
