/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkOpenGLGPUVolumeRayCastMapperInternals.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkOpenGLGPUVolumeRayCastMapperInternals
 * @brief   Implementation details for vtkOpenGLGPUVolumeRayCastMapper
 *
 * vtkOpenGLGPUVolumeRayCastMapperInternals is used in internal implementation
 * of vtkOpenGLGPUVolumeRayCastMapper. This should only be accessed by friends
 * and sub-classes of that class.
*/

#ifndef vtkOpenGLGPUVolumeRayCastMapperInternals_h
#define vtkOpenGLGPUVolumeRayCastMapperInternals_h

// VTK includes
#include "vtkNew.h"
#include "vtkSmartPointer.h"
#include "vtkTimeStamp.h"

// GLEW
#include <vtk_glew.h>

// STL includes
#include <limits>
#include <map>
#include <sstream>
#include <string>
#include <vector>

// Forward declarations
class vtkActor;
class vtkContourFilter;
class vtkDataArray;
class vtkImageData;
class vtkMatrix4x4;
class vtkOpenGLBufferObject;
class vtkOpenGLFramebufferObject;
class vtkOpenGLGPUVolumeRayCastMapper;
class vtkOpenGLShaderCache;
class vtkOpenGLTransferFunctions2D;
class vtkOpenGLVertexArrayObject;
class vtkOpenGLVolumeGradientOpacityTables;
class vtkOpenGLVolumeOpacityTables;
class vtkOpenGLVolumeRGBTable;
class vtkOpenGLVolumeRGBTables;
class vtkPolyData;
class vtkPolyDataMapper;
class vtkRenderWindow;
class vtkRenderer;
class vtkShaderProgram;
class vtkTextureObject;
class vtkVolume;
class vtkVolumeProperty;
class vtkVolumeTexture;
class vtkWindow;

class vtkOpenGLGPUVolumeRayCastMapperInternals
{
public:
  vtkOpenGLGPUVolumeRayCastMapperInternals(
    vtkOpenGLGPUVolumeRayCastMapper* parent);
  ~vtkOpenGLGPUVolumeRayCastMapperInternals();

  // Helper methods
  //--------------------------------------------------------------------------
  template<typename T>
  static void ToFloat(const T& in1, const T& in2, float (&out)[2]);
  template<typename T>
  static void ToFloat(const T& in1,
    const T& in2,
    const T& in3,
    float (&out)[3]);
  template<typename T>
  static void ToFloat(T* in, float* out, int noOfComponents);
  template<typename T>
  static void ToFloat(T (&in)[3], float (&out)[3]);
  template<typename T>
  static void ToFloat(T (&in)[2], float (&out)[2]);
  template<typename T>
  static void ToFloat(T& in, float& out);
  template<typename T>
  static void ToFloat(T (&in)[4][2], float (&out)[4][2]);

  ///@{
  /**
   * \brief Setup and clean-up 1D and 2D transfer functions.
   */
  void InitializeTransferFunction(vtkRenderer* ren,
    vtkVolume* vol,
    int noOfComponents,
    int independentComponents);

  void UpdateTransferFunction(vtkRenderer* ren,
    vtkVolume* vol,
    int noOfComponents,
    int independentComponents);

  void ActivateTransferFunction(vtkShaderProgram* prog,
    vtkVolumeProperty* volProp,
    int numSamplers);

  void DeactivateTransferFunction(vtkVolumeProperty* volumeProperty,
    int numSamplers);

  void SetupTransferFunction1D(vtkRenderer* ren,
    int noOfComponents,
    int independentComponents);
  void ReleaseGraphicsTransfer1D(vtkWindow* window);
  void DeleteTransfer1D();

  void SetupTransferFunction2D(vtkRenderer* ren,
    int noOfComponents,
    int independentComponents);
  void ReleaseGraphicsTransfer2D(vtkWindow* window);
  void DeleteTransfer2D();
  ///@}

  bool LoadMask(vtkRenderer* ren,
    vtkImageData* input,
    vtkImageData* maskInput,
    vtkVolume* volume);

  bool LoadData(vtkRenderer* ren,
    vtkVolume* vol,
    vtkVolumeProperty* volProp,
    vtkImageData* input,
    vtkDataArray* scalars);

  void ComputeBounds(vtkImageData* input);

  // Update OpenGL volume information
  void UpdateVolume(vtkVolumeProperty* volumeProperty);

  // Update transfer color function based on the incoming inputs
  // and number of scalar components.
  int UpdateColorTransferFunction(vtkRenderer* ren,
    vtkVolume* vol,
    unsigned int component);

  // Scalar opacity
  int UpdateOpacityTransferFunction(vtkRenderer* ren,
    vtkVolume* vol,
    unsigned int component);

  int UpdateGradientOpacityTransferFunction(vtkRenderer* ren,
    vtkVolume* vol,
    unsigned int component);

  void UpdateTransferFunction2D(vtkRenderer* ren,
    vtkVolume* vol,
    unsigned int component);

  // Update noise texture (used to reduce rendering artifacts
  // specifically banding effects)
  void CreateNoiseTexture(vtkRenderer* ren);

  // Update depth texture (used for early termination of the ray)
  void CaptureDepthTexture(vtkRenderer* ren, vtkVolume* vol);

  // Test if camera is inside the volume geometry
  bool IsCameraInside(vtkRenderer* ren, vtkVolume* vol);

  // Compute transformation from cell texture-coordinates to point
  // texture-coords
  // (CTP). Cell data maps correctly to OpenGL cells, point data does not (VTK
  // defines points at the cell corners). To set the point data in the center of
  // the
  // OpenGL texels, a translation of 0.5 texels is applied, and the range is
  // rescaled
  // to the point range.
  //
  // delta = TextureExtentsMax - TextureExtentsMin;
  // min   = vec3(0.5) / delta;
  // max   = (delta - vec3(0.5)) / delta;
  // range = max - min
  //
  // CTP = translation * Scale
  // CTP = range.x,        0,        0,  min.x
  //             0,  range.y,        0,  min.y
  //             0,        0,  range.z,  min.z
  //             0,        0,        0,    1.0
  void ComputeCellToPointMatrix();

  // Update parameters for lighting that will be used in the shader.
  void SetLightingParameters(vtkRenderer* ren,
    vtkShaderProgram* prog,
    vtkVolume* vol);

  // Update the volume geometry
  void RenderVolumeGeometry(vtkRenderer* ren,
    vtkShaderProgram* prog,
    vtkVolume* vol);

  // Update cropping params to shader
  void SetCroppingRegions(vtkRenderer* ren,
    vtkShaderProgram* prog,
    vtkVolume* vol);

  // Update clipping params to shader
  void SetClippingPlanes(vtkRenderer* ren,
    vtkShaderProgram* prog,
    vtkVolume* vol);

  // Check if the mapper should enter picking mode.
  void CheckPickingState(vtkRenderer* ren);

  // Look for property keys used to control the mapper's state.
  // This is necessary for some render passes which need to ensure
  // a specific OpenGL state when rendering through this mapper.
  void CheckPropertyKeys(vtkVolume* vol);

  // Configure the vtkHardwareSelector to begin a picking pass.
  void BeginPicking(vtkRenderer* ren);

  // Update the prop Id if hardware selection is enabled.
  void SetPickingId(vtkRenderer* ren);

  // Configure the vtkHardwareSelector to end a picking pass.
  void EndPicking(vtkRenderer* ren);

  // Load OpenGL extensiosn required to grab depth sampler buffer
  void LoadRequireDepthTextureExtensions(vtkRenderWindow* renWin);

  // Create GL buffers
  void CreateBufferObjects();

  // Dispose / free GL buffers
  void DeleteBufferObjects();

  // Convert vtkTextureObject to vtkImageData
  void ConvertTextureToImageData(vtkTextureObject* texture,
    vtkImageData* output);

  // Render to texture for final rendering
  void SetupRenderToTexture(vtkRenderer* ren);
  void ExitRenderToTexture(vtkRenderer* ren);

  // Render to texture for depth pass
  void SetupDepthPass(vtkRenderer* ren);
  void ExitDepthPass(vtkRenderer* ren);

  //@{
  /**
   * Image XY-Sampling
   * Render to an internal framebuffer with lower resolution than the currently
   * bound one (hence casting less rays and improving performance). The rendered
   * image is subsequently rendered as a texture-mapped quad (linearly
   * interpolated) to the default (or previously attached) framebuffer. If a
   * vtkOpenGLRenderPass is attached, a variable number of render targets are
   * supported (as specified by the RenderPass). The render targets are assumed
   * to be ordered from GL_COLOR_ATTACHMENT0 to GL_COLOR_ATTACHMENT$N$, where
   * $N$ is the number of targets specified (targets of the previously bound
   * framebuffer as activated through ActivateDrawBuffers(int)). Without a
   * RenderPass attached, it relies on FramebufferObject to re-activate the
   * appropriate previous DrawBuffer.
   *
   * \sa vtkOpenGLRenderPass vtkOpenGLFramebufferObject
   */
  void BeginImageSample(vtkRenderer* ren, vtkVolume* vol);
  bool InitializeImageSampleFBO(vtkRenderer* ren);
  void EndImageSample(vtkRenderer* ren);
  size_t GetNumImageSampleDrawBuffers(vtkVolume* vol);
  //@}

  void ReleaseRenderToTextureGraphicsResources(vtkWindow* win);
  void ReleaseImageSampleGraphicsResources(vtkWindow* win);
  void ReleaseDepthPassGraphicsResources(vtkWindow* win);

  // Member variables
  //--------------------------------------------------------------------------
  vtkOpenGLGPUVolumeRayCastMapper* Parent;

  bool ValidTransferFunction;
  bool LoadDepthTextureExtensionsSucceeded;
  bool CameraWasInsideInLastUpdate;

  GLuint CubeVBOId;
  GLuint CubeVAOId;
  GLuint CubeIndicesId;

  vtkTextureObject* NoiseTextureObject;
  vtkTextureObject* DepthTextureObject;

  int TextureWidth;

  std::vector<double> Scale;
  std::vector<double> Bias;

  float* NoiseTextureData;

  float ActualSampleDistance;

  int LastProjectionParallel;
  int Dimensions[3];
  int TextureSize[3];
  int WindowLowerLeft[2];
  int WindowSize[2];
  int LastDepthPassWindowSize[2];
  int LastRenderToImageWindowSize[2];

  double LoadedBounds[6];
  int Extents[6];
  double DatasetStepSize[3];
  double CellScale[3];
  double CellStep[3];
  double CellSpacing[3];

  int NumberOfLights;
  int LightComplexity;

  std::ostringstream ExtensionsStringStream;

  vtkOpenGLVolumeRGBTables* RGBTables;
  std::map<int, std::string> RGBTablesMap;

  vtkOpenGLVolumeOpacityTables* OpacityTables;
  std::map<int, std::string> OpacityTablesMap;

  vtkOpenGLVolumeRGBTable* Mask1RGBTable;
  vtkOpenGLVolumeRGBTable* Mask2RGBTable;
  vtkOpenGLVolumeGradientOpacityTables* GradientOpacityTables;
  std::map<int, std::string> GradientOpacityTablesMap;

  vtkOpenGLTransferFunctions2D* TransferFunctions2D;
  std::map<int, std::string> TransferFunctions2DMap;
  vtkTimeStamp Transfer2DTime;

  vtkTimeStamp ShaderBuildTime;

  vtkNew<vtkMatrix4x4> TextureToDataSetMat;
  vtkNew<vtkMatrix4x4> InverseTextureToDataSetMat;

  vtkNew<vtkMatrix4x4> InverseProjectionMat;
  vtkNew<vtkMatrix4x4> InverseModelViewMat;
  vtkNew<vtkMatrix4x4> InverseVolumeMat;

  vtkNew<vtkMatrix4x4> TextureToEyeTransposeInverse;

  vtkNew<vtkMatrix4x4> TempMatrix1;

  vtkNew<vtkMatrix4x4> CellToPointMatrix;
  float AdjustedTexMin[4];
  float AdjustedTexMax[4];

  vtkSmartPointer<vtkPolyData> BBoxPolyData;

  vtkSmartPointer<vtkVolumeTexture> CurrentMask;

  vtkTimeStamp InitializationTime;
  vtkTimeStamp InputUpdateTime;
  vtkTimeStamp VolumeUpdateTime;
  vtkTimeStamp MaskUpdateTime;
  vtkTimeStamp ReleaseResourcesTime;
  vtkTimeStamp DepthPassTime;
  vtkTimeStamp DepthPassSetupTime;
  vtkTimeStamp SelectionStateTime;
  int CurrentSelectionPass;
  bool IsPicking;

  bool NeedToInitializeResources;
  bool PreserveViewport;
  bool PreserveGLState;

  vtkShaderProgram* ShaderProgram;
  vtkOpenGLShaderCache* ShaderCache;

  vtkOpenGLFramebufferObject* FBO;
  vtkTextureObject* RTTDepthBufferTextureObject;
  vtkTextureObject* RTTDepthTextureObject;
  vtkTextureObject* RTTColorTextureObject;
  int RTTDepthTextureType;

  vtkOpenGLFramebufferObject* DPFBO;
  vtkTextureObject* DPDepthBufferTextureObject;
  vtkTextureObject* DPColorTextureObject;

  vtkOpenGLFramebufferObject* ImageSampleFBO = nullptr;
  std::vector<vtkSmartPointer<vtkTextureObject>> ImageSampleTexture;
  std::vector<std::string> ImageSampleTexNames;
  vtkShaderProgram* ImageSampleProg = nullptr;
  vtkOpenGLVertexArrayObject* ImageSampleVAO = nullptr;
  vtkOpenGLBufferObject* ImageSampleVBO = nullptr;
  size_t NumImageSampleDrawBuffers = 0;
  bool RebuildImageSampleProg = false;
  bool RenderPassAttached = false;

  vtkNew<vtkContourFilter> ContourFilter;
  vtkNew<vtkPolyDataMapper> ContourMapper;
  vtkNew<vtkActor> ContourActor;
};

#endif // vtkOpenGLGPUVolumeRayCastMapperInternals_h
// VTK-HeaderTest-Exclude: vtkOpenGLGPUVolumeRayCastMapperInternals.h
