/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkOpenGLGPUVolumeRayCastMapperInternals.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkOpenGLGPUVolumeRayCastMapperInternals.h"

#include "vtkOpenGLTransferFunction2D.h"
#include "vtkOpenGLVolumeGradientOpacityTable.h"
#include "vtkOpenGLVolumeOpacityTable.h"
#include "vtkOpenGLVolumeRGBTable.h"
#include "vtkVolumeShaderComposer.h"
#include "vtkVolumeStateRAII.h"

#include "vtkActor.h"
#include "vtkClipConvexPolyData.h"
#include "vtkContourFilter.h"
#include "vtkDensifyPolyData.h"
#include "vtkHardwareSelector.h"
#include "vtkInformation.h"
#include "vtkLight.h"
#include "vtkMatrix4x4.h"
#include "vtkOpenGLActor.h"
#include "vtkOpenGLBufferObject.h"
#include "vtkOpenGLError.h"
#include "vtkOpenGLFramebufferObject.h"
#include "vtkOpenGLGPUVolumeRayCastMapper.h"
#include "vtkOpenGLRenderPass.h"
#include "vtkOpenGLRenderUtilities.h"
#include "vtkOpenGLRenderWindow.h"
#include "vtkOpenGLShaderCache.h"
#include "vtkOpenGLVertexArrayObject.h"
#include "vtkPerlinNoise.h"
#include "vtkPixelBufferObject.h"
#include "vtkPixelExtent.h"
#include "vtkPixelTransfer.h"
#include "vtkPlane.h"
#include "vtkPlaneCollection.h"
#include "vtkPolyData.h"
#include "vtkPolyDataMapper.h"
#include "vtkRenderer.h"
#include "vtkShaderProgram.h"
#include "vtkTessellatedBoxSource.h"
#include "vtkTextureObject.h"
#include "vtkTimeStamp.h"
#include "vtkTransform.h"
#include "vtkUnsignedIntArray.h"
#include "vtkVolume.h"
#include "vtkVolumeMask.h"
#include "vtkVolumeTexture.h"

//-----------------------------------------------------------------------------
vtkOpenGLGPUVolumeRayCastMapperInternals::
  vtkOpenGLGPUVolumeRayCastMapperInternals(
    vtkOpenGLGPUVolumeRayCastMapper* parent)
{
  this->Parent = parent;
  this->ValidTransferFunction = false;
  this->LoadDepthTextureExtensionsSucceeded = false;
  this->CameraWasInsideInLastUpdate = false;
  this->CubeVBOId = 0;
  this->CubeVAOId = 0;
  this->CubeIndicesId = 0;
  this->NoiseTextureObject = nullptr;
  this->DepthTextureObject = nullptr;
  this->TextureWidth = 1024;
  this->ActualSampleDistance = 1.0;
  this->ReductionFactor = 1.0;
  this->RGBTables = nullptr;
  this->OpacityTables = nullptr;
  this->Mask1RGBTable = nullptr;
  this->Mask2RGBTable = nullptr;
  this->GradientOpacityTables = nullptr;
  this->TransferFunctions2D = nullptr;
  this->CurrentMask = nullptr;
  this->Dimensions[0] = this->Dimensions[1] = this->Dimensions[2] = -1;
  this->TextureSize[0] = this->TextureSize[1] = this->TextureSize[2] = -1;
  this->WindowLowerLeft[0] = this->WindowLowerLeft[1] = 0;
  this->WindowSize[0] = this->WindowSize[1] = 0;
  this->LastDepthPassWindowSize[0] = this->LastDepthPassWindowSize[1] = 0;
  this->LastRenderToImageWindowSize[0] = 0;
  this->LastRenderToImageWindowSize[1] = 0;
  this->CurrentSelectionPass = vtkHardwareSelector::MIN_KNOWN_PASS - 1;

  this->CellScale[0] = this->CellScale[1] = this->CellScale[2] = 0.0;
  this->NoiseTextureData = nullptr;

  this->NumberOfLights = 0;
  this->LightComplexity = 0;

  this->Extents[0] = VTK_INT_MAX;
  this->Extents[1] = VTK_INT_MIN;
  this->Extents[2] = VTK_INT_MAX;
  this->Extents[3] = VTK_INT_MIN;
  this->Extents[4] = VTK_INT_MAX;
  this->Extents[5] = VTK_INT_MIN;

  this->CellToPointMatrix->Identity();
  this->AdjustedTexMin[0] = this->AdjustedTexMin[1] = this->AdjustedTexMin[2] =
    0.0f;
  this->AdjustedTexMin[3] = 1.0f;
  this->AdjustedTexMax[0] = this->AdjustedTexMax[1] = this->AdjustedTexMax[2] =
    1.0f;
  this->AdjustedTexMax[3] = 1.0f;

  this->Scale.clear();
  this->Bias.clear();

  this->NeedToInitializeResources = false;
  this->ShaderCache = nullptr;

  this->FBO = nullptr;
  this->RTTDepthBufferTextureObject = nullptr;
  this->RTTDepthTextureObject = nullptr;
  this->RTTColorTextureObject = nullptr;
  this->RTTDepthTextureType = -1;

  this->DPFBO = nullptr;
  this->DPDepthBufferTextureObject = nullptr;
  this->DPColorTextureObject = nullptr;
  this->PreserveViewport = false;
  this->PreserveGLState = false;
}

//-----------------------------------------------------------------------------
vtkOpenGLGPUVolumeRayCastMapperInternals::
  ~vtkOpenGLGPUVolumeRayCastMapperInternals()
{
  delete[] this->NoiseTextureData;

  if (this->NoiseTextureObject)
  {
    this->NoiseTextureObject->Delete();
    this->NoiseTextureObject = nullptr;
  }

  if (this->DepthTextureObject)
  {
    this->DepthTextureObject->Delete();
    this->DepthTextureObject = nullptr;
  }

  if (this->FBO)
  {
    this->FBO->Delete();
    this->FBO = nullptr;
  }

  if (this->RTTDepthBufferTextureObject)
  {
    this->RTTDepthBufferTextureObject->Delete();
    this->RTTDepthBufferTextureObject = nullptr;
  }

  if (this->RTTDepthTextureObject)
  {
    this->RTTDepthTextureObject->Delete();
    this->RTTDepthTextureObject = nullptr;
  }

  if (this->RTTColorTextureObject)
  {
    this->RTTColorTextureObject->Delete();
    this->RTTColorTextureObject = nullptr;
  }

  if (this->ImageSampleFBO)
  {
    this->ImageSampleFBO->Delete();
    this->ImageSampleFBO = nullptr;
  }

  for (auto& tex : this->ImageSampleTexture)
  {
    tex = nullptr;
  }
  this->ImageSampleTexture.clear();
  this->ImageSampleTexNames.clear();

  if (this->ImageSampleVBO)
  {
    this->ImageSampleVBO->Delete();
    this->ImageSampleVBO = nullptr;
  }

  if (this->ImageSampleVAO)
  {
    this->ImageSampleVAO->Delete();
    this->ImageSampleVAO = nullptr;
  }
  this->DeleteTransfer1D();
  this->DeleteTransfer2D();

  this->Scale.clear();
  this->Bias.clear();

  // Do not delete the shader programs - Let the cache clean them up.
  this->ImageSampleProg = nullptr;
}

//----------------------------------------------------------------------------
template<typename T>
void vtkOpenGLGPUVolumeRayCastMapperInternals::ToFloat(const T& in1,
  const T& in2,
  float (&out)[2])
{
  out[0] = static_cast<float>(in1);
  out[1] = static_cast<float>(in2);
}

template<typename T>
void vtkOpenGLGPUVolumeRayCastMapperInternals::ToFloat(const T& in1,
  const T& in2,
  const T& in3,
  float (&out)[3])
{
  out[0] = static_cast<float>(in1);
  out[1] = static_cast<float>(in2);
  out[2] = static_cast<float>(in3);
}

//----------------------------------------------------------------------------
template<typename T>
void vtkOpenGLGPUVolumeRayCastMapperInternals::ToFloat(T* in,
  float* out,
  int noOfComponents)
{
  for (int i = 0; i < noOfComponents; ++i)
  {
    out[i] = static_cast<float>(in[i]);
  }
}

//----------------------------------------------------------------------------
template<typename T>
void vtkOpenGLGPUVolumeRayCastMapperInternals::ToFloat(T (&in)[3],
  float (&out)[3])
{
  out[0] = static_cast<float>(in[0]);
  out[1] = static_cast<float>(in[1]);
  out[2] = static_cast<float>(in[2]);
}

//----------------------------------------------------------------------------
template<typename T>
void vtkOpenGLGPUVolumeRayCastMapperInternals::ToFloat(T (&in)[2],
  float (&out)[2])
{
  out[0] = static_cast<float>(in[0]);
  out[1] = static_cast<float>(in[1]);
}

//----------------------------------------------------------------------------
template<typename T>
void vtkOpenGLGPUVolumeRayCastMapperInternals::ToFloat(T& in, float& out)
{
  out = static_cast<float>(in);
}

//----------------------------------------------------------------------------
template<typename T>
void vtkOpenGLGPUVolumeRayCastMapperInternals::ToFloat(T (&in)[4][2],
  float (&out)[4][2])
{
  out[0][0] = static_cast<float>(in[0][0]);
  out[0][1] = static_cast<float>(in[0][1]);
  out[1][0] = static_cast<float>(in[1][0]);
  out[1][1] = static_cast<float>(in[1][1]);
  out[2][0] = static_cast<float>(in[2][0]);
  out[2][1] = static_cast<float>(in[2][1]);
  out[3][0] = static_cast<float>(in[3][0]);
  out[3][1] = static_cast<float>(in[3][1]);
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::SetupTransferFunction1D(
  vtkRenderer* ren,
  int noOfComponents,
  int independentComponents)
{
  this->ReleaseGraphicsTransfer1D(ren->GetRenderWindow());
  this->DeleteTransfer1D();

  // Check component mode (independent or dependent)
  noOfComponents =
    noOfComponents > 1 && independentComponents ? noOfComponents : 1;

  // Create RGB and opacity (scalar and gradient) lookup tables. We support up
  // to four components in independentComponents mode.
  this->RGBTables = new vtkOpenGLVolumeRGBTables(noOfComponents);
  this->OpacityTables = new vtkOpenGLVolumeOpacityTables(noOfComponents);
  this->GradientOpacityTables =
    new vtkOpenGLVolumeGradientOpacityTables(noOfComponents);

  this->OpacityTablesMap.clear();
  this->RGBTablesMap.clear();
  this->GradientOpacityTablesMap.clear();

  if (this->Parent->GetMaskInput() != nullptr &&
    this->Parent->GetMaskType() == vtkGPUVolumeRayCastMapper::LabelMapMaskType)
  {
    if (this->Mask1RGBTable == nullptr)
    {
      this->Mask1RGBTable = vtkOpenGLVolumeRGBTable::New();
    }
    if (this->Mask2RGBTable == nullptr)
    {
      this->Mask2RGBTable = vtkOpenGLVolumeRGBTable::New();
    }
  }

  std::ostringstream numeric;
  for (int i = 0; i < noOfComponents; ++i)
  {
    numeric << i;
    if (i > 0)
    {
      this->OpacityTablesMap[i] =
        std::string("in_opacityTransferFunc") + numeric.str();
      this->RGBTablesMap[i] =
        std::string("in_colorTransferFunc") + numeric.str();
      this->GradientOpacityTablesMap[i] =
        std::string("in_gradientTransferFunc") + numeric.str();
    }
    else
    {
      this->OpacityTablesMap[i] = std::string("in_opacityTransferFunc");
      this->RGBTablesMap[i] = std::string("in_colorTransferFunc");
      this->GradientOpacityTablesMap[i] =
        std::string("in_gradientTransferFunc");
    }
    numeric.str("");
    numeric.clear();
  }

  this->InitializationTime.Modified();
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::SetupTransferFunction2D(
  vtkRenderer* ren,
  int noOfComponents,
  int independentComponents)
{
  this->ReleaseGraphicsTransfer2D(ren->GetRenderWindow());
  this->DeleteTransfer2D();

  unsigned int const num =
    (noOfComponents > 1 && independentComponents) ? noOfComponents : 1;
  this->TransferFunctions2D = new vtkOpenGLTransferFunctions2D(num);

  std::ostringstream indexStream;
  const std::string baseName = "in_transfer2D";
  for (unsigned int i = 0; i < num; i++)
  {
    if (i > 0)
    {
      indexStream << i;
    }
    this->TransferFunctions2DMap[0] = baseName + indexStream.str();
    indexStream.str("");
    indexStream.clear();
  }

  this->Transfer2DTime.Modified();
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::InitializeTransferFunction(
  vtkRenderer* ren,
  vtkVolume* vol,
  int noOfComponents,
  int independentComponents)
{
  const int transferMode = vol->GetProperty()->GetTransferFunctionMode();
  switch (transferMode)
  {
    case vtkVolumeProperty::TF_2D:
      this->SetupTransferFunction2D(ren, noOfComponents, independentComponents);
      break;

    case vtkVolumeProperty::TF_1D:
    default:
      this->SetupTransferFunction1D(ren, noOfComponents, independentComponents);
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::UpdateTransferFunction(
  vtkRenderer* ren,
  vtkVolume* vol,
  int noOfComponents,
  int independentComponents)
{
  int const transferMode = vol->GetProperty()->GetTransferFunctionMode();
  switch (transferMode)
  {
    case vtkVolumeProperty::TF_1D:
      if (independentComponents)
      {
        for (int i = 0; i < noOfComponents; ++i)
        {
          this->UpdateOpacityTransferFunction(ren, vol, i);
          this->UpdateGradientOpacityTransferFunction(ren, vol, i);
          this->UpdateColorTransferFunction(ren, vol, i);
        }
      }
      else
      {
        if (noOfComponents == 2 || noOfComponents == 4)
        {
          this->UpdateOpacityTransferFunction(ren, vol, noOfComponents - 1);
          this->UpdateGradientOpacityTransferFunction(
            ren, vol, noOfComponents - 1);
          this->UpdateColorTransferFunction(ren, vol, 0);
        }
      }
      break;

    case vtkVolumeProperty::TF_2D:
      if (independentComponents)
      {
        for (int i = 0; i < noOfComponents; ++i)
        {
          this->UpdateTransferFunction2D(ren, vol, i);
        }
      }
      else
      {
        if (noOfComponents == 2 || noOfComponents == 4)
        {
          this->UpdateTransferFunction2D(ren, vol, 0);
        }
      }
      break;
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::ActivateTransferFunction(
  vtkShaderProgram* prog,
  vtkVolumeProperty* volumeProperty,
  int numberOfSamplers)
{
  int const transferMode = volumeProperty->GetTransferFunctionMode();
  switch (transferMode)
  {
    case vtkVolumeProperty::TF_1D:
      for (int i = 0; i < numberOfSamplers; ++i)
      {
        this->OpacityTables->GetTable(i)->Activate();
        prog->SetUniformi(this->OpacityTablesMap[i].c_str(),
          this->OpacityTables->GetTable(i)->GetTextureUnit());

        if (this->Parent->GetBlendMode() !=
          vtkGPUVolumeRayCastMapper::ADDITIVE_BLEND)
        {
          this->RGBTables->GetTable(i)->Activate();
          prog->SetUniformi(this->RGBTablesMap[i].c_str(),
            this->RGBTables->GetTable(i)->GetTextureUnit());
        }

        if (this->GradientOpacityTables)
        {
          this->GradientOpacityTables->GetTable(i)->Activate();
          prog->SetUniformi(this->GradientOpacityTablesMap[i].c_str(),
            this->GradientOpacityTables->GetTable(i)->GetTextureUnit());
        }
      }
      break;
    case vtkVolumeProperty::TF_2D:
      for (int i = 0; i < numberOfSamplers; ++i)
      {
        vtkOpenGLTransferFunction2D* table =
          this->TransferFunctions2D->GetTable(i);
        table->Activate();
        prog->SetUniformi(
          this->TransferFunctions2DMap[i].c_str(), table->GetTextureUnit());
      }
      break;
  }
}

//-----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::DeactivateTransferFunction(
  vtkVolumeProperty* volumeProperty,
  int numberOfSamplers)
{
  int const transferMode = volumeProperty->GetTransferFunctionMode();
  switch (transferMode)
  {
    case vtkVolumeProperty::TF_1D:
      for (int i = 0; i < numberOfSamplers; ++i)
      {
        this->OpacityTables->GetTable(i)->Deactivate();
        if (this->Parent->GetBlendMode() !=
          vtkGPUVolumeRayCastMapper::ADDITIVE_BLEND)
        {
          this->RGBTables->GetTable(i)->Deactivate();
        }
        if (this->GradientOpacityTables)
        {
          this->GradientOpacityTables->GetTable(i)->Deactivate();
        }
      }
      break;
    case vtkVolumeProperty::TF_2D:
      for (int i = 0; i < numberOfSamplers; ++i)
      {
        this->TransferFunctions2D->GetTable(i)->Deactivate();
      }
      break;
  }
}

//-----------------------------------------------------------------------------
bool vtkOpenGLGPUVolumeRayCastMapperInternals::LoadMask(vtkRenderer* ren,
  vtkImageData* vtkNotUsed(input),
  vtkImageData* maskInput,
  vtkVolume* vtkNotUsed(volume))
{
  bool result = true;
  if (maskInput && (maskInput->GetMTime() > this->MaskUpdateTime))
  {
    if (!this->CurrentMask)
    {
      this->CurrentMask = vtkSmartPointer<vtkVolumeTexture>::New();
      this->CurrentMask->SetMapper(this->Parent);

      const auto& part = this->Parent->GetVolumeTexture()->GetPartitions();
      this->CurrentMask->SetPartitions(part[0], part[1], part[2]);
    }

    vtkDataArray* arr = this->Parent->GetScalars(maskInput,
      this->Parent->GetScalarMode(),
      this->Parent->GetArrayAccessMode(),
      this->Parent->GetArrayId(),
      this->Parent->GetArrayName(),
      this->Parent->GetCellFlag());

    result = this->CurrentMask->LoadVolume(
      ren, maskInput, arr, VTK_NEAREST_INTERPOLATION);

    this->MaskUpdateTime.Modified();
  }

  return result;
}

//----------------------------------------------------------------------------
bool vtkOpenGLGPUVolumeRayCastMapperInternals::LoadData(vtkRenderer* ren,
  vtkVolume* vol,
  vtkVolumeProperty* volProp,
  vtkImageData* input,
  vtkDataArray* scalars)
{
  // Update bounds, data, and geometry
  input->GetDimensions(this->Dimensions);
  bool success = this->Parent->GetVolumeTexture()->LoadVolume(
    ren, input, scalars, volProp->GetInterpolationType());

  this->ComputeBounds(input);
  this->ComputeCellToPointMatrix();
  this->LoadMask(ren, input, this->Parent->GetMaskInput(), vol);
  this->InputUpdateTime.Modified();

  return success;
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::ReleaseGraphicsTransfer1D(
  vtkWindow* window)
{
  if (this->RGBTables)
  {
    this->RGBTables->ReleaseGraphicsResources(window);
  }

  if (this->Mask1RGBTable)
  {
    this->Mask1RGBTable->ReleaseGraphicsResources(window);
  }

  if (this->Mask2RGBTable)
  {
    this->Mask2RGBTable->ReleaseGraphicsResources(window);
  }

  if (this->OpacityTables)
  {
    this->OpacityTables->ReleaseGraphicsResources(window);
  }

  if (this->GradientOpacityTables)
  {
    this->GradientOpacityTables->ReleaseGraphicsResources(window);
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::DeleteTransfer1D()
{
  delete this->RGBTables;
  this->RGBTables = nullptr;

  if (this->Mask1RGBTable)
  {
    this->Mask1RGBTable->Delete();
    this->Mask1RGBTable = nullptr;
  }

  if (this->Mask2RGBTable)
  {
    this->Mask2RGBTable->Delete();
    this->Mask2RGBTable = nullptr;
  }

  delete this->OpacityTables;
  this->OpacityTables = nullptr;

  delete this->GradientOpacityTables;
  this->GradientOpacityTables = nullptr;
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::ReleaseGraphicsTransfer2D(
  vtkWindow* window)
{
  if (this->TransferFunctions2D)
  {
    this->TransferFunctions2D->ReleaseGraphicsResources(window);
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::DeleteTransfer2D()
{
  delete this->TransferFunctions2D;
  this->TransferFunctions2D = nullptr;
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::ComputeBounds(
  vtkImageData* input)
{
  double origin[3];

  input->GetSpacing(this->CellSpacing);
  input->GetOrigin(origin);
  input->GetExtent(this->Extents);

  int swapBounds[3];
  swapBounds[0] = (this->CellSpacing[0] < 0);
  swapBounds[1] = (this->CellSpacing[1] < 0);
  swapBounds[2] = (this->CellSpacing[2] < 0);

  // Loaded data represents points
  if (!this->Parent->GetCellFlag())
  {
    // If spacing is negative, we may have to rethink the equation
    // between real point and texture coordinate...
    this->LoadedBounds[0] = origin[0] +
      static_cast<double>(this->Extents[0 + swapBounds[0]]) *
        this->CellSpacing[0];
    this->LoadedBounds[2] = origin[1] +
      static_cast<double>(this->Extents[2 + swapBounds[1]]) *
        this->CellSpacing[1];
    this->LoadedBounds[4] = origin[2] +
      static_cast<double>(this->Extents[4 + swapBounds[2]]) *
        this->CellSpacing[2];
    this->LoadedBounds[1] = origin[0] +
      static_cast<double>(this->Extents[1 - swapBounds[0]]) *
        this->CellSpacing[0];
    this->LoadedBounds[3] = origin[1] +
      static_cast<double>(this->Extents[3 - swapBounds[1]]) *
        this->CellSpacing[1];
    this->LoadedBounds[5] = origin[2] +
      static_cast<double>(this->Extents[5 - swapBounds[2]]) *
        this->CellSpacing[2];
  }
  // Loaded extents represent cells
  else
  {
    int i = 0;
    while (i < 3)
    {
      this->LoadedBounds[2 * i + swapBounds[i]] = origin[i] +
        (static_cast<double>(this->Extents[2 * i])) * this->CellSpacing[i];

      this->LoadedBounds[2 * i + 1 - swapBounds[i]] = origin[i] +
        (static_cast<double>(this->Extents[2 * i + 1]) + 1.0) *
          this->CellSpacing[i];

      i++;
    }
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::UpdateVolume(
  vtkVolumeProperty* volumeProperty)
{
  if (volumeProperty->GetMTime() > this->VolumeUpdateTime.GetMTime())
  {
    int const newInterp = volumeProperty->GetInterpolationType();
    this->Parent->GetVolumeTexture()->UpdateInterpolationType(newInterp);
  }

  this->VolumeUpdateTime.Modified();
}

//----------------------------------------------------------------------------
int vtkOpenGLGPUVolumeRayCastMapperInternals::UpdateColorTransferFunction(
  vtkRenderer* ren,
  vtkVolume* vol,
  unsigned int component)
{
  // Volume property cannot be null.
  vtkVolumeProperty* volumeProperty = vol->GetProperty();

  // Build the colormap in a 1D texture.
  // 1D RGB-texture=mapping from scalar values to color values
  // build the table.
  vtkColorTransferFunction* colorTransferFunction =
    volumeProperty->GetRGBTransferFunction(component);

  double componentRange[2];
  for (int i = 0; i < 2; ++i)
  {
    componentRange[i] =
      this->Parent->GetVolumeTexture()->ScalarRange[component][i];
  }

  // Add points only if its not being added before
  if (colorTransferFunction->GetSize() < 1)
  {
    colorTransferFunction->AddRGBPoint(componentRange[0], 0.0, 0.0, 0.0);
    colorTransferFunction->AddRGBPoint(componentRange[1], 1.0, 1.0, 1.0);
  }

  int filterVal =
    volumeProperty->GetInterpolationType() == VTK_LINEAR_INTERPOLATION
    ? vtkTextureObject::Linear
    : vtkTextureObject::Nearest;

  this->RGBTables->GetTable(component)->Update(
    volumeProperty->GetRGBTransferFunction(component),
    componentRange,
#if GL_ES_VERSION_3_0 != 1
    filterVal,
#else
    vtkTextureObject::Nearest,
#endif
    vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));

  if (this->Parent->GetMaskInput() != nullptr &&
    this->Parent->GetMaskType() == vtkGPUVolumeRayCastMapper::LabelMapMaskType)
  {
    vtkColorTransferFunction* colorTransferFunc =
      volumeProperty->GetRGBTransferFunction(1);
    this->Mask1RGBTable->Update(colorTransferFunc,
      componentRange,
      vtkTextureObject::Nearest,
      vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));

    colorTransferFunc = volumeProperty->GetRGBTransferFunction(2);
    this->Mask2RGBTable->Update(colorTransferFunc,
      componentRange,
      vtkTextureObject::Nearest,
      vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));
  }

  return 0;
}

//----------------------------------------------------------------------------
int vtkOpenGLGPUVolumeRayCastMapperInternals::UpdateOpacityTransferFunction(
  vtkRenderer* ren,
  vtkVolume* vol,
  unsigned int component)
{
  vtkVolumeProperty* volumeProperty = vol->GetProperty();

  // Transfer function table index based on whether independent / dependent
  // components. If dependent, use the first scalar opacity transfer function
  unsigned int lookupTableIndex =
    volumeProperty->GetIndependentComponents() ? component : 0;
  vtkPiecewiseFunction* scalarOpacity =
    volumeProperty->GetScalarOpacity(lookupTableIndex);

  double componentRange[2];
  for (int i = 0; i < 2; ++i)
  {
    componentRange[i] =
      this->Parent->GetVolumeTexture()->ScalarRange[component][i];
  }

  if (scalarOpacity->GetSize() < 1)
  {
    scalarOpacity->AddPoint(componentRange[0], 0.0);
    scalarOpacity->AddPoint(componentRange[1], 0.5);
  }

  int filterVal =
    volumeProperty->GetInterpolationType() == VTK_LINEAR_INTERPOLATION
    ? vtkTextureObject::Linear
    : vtkTextureObject::Nearest;

  this->OpacityTables->GetTable(lookupTableIndex)
    ->Update(scalarOpacity,
      this->Parent->GetBlendMode(),
      this->ActualSampleDistance,
      componentRange,
      volumeProperty->GetScalarOpacityUnitDistance(component),
#if GL_ES_VERSION_3_0 != 1
      filterVal,
#else
      vtkTextureObject::Nearest,
#endif
      vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));

  return 0;
}

//------------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::UpdateTransferFunction2D(
  vtkRenderer* ren,
  vtkVolume* vol,
  unsigned int component)
{
  vtkVolumeProperty* prop = vol->GetProperty();
  int const transferMode = prop->GetTransferFunctionMode();
  if (transferMode != vtkVolumeProperty::TF_2D)
  {
    return;
  }

  // Use the first LUT when using dependent components
  unsigned int const lutIndex =
    prop->GetIndependentComponents() ? component : 0;

  vtkImageData* transfer2D = prop->GetTransferFunction2D(lutIndex);
#if GL_ES_VERSION_3_0 != 1
  int const interp = prop->GetInterpolationType() == VTK_LINEAR_INTERPOLATION
    ? vtkTextureObject::Linear
    : vtkTextureObject::Nearest;
#else
  int const interp = vtkTextureObject::Nearest;
#endif

  this->TransferFunctions2D->GetTable(lutIndex)->Update(transfer2D,
    interp,
    vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));
}

//----------------------------------------------------------------------------
int vtkOpenGLGPUVolumeRayCastMapperInternals::
  UpdateGradientOpacityTransferFunction(vtkRenderer* ren,
    vtkVolume* vol,
    unsigned int component)
{
  vtkVolumeProperty* volumeProperty = vol->GetProperty();

  // Transfer function table index based on whether independent / dependent
  // components. If dependent, use the first gradient opacity transfer function
  unsigned int lookupTableIndex =
    volumeProperty->GetIndependentComponents() ? component : 0;

  if (!volumeProperty->HasGradientOpacity(lookupTableIndex) ||
    !this->GradientOpacityTables)
  {
    return 1;
  }

  vtkPiecewiseFunction* gradientOpacity =
    volumeProperty->GetGradientOpacity(lookupTableIndex);

  double componentRange[2];
  for (int i = 0; i < 2; ++i)
  {
    componentRange[i] =
      this->Parent->GetVolumeTexture()->ScalarRange[component][i];
  }

  if (gradientOpacity->GetSize() < 1)
  {
    gradientOpacity->AddPoint(componentRange[0], 0.0);
    gradientOpacity->AddPoint(componentRange[1], 0.5);
  }

  int filterVal =
    volumeProperty->GetInterpolationType() == VTK_LINEAR_INTERPOLATION
    ? vtkTextureObject::Linear
    : vtkTextureObject::Nearest;

  this->GradientOpacityTables->GetTable(lookupTableIndex)
    ->Update(gradientOpacity,
      this->ActualSampleDistance,
      componentRange,
      volumeProperty->GetScalarOpacityUnitDistance(component),
#if GL_ES_VERSION_3_0 != 1
      filterVal,
#else
      vtkTextureObject::Nearest,
#endif
      vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));

  return 0;
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::CreateNoiseTexture(
  vtkRenderer* ren)
{
  vtkOpenGLRenderWindow* glWindow =
    vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow());

  if (!this->NoiseTextureObject)
  {
    this->NoiseTextureObject = vtkTextureObject::New();
  }
  this->NoiseTextureObject->SetContext(glWindow);

  bool updateSize = false;
  bool useUserSize = this->Parent->GetNoiseTextureSize()[0] > 0 &&
    this->Parent->GetNoiseTextureSize()[1] > 0;
  if (useUserSize)
  {
    int const twidth = this->NoiseTextureObject->GetWidth();
    int const theight = this->NoiseTextureObject->GetHeight();
    updateSize = this->Parent->GetNoiseTextureSize()[0] != twidth ||
      this->Parent->GetNoiseTextureSize()[1] != theight;
  }

  if (!this->NoiseTextureObject->GetHandle() || updateSize ||
    this->NoiseTextureObject->GetMTime() <
      this->Parent->GetNoiseGenerator()->GetMTime())
  {
    int* winSize = ren->GetRenderWindow()->GetSize();
    int sizeX =
      useUserSize ? this->Parent->GetNoiseTextureSize()[0] : winSize[0];
    int sizeY =
      useUserSize ? this->Parent->GetNoiseTextureSize()[1] : winSize[1];

    int const maxSize = vtkTextureObject::GetMaximumTextureSize(glWindow);
    if (sizeX > maxSize || sizeY > maxSize)
    {
      sizeX = vtkMath::Max(sizeX, maxSize);
      sizeY = vtkMath::Max(sizeY, maxSize);
    }

    // Allocate buffer. After controlling for the maximum supported size sizeX/Y
    // might have changed, so an additional check is needed.
    int const twidth = this->NoiseTextureObject->GetWidth();
    int const theight = this->NoiseTextureObject->GetHeight();
    bool sizeChanged = sizeX != twidth || sizeY != theight;
    if (sizeChanged || !this->NoiseTextureData)
    {
      delete[] this->NoiseTextureData;
      this->NoiseTextureData = nullptr;
      this->NoiseTextureData = new float[sizeX * sizeY];
    }

    // Generate jitter noise
    if (!this->Parent->GetNoiseGenerator())
    {
      // Use default settings
      vtkPerlinNoise* perlinNoise = vtkPerlinNoise::New();
      perlinNoise->SetPhase(0.0, 0.0, 0.0);
      perlinNoise->SetFrequency(sizeX, sizeY, 1.0);
      perlinNoise->SetAmplitude(0.5); /* [-n, n] */
      this->Parent->SetNoiseGenerator(perlinNoise);
    }

    int const bufferSize = sizeX * sizeY;
    for (int i = 0; i < bufferSize; i++)
    {
      int const x = i % sizeX;
      int const y = i / sizeY;
      this->NoiseTextureData[i] = static_cast<float>(
        this->Parent->GetNoiseGenerator()->EvaluateFunction(x, y, 0.0) + 0.1);
    }

    // Prepare texture
    this->NoiseTextureObject->Create2DFromRaw(
      sizeX, sizeY, 1, VTK_FLOAT, this->NoiseTextureData);

    this->NoiseTextureObject->SetWrapS(vtkTextureObject::Repeat);
    this->NoiseTextureObject->SetWrapT(vtkTextureObject::Repeat);
    this->NoiseTextureObject->SetMagnificationFilter(vtkTextureObject::Nearest);
    this->NoiseTextureObject->SetMinificationFilter(vtkTextureObject::Nearest);
    this->NoiseTextureObject->SetBorderColor(0.0f, 0.0f, 0.0f, 0.0f);
    this->NoiseTextureObject->Modified();
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::CaptureDepthTexture(
  vtkRenderer* ren,
  vtkVolume* vtkNotUsed(vol))
{
  // Make sure our render window is the current OpenGL context
  ren->GetRenderWindow()->MakeCurrent();

  // Load required extensions for grabbing depth sampler buffer
  if (!this->LoadDepthTextureExtensionsSucceeded)
  {
    this->LoadRequireDepthTextureExtensions(ren->GetRenderWindow());
  }

  // If we can't load the necessary extensions, provide
  // feedback on why it failed.
  if (!this->LoadDepthTextureExtensionsSucceeded)
  {
    std::cerr << this->ExtensionsStringStream.str() << std::endl;
    return;
  }

  if (!this->DepthTextureObject)
  {
    this->DepthTextureObject = vtkTextureObject::New();
  }

  this->DepthTextureObject->SetContext(
    vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));
  if (!this->DepthTextureObject->GetHandle())
  {
    // First set the parameters
    this->DepthTextureObject->SetWrapS(vtkTextureObject::ClampToEdge);
    this->DepthTextureObject->SetWrapT(vtkTextureObject::ClampToEdge);
    this->DepthTextureObject->SetMagnificationFilter(vtkTextureObject::Linear);
    this->DepthTextureObject->SetMinificationFilter(vtkTextureObject::Linear);
    this->DepthTextureObject->AllocateDepth(
      this->WindowSize[0], this->WindowSize[1], 4);
  }

#if GL_ES_VERSION_3_0 != 1
  // currently broken on ES
  this->DepthTextureObject->CopyFromFrameBuffer(this->WindowLowerLeft[0],
    this->WindowLowerLeft[1],
    0,
    0,
    this->WindowSize[0],
    this->WindowSize[1]);
#endif
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::SetLightingParameters(
  vtkRenderer* ren,
  vtkShaderProgram* prog,
  vtkVolume* vol)
{
  if (!ren || !prog || !vol)
  {
    return;
  }

  if (vol && !vol->GetProperty()->GetShade())
  {
    return;
  }

  prog->SetUniformi("in_twoSidedLighting", ren->GetTwoSidedLighting());

  // for lightkit case there are some parameters to set
  vtkCamera* cam = ren->GetActiveCamera();
  vtkTransform* viewTF = cam->GetModelViewTransformObject();

  // Bind some light settings
  int numberOfLights = 0;
  vtkLightCollection* lc = ren->GetLights();
  vtkLight* light;

  vtkCollectionSimpleIterator sit;
  float lightAmbientColor[6][3];
  float lightDiffuseColor[6][3];
  float lightSpecularColor[6][3];
  float lightDirection[6][3];
  for (lc->InitTraversal(sit); (light = lc->GetNextLight(sit));)
  {
    float status = light->GetSwitch();
    if (status > 0.0)
    {
      double* aColor = light->GetAmbientColor();
      double* dColor = light->GetDiffuseColor();
      double* sColor = light->GetDiffuseColor();
      double intensity = light->GetIntensity();
      lightAmbientColor[numberOfLights][0] = aColor[0] * intensity;
      lightAmbientColor[numberOfLights][1] = aColor[1] * intensity;
      lightAmbientColor[numberOfLights][2] = aColor[2] * intensity;
      lightDiffuseColor[numberOfLights][0] = dColor[0] * intensity;
      lightDiffuseColor[numberOfLights][1] = dColor[1] * intensity;
      lightDiffuseColor[numberOfLights][2] = dColor[2] * intensity;
      lightSpecularColor[numberOfLights][0] = sColor[0] * intensity;
      lightSpecularColor[numberOfLights][1] = sColor[1] * intensity;
      lightSpecularColor[numberOfLights][2] = sColor[2] * intensity;
      // Get required info from light
      double* lfp = light->GetTransformedFocalPoint();
      double* lp = light->GetTransformedPosition();
      double lightDir[3];
      vtkMath::Subtract(lfp, lp, lightDir);
      vtkMath::Normalize(lightDir);
      double* tDir = viewTF->TransformNormal(lightDir);
      lightDirection[numberOfLights][0] = tDir[0];
      lightDirection[numberOfLights][1] = tDir[1];
      lightDirection[numberOfLights][2] = tDir[2];
      numberOfLights++;
    }
  }

  prog->SetUniform3fv(
    "in_lightAmbientColor", numberOfLights, lightAmbientColor);
  prog->SetUniform3fv(
    "in_lightDiffuseColor", numberOfLights, lightDiffuseColor);
  prog->SetUniform3fv(
    "in_lightSpecularColor", numberOfLights, lightSpecularColor);
  prog->SetUniform3fv("in_lightDirection", numberOfLights, lightDirection);
  prog->SetUniformi("in_numberOfLights", numberOfLights);

  // we are done unless we have positional lights
  if (this->LightComplexity < 3)
  {
    return;
  }

  // if positional lights pass down more parameters
  float lightAttenuation[6][3];
  float lightPosition[6][3];
  float lightConeAngle[6];
  float lightExponent[6];
  int lightPositional[6];
  numberOfLights = 0;
  for (lc->InitTraversal(sit); (light = lc->GetNextLight(sit));)
  {
    float status = light->GetSwitch();
    if (status > 0.0)
    {
      double* attn = light->GetAttenuationValues();
      lightAttenuation[numberOfLights][0] = attn[0];
      lightAttenuation[numberOfLights][1] = attn[1];
      lightAttenuation[numberOfLights][2] = attn[2];
      lightExponent[numberOfLights] = light->GetExponent();
      lightConeAngle[numberOfLights] = light->GetConeAngle();
      double* lp = light->GetTransformedPosition();
      double* tlp = viewTF->TransformPoint(lp);
      lightPosition[numberOfLights][0] = tlp[0];
      lightPosition[numberOfLights][1] = tlp[1];
      lightPosition[numberOfLights][2] = tlp[2];
      lightPositional[numberOfLights] = light->GetPositional();
      numberOfLights++;
    }
  }
  prog->SetUniform3fv("in_lightAttenuation", numberOfLights, lightAttenuation);
  prog->SetUniform1iv("in_lightPositional", numberOfLights, lightPositional);
  prog->SetUniform3fv("in_lightPosition", numberOfLights, lightPosition);
  prog->SetUniform1fv("in_lightExponent", numberOfLights, lightExponent);
  prog->SetUniform1fv("in_lightConeAngle", numberOfLights, lightConeAngle);
}

//-----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::ComputeCellToPointMatrix()
{
  this->CellToPointMatrix->Identity();
  this->AdjustedTexMin[0] = this->AdjustedTexMin[1] = this->AdjustedTexMin[2] =
    0.0f;
  this->AdjustedTexMin[3] = 1.0f;
  this->AdjustedTexMax[0] = this->AdjustedTexMax[1] = this->AdjustedTexMax[2] =
    1.0f;
  this->AdjustedTexMax[3] = 1.0f;

  if (!this->Parent->GetCellFlag()) // point data
  {
    float delta[3];
    delta[0] = this->Extents[1] - this->Extents[0];
    delta[1] = this->Extents[3] - this->Extents[2];
    delta[2] = this->Extents[5] - this->Extents[4];

    float min[3];
    min[0] = 0.5f / delta[0];
    min[1] = 0.5f / delta[1];
    min[2] = 0.5f / delta[2];

    float range[3]; // max - min
    range[0] = (delta[0] - 0.5f) / delta[0] - min[0];
    range[1] = (delta[1] - 0.5f) / delta[1] - min[1];
    range[2] = (delta[2] - 0.5f) / delta[2] - min[2];

    this->CellToPointMatrix->SetElement(0, 0, range[0]); // Scale diag
    this->CellToPointMatrix->SetElement(1, 1, range[1]);
    this->CellToPointMatrix->SetElement(2, 2, range[2]);
    this->CellToPointMatrix->SetElement(0, 3, min[0]); // t vector
    this->CellToPointMatrix->SetElement(1, 3, min[1]);
    this->CellToPointMatrix->SetElement(2, 3, min[2]);

    // Adjust limit coordinates for texture access.
    float const zeros[4] = { 0.0f, 0.0f, 0.0f, 1.0f }; // GL tex min
    float const ones[4] = { 1.0f, 1.0f, 1.0f, 1.0f };  // GL tex max
    this->CellToPointMatrix->MultiplyPoint(zeros, this->AdjustedTexMin);
    this->CellToPointMatrix->MultiplyPoint(ones, this->AdjustedTexMax);
  }
}

//----------------------------------------------------------------------------
bool vtkOpenGLGPUVolumeRayCastMapperInternals::IsCameraInside(vtkRenderer* ren,
  vtkVolume* vol)
{
  this->TempMatrix1->DeepCopy(vol->GetMatrix());
  this->TempMatrix1->Invert();

  vtkCamera* cam = ren->GetActiveCamera();
  double camWorldRange[2];
  double camWorldPos[4];
  double camFocalWorldPoint[4];
  double camWorldDirection[4];
  double camPos[4];
  double camPlaneNormal[4];

  cam->GetPosition(camWorldPos);
  camWorldPos[3] = 1.0;
  this->TempMatrix1->MultiplyPoint(camWorldPos, camPos);

  cam->GetFocalPoint(camFocalWorldPoint);
  camFocalWorldPoint[3] = 1.0;

  // The range (near/far) must also be transformed
  // into the local coordinate system.
  camWorldDirection[0] = camFocalWorldPoint[0] - camWorldPos[0];
  camWorldDirection[1] = camFocalWorldPoint[1] - camWorldPos[1];
  camWorldDirection[2] = camFocalWorldPoint[2] - camWorldPos[2];
  camWorldDirection[3] = 0.0;

  // Compute the normalized near plane normal
  this->TempMatrix1->MultiplyPoint(camWorldDirection, camPlaneNormal);

  vtkMath::Normalize(camWorldDirection);
  vtkMath::Normalize(camPlaneNormal);

  double camNearWorldPoint[4];
  double camNearPoint[4];

  cam->GetClippingRange(camWorldRange);
  camNearWorldPoint[0] =
    camWorldPos[0] + camWorldRange[0] * camWorldDirection[0];
  camNearWorldPoint[1] =
    camWorldPos[1] + camWorldRange[0] * camWorldDirection[1];
  camNearWorldPoint[2] =
    camWorldPos[2] + camWorldRange[0] * camWorldDirection[2];
  camNearWorldPoint[3] = 1.;

  this->TempMatrix1->MultiplyPoint(camNearWorldPoint, camNearPoint);

  int const result = vtkMath::PlaneIntersectsAABB(
    this->LoadedBounds, camPlaneNormal, camNearPoint);

  if (result == 0)
  {
    return true;
  }

  return false;
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::RenderVolumeGeometry(
  vtkRenderer* ren,
  vtkShaderProgram* prog,
  vtkVolume* vol)
{
  if (this->NeedToInitializeResources || !this->BBoxPolyData ||
    this->Parent->GetVolumeTexture()->UploadTime >
      this->BBoxPolyData->GetMTime() ||
    this->IsCameraInside(ren, vol) || this->CameraWasInsideInLastUpdate)
  {
    vtkNew<vtkTessellatedBoxSource> boxSource;
    boxSource->SetBounds(this->LoadedBounds);
    boxSource->QuadsOn();
    boxSource->SetLevel(0);

    vtkNew<vtkDensifyPolyData> densityPolyData;

    if (this->IsCameraInside(ren, vol))
    {
      // Normals should be transformed using the transpose of inverse
      // InverseVolumeMat
      this->TempMatrix1->DeepCopy(vol->GetMatrix());
      this->TempMatrix1->Invert();

      vtkCamera* cam = ren->GetActiveCamera();
      double camWorldRange[2];
      double camWorldPos[4];
      double camFocalWorldPoint[4];
      double camWorldDirection[4];
      double camPos[4];
      double camPlaneNormal[4];

      cam->GetPosition(camWorldPos);
      camWorldPos[3] = 1.0;
      this->TempMatrix1->MultiplyPoint(camWorldPos, camPos);

      cam->GetFocalPoint(camFocalWorldPoint);
      camFocalWorldPoint[3] = 1.0;

      // The range (near/far) must also be transformed
      // into the local coordinate system.
      camWorldDirection[0] = camFocalWorldPoint[0] - camWorldPos[0];
      camWorldDirection[1] = camFocalWorldPoint[1] - camWorldPos[1];
      camWorldDirection[2] = camFocalWorldPoint[2] - camWorldPos[2];
      camWorldDirection[3] = 0.0;

      // Compute the normalized near plane normal
      this->TempMatrix1->MultiplyPoint(camWorldDirection, camPlaneNormal);

      vtkMath::Normalize(camWorldDirection);
      vtkMath::Normalize(camPlaneNormal);

      double camNearWorldPoint[4];
      double camFarWorldPoint[4];
      double camNearPoint[4];
      double camFarPoint[4];

      cam->GetClippingRange(camWorldRange);
      camNearWorldPoint[0] =
        camWorldPos[0] + camWorldRange[0] * camWorldDirection[0];
      camNearWorldPoint[1] =
        camWorldPos[1] + camWorldRange[0] * camWorldDirection[1];
      camNearWorldPoint[2] =
        camWorldPos[2] + camWorldRange[0] * camWorldDirection[2];
      camNearWorldPoint[3] = 1.;

      camFarWorldPoint[0] =
        camWorldPos[0] + camWorldRange[1] * camWorldDirection[0];
      camFarWorldPoint[1] =
        camWorldPos[1] + camWorldRange[1] * camWorldDirection[1];
      camFarWorldPoint[2] =
        camWorldPos[2] + camWorldRange[1] * camWorldDirection[2];
      camFarWorldPoint[3] = 1.;

      this->TempMatrix1->MultiplyPoint(camNearWorldPoint, camNearPoint);
      this->TempMatrix1->MultiplyPoint(camFarWorldPoint, camFarPoint);

      vtkNew<vtkPlane> nearPlane;

      // We add an offset to the near plane to avoid hardware clipping of the
      // near plane due to floating-point precision.
      // camPlaneNormal is a unit vector, if the offset is larger than the
      // distance between near and far point, it will not work. Hence, we choose
      // a fraction of the near-far distance. However, care should be taken
      // to avoid hardware clipping in volumes with very small spacing where the
      // distance between near and far plane is also very small. In that case,
      // a minimum offset is chosen. This is chosen based on the typical
      // epsilon values on x86 systems.
      double offset =
        sqrt(vtkMath::Distance2BetweenPoints(camNearPoint, camFarPoint)) /
        1000.0;
      // Minimum offset to avoid floating point precision issues for volumes
      // with very small spacing
      double minOffset =
        static_cast<double>(std::numeric_limits<float>::epsilon()) * 1000.0;
      offset = offset < minOffset ? minOffset : offset;

      camNearPoint[0] += camPlaneNormal[0] * offset;
      camNearPoint[1] += camPlaneNormal[1] * offset;
      camNearPoint[2] += camPlaneNormal[2] * offset;

      nearPlane->SetOrigin(camNearPoint);
      nearPlane->SetNormal(camPlaneNormal);

      vtkNew<vtkPlaneCollection> planes;
      planes->RemoveAllItems();
      planes->AddItem(nearPlane.GetPointer());

      vtkNew<vtkClipConvexPolyData> clip;
      clip->SetInputConnection(boxSource->GetOutputPort());
      clip->SetPlanes(planes.GetPointer());

      densityPolyData->SetInputConnection(clip->GetOutputPort());

      this->CameraWasInsideInLastUpdate = true;
    }
    else
    {
      densityPolyData->SetInputConnection(boxSource->GetOutputPort());
      this->CameraWasInsideInLastUpdate = false;
    }

    densityPolyData->SetNumberOfSubdivisions(2);
    densityPolyData->Update();

    this->BBoxPolyData = vtkSmartPointer<vtkPolyData>::New();
    this->BBoxPolyData->ShallowCopy(densityPolyData->GetOutput());
    vtkPoints* points = this->BBoxPolyData->GetPoints();
    vtkCellArray* cells = this->BBoxPolyData->GetPolys();

    vtkNew<vtkUnsignedIntArray> polys;
    polys->SetNumberOfComponents(3);
    vtkIdType npts;
    vtkIdType* pts;

    // See if the volume transform is orientation-preserving
    // and orient polygons accordingly
    vtkMatrix4x4* volMat = vol->GetMatrix();
    double det = vtkMath::Determinant3x3(volMat->GetElement(0, 0),
      volMat->GetElement(0, 1),
      volMat->GetElement(0, 2),
      volMat->GetElement(1, 0),
      volMat->GetElement(1, 1),
      volMat->GetElement(1, 2),
      volMat->GetElement(2, 0),
      volMat->GetElement(2, 1),
      volMat->GetElement(2, 2));
    bool preservesOrientation = det > 0.0;

    const vtkIdType indexMap[3] = {
      preservesOrientation ? 0 : 2, 1, preservesOrientation ? 2 : 0
    };

    while (cells->GetNextCell(npts, pts))
    {
      polys->InsertNextTuple3(
        pts[indexMap[0]], pts[indexMap[1]], pts[indexMap[2]]);
    }

    // Dispose any previously created buffers
    this->DeleteBufferObjects();

    // Now create new ones
    this->CreateBufferObjects();

// TODO: should really use the built in VAO class
// which handles these apple issues internally
#ifdef __APPLE__
    if (vtkOpenGLRenderWindow::GetContextSupportsOpenGL32())
#endif
    {
      glBindVertexArray(this->CubeVAOId);
    }

    // Pass cube vertices to buffer object memory
    glBindBuffer(GL_ARRAY_BUFFER, this->CubeVBOId);
    glBufferData(GL_ARRAY_BUFFER,
      points->GetData()->GetDataSize() * points->GetData()->GetDataTypeSize(),
      points->GetData()->GetVoidPointer(0),
      GL_STATIC_DRAW);

    prog->EnableAttributeArray("in_vertexPos");
    prog->UseAttributeArray(
      "in_vertexPos", 0, 0, VTK_FLOAT, 3, vtkShaderProgram::NoNormalize);

    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, this->CubeIndicesId);
    glBufferData(GL_ELEMENT_ARRAY_BUFFER,
      polys->GetDataSize() * polys->GetDataTypeSize(),
      polys->GetVoidPointer(0),
      GL_STATIC_DRAW);
  }
  else
  {
#ifdef __APPLE__
    if (!vtkOpenGLRenderWindow::GetContextSupportsOpenGL32())
    {
      glBindBuffer(GL_ARRAY_BUFFER, this->CubeVBOId);
      prog->EnableAttributeArray("in_vertexPos");
      prog->UseAttributeArray(
        "in_vertexPos", 0, 0, VTK_FLOAT, 3, vtkShaderProgram::NoNormalize);
      glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, this->CubeIndicesId);
    }
    else
#endif
    {
      glBindVertexArray(this->CubeVAOId);
    }
  }

  glDrawElements(GL_TRIANGLES,
    this->BBoxPolyData->GetNumberOfCells() * 3,
    GL_UNSIGNED_INT,
    nullptr);

  vtkOpenGLStaticCheckErrorMacro("Error after glDrawElements in"
                                 " RenderVolumeGeometry!");
#ifdef __APPLE__
  if (!vtkOpenGLRenderWindow::GetContextSupportsOpenGL32())
  {
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
  }
  else
#endif
  {
    glBindVertexArray(0);
    glBindBuffer(GL_ARRAY_BUFFER, 0);
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, 0);
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::SetCroppingRegions(
  vtkRenderer* vtkNotUsed(ren),
  vtkShaderProgram* prog,
  vtkVolume* vtkNotUsed(vol))
{
  if (this->Parent->GetCropping())
  {
    int cropFlags = this->Parent->GetCroppingRegionFlags();
    double croppingRegionPlanes[6];
    this->Parent->GetCroppingRegionPlanes(croppingRegionPlanes);

    // Clamp it
    croppingRegionPlanes[0] = croppingRegionPlanes[0] < this->LoadedBounds[0]
      ? this->LoadedBounds[0]
      : croppingRegionPlanes[0];
    croppingRegionPlanes[0] = croppingRegionPlanes[0] > this->LoadedBounds[1]
      ? this->LoadedBounds[1]
      : croppingRegionPlanes[0];
    croppingRegionPlanes[1] = croppingRegionPlanes[1] < this->LoadedBounds[0]
      ? this->LoadedBounds[0]
      : croppingRegionPlanes[1];
    croppingRegionPlanes[1] = croppingRegionPlanes[1] > this->LoadedBounds[1]
      ? this->LoadedBounds[1]
      : croppingRegionPlanes[1];

    croppingRegionPlanes[2] = croppingRegionPlanes[2] < this->LoadedBounds[2]
      ? this->LoadedBounds[2]
      : croppingRegionPlanes[2];
    croppingRegionPlanes[2] = croppingRegionPlanes[2] > this->LoadedBounds[3]
      ? this->LoadedBounds[3]
      : croppingRegionPlanes[2];
    croppingRegionPlanes[3] = croppingRegionPlanes[3] < this->LoadedBounds[2]
      ? this->LoadedBounds[2]
      : croppingRegionPlanes[3];
    croppingRegionPlanes[3] = croppingRegionPlanes[3] > this->LoadedBounds[3]
      ? this->LoadedBounds[3]
      : croppingRegionPlanes[3];

    croppingRegionPlanes[4] = croppingRegionPlanes[4] < this->LoadedBounds[4]
      ? this->LoadedBounds[4]
      : croppingRegionPlanes[4];
    croppingRegionPlanes[4] = croppingRegionPlanes[4] > this->LoadedBounds[5]
      ? this->LoadedBounds[5]
      : croppingRegionPlanes[4];
    croppingRegionPlanes[5] = croppingRegionPlanes[5] < this->LoadedBounds[4]
      ? this->LoadedBounds[4]
      : croppingRegionPlanes[5];
    croppingRegionPlanes[5] = croppingRegionPlanes[5] > this->LoadedBounds[5]
      ? this->LoadedBounds[5]
      : croppingRegionPlanes[5];

    float cropPlanes[6] = { static_cast<float>(croppingRegionPlanes[0]),
      static_cast<float>(croppingRegionPlanes[1]),
      static_cast<float>(croppingRegionPlanes[2]),
      static_cast<float>(croppingRegionPlanes[3]),
      static_cast<float>(croppingRegionPlanes[4]),
      static_cast<float>(croppingRegionPlanes[5]) };

    prog->SetUniform1fv("in_croppingPlanes", 6, cropPlanes);
    const int numberOfRegions = 32;
    int cropFlagsArray[numberOfRegions];
    cropFlagsArray[0] = 0;
    int i = 1;
    while (cropFlags && i < 32)
    {
      cropFlagsArray[i] = cropFlags & 1;
      cropFlags = cropFlags >> 1;
      ++i;
    }
    for (; i < 32; ++i)
    {
      cropFlagsArray[i] = 0;
    }

    prog->SetUniform1iv("in_croppingFlags", numberOfRegions, cropFlagsArray);
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::SetClippingPlanes(
  vtkRenderer* vtkNotUsed(ren),
  vtkShaderProgram* prog,
  vtkVolume* vtkNotUsed(vol))
{
  if (this->Parent->GetClippingPlanes())
  {
    std::vector<float> clippingPlanes;
    // Currently we don't have any clipping plane
    clippingPlanes.push_back(0);

    this->Parent->GetClippingPlanes()->InitTraversal();
    vtkPlane* plane;
    while ((plane = this->Parent->GetClippingPlanes()->GetNextItem()))
    {
      // Planes are in world coordinates
      double planeOrigin[3], planeNormal[3];
      plane->GetOrigin(planeOrigin);
      plane->GetNormal(planeNormal);

      clippingPlanes.push_back(planeOrigin[0]);
      clippingPlanes.push_back(planeOrigin[1]);
      clippingPlanes.push_back(planeOrigin[2]);
      clippingPlanes.push_back(planeNormal[0]);
      clippingPlanes.push_back(planeNormal[1]);
      clippingPlanes.push_back(planeNormal[2]);
    }

    clippingPlanes[0] = clippingPlanes.size() > 1
      ? static_cast<int>(clippingPlanes.size() - 1)
      : 0;

    prog->SetUniform1fv("in_clippingPlanes",
      static_cast<int>(clippingPlanes.size()),
      &clippingPlanes[0]);
  }
}

// -----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::CheckPropertyKeys(vtkVolume* vol)
{
  // Check the property keys to see if we should modify the blend/etc state:
  // Otherwise this breaks volume/translucent geo depth peeling.
  vtkInformation* volumeKeys = vol->GetPropertyKeys();
  this->PreserveGLState = false;
  if (volumeKeys && volumeKeys->Has(vtkOpenGLActor::GLDepthMaskOverride()))
  {
    int override = volumeKeys->Get(vtkOpenGLActor::GLDepthMaskOverride());
    if (override != 0 && override != 1)
    {
      this->PreserveGLState = true;
    }
  }

  // Some render passes (e.g. DualDepthPeeling) adjust the viewport for
  // intermediate passes so it is necessary to preserve it. This is a
  // temporary fix for vtkDualDepthPeelingPass to work when various viewports
  // are defined.  The correct way of fixing this would be to avoid setting the
  // viewport within the mapper.  It is enough for now to check for the
  // RenderPasses() vtkInfo given that vtkDualDepthPeelingPass is the only pass
  // currently supported by this mapper, the viewport will have to be adjusted
  // externally before adding support for other passes.
  vtkInformation* info = vol->GetPropertyKeys();
  this->PreserveViewport =
    info && info->Has(vtkOpenGLRenderPass::RenderPasses());
}

// -----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::CheckPickingState(
  vtkRenderer* ren)
{
  vtkHardwareSelector* selector = ren->GetSelector();
  bool selectorPicking = selector != nullptr;
  if (selector)
  {
    // this mapper currently only supports cell picking
    selectorPicking &=
      selector->GetFieldAssociation() == vtkDataObject::FIELD_ASSOCIATION_CELLS;
  }

  this->IsPicking = selectorPicking || ren->GetRenderWindow()->GetIsPicking();
  if (this->IsPicking)
  {
    // rebuild the shader on every pass
    this->SelectionStateTime.Modified();
    this->CurrentSelectionPass =
      selector ? selector->GetCurrentPass() : vtkHardwareSelector::ACTOR_PASS;
  }
  else if (this->CurrentSelectionPass !=
    vtkHardwareSelector::MIN_KNOWN_PASS - 1)
  {
    // return to the regular rendering state
    this->SelectionStateTime.Modified();
    this->CurrentSelectionPass = vtkHardwareSelector::MIN_KNOWN_PASS - 1;
  }
}

// -----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::BeginPicking(vtkRenderer* ren)
{
  vtkHardwareSelector* selector = ren->GetSelector();
  if (selector && this->IsPicking)
  {
    selector->BeginRenderProp();

    if (this->CurrentSelectionPass >= vtkHardwareSelector::ID_LOW24)
    {
      selector->RenderAttributeId(0);
    }
  }
}

//------------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::SetPickingId(vtkRenderer* ren)
{
  float propIdColor[3] = { 0.0, 0.0, 0.0 };
  vtkHardwareSelector* selector = ren->GetSelector();

  if (selector && this->IsPicking)
  {
    // query the selector for the appropriate id
    selector->GetPropColorValue(propIdColor);
  }
  else // RenderWindow is picking
  {
    unsigned int const idx = ren->GetCurrentPickId();
    vtkHardwareSelector::Convert(idx, propIdColor);
  }

  this->ShaderProgram->SetUniform3f("in_propId", propIdColor);
}

// ---------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::EndPicking(vtkRenderer* ren)
{
  vtkHardwareSelector* selector = ren->GetSelector();
  if (selector && this->IsPicking)
  {
    if (this->CurrentSelectionPass >= vtkHardwareSelector::ID_LOW24)
    {
      // tell the selector the maximum number of cells that the mapper could
      // render
      unsigned int const numVoxels = (this->Extents[1] - this->Extents[0]) *
        (this->Extents[3] - this->Extents[2]) *
        (this->Extents[5] - this->Extents[4]);
      selector->RenderAttributeId(numVoxels);
    }
    selector->EndRenderProp();
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::
  LoadRequireDepthTextureExtensions(vtkRenderWindow* vtkNotUsed(renWin))
{
  // Reset the message stream for extensions
  if (vtkOpenGLRenderWindow::GetContextSupportsOpenGL32())
  {
    this->LoadDepthTextureExtensionsSucceeded = true;
    return;
  }

  this->ExtensionsStringStream.str("");
  this->ExtensionsStringStream.clear();

#if GL_ES_VERSION_3_0 != 1
  // Check for float texture support. This extension became core
  // in 3.0
  if (!glewIsSupported("GL_ARB_texture_float"))
  {
    this->ExtensionsStringStream << "Required extension "
                                 << " GL_ARB_texture_float is not supported";
    return;
  }
#endif

  // NOTE: Support for depth sampler texture made into the core since version
  // 1.4 and therefore we are no longer checking for it.
  this->LoadDepthTextureExtensionsSucceeded = true;
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::CreateBufferObjects()
{
#ifdef __APPLE__
  if (vtkOpenGLRenderWindow::GetContextSupportsOpenGL32())
#endif
  {
    glGenVertexArrays(1, &this->CubeVAOId);
  }
  glGenBuffers(1, &this->CubeVBOId);
  glGenBuffers(1, &this->CubeIndicesId);
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::DeleteBufferObjects()
{
  if (this->CubeVBOId)
  {
    glBindBuffer(GL_ARRAY_BUFFER, this->CubeVBOId);
    glDeleteBuffers(1, &this->CubeVBOId);
    this->CubeVBOId = 0;
  }

  if (this->CubeIndicesId)
  {
    glBindBuffer(GL_ELEMENT_ARRAY_BUFFER, this->CubeIndicesId);
    glDeleteBuffers(1, &this->CubeIndicesId);
    this->CubeIndicesId = 0;
  }

  if (this->CubeVAOId)
  {
#ifdef __APPLE__
    if (vtkOpenGLRenderWindow::GetContextSupportsOpenGL32())
#endif
    {
      glDeleteVertexArrays(1, &this->CubeVAOId);
    }
    this->CubeVAOId = 0;
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::ConvertTextureToImageData(
  vtkTextureObject* texture,
  vtkImageData* output)
{
  if (!texture)
  {
    return;
  }
  unsigned int tw = texture->GetWidth();
  unsigned int th = texture->GetHeight();
  unsigned int tnc = texture->GetComponents();
  int tt = texture->GetVTKDataType();

  vtkPixelExtent texExt(0U, tw - 1U, 0U, th - 1U);

  int dataExt[6] = { 0, 0, 0, 0, 0, 0 };
  texExt.GetData(dataExt);

  double dataOrigin[6] = { 0, 0, 0, 0, 0, 0 };

  vtkImageData* id = vtkImageData::New();
  id->SetOrigin(dataOrigin);
  id->SetDimensions(tw, th, 1);
  id->SetExtent(dataExt);
  id->AllocateScalars(tt, tnc);

  vtkPixelBufferObject* pbo = texture->Download();

  vtkPixelTransfer::Blit(texExt,
    texExt,
    texExt,
    texExt,
    tnc,
    tt,
    pbo->MapPackedBuffer(),
    tnc,
    tt,
    id->GetScalarPointer(0, 0, 0));

  pbo->UnmapPackedBuffer();
  pbo->Delete();

  if (!output)
  {
    output = vtkImageData::New();
  }
  output->DeepCopy(id);
  id->Delete();
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::BeginImageSample(
  vtkRenderer* ren,
  vtkVolume* vol)
{
  const auto numBuffers = this->GetNumImageSampleDrawBuffers(vol);
  if (numBuffers != this->NumImageSampleDrawBuffers)
  {
    if (numBuffers > this->NumImageSampleDrawBuffers)
    {
      this->ReleaseImageSampleGraphicsResources(ren->GetRenderWindow());
    }

    this->NumImageSampleDrawBuffers = numBuffers;
    this->RebuildImageSampleProg = true;
  }

  float const xySampleDist = this->Parent->GetImageSampleDistance();
  if (xySampleDist != 1.f && this->InitializeImageSampleFBO(ren))
  {
    this->ImageSampleFBO->SaveCurrentBindingsAndBuffers(GL_DRAW_FRAMEBUFFER);
    this->ImageSampleFBO->DeactivateDrawBuffers();
    this->ImageSampleFBO->Bind(GL_DRAW_FRAMEBUFFER);
    this->ImageSampleFBO->ActivateDrawBuffers(
      static_cast<unsigned int>(this->NumImageSampleDrawBuffers));

    glClearColor(0.0, 0.0, 0.0, 0.0);
    glClear(GL_COLOR_BUFFER_BIT);
  }
}

//----------------------------------------------------------------------------
bool vtkOpenGLGPUVolumeRayCastMapperInternals::InitializeImageSampleFBO(
  vtkRenderer* ren)
{
  // Set the FBO viewport size. These are used in the shader to normalize the
  // fragment coordinate, the normalized coordinate is used to fetch the depth
  // buffer.
  this->WindowSize[0] /= this->Parent->GetImageSampleDistance();
  this->WindowSize[1] /= this->Parent->GetImageSampleDistance();
  this->WindowLowerLeft[0] = 0;
  this->WindowLowerLeft[1] = 0;

  // Set FBO viewport
  glViewport(this->WindowLowerLeft[0],
    this->WindowLowerLeft[1],
    this->WindowSize[0],
    this->WindowSize[1]);

  if (!this->ImageSampleFBO)
  {
    vtkOpenGLRenderWindow* win =
      vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow());

    this->ImageSampleTexture.reserve(this->NumImageSampleDrawBuffers);
    this->ImageSampleTexNames.reserve(this->NumImageSampleDrawBuffers);
    for (size_t i = 0; i < this->NumImageSampleDrawBuffers; i++)
    {
      auto tex = vtkSmartPointer<vtkTextureObject>::New();
      tex->SetContext(win);
      tex->Create2D(
        this->WindowSize[0], this->WindowSize[1], 4, VTK_UNSIGNED_CHAR, false);
      tex->Activate();
      tex->SetMinificationFilter(vtkTextureObject::Linear);
      tex->SetMagnificationFilter(vtkTextureObject::Linear);
      tex->SetWrapS(vtkTextureObject::ClampToEdge);
      tex->SetWrapT(vtkTextureObject::ClampToEdge);
      this->ImageSampleTexture.push_back(tex);

      std::stringstream ss;
      ss << i;
      const std::string name = "renderedTex_" + ss.str();
      this->ImageSampleTexNames.push_back(name);
    }

    this->ImageSampleFBO = vtkOpenGLFramebufferObject::New();
    this->ImageSampleFBO->SetContext(win);
    this->ImageSampleFBO->SaveCurrentBindingsAndBuffers(GL_FRAMEBUFFER);
    this->ImageSampleFBO->Bind(GL_FRAMEBUFFER);
    this->ImageSampleFBO->InitializeViewport(
      this->WindowSize[0], this->WindowSize[1]);

    auto num = static_cast<unsigned int>(this->NumImageSampleDrawBuffers);
    for (unsigned int i = 0; i < num; i++)
    {
      this->ImageSampleFBO->AddColorAttachment(
        GL_FRAMEBUFFER, i, this->ImageSampleTexture[i]);
    }

    // Verify completeness
    const int complete =
      this->ImageSampleFBO->CheckFrameBufferStatus(GL_FRAMEBUFFER);
    for (auto& tex : this->ImageSampleTexture)
    {
      tex->Deactivate();
    }
    this->ImageSampleFBO->RestorePreviousBindingsAndBuffers(GL_FRAMEBUFFER);

    if (!complete)
    {
      vtkGenericWarningMacro(<< "Failed to attach ImageSampleFBO!");
      this->ReleaseImageSampleGraphicsResources(win);
      return false;
    }

    this->RebuildImageSampleProg = true;
    return true;
  }

  // Resize if necessary
  int lastSize[2];
  this->ImageSampleFBO->GetLastSize(lastSize);
  if (lastSize[0] != this->WindowSize[0] || lastSize[1] != this->WindowSize[1])
  {
    this->ImageSampleFBO->Resize(this->WindowSize[0], this->WindowSize[1]);
  }

  return true;
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::EndImageSample(vtkRenderer* ren)
{
  if (this->Parent->GetImageSampleDistance() != 1.f)
  {
    this->ImageSampleFBO->DeactivateDrawBuffers();
    this->ImageSampleFBO->RestorePreviousBindingsAndBuffers(
      GL_DRAW_FRAMEBUFFER);
    if (this->RenderPassAttached)
    {
      this->ImageSampleFBO->ActivateDrawBuffers(
        static_cast<unsigned int>(this->NumImageSampleDrawBuffers));
    }

    // Render the contents of ImageSampleFBO as a quad to intermix with the
    // rest of the scene.
    typedef vtkOpenGLRenderUtilities GLUtil;
    vtkOpenGLRenderWindow* win =
      static_cast<vtkOpenGLRenderWindow*>(ren->GetRenderWindow());

    if (this->RebuildImageSampleProg)
    {
      std::string frag = GLUtil::GetFullScreenQuadFragmentShaderTemplate();

      vtkShaderProgram::Substitute(frag,
        "//VTK::FSQ::Decl",
        vtkvolume::ImageSampleDeclarationFrag(
          this->ImageSampleTexNames, this->NumImageSampleDrawBuffers));
      vtkShaderProgram::Substitute(frag,
        "//VTK::FSQ::Impl",
        vtkvolume::ImageSampleImplementationFrag(
          this->ImageSampleTexNames, this->NumImageSampleDrawBuffers));

      this->ImageSampleProg = win->GetShaderCache()->ReadyShaderProgram(
        GLUtil::GetFullScreenQuadVertexShader().c_str(),
        frag.c_str(),
        GLUtil::GetFullScreenQuadGeometryShader().c_str());
    }
    else
    {
      win->GetShaderCache()->ReadyShaderProgram(this->ImageSampleProg);
    }

    if (!this->ImageSampleProg)
    {
      vtkGenericWarningMacro(<< "Failed to initialize ImageSampleProgram!");
      return;
    }

    if (!this->ImageSampleVAO)
    {
      this->ImageSampleVBO = vtkOpenGLBufferObject::New();
      this->ImageSampleVAO = vtkOpenGLVertexArrayObject::New();
      GLUtil::PrepFullScreenVAO(
        this->ImageSampleVBO, this->ImageSampleVAO, this->ImageSampleProg);
    }

    // Adjust the GL viewport to VTK's defined viewport
    ren->GetTiledSizeAndOrigin(this->WindowSize,
      this->WindowSize + 1,
      this->WindowLowerLeft,
      this->WindowLowerLeft + 1);
    glViewport(this->WindowLowerLeft[0],
      this->WindowLowerLeft[1],
      this->WindowSize[0],
      this->WindowSize[1]);

    // Bind objects and draw
    glEnable(GL_BLEND);
    glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);
    glDisable(GL_DEPTH_TEST);

    for (size_t i = 0; i < this->NumImageSampleDrawBuffers; i++)
    {
      this->ImageSampleTexture[i]->Activate();
      this->ImageSampleProg->SetUniformi(this->ImageSampleTexNames[i].c_str(),
        this->ImageSampleTexture[i]->GetTextureUnit());
    }

    this->ImageSampleVAO->Bind();
    GLUtil::DrawFullScreenQuad();
    this->ImageSampleVAO->Release();
    vtkOpenGLStaticCheckErrorMacro("Error after DrawFullScreenQuad()!");

    for (auto& tex : this->ImageSampleTexture)
    {
      tex->Deactivate();
    }
  }
}

//------------------------------------------------------------------------------
size_t vtkOpenGLGPUVolumeRayCastMapperInternals::GetNumImageSampleDrawBuffers(
  vtkVolume* vol)
{
  if (this->RenderPassAttached)
  {
    vtkInformation* info = vol->GetPropertyKeys();
    const int num = info->Length(vtkOpenGLRenderPass::RenderPasses());
    vtkObjectBase* rpBase =
      info->Get(vtkOpenGLRenderPass::RenderPasses(), num - 1);
    vtkOpenGLRenderPass* rp = static_cast<vtkOpenGLRenderPass*>(rpBase);
    return static_cast<size_t>(rp->GetActiveDrawBuffers());
  }

  return 1;
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::SetupRenderToTexture(
  vtkRenderer* ren)
{
  if (this->Parent->GetRenderToImage() &&
    this->Parent->GetCurrentPass() ==
      vtkOpenGLGPUVolumeRayCastMapper::RenderPass)
  {
    if (this->Parent->GetImageSampleDistance() != 1.f)
    {
      this->WindowSize[0] /= this->Parent->GetImageSampleDistance();
      this->WindowSize[1] /= this->Parent->GetImageSampleDistance();
    }

    if ((this->LastRenderToImageWindowSize[0] != this->WindowSize[0]) ||
      (this->LastRenderToImageWindowSize[1] != this->WindowSize[1]))
    {
      this->LastRenderToImageWindowSize[0] = this->WindowSize[0];
      this->LastRenderToImageWindowSize[1] = this->WindowSize[1];
      this->ReleaseRenderToTextureGraphicsResources(ren->GetRenderWindow());
    }

    if (!this->FBO)
    {
      this->FBO = vtkOpenGLFramebufferObject::New();
    }

    this->FBO->SetContext(
      vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));

    this->FBO->SaveCurrentBindingsAndBuffers();
    this->FBO->Bind(GL_FRAMEBUFFER);
    this->FBO->InitializeViewport(this->WindowSize[0], this->WindowSize[1]);

    int depthImageScalarType = this->Parent->GetDepthImageScalarType();
    bool initDepthTexture = true;
    // Re-instantiate the depth texture object if the scalar type requested has
    // changed from the last frame
    if (this->RTTDepthTextureObject &&
      this->RTTDepthTextureType == depthImageScalarType)
    {
      initDepthTexture = false;
    }

    if (initDepthTexture)
    {
      if (this->RTTDepthTextureObject)
      {
        this->RTTDepthTextureObject->Delete();
        this->RTTDepthTextureObject = nullptr;
      }
      this->RTTDepthTextureObject = vtkTextureObject::New();
      this->RTTDepthTextureObject->SetContext(
        vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));
      this->RTTDepthTextureObject->Create2D(this->WindowSize[0],
        this->WindowSize[1],
        1,
        depthImageScalarType,
        false);
      this->RTTDepthTextureObject->Activate();
      this->RTTDepthTextureObject->SetMinificationFilter(
        vtkTextureObject::Nearest);
      this->RTTDepthTextureObject->SetMagnificationFilter(
        vtkTextureObject::Nearest);
      this->RTTDepthTextureObject->SetAutoParameters(0);

      // Cache the value of the scalar type
      this->RTTDepthTextureType = depthImageScalarType;
    }

    if (!this->RTTColorTextureObject)
    {
      this->RTTColorTextureObject = vtkTextureObject::New();

      this->RTTColorTextureObject->SetContext(
        vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));
      this->RTTColorTextureObject->Create2D(
        this->WindowSize[0], this->WindowSize[1], 4, VTK_UNSIGNED_CHAR, false);
      this->RTTColorTextureObject->Activate();
      this->RTTColorTextureObject->SetMinificationFilter(
        vtkTextureObject::Nearest);
      this->RTTColorTextureObject->SetMagnificationFilter(
        vtkTextureObject::Nearest);
      this->RTTColorTextureObject->SetAutoParameters(0);
    }

    if (!this->RTTDepthBufferTextureObject)
    {
      this->RTTDepthBufferTextureObject = vtkTextureObject::New();
      this->RTTDepthBufferTextureObject->SetContext(
        vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));
      this->RTTDepthBufferTextureObject->AllocateDepth(
        this->WindowSize[0], this->WindowSize[1], vtkTextureObject::Float32);
      this->RTTDepthBufferTextureObject->Activate();
      this->RTTDepthBufferTextureObject->SetMinificationFilter(
        vtkTextureObject::Nearest);
      this->RTTDepthBufferTextureObject->SetMagnificationFilter(
        vtkTextureObject::Nearest);
      this->RTTDepthBufferTextureObject->SetAutoParameters(0);
    }

    this->FBO->Bind(GL_FRAMEBUFFER);
    this->FBO->AddDepthAttachment(
      GL_FRAMEBUFFER, this->RTTDepthBufferTextureObject);
    this->FBO->AddColorAttachment(
      GL_FRAMEBUFFER, 0U, this->RTTColorTextureObject);
    this->FBO->AddColorAttachment(
      GL_FRAMEBUFFER, 1U, this->RTTDepthTextureObject);
    this->FBO->ActivateDrawBuffers(2);

    this->FBO->CheckFrameBufferStatus(GL_FRAMEBUFFER);

    glClearColor(1.0, 1.0, 1.0, 0.0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::ExitRenderToTexture(
  vtkRenderer* vtkNotUsed(ren))
{
  if (this->Parent->GetRenderToImage() &&
    this->Parent->GetCurrentPass() ==
      vtkOpenGLGPUVolumeRayCastMapper::RenderPass)
  {
    this->FBO->RemoveTexDepthAttachment(GL_FRAMEBUFFER);
    this->FBO->RemoveTexColorAttachment(GL_FRAMEBUFFER, 0U);
    this->FBO->RemoveTexColorAttachment(GL_FRAMEBUFFER, 1U);
    this->FBO->DeactivateDrawBuffers();
    this->FBO->RestorePreviousBindingsAndBuffers();

    this->RTTDepthBufferTextureObject->Deactivate();
    this->RTTColorTextureObject->Deactivate();
    this->RTTDepthTextureObject->Deactivate();
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::SetupDepthPass(vtkRenderer* ren)
{
  if (this->Parent->GetImageSampleDistance() != 1.f)
  {
    this->WindowSize[0] /= this->Parent->GetImageSampleDistance();
    this->WindowSize[1] /= this->Parent->GetImageSampleDistance();
  }

  if ((this->LastDepthPassWindowSize[0] != this->WindowSize[0]) ||
    (this->LastDepthPassWindowSize[1] != this->WindowSize[1]))
  {
    this->LastDepthPassWindowSize[0] = this->WindowSize[0];
    this->LastDepthPassWindowSize[1] = this->WindowSize[1];
    this->ReleaseDepthPassGraphicsResources(ren->GetRenderWindow());
  }

  if (!this->DPFBO)
  {
    this->DPFBO = vtkOpenGLFramebufferObject::New();
  }

  this->DPFBO->SetContext(
    vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));

  this->DPFBO->SaveCurrentBindingsAndBuffers();
  this->DPFBO->Bind(GL_FRAMEBUFFER);
  this->DPFBO->InitializeViewport(this->WindowSize[0], this->WindowSize[1]);

  if (!this->DPDepthBufferTextureObject || !this->DPColorTextureObject)
  {
    this->DPDepthBufferTextureObject = vtkTextureObject::New();
    this->DPDepthBufferTextureObject->SetContext(
      vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));
    this->DPDepthBufferTextureObject->AllocateDepth(
      this->WindowSize[0], this->WindowSize[1], vtkTextureObject::Native);
    this->DPDepthBufferTextureObject->Activate();
    this->DPDepthBufferTextureObject->SetMinificationFilter(
      vtkTextureObject::Nearest);
    this->DPDepthBufferTextureObject->SetMagnificationFilter(
      vtkTextureObject::Nearest);
    this->DPDepthBufferTextureObject->SetAutoParameters(0);
    this->DPDepthBufferTextureObject->Bind();

    this->DPColorTextureObject = vtkTextureObject::New();

    this->DPColorTextureObject->SetContext(
      vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow()));
    this->DPColorTextureObject->Create2D(
      this->WindowSize[0], this->WindowSize[1], 4, VTK_UNSIGNED_CHAR, false);
    this->DPColorTextureObject->Activate();
    this->DPColorTextureObject->SetMinificationFilter(
      vtkTextureObject::Nearest);
    this->DPColorTextureObject->SetMagnificationFilter(
      vtkTextureObject::Nearest);
    this->DPColorTextureObject->SetAutoParameters(0);

    this->DPFBO->AddDepthAttachment(
      GL_FRAMEBUFFER, this->DPDepthBufferTextureObject);

    this->DPFBO->AddColorAttachment(
      GL_FRAMEBUFFER, 0U, this->DPColorTextureObject);
  }

  this->DPFBO->ActivateDrawBuffers(1);
  this->DPFBO->CheckFrameBufferStatus(GL_FRAMEBUFFER);

  // Setup the contour polydata mapper to render to DPFBO
  this->ContourMapper->SetInputConnection(this->ContourFilter->GetOutputPort());

  glClearColor(0.0, 0.0, 0.0, 0.0);
  glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
  glEnable(GL_DEPTH_TEST);
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::ExitDepthPass(
  vtkRenderer* vtkNotUsed(ren))
{
  this->DPFBO->DeactivateDrawBuffers();
  this->DPFBO->RestorePreviousBindingsAndBuffers();

  this->DPDepthBufferTextureObject->Deactivate();
  this->DPColorTextureObject->Deactivate();
  glDisable(GL_DEPTH_TEST);
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::
  ReleaseRenderToTextureGraphicsResources(vtkWindow* win)
{
  vtkOpenGLRenderWindow* rwin = vtkOpenGLRenderWindow::SafeDownCast(win);

  if (rwin)
  {
    if (this->FBO)
    {
      this->FBO->Delete();
      this->FBO = nullptr;
    }

    if (this->RTTDepthBufferTextureObject)
    {
      this->RTTDepthBufferTextureObject->ReleaseGraphicsResources(win);
      this->RTTDepthBufferTextureObject->Delete();
      this->RTTDepthBufferTextureObject = nullptr;
    }

    if (this->RTTDepthTextureObject)
    {
      this->RTTDepthTextureObject->ReleaseGraphicsResources(win);
      this->RTTDepthTextureObject->Delete();
      this->RTTDepthTextureObject = nullptr;
    }

    if (this->RTTColorTextureObject)
    {
      this->RTTColorTextureObject->ReleaseGraphicsResources(win);
      this->RTTColorTextureObject->Delete();
      this->RTTColorTextureObject = nullptr;
    }
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::
  ReleaseDepthPassGraphicsResources(vtkWindow* win)
{
  vtkOpenGLRenderWindow* rwin = vtkOpenGLRenderWindow::SafeDownCast(win);

  if (rwin)
  {
    if (this->DPFBO)
    {
      this->DPFBO->Delete();
      this->DPFBO = nullptr;
    }

    if (this->DPDepthBufferTextureObject)
    {
      this->DPDepthBufferTextureObject->ReleaseGraphicsResources(win);
      this->DPDepthBufferTextureObject->Delete();
      this->DPDepthBufferTextureObject = nullptr;
    }

    if (this->DPColorTextureObject)
    {
      this->DPColorTextureObject->ReleaseGraphicsResources(win);
      this->DPColorTextureObject->Delete();
      this->DPColorTextureObject = nullptr;
    }

    this->ContourMapper->ReleaseGraphicsResources(win);
  }
}

//----------------------------------------------------------------------------
void vtkOpenGLGPUVolumeRayCastMapperInternals::
  ReleaseImageSampleGraphicsResources(vtkWindow* win)
{
  vtkOpenGLRenderWindow* rwin = vtkOpenGLRenderWindow::SafeDownCast(win);

  if (rwin)
  {
    if (this->ImageSampleFBO)
    {
      this->ImageSampleFBO->Delete();
      this->ImageSampleFBO = nullptr;
    }

    for (auto& tex : this->ImageSampleTexture)
    {
      tex->ReleaseGraphicsResources(win);
      tex = nullptr;
    }
    this->ImageSampleTexture.clear();
    this->ImageSampleTexNames.clear();

    if (this->ImageSampleVBO)
    {
      this->ImageSampleVBO->Delete();
      this->ImageSampleVBO = nullptr;
    }

    if (this->ImageSampleVAO)
    {
      this->ImageSampleVAO->Delete();
      this->ImageSampleVAO = nullptr;
    }

    // Do not delete the shader program - Let the cache clean it up.
    this->ImageSampleProg = nullptr;
  }
}
