#include <random>
#include <iostream>
#include <mpi.h>
#include <vtkDoubleArray.h>
#include <vtkImageData.h>
#include <vtkPointData.h>
#include "Error.h"
#include "Histogram.h"
#include "VTKDataAdaptor.h"

//#define GENERATE_SEQUENCE
//#define GENERATE_HISTOGRAM

extern unsigned int gNBins;
extern unsigned int gHist[];
extern double gMin;
extern double gMax;
extern unsigned int gNx;
extern unsigned int gNy;
extern unsigned int gNz;
extern unsigned int gSequenceLen;
extern double gSequence[];

template<typename n_t>
int getSequence(std::vector<n_t> &vals)
{
  int rank = 0;
  int nRanks = 1;

  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  MPI_Comm_size(MPI_COMM_WORLD, &nRanks);

  long blockSize = gSequenceLen/nRanks;
  long nLarge = gSequenceLen%nRanks;
  long nLocal = blockSize + (rank < nLarge ? 1 : 0);
  long start = rank*blockSize + (rank < nLarge ? rank : nLarge);
  long end = start + nLocal;

#if defined(GENERATE_SEQUENCE)
  std::random_device dev;
  std::mt19937 gen(dev());
  std::normal_distribution<n_t> dist(5,2);
  for (long i = 0; i < nLocal; ++i)
    {
    n_t val = dist(gen);
    vals.push_back(val);
    std::cerr << val << " ";
    }
  std::cerr << std::endl;
#else
  for (long i = start; i < end; ++i)
    vals.push_back(gSequence[i]);
#endif
  return 0;
}

int validateHistogram(double min, double max, std::vector<unsigned int> &bins)
{
#if defined(GENERATE_HISTOGRAM)
  unsigned int nBins = bins.size();
  for (unsigned int i = 0; i < nBins; ++i)
    std::cerr << bins[i] << ", ";
  std::cerr << std::endl;
  std::cerr << std::setprecision(std::numeric_limits<long double>::digits10 + 1) << min << endl;
  std::cerr << std::setprecision(std::numeric_limits<long double>::digits10 + 1) << max << endl;
#else
  int rank = 0;
  MPI_Comm_rank(MPI_COMM_WORLD, &rank);
  if (rank != 0)
    return 0;

  if (bins.size() != gNBins)
    {
    SENSEI_ERROR("wrong number of bins " << bins.size())
    return -1;
    }

  unsigned int gNBins = 10;
  unsigned int gHist[] = {8, 55, 414, 1360, 2669, 2867, 1841, 640, 132, 14};

  for (unsigned int i = 0; i < gNBins; ++i)
    {
    if (gHist[i] != bins[i])
      {
      SENSEI_ERROR("Bin count is wrong at bin " << i)
      return -1;
      }
    }

  if (fabs(gMin - min) > 1.0e-6)
    {
    SENSEI_ERROR("Incorrect minimum")
    return -1;
    }

  if (fabs(gMax - max) > 1.0e-6)
    {
    SENSEI_ERROR("Incorrect maximum")
    return -1;
    }

#endif
  return 0;
}


int main(int argc, char **argv)
{
  MPI_Init(&argc, &argv);

  std::vector<double> vals;
  getSequence(vals);
  unsigned int nVals = vals.size();

  vtkDoubleArray *da = vtkDoubleArray::New();
  da->SetNumberOfTuples(nVals);
  da->SetName("normal");
  for (unsigned int i = 0; i < nVals; ++i)
    *da->GetPointer(i) = vals[i];

  vtkImageData *im = vtkImageData::New();
  im->SetDimensions(gNx, gNy, gNz);
  im->GetPointData()->AddArray(da);
  da->Delete();

  sensei::VTKDataAdaptor *dataAdaptor = sensei::VTKDataAdaptor::New();
  dataAdaptor->SetDataObject("mesh", im);
  im->Delete();

  sensei::Histogram *analysisAdaptor = sensei::Histogram::New();

  analysisAdaptor->Initialize(gNBins, "mesh", vtkDataObject::POINT,
     "normal", "");

  analysisAdaptor->Execute(dataAdaptor);
  dataAdaptor->Delete();

  double min = 0.0;
  double max = 0.0;
  std::vector<unsigned int> bins;
  analysisAdaptor->GetHistogram(min, max, bins);

  int testResult = validateHistogram(min, max, bins);

  analysisAdaptor->Finalize();
  analysisAdaptor->Delete();

  MPI_Finalize();

  return testResult;

}

// the histogram of the below data as calculated by SENSEI
unsigned int gNBins = 10;
unsigned int gHist[] = {8, 55, 414, 1360, 2669, 2867, 1841, 640, 132, 14};
double gMin = -2.877769999999999939;
double gMax = 12.40249999999999986;

// data from a normal distribution on a Cartesian mesh
unsigned int gNx = 10;
unsigned int gNy = 10;
unsigned int gNz = 100;
unsigned int gSequenceLen = gNx*gNy*gNz;
double gSequence[] = {4.88154, 4.94947, 7.56139, 7.9841, 6.91428, 2.07837,
  6.00519, 5.33419, 8.00533, 2.67247, 3.34669, 5.11191, 3.53075, 3.52474,
  3.74709, 2.79868, 3.55827, 4.94499, 4.23562, 6.57719, 6.14486, 7.6833,
  5.64141, 0.686811, 6.27175, 7.36018, 8.78421, 7.08961, 2.83816, 7.56889,
  5.98855, 2.97578, 2.70819, 5.1122, 4.26025, 3.21616, 5.86099, 3.06063,
  3.70063, 3.01617, 2.65363, 4.67202, 5.39182, 5.88548, 5.30704, 7.44874,
  3.46047, 4.5671, 4.39995, 0.816105, 3.20079, 4.89209, 3.90581, 2.15088,
  7.67066, 1.90373, 5.7932, 2.3447, 7.39718, 3.99207, 6.24623, 4.70216, 6.9141,
  5.31974, 5.58113, 6.96671, 7.83091, 1.73157, 3.66478, 4.67923, 6.81851,
  4.11632, 1.51874, 4.73193, 8.34513, 4.46826, 2.76498, 6.84334, 2.87384,
  3.48703, 5.28348, 1.856, 5.89946, 5.13855, 4.95804, 7.70441, 4.98032,
  0.668146, 1.85475, 5.35523, 4.58994, 1.9614, 5.83351, 4.50062, 4.54014,
  5.73923, 2.71621, 5.66944, 6.49558, 8.88723, 5.44452, 5.36531, 4.65346,
  5.37221, 6.78594, 3.68444, 2.1914, 3.54954, 5.12264, 10.3208, 6.25259,
  7.64545, 3.79109, 8.8532, 1.5399, 4.84575, 3.13395, 6.25296, 6.06892,
  4.59019, 4.17555, 0.759454, 3.959, 7.43717, 6.84334, 5.81382, 6.42337,
  4.16896, 5.61918, 2.93304, 7.51623, 4.01406, 7.74423, 3.73676, 4.6251,
  3.59861, 3.30651, 3.70951, 6.29639, 5.95889, 7.74647, 4.71205, 2.76188,
  4.83835, 3.9055, 7.24401, 6.91164, 2.95911, 1.74768, 4.50883, 7.06724,
  5.67211, 3.97181, 5.16168, 4.89055, 4.90464, 6.56012, 10.2811, 3.66731,
  2.06304, 6.45104, 6.31055, 3.96666, 9.6154, 2.37532, 2.22702, 6.92389,
  8.37899, 3.56871, 4.51459, 5.34746, 3.05125, 0.66664, 7.02948, 4.83889,
  7.37728, 5.3667, 5.5496, 4.11866, 5.17672, 0.206375, 4.79405, 3.27049,
  5.38239, 5.10665, 2.4656, 5.26244, 8.44464, 5.4579, 4.94132, 6.43088, 6.1057,
  7.60027, 1.389, 4.52499, 4.73346, 5.47057, 3.16639, 6.4386, 5.07838, 5.21516,
  4.92122, 7.91391, 4.09213, 1.68856, 3.76848, 2.99479, 6.81313, 8.73263,
  4.79413, 6.50187, 5.6506, 3.72192, 5.85268, 7.71059, 5.491, 4.83934, 4.28942,
  7.35838, 3.31111, 4.17939, 4.40919, 2.69356, 5.07702, 4.29882, 2.12523,
  7.34899, 3.01987, 4.35401, 6.35328, 2.86928, 6.90462, 6.70909, 5.58176,
  3.14488, 5.20216, 4.34982, 4.40767, 7.37696, 4.91058, 2.70181, 6.35881,
  7.4202, 6.27222, 5.17285, 5.35214, 3.67828, 8.1193, 7.11646, 4.7242, 2.11307,
  8.30726, 6.98093, 6.90649, 2.16255, 4.37479, 5.52714, 7.66752, -0.788683,
  9.10236, 8.39524, 4.23873, 7.3376, 5.13592, 5.40637, 5.58369, 7.00598,
  3.83522, 7.57133, 3.51924, 3.02835, 5.65458, 3.29211, 1.67809, 3.14193,
  3.97109, 5.52086, 3.98568, 6.89826, 5.16403, 6.2259, 1.77575, 6.02259,
  6.35635, 6.1201, 4.79691, 5.99734, 3.26259, 6.18889, 5.51009, 3.61456,
  2.4684, 7.32342, 3.74219, 9.48481, 7.12861, 4.32006, 6.19187, 4.59343,
  6.52992, 3.64729, 4.52582, 6.38421, 6.79095, 6.35858, 3.39446, 4.65166,
  6.49457, 3.1217, 3.30714, 0.713285, 2.42131, 5.06737, 5.7654, 5.30596,
  4.15872, 8.55615, 6.11646, 5.75356, 5.90323, 4.34727, 7.2129, 4.2887,
  7.72424, 2.22659, 3.98272, 2.48537, 5.4009, 2.15683, 6.29116, 2.64897,
  6.65256, 5.04562, 7.39369, 5.91269, 5.65017, 4.9545, 5.20653, 6.73288,
  4.06639, 4.40655, 8.47115, 6.25024, 4.69538, 5.2697, 6.38531, 3.02886,
  3.57652, 4.58909, 3.78946, 5.03056, 5.81719, 4.61786, 3.75375, 3.96148,
  3.2721, 6.32358, 3.30819, 8.8121, 5.05087, 5.41544, 5.97314, 6.83566,
  4.75241, 5.41468, 5.94655, 3.99276, 4.51385, 5.75728, 2.32002, 6.75658,
  5.11504, 3.503, 3.08793, 1.72131, 7.01086, 2.3222, 6.82979, 4.19523, 5.39674,
  3.93225, 9.81857, 6.10813, 2.93364, 1.99352, 4.79748, 5.12659, 5.09687,
  4.15366, 0.306607, 6.38503, 3.6489, 4.78332, 2.07933, 2.80674, 5.87714,
  6.21846, 2.61557, 8.58292, 8.85583, 4.17811, 0.681691, 3.72591, 0.421585,
  4.80468, 2.05066, 3.96148, 7.70291, 4.11964, 6.00843, 3.38972, 5.80853,
  4.92803, 3.35539, 7.90591, 9.65696, 3.77526, 4.6223, 6.66301, 3.25016,
  5.61298, 2.95536, 4.56237, 6.64156, 2.05691, 1.33306, 3.49984, 4.56855,
  6.96133, 5.84642, 5.75811, 7.31666, 3.22517, 6.84812, 3.14061, 6.12654,
  2.64788, 4.69849, 7.73183, 5.26457, 4.98213, 0.801835, 2.44807, 4.7951,
  7.03033, 4.58953, 3.14746, 5.65378, 4.51554, 7.47046, 7.84321, 7.16349,
  5.21224, 5.35306, 6.09308, 3.33394, 3.31392, 2.51508, 3.436, 4.83728, 4.7927,
  5.88819, 6.08155, 2.6348, 6.66377, -2.13925, 5.29164, 4.76191, 7.8312,
  2.30158, 2.85652, 5.9676, 8.23954, 8.89061, 8.30121, 0.550212, 5.77502,
  4.87845, 4.71395, 4.68871, 2.74119, 2.2249, 5.90209, 4.48979, 4.61213,
  3.50344, 8.47185, 0.61498, 4.98656, 6.8359, 3.74282, 8.04957, 7.14786,
  11.8541, 6.89014, 4.37192, 4.89167, 4.82382, 5.11821, 4.1857, 2.44102,
  3.10162, 5.14623, 3.93739, 6.89129, 5.52232, 1.73932, 3.91581, 4.19169,
  6.37432, 6.07379, 8.66665, 4.10255, 9.5634, 5.3549, 7.83966, 5.34233,
  5.58378, 1.78713, 6.85784, 5.3677, 4.08969, 0.416117, 4.07443, 7.81187,
  5.61073, 7.88804, 9.20878, 5.32332, 4.36384, 7.33569, 5.37935, 7.4894,
  3.57097, 8.8339, 4.3979, 6.63842, 5.39831, 6.24112, 3.50053, 7.51922,
  5.76123, -0.0186858, 3.06607, 8.19989, 3.78601, -1.13949, 4.92208, 4.56117,
  4.12832, 5.0611, 3.60266, 4.2879, 4.19595, 4.0581, 6.17177, 5.70378, 4.51933,
  12.0066, 5.64205, 4.36745, 4.35915, 5.81551, 5.53485, 4.37501, 7.80743,
  5.25775, 6.67965, 0.119288, 4.37295, 6.10835, 7.35967, 2.6451, 7.1322,
  2.47571, 5.39241, 5.46394, 6.23885, 6.44717, 2.08434, 4.38482, 5.39801,
  7.05102, 4.00472, 6.415, 6.6817, 5.89446, 7.40958, 5.82279, 0.189165,
  7.38395, 3.19271, 4.25807, 7.42484, 2.06788, 9.81218, 5.755, 8.27658,
  8.11687, 2.0871, 8.94268, 2.80134, 5.40693, 5.84839, 4.97288, 2.09877,
  1.91612, 6.69034, 4.24384, 5.86845, 5.00873, 3.4701, 3.72209, 2.33657,
  4.79239, 4.6944, 3.58688, 4.12109, 6.34466, 5.85874, 1.86148, 4.4127, 3.8003,
  4.88799, 5.06014, 9.05026, 5.61424, 3.76246, 5.90303, 3.37658, 8.62403,
  4.81756, 5.33126, 3.21886, 4.26171, 6.82324, 3.77727, 10.0185, 6.97975,
  0.420268, 3.51727, 7.01487, 2.76939, 1.21851, 3.645, 5.22996, 3.45392,
  4.41616, 2.73922, 8.6724, 3.64427, 6.06691, 5.9378, 5.43805, 6.82995,
  6.98546, 8.17956, 4.17883, 4.75696, 6.88619, 3.47025, 6.19091, 3.92965,
  6.29759, 3.16992, 2.56419, 3.68462, 3.10913, 5.61798, 7.52015, 3.2139,
  7.53476, 2.38829, 2.09276, 8.95632, 6.3064, 1.19987, 6.7309, 2.29951,
  4.15624, 4.13206, 4.53124, 6.05644, -1.90465, 1.71581, 4.12377, 1.07229,
  6.80486, 5.14741, 4.38234, 3.3064, 6.05102, 6.87142, 7.3201, 5.36097,
  1.52019, 6.11633, 7.85423, 4.6221, 7.01971, 4.47624, 3.47749, 0.910874,
  3.06026, 7.57805, 7.16333, 5.3667, 5.8969, 6.30045, 5.70619, 6.72198,
  3.73866, 6.24655, 6.46487, 3.14738, 6.58113, 6.708, 3.98711, 7.63113,
  2.45237, 1.24626, 1.22343, 4.84554, 6.04401, 2.18343, 5.35017, 7.06024,
  3.38051, 4.87, 5.75637, 1.58532, 6.9312, 2.88867, 4.74993, 4.63291, 5.64503,
  3.81665, 4.77247, 4.52913, 8.22549, 4.68543, 5.47514, 5.15918, 2.34614,
  5.3095, 5.05032, 8.52623, 4.35321, 4.48403, 4.38329, 5.67907, 7.38607,
  5.18714, 3.72956, 6.83954, 4.0576, 5.6418, 3.55189, 6.58478, 6.35577,
  4.09408, 5.40974, 5.89814, 7.08551, 4.67735, 4.58846, 4.42519, 6.94955,
  4.80813, 4.59311, 7.16214, 2.21199, 6.36967, 8.97095, 4.27619, 6.77198,
  7.27055, 4.65481, 6.0473, 9.63262, 8.1132, 1.48563, 6.25875, 7.91631,
  4.80353, 2.65121, 2.74551, 6.19977, 5.0105, 4.04953, 5.46025, 7.6514,
  3.57465, 6.94237, 4.7901, 7.13129, 7.98504, 6.48375, 1.62861, 6.63607,
  2.12066, 5.2925, 3.59181, 0.00297926, 4.78104, 4.57661, 4.80241, 3.77357,
  5.84889, -0.626752, 6.3128, 0.159584, 4.07031, 4.09114, 9.49379, 5.16034,
  6.66606, 4.08245, 1.17198, 3.52839, 9.36529, 2.18137, 5.33014, 3.89577,
  4.08912, 7.59328, 5.46541, 6.56337, 7.70258, 5.4269, 4.76789, 6.82594,
  10.2554, 3.65459, 4.17466, 6.33624, 7.56341, 5.49337, 8.33373, 6.96304,
  5.55558, 2.06003, 6.46427, 3.07287, 5.89236, 2.74692, 4.52766, 4.44681,
  8.41913, 4.24818, 4.94767, 6.2793, 5.63718, 3.15286, 5.6984, 3.40121,
  6.87715, 2.88621, 4.96168, 7.02604, 1.76525, 6.50918, 2.75248, 4.94434,
  5.10527, 7.57756, 4.18324, 3.62225, 1.69439, 4.6764, 4.19606, 5.84203,
  5.04625, 4.78958, 5.04818, 4.32927, 5.90165, 4.58902, 9.08788, 6.71742,
  3.21433, 6.78613, 8.03792, 2.64767, 5.62456, 6.43671, 6.4495, 6.68789, 5.971,
  8.14686, 4.21554, 3.34002, 4.22824, 4.5851, -2.24515, 2.40925, 5.37589,
  4.09434, 4.34659, 2.8242, 3.70561, 3.88643, 4.78368, 1.62928, 2.08132,
  4.28451, 3.7367, 6.48245, 6.10118, 6.38492, 3.05272, 5.27941, 0.971248,
  4.55078, 2.10792, 5.11068, 5.53604, 4.75108, 8.18838, 7.32767, 4.11913,
  2.54014, 7.94723, 6.68405, 5.58082, 5.29769, 2.08386, 3.66817, 4.64231,
  6.63516, 5.00424, 1.48259, 5.96132, 2.47831, 3.38693, 3.89216, 2.01703,
  4.13814, 5.01779, 4.29339, 9.88255, 3.89878, 5.51351, 1.55974, 7.70853,
  4.85013, 1.99317, 4.4853, 3.86401, 2.79729, 6.89581, 5.12359, 0.231104,
  6.82594, 4.40124, 2.33259, 5.70954, 3.91352, 6.3735, 5.71372, 8.00508,
  10.5743, 5.27479, 9.37476, 3.13131, 2.16698, 3.55701, 4.32907, 2.0579,
  4.56606, 5.52994, 3.69193, 1.97996, 1.8649, 8.98662, 4.5851, 7.33019,
  3.00487, 7.01168, 4.18177, 5.64302, 6.69911, 5.41771, 4.47928, 6.84507,
  4.54443, 5.1692, 3.00686, 4.28736, 4.83584, 1.42119, 4.0525, 4.65022,
  6.50104, 4.35764, 3.05431, 1.49941, 3.25103, 2.32849, 4.2665, 5.02976,
  1.39597, 1.52931, 6.88976, 6.02499, 4.79382, 4.10851, 6.42225, 8.19684,
  4.06516, 2.00718, 8.40659, 7.35928, 6.53932, 6.10469, 6.07167, 6.70644,
  3.0908, 8.49434, 4.47934, 6.77576, 7.81866, 2.41937, 2.72619, 5.91099,
  3.00522, 1.40483, 5.60986, 4.09169, 5.92426, 3.76788, 3.17984, 6.89162,
  5.193, 5.07186, 0.22824, 7.08377, 4.63143, 6.63451, 4.59116, 4.52844,
  4.65585, 5.52586, 4.89485, 2.26208, 7.10703, 3.82849, 6.3687, 6.91152,
  6.45165, 4.91604, 4.01694, 3.68331, 7.21954, 8.6709, 5.38411, 3.22248,
  2.94674, 7.59643, 7.53398, 8.31375, 2.61544, 5.88837, 1.02782, 7.39455,
  2.24348, 4.353, 1.31689, 7.8785, 4.1582, 3.50486, 5.07376, 1.99124, 7.65007,
  2.12596, 4.08811, 4.64828, 7.95139, 2.78546, 6.29828, 5.36246, 1.1226,
  4.79323, 2.47391, 3.27441, 5.14006, 6.93545, 4.81898, 8.75772, 2.17176,
  5.82125, 4.52329, 5.1125, 8.19909, 5.81047, 2.61882, 5.64324, 7.04078,
  6.35918, 4.19305, 3.32602, 5.10958, 7.12597, 9.22071, 7.04912, 3.17369,
  4.47552, 7.18552, 6.66736, 3.69979, 4.82106, 6.11962, 3.98052, 6.80041,
  1.76199, 4.11283, 7.07258, 2.78655, 4.07542, 4.67615, 1.86454, 5.55116,
  7.76278, 3.9939, 6.79852, 5.72475, 4.3566, 6.66904, 2.5714, 1.72731, 4.66124,
  2.44179, 2.76769, 1.90043, 3.22361, 6.05199, 6.87405, 4.93932, 1.22793,
  3.42453, 6.59906, 5.1272, 4.34882, 2.62179, 7.30908, 7.92923, 4.85004,
  4.55847, 6.42952, 4.82356, 4.65924, 4.08627, 8.96044, 4.46136, 6.05107,
  3.88982, 2.48103, 7.97128, 3.38881, 4.19732, 4.22472, 5.00197, 3.74277,
  3.52777, 5.19031, 6.22834, 5.02843, 4.82193, 5.49499, 4.14651, 3.68326,
  6.26329, 3.60767, 3.77965, 3.26093, 4.06359, 6.98338, 6.48535, 3.93452,
  5.94814, 3.41452, 9.02774, 3.14196, 0.0424577, 7.32858, 2.88959, 6.6547,
  4.87486, 5.6277, 2.60069, 4.8144, 5.15745, 5.10426, 6.42615, 5.32986,
  4.12702, 4.86039, 2.67322, 2.00217, 7.71715, 2.64908, 2.59941, 3.02833,
  4.41809, 5.57768, 4.87148, 10.0912, 5.46073, 5.11031, 6.97261, 5.09458,
  4.0714, 5.34137, 5.01775, 4.56572, 1.65355, 3.68485, 1.47939, 5.60916,
  6.48019, 3.02573, 5.41949, 4.0761, 6.43555, 4.42052, 6.22229, 5.30451,
  5.67207, 2.21204, 3.28932, 2.26355, 5.67427, 4.02898, 7.61985, 3.84354,
  2.84526, 4.28251, 5.47481, 6.49783, 4.32496, 4.64823, 5.81346, 2.21417,
  4.06934, 3.08845, 6.67102, 4.64625, 2.29964, 3.08188, 1.92148, 4.29558,
  3.56305, 4.28311, 4.93012, 6.34047, 3.58134, 4.37954, 7.12496, 4.43575,
  7.10165, 3.78768, 6.18308, 4.84597, 3.28799, 4.21272, 4.89839, 5.85686,
  4.4455, 2.5532, 2.60335, 6.44858, 2.89654, 4.37015, 4.30679, 1.59233,
  8.69207, 3.22789, 5.1847, 3.85073, 5.47912, 1.6652, 2.99366, 5.0317, 2.55975,
  7.22412, 4.67901, 5.92513, 4.44202, 6.92022, 7.07432, 3.18522, 3.33278,
  5.73917, 3.65189, 2.40191, 7.39487, 6.88197, 2.40007, 6.56892, 6.06482,
  6.05591, 6.56139, 3.27342, 10.4754, 6.20287, 5.69382, 3.04605, 3.26881,
  2.37601, 8.74265, 2.17714, 6.26002, 0.373803, 2.5538, 6.39814, 7.85801,
  4.82175, 5.74224, 5.06728, 1.67083, 7.30234, 7.02466, 8.20138, 8.44143,
  5.80598, 6.38692, 4.09577, 6.04593, 3.44423, 2.2584, 8.25171, 4.88754,
  2.97038, 5.26295, 5.20687, 5.31945, 6.07749, 5.20848, 6.47827, 3.37248,
  8.52357, 6.71349, 6.72629, 3.89551, 3.98912, 7.48266, 1.18251, 1.75309,
  7.49191, 5.67101, 3.37748, 4.42391, 7.6323, 8.911, 5.2653, 1.55344, 6.37865,
  5.56418, 10.3165, 5.54349, 7.83758, 8.09333, 7.99592, 4.60571, 6.02802,
  5.96747, 1.11548, 6.28755, 6.62454, 4.19833, 2.08946, 6.48763, 4.81114,
  1.64886, 6.83922, 6.29157, 8.25223, 6.88387, 4.04963, 9.08569, 8.77975,
  1.60539, 3.18664, 7.00857, 4.69164, 4.27561, 4.22168, 7.26343, 7.178,
  6.59893, 4.62415, 6.10529, 5.58765, 4.96238, 5.27216, 3.32578, 3.35605,
  3.42295, 7.09016, 4.24513, 8.63905, 3.5828, 5.1766, 8.97839, 2.53727,
  3.93064, 7.82243, 3.24417, 7.47556, 5.32569, 4.72773, 2.98966, 4.72346,
  7.5928, 5.17709, 5.69342, 4.31004, 6.55091, 4.06949, 4.40824, 6.22479,
  2.29491, 3.15106, 4.86649, 3.50497, 3.7256, 1.2507, 5.11474, 6.08769,
  5.25908, 4.01078, 6.10197, 5.68509, 7.59818, 5.72656, 2.49445, 10.8269,
  3.41813, 6.96457, 3.61174, 2.95629, 5.96423, 1.82084, 5.35853, 3.02372,
  7.25029, 5.94226, 5.44216, 2.95594, 5.1717, 3.11019, 3.80805, 1.62879,
  6.72072, 7.88153, 7.58216, 5.1778, 8.25226, 5.7895, 7.92041, 5.13632,
  8.52186, 5.88962, 6.69429, 6.87092, 5.96651, 4.54313, 5.22173, 4.21178,
  3.50397, 6.41485, 5.56533, 3.00708, 4.62533, 5.44598, 6.93011, 3.01892,
  3.08067, 5.52291, 1.24215, 2.87941, 4.83244, 4.05568, 6.34944, 3.91127,
  1.76685, 7.51639, 4.50284, 5.39974, 7.81186, 5.5164, 4.35421, 4.69563,
  3.02186, 7.01515, -0.564118, 7.31156, 8.65969, 6.99198, 4.75055, 4.51883,
  5.23132, 4.93807, 5.8656, 3.41734, 6.52122, 5.013, 3.88146, 5.15436, 3.2654,
  6.65347, 4.62565, 2.57578, 2.21931, 5.86491, 3.90602, 7.3484, 3.41802,
  2.21295, 5.52434, 5.83113, 6.3851, 6.39931, 8.67596, 1.4368, 3.03237,
  5.75209, 5.16774, 7.05123, 2.52969, 5.59116, 6.47145, 3.67084, 6.93228,
  6.74821, 3.77602, 5.25196, 3.47354, 7.47141, 9.98666, 7.17375, 6.44941,
  5.88441, 6.20519, 7.26469, 1.46973, 8.59211, 5.73516, 2.12625, 2.54978,
  0.898831, 6.93199, 7.13967, 7.0415, 4.976, 4.20054, 4.32356, 2.20645, 1.8691,
  4.71508, 4.60341, 4.85985, 3.52563, 3.40236, 2.97872, 1.14042, 4.22341,
  7.00144, 3.67856, 4.19043, 3.84761, 3.29171, 3.47295, 7.46927, 8.40541,
  4.62598, 3.93578, 4.68389, 2.81527, 3.04042, 6.2199, 1.36306, 3.23539,
  4.58309, 5.23826, 5.98439, 9.40782, 3.5805, 7.91056, 5.19099, 6.39203,
  4.68391, 4.17008, 10.7381, 3.05627, 7.17043, 7.65775, 8.59059, 5.62235,
  3.46068, 1.8853, 6.23979, 2.4386, 2.24764, 2.44315, 6.36524, 5.97285,
  3.21303, 8.80724, 7.91661, 6.58618, 3.69167, 7.61526, 4.3446, 6.46919,
  6.11509, 7.99398, 4.965, 6.11633, 6.90064, 6.08802, 3.89254, 5.19119,
  7.30033, 3.15028, 7.85867, 2.87434, 2.24643, 5.55053, 6.04563, 6.81196,
  6.31374, 3.48637, 4.15553, 7.20448, 2.22462, 4.83862, 3.8597, 3.70306,
  4.72407, 6.54724, 5.46386, 5.21889, 3.12437, 2.11722, 4.16865, 7.7348,
  7.03138, 3.75343, 4.82201, 6.99521, 6.71384, 4.4705, 2.98368, 7.98544,
  5.51526, 4.60344, 3.64244, 4.4154, 2.99188, 8.22232, 3.24741, 4.24564,
  9.98356, 9.51215, 4.92378, 2.40925, 5.59145, 6.70644, 6.56179, 4.38528,
  6.19989, 5.9426, 2.8937, 4.31135, 5.23354, 1.15435, 6.53164, 3.28222,
  5.06499, 3.20512, 6.16665, 5.992, 3.70196, 7.11582, 5.8127, 4.4564, 6.47096,
  4.81035, 5.70806, 4.67703, 8.94812, 4.94331, 5.92446, 4.07904, 4.2731,
  7.55517, 2.94175, 7.43693, 7.68204, 0.839203, 3.2693, 4.71136, 6.79603,
  7.56479, 7.02583, 5.74088, 6.75594, 1.84665, 6.39782, 1.64212, 3.57746,
  6.53432, 2.70582, 6.03356, 6.18299, 4.66414, 6.16511, 3.97417, 8.1313,
  5.09757, 5.46723, 7.10475, 6.38856, 3.53092, 3.35923, 3.91003, 1.3881,
  4.9121, 6.10832, 6.12047, 3.10967, 6.65904, 5.0614, 4.84625, 6.09923,
  8.40824, 5.43156, 8.19077, 3.31798, 5.66185, 7.19637, 3.42912, 4.97083,
  5.97233, 5.32979, 7.86117, 6.65134, 6.10536, 4.0415, 9.23613, 5.27747,
  4.58735, 2.09865, 6.55497, 4.26432, 4.98472, 0.273541, 7.11077, 6.37127,
  4.65752, 6.00199, 4.02673, 3.1672, 7.54399, 4.58074, 2.15945, 3.8996,
  6.28636, 9.35027, 4.03344, 6.60907, 7.44739, 7.58176, 8.26496, 8.29587,
  3.39543, 3.48717, 4.66802, 3.63736, 1.69231, 9.10559, 4.03186, 5.37425,
  2.3736, 2.4598, 4.12833, 1.52459, 6.52663, 1.77177, 6.16366, 4.4246, 6.03786,
  5.40506, 6.03565, 6.50012, 8.49733, 5.22438, 3.01735, 2.24089, 4.42788,
  4.67712, 7.96817, 7.13158, 7.7273, 1.11256, 5.51739, 6.9424, 3.08279,
  8.19273, 7.54177, 4.33547, 4.17157, 5.79047, 4.04143, 3.79865, 2.26662,
  5.11383, 2.16049, 5.18591, 4.39871, 7.02382, 5.12244, 3.5161, 5.26747,
  6.15076, 7.60347, 9.18443, 7.09836, 5.23058, 5.60084, 3.44582, -1.29286,
  7.54364, 2.52936, 5.23398, 1.35801, 4.1336, 3.23219, 4.06714, 6.64046,
  7.73564, 1.77598, 4.92537, 4.77864, 5.82273, 3.46182, 8.94114, 4.48913,
  3.97327, 5.57701, 6.19078, 3.92078, 5.62812, 1.49357, 3.62496, 6.73085,
  5.81591, 6.25325, 7.42509, 5.56732, 4.47828, 6.48567, 4.47245, 5.03164,
  5.94958, 3.0806, 3.10651, 6.33981, 8.29829, 6.07584, 9.8013, 2.98071, 4.551,
  9.64217, 4.09179, 6.55166, 7.00992, 6.29026, 2.97364, 2.35389, 4.39457,
  4.54069, 8.52871, 4.74566, 8.42577, 2.05477, 2.78852, 6.6592, 5.68426,
  4.87939, 6.47707, 6.15381, 5.33724, 5.26353, 4.55256, 7.21669, 5.66173,
  6.28684, 6.18921, 2.31273, 6.43487, 4.50418, 5.89914, 8.0587, 3.93594,
  4.53656, 7.11558, 7.96155, 5.62966, 5.36964, 6.16781, 6.91611, 3.23392,
  5.309, 4.12914, 0.315609, 5.20956, 6.41766, 5.39866, 7.59313, 5.77976,
  2.85409, 7.50835, 5.59906, 5.99413, 6.6454, 4.95828, 5.79571, 5.61256,
  2.5007, 6.15141, 2.36813, 3.96289, 0.926669, 7.46196, 4.21199, 2.74669,
  4.50106, 4.75586, 5.48896, 5.55648, 5.17473, 4.75609, 2.11816, 7.78605,
  1.08081, 8.96227, 4.41885, 5.66508, 3.03549, 6.6442, 7.18361, 8.75799,
  2.11581, 4.59071, 4.29426, 1.54811, 5.62163, 2.61924, 5.14942, 0.638955,
  4.69045, 6.1768, 4.63018, 5.60652, 6.38495, 4.54196, 4.16909, 4.30334,
  4.98558, 3.96731, 2.46573, 1.80273, 6.69164, 6.66166, 2.47577, 3.83383,
  5.68164, 7.00604, 4.66429, 4.1242, 4.68939, 4.56949, 3.0172, 1.92501,
  1.36831, 0.790734, 5.03261, 2.78073, 2.54282, 7.41134, 8.62935, 4.13798,
  7.37411, 4.18672, 6.86611, 6.01404, 4.57135, 4.46998, 7.50938, 2.98978,
  7.50034, 5.70205, 6.98135, 7.87458, 5.2223, 3.56936, 4.90222, 4.73987,
  3.82051, 6.13088, 0.399438, 5.84131, 6.7993, 8.35132, 3.97973, 5.12334,
  3.61886, 7.01756, 6.5661, 6.40729, 4.47844, 4.60654, 6.07652, 3.00842,
  6.66607, 3.92333, 6.56272, 1.28767, 6.47036, 6.56844, 6.41715, 4.70869,
  5.64565, 3.70001, 3.25758, 6.02959, 10.6061, 6.86784, 5.11798, 8.57357,
  7.90212, 6.96497, 6.60451, 8.07119, 3.87116, 2.01992, 6.01503, 5.74799,
  3.60944, 4.03452, 6.13821, 7.09827, 4.57692, 3.1394, 3.41322, 7.70247,
  2.45073, 1.36432, 5.79323, 7.67079, 6.07999, 0.204987, 3.29656, 4.2812,
  2.65519, 7.28103, 2.99303, 4.58003, 4.19275, 5.83039, 5.93929, 6.86703,
  5.70363, 4.47189, 5.98456, 3.31801, 3.26091, 8.01137, 5.94246, 4.9615,
  7.37763, 7.49832, 4.4905, 5.44119, 2.47221, 8.82642, 5.45694, 2.4531,
  3.30832, 7.58774, 4.93772, 8.47857, 6.00723, 2.86418, 2.87329, 6.32187,
  3.74351, 4.63357, 3.5528, 8.48303, 4.9344, 6.8899, 3.98647, 2.89285, 5.51894,
  5.65245, 6.86438, 0.170575, 1.38375, 7.43708, 6.4836, 5.59532, 5.50093,
  7.27429, 2.71135, 6.16939, 2.1822, 3.30944, 6.65176, 0.837054, 8.20267,
  7.66983, 2.81618, 8.14456, 4.4675, 4.19315, 4.17012, 2.22108, 3.5729,
  4.30172, 4.29336, 4.85058, 9.18505, 2.21603, 5.5897, 3.5005, 5.67669,
  6.29004, 4.48466, 3.71869, 5.99947, 1.27504, 3.69224, 1.65572, 5.57186,
  5.14671, 3.45296, 3.025, 3.57575, 3.1965, 3.85282, 4.87653, 6.91711, 8.23525,
  8.08968, 8.64059, 7.74108, 4.67445, 7.15274, 3.27766, 4.14786, 3.98624,
  1.79536, 5.72775, 5.54604, 4.88014, 1.89583, 6.91252, 7.69065, 6.33045,
  4.4759, 5.48978, 2.05273, 4.11984, 5.42907, 3.32828, 5.27979, 5.49714,
  3.38431, 7.28323, 3.4934, 4.52684, 2.59062, 4.94224, 5.80142, 8.46657,
  2.61041, 9.76213, 6.11447, 3.54776, 1.46068, 5.19048, 4.47386, 5.90229,
  6.8046, 6.30724, 3.64489, 4.76524, 0.322816, 4.76627, 3.75756, 7.72346,
  1.73921, 8.30962, 6.49019, 4.89437, 4.32257, 4.39097, 6.86554, 3.02017,
  5.97024, 5.44573, 3.42289, 4.91171, 4.76985, 2.32524, 3.55723, 4.00757,
  4.5038, 5.01365, 6.22003, 3.81504, 4.5753, 4.58275, 1.81615, 6.06136, 3.8205,
  5.50978, 5.41589, 3.72218, 7.2638, 7.85555, 4.72808, 6.57073, 7.1158,
  6.52719, 4.36072, 6.3056, 4.77796, 6.06424, 9.90121, 5.37465, 2.67216, 4.116,
  7.84878, 4.42671, 6.99583, 5.31557, 4.87557, 5.97387, 2.89548, 4.38294,
  3.2409, 5.26, 5.29224, 5.86253, 4.99988, 3.61211, 6.22261, 4.70265, 5.78043,
  5.1802, 3.74106, 5.93745, 6.30984, 5.27074, 6.39774, 1.73875, 2.61963,
  3.4381, 9.57708, 4.17555, 1.51078, 2.87152, 6.52345, 3.73572, 5.83583,
  1.66958, 4.44002, 3.15225, 3.06436, 6.33996, 1.51078, 3.3846, 4.48717,
  1.52576, 6.10216, 7.26415, 6.27635, 5.74042, 3.58519, 3.901, 2.8735, 7.94202,
  5.03026, 6.2437, 6.39904, 1.57619, 4.31701, 5.73623, 4.62851, 3.35453,
  4.0963, 8.68226, 4.41983, 3.24066, 2.99478, 7.279, 2.69883, 7.97653, 5.76888,
  2.35776, 2.2623, 3.87881, 4.50152, 4.36484, 4.396, 5.23053, 4.22807, 3.18149,
  3.31654, 5.7636, 8.53841, 1.72607, 5.71052, 3.83233, 3.87057, 5.18933,
  4.65871, 2.54152, 5.20936, 5.16189, 7.20077, 2.57793, 1.51577, 5.39563,
  6.83245, 4.68046, 7.52223, 0.353965, 4.06506, 5.20138, -0.812051, 0.412252,
  5.84964, 1.18787, 2.19579, 3.43453, 6.82373, 7.42782, 5.11288, 6.00439,
  5.36698, 5.21617, 5.07787, 5.01113, 3.50346, 3.58511, 4.33995, 1.76542,
  5.75214, 4.50396, 3.7433, 5.09151, 4.22014, 0.60793, 3.46371, 5.25403,
  4.25141, 4.28056, 7.45188, 5.02883, 5.38108, 2.95164, 2.49677, 3.86118,
  4.61038, 4.35064, 6.40263, 4.97078, 4.85222, 2.53487, 8.70189, 6.52742,
  7.66825, 7.81079, 4.3199, 2.46415, 3.35796, 5.67139, 4.36208, 4.65678,
  2.14009, 8.58682, 5.85952, 3.95321, 3.72667, 3.74205, 7.82684, 5.48638,
  6.98603, 7.33652, 3.02199, 5.40222, 7.69852, 6.39606, 1.90345, 5.20687,
  5.31637, 3.14277, 5.68954, 5.02055, 5.05906, 4.51873, 2.29336, 3.57199,
  4.05578, 3.17099, 3.77705, 8.01837, 7.52059, 5.89362, 2.86934, 4.53312,
  5.84596, 6.0646, 6.51568, 4.70641, 2.95327, 3.25502, 4.40956, 2.72944,
  5.54907, 4.53096, 7.62674, 2.11878, 7.28917, 4.40167, 5.25314, 4.60474,
  5.51606, 3.97941, 2.5122, 5.96804, 4.00449, 3.32532, 5.36028, 3.53998,
  8.83724, 6.38803, 6.89488, 3.03616, 4.21235, 6.03171, 4.1896, 3.42822,
  4.71867, 8.81413, 4.39973, 1.40806, 1.2792, 8.4382, 6.61264, 5.13921,
  6.93953, 6.90851, 4.91246, 6.50745, 8.21345, 7.48608, 6.16879, 4.70121,
  2.26412, 6.29921, 5.4162, 3.94515, 5.19652, 4.95339, 5.63374, 1.93412,
  4.27624, 1.23956, 4.27838, 4.39027, 5.81464, 5.6025, 6.94581, -0.297808,
  4.77109, 2.96158, 3.21116, 2.43218, 3.57231, 6.34549, 6.05439, 6.35988,
  7.81617, 3.33348, 4.24218, 5.10058, 4.57944, 5.1629, 3.74417, 1.45102,
  7.22291, 7.21409, 6.23013, 4.7624, 5.42497, 8.00879, 6.92539, 3.45628,
  5.49782, 4.70234, 1.65076, 4.37051, 8.4663, 4.90564, 4.91367, 8.2656,
  3.17442, 5.51464, 5.43806, 7.38924, 4.70162, 5.34183, 3.47291, 7.31125,
  5.55167, 4.29375, 4.42373, 5.69264, 1.20882, 3.52, -0.385871, 3.38403,
  4.14035, 4.31881, 4.56907, 7.61019, 5.8594, 6.69114, 6.43164, 8.12471,
  5.02313, 9.7024, 8.12361, 7.12929, 6.92297, 1.88344, 3.89633, 4.51672,
  8.82912, 5.09498, 5.51762, 3.97953, 4.06503, 8.62407, 8.19223, 5.77026,
  7.10435, 4.09536, 2.43203, 2.48737, 3.56949, 3.5546, 7.62175, 5.47453,
  6.49149, 6.68599, 5.69955, 1.47345, 10.2525, 2.22088, 8.35474, 4.39226,
  5.60076, 4.03412, 7.67268, 1.77248, 6.74423, 3.60064, 8.40647, 4.79629,
  7.32305, 7.38378, 6.32175, 2.75952, 3.34629, 1.75291, 7.82793, 0.607309,
  0.602474, 3.91681, 3.5337, 6.04435, 1.97213, 2.13309, 9.45619, 3.88427,
  7.2917, 1.69369, 6.18886, 5.57132, 3.85023, 5.66408, 2.09487, 7.63157,
  5.09554, 3.20632, 2.13477, 6.01114, 3.79811, 2.20151, 4.38595, 6.51777,
  6.77188, 2.76416, 3.53019, 6.11666, 3.22656, 5.80942, 1.0442, 3.96942,
  1.54198, 8.33695, 2.70607, 7.43182, 0.47177, 2.27632, 3.15197, 3.54528,
  6.49965, 8.99673, 5.51359, 2.15063, 2.46285, 7.32307, 4.19325, 1.85857,
  6.656, 3.45518, 4.98439, 7.53757, 5.39552, 5.07594, 8.52348, 4.65538,
  2.64222, 1.99762, 8.36735, 5.81388, 4.60596, 7.82874, 7.67067, 7.1618,
  2.58316, 6.1686, 5.45117, 1.50103, 4.57852, 4.51698, 4.5662, 5.1747, 7.26003,
  5.69699, 7.03205, 4.98864, 5.28197, 6.76352, 5.47169, 3.67879, 6.40495,
  8.06616, 2.33779, 4.90196, 2.36995, 5.6813, 4.23334, 3.55887, 4.3871,
  2.22116, 6.78832, 3.55094, 0.315962, 9.80347, 3.54259, 5.8237, 1.06056,
  3.4894, 1.49676, 4.85891, 5.9521, 7.08178, 4.40152, 2.79072, 4.39895,
  4.31702, 4.80764, 1.59862, 5.29798, 2.52386, 5.29801, 4.57261, 2.09607,
  4.22989, 5.34526, 3.43984, 5.21092, 7.51023, 5.97588, 5.66698, 4.29855,
  -0.0813865, 3.84117, 6.70365, 5.66888, 8.09131, 7.42114, 2.06904, 3.87357,
  3.8993, 6.83524, 4.44582, 3.96876, 1.37806, 5.52088, 4.30473, 4.73439,
  5.51251, 5.58194, 4.35105, 3.72779, 6.08613, 4.18113, 5.56087, 3.38493,
  7.02748, 3.11666, 9.13636, 5.6691, 7.12228, 4.20668, 7.12233, 6.74326,
  5.74104, 5.54387, 1.65439, 4.91313, 9.44348, 5.10059, 5.622, 5.0327, 4.30903,
  4.02955, 7.90794, 3.64801, 4.52716, 6.81641, 5.58483, 2.9425, 5.52989,
  4.8536, 3.84307, 5.70704, 6.34276, 6.74103, 4.53111, 3.1831, 6.55918, 2.9041,
  7.86259, 8.58881, 7.41344, 3.99193, 4.72482, 5.08214, 2.36926, 10.8337,
  7.9308, 7.99064, 3.00763, 3.51406, 5.47074, 7.20762, 6.46029, 6.61992,
  3.78411, 6.9894, 3.06644, 6.6482, 4.05183, 2.8252, 5.91243, 3.63773, 5.09688,
  8.13409, 3.53115, 6.70167, 4.92044, 7.44228, 7.87985, 2.92226, 6.95401,
  4.50858, 4.41394, 9.80655, 4.22876, 3.67372, 1.74527, 5.6185, 2.13044,
  5.30092, 5.37911, 1.66642, 6.55369, 6.13397, 4.26022, 6.65209, 5.89098,
  7.52136, 6.03759, 6.56729, 6.19947, 7.72448, 5.19604, 7.90387, 5.53197,
  4.42975, 4.5129, 4.32183, 3.89419, 8.76544, 4.90719, 7.29892, 5.87655,
  6.35661, 10.0418, 4.73705, 2.90834, 0.696163, 6.58608, 2.53497, 6.52248,
  7.08545, 3.67623, 6.17511, 6.38953, 5.32917, 7.54982, 4.40258, 5.68719,
  4.7333, 4.43266, 3.7132, 6.72821, 4.5895, 3.08871, 3.9157, 6.83386, 4.26392,
  6.21673, 5.15833, 4.95789, 2.45285, 5.00689, 4.79245, 6.89039, 2.49173,
  3.85872, 3.51403, 3.83107, 4.29869, 3.71623, 0.112603, 4.77408, 7.09724,
  2.40106, 4.49589, 3.09472, 8.35935, 5.32016, 7.99163, 7.91505, 5.18213,
  6.03262, 6.03071, 3.086, 2.9031, 6.84293, 3.38942, 7.17377, 5.75869, 8.56391,
  4.25459, 7.57382, 6.31136, 4.34756, 7.26474, 2.76427, 3.47055, 5.31144,
  8.49383, 1.38025, 6.04129, 2.45478, 3.51957, 6.29301, 6.7406, 5.62619,
  7.34569, 3.6385, 1.86228, 3.09436, 4.62259, 4.86288, 7.14208, 3.19675,
  3.61908, 3.93615, 7.09662, 5.46686, 6.013, 6.22101, 5.59805, 5.90538,
  5.50732, 5.59347, 2.71047, 7.13235, 1.86929, 7.54439, 6.46603, 1.75706,
  3.60492, 2.64173, 3.13202, 3.77334, 6.29999, 7.12721, 7.40879, 2.77258,
  7.46988, 7.95459, 2.5666, 2.61932, 9.10625, 3.56083, 5.6223, 4.36819,
  7.18477, 4.77222, 7.68519, 5.43266, 3.93732, 8.71635, 3.37908, 4.77057,
  2.84124, 8.2141, 4.63219, 4.9514, 4.29131, 5.82078, 7.41793, 5.65357,
  5.67636, 5.52343, 3.47942, 6.88788, 6.0677, 0.0838241, 5.84513, 4.62354,
  5.43013, 7.30816, 4.69701, 8.38796, 5.01029, 4.69271, 5.97871, 7.65269,
  4.70588, 3.93811, 4.62395, 3.43031, 4.54738, 5.93216, 3.9061, 3.17133,
  2.24968, 3.5152, 6.08197, 5.85206, -0.0105933, 4.11806, 6.71578, 6.51314,
  5.02417, 5.25241, 5.16249, 5.37189, 8.90319, 7.04198, 6.82706, 6.33961,
  6.61063, 2.08632, 6.10754, 4.74096, 3.49494, 2.03499, 2.12466, 8.44559,
  3.58811, 5.22746, 6.5918, 5.89659, 2.60489, 2.97912, 7.64799, 6.11755,
  1.24328, 3.30195, 3.73002, 3.94708, 5.05846, 2.36084, 3.78757, 4.39113,
  6.7271, 6.94179, 5.80165, -0.0575951, 6.64599, 6.15659, 8.46349, 2.75071,
  4.15743, 2.89798, 6.15515, 6.06404, 6.22574, 3.85594, 3.54418, 5.08333,
  3.30811, 3.57916, 6.04342, 2.51494, 2.87153, 5.85102, 6.43839, 6.08071,
  1.4477, 5.89732, -0.675635, 7.79594, 5.62868, 4.89573, 2.64041, 7.37362,
  7.02758, 6.96715, 6.40103, 4.41635, 5.28168, 2.19978, 4.67358, 2.9438,
  7.70822, 5.72754, 5.54302, 6.03395, 4.28047, 5.64684, 6.78482, 2.73353,
  6.36243, 3.36347, 5.93221, 3.86496, 8.15821, 6.40405, 6.83075, 7.25049,
  2.05342, 4.42387, 4.10878, 8.29974, 7.08043, 6.23152, 2.92203, 3.1239,
  6.52877, 4.47066, 2.06468, 5.05615, 5.98179, 6.85493, 1.60719, 6.17595,
  5.18262, 1.70562, 2.79126, 5.69026, 6.50114, 6.80319, 6.2804, 3.08289,
  6.28107, 1.22809, 6.07906, 7.16152, 5.1297, 5.21326, 6.10649, 3.46478,
  4.73309, 2.70736, 3.76661, 3.29394, 5.35233, 6.49574, 1.97603, 5.55788,
  8.28987, 8.25778, 5.21729, 5.85654, 2.74547, 8.61718, 4.31616, 6.68856,
  8.18993, 5.59487, 6.80049, 6.3014, 5.33254, 6.57642, 6.08077, 4.68294,
  2.68996, 4.85325, 6.2401, 5.65525, 8.50448, 3.77751, 5.02955, 6.71932,
  8.8797, 5.13418, 4.25547, 5.07821, 2.54965, 4.31822, -1.02445, 2.97998,
  5.75972, 3.81622, 2.18225, 5.41789, 4.40854, 3.40324, 3.22225, 0.650296,
  4.3224, 8.77808, 5.00737, 3.37556, 6.43124, 6.09255, 2.51721, 3.69698,
  7.11545, 6.70998, 3.72657, 6.42916, 6.3467, 4.96763, 3.25058, 5.40374,
  6.5246, 6.14149, 10.5918, 5.46988, 2.85084, 5.31408, 5.43926, 7.70235,
  5.56415, 6.45925, 3.62964, 2.93557, 4.8125, 7.15436, 8.28498, 4.94587,
  3.05285, 2.23681, 6.13937, 4.81796, 4.83048, 5.26442, 6.34615, 7.48475,
  4.46257, 1.53744, 3.6143, 5.79636, 6.39815, 7.4383, 4.87042, 3.15379,
  5.40727, 9.06324, 6.17518, 5.59322, 6.43768, 5.23839, 7.33506, 2.70663,
  5.56379, 4.30075, -0.0482929, 2.1159, 5.34701, 5.19154, 4.50906, 1.41426,
  3.43158, 6.18056, 4.87832, 5.87873, 7.19107, 3.9665, 2.36118, 7.58706,
  6.75431, 4.9701, 5.71024, 6.32618, 4.97659, 6.06961, 2.54127, 3.92704,
  5.64422, 3.94323, 4.21718, 3.58854, 6.95186, 9.18553, 8.09939, 2.624,
  1.85891, 6.47182, 3.8262, 7.3016, 4.2535, 4.31863, 1.65625, 4.73029, 6.72489,
  8.72117, 5.66421, 8.22265, 10.2922, 7.51799, 4.63663, 4.53138, 7.35949,
  5.16236, 8.08931, 1.00122, 6.0368, 4.66993, 6.54106, 5.92568, 6.56098,
  2.41821, 1.54859, 4.95484, 5.66331, 4.50178, 4.14268, 3.57145, 6.83918,
  2.69052, 2.58256, 5.938, 7.80827, 6.68182, 8.63762, 7.0752, 4.84674, 1.9196,
  6.0766, 2.22909, 5.39106, 3.32967, 7.90651, 6.02754, 3.44225, 3.02364,
  3.98488, 5.61039, 4.35802, 6.0015, 0.497204, 6.35551, 4.97246, 1.65017,
  7.02003, 7.30342, 5.37262, 7.91431, 3.03579, 3.32196, 3.82028, 6.45995,
  5.25458, 4.86552, 3.82041, 5.78156, 7.53522, 2.90311, 2.1963, 4.40437,
  6.12009, 5.10248, 3.31903, 3.28913, 5.26955, 1.37861, 4.52027, 2.26834,
  7.63194, 7.69966, 4.68183, 6.05285, 3.21245, 6.67154, 7.92001, 7.20963,
  4.99926, 4.56526, 6.91862, 6.32819, 4.35894, 8.73677, 2.69915, 5.24511,
  4.47224, 2.07831, 4.20231, 3.41586, 6.15955, 4.33486, 4.64563, 6.59115,
  1.98723, 6.21301, 3.32541, 5.81679, 3.90294, 3.45335, 2.42653, 4.97246,
  3.84066, 3.80883, 6.63055, 6.86833, 0.0490323, 6.00081, 8.75295, 4.97785,
  3.14111, 3.65883, 7.10538, 6.02874, 2.76023, 1.51801, 5.11145, 3.72873,
  5.08549, 6.19709, 4.47556, 5.72935, 4.43046, 0.290834, 4.13414, 3.88926,
  4.3674, 4.25883, 7.13457, 3.81295, 4.14779, 6.70474, 6.59082, 3.51187,
  7.22844, 3.25366, 6.27399, 4.72945, 3.64857, 4.3331, 6.30724, 2.41936,
  2.8753, 8.11461, 6.40853, 3.49033, 0.805628, 2.53364, 8.79788, 2.22981,
  2.23287, 2.72509, 7.45157, 4.74358, 5.84345, 8.00608, 4.61212, 3.01459,
  5.71266, 2.73404, 2.91413, 4.1109, 5.3538, 5.13183, 2.91202, 4.54294, 7.8353,
  6.04889, 2.72434, 9.40142, 4.78633, 4.2058, 9.55936, 4.54779, 5.55554,
  3.69634, 2.53036, 5.90061, 6.9561, 6.24328, 8.08031, 4.94404, 7.39661,
  9.17954, 8.44155, 8.85287, 4.53055, 4.3513, 5.43307, 2.01366, 3.80971,
  4.1445, 5.48338, 4.36156, 2.92909, 4.63302, 8.59558, 6.03507, 4.64072,
  5.9016, 11.209, 2.60904, 3.52618, 5.03192, 8.40543, 2.78437, 8.93664,
  6.02889, 5.86865, 7.59712, 8.78535, 7.73039, 4.93445, 2.86615, 5.24174,
  4.55124, 5.16207, 1.752, 5.44632, 4.57754, 5.32246, 5.02724, 8.00581,
  7.01935, 3.70965, 7.85495, 7.96349, 3.71397, 7.28164, 5.20146, 6.03891,
  3.54662, 2.05964, 4.72066, 2.648, 7.1865, 5.84235, 3.68725, 3.03729, 5.20937,
  3.95051, 4.44109, 3.29545, 1.17485, 6.52317, 5.38165, 7.49836, 4.61435,
  7.29968, 2.45238, 7.1282, 3.12355, 3.49293, 3.31293, 7.95339, 0.136371,
  5.10213, 4.93555, 6.33284, 2.99253, 3.80193, 7.11612, 2.1026, 6.31929,
  6.30617, 3.52538, 5.4944, 6.32807, 5.08541, 5.78255, 4.93106, 6.02952,
  4.30603, 2.50862, 5.74405, 3.59849, 1.61505, 1.9301, 8.15047, 5.16764,
  4.67466, 7.12293, 5.31768, 2.42771, 6.56537, 4.12613, 6.16329, 6.44448,
  4.74008, 7.63577, 3.28323, 8.05839, 3.92447, 3.32382, 8.04982, 5.83035,
  2.51272, 6.41089, 4.34173, 3.64821, 4.32514, 1.07661, 4.29783, 5.09711,
  3.24998, 4.33301, 4.88601, 6.64175, 10.1302, 4.56451, 10.2429, 7.00453,
  6.11112, 3.38435, 10.8901, 3.93671, 7.60698, 5.50772, 1.43101, 3.22127,
  4.38171, 5.09381, 1.3323, 5.41123, 8.108, 5.29515, 5.67676, 3.6537, 4.82366,
  3.97808, 2.36742, 5.89716, 8.65808, 6.95016, 5.22875, 6.33183, 2.16663,
  6.25872, 3.77752, 3.74124, 5.40022, 5.99024, 3.79771, 7.18494, 4.41104,
  5.88052, 7.20871, 4.72416, 8.5323, 7.78575, 4.87902, 4.97189, 3.11589,
  5.16676, 9.3269, 7.47978, 7.57571, 4.75105, 5.34104, 6.66612, 7.74549,
  2.88201, 6.32688, 2.57062, 2.10497, 6.58352, 9.02414, 7.23543, 5.43953,
  3.94352, 6.9019, 7.25848, 6.47283, 4.21562, 2.98279, 4.66009, 5.11162,
  5.37385, 6.68703, 5.66349, 1.25364, 3.98346, 3.54667, 3.83744, 4.44799,
  6.02639, 4.876, 7.03825, 2.09583, 5.45449, 7.87966, 5.65819, 4.7415, 4.2264,
  7.85656, 1.33459, 2.89885, 8.08308, 5.1536, 1.92848, 3.11748, 2.57844,
  4.72269, 3.35455, 8.54549, 9.51249, 8.48722, 3.80191, 3.86049, 3.84766,
  7.33384, 4.87778, 3.64488, 5.13191, 2.91599, 4.42579, 5.73401, 2.81711,
  7.28272, 1.72052, 2.38256, 5.91686, 1.51243, 4.82428, 8.23157, 5.87597,
  4.77875, 1.91141, 4.58641, 4.24724, 5.1873, 2.67086, 4.70808, 4.53738,
  7.33476, 9.91527, 6.23812, 3.53413, 7.75898, 7.1428, 2.23571, 8.19004,
  3.0196, 4.6806, 5.29263, 3.87876, 5.51819, 6.5778, 0.841833, 6.00596,
  5.90093, 7.35295, 3.38845, 4.34587, 2.49449, 4.39637, 2.10036, 4.43809,
  5.17004, 4.36385, 5.73832, 6.68945, 1.19743, 6.05641, 2.43698, 3.62824,
  1.88401, 6.41749, 5.27716, 5.36866, 8.65513, 8.78391, 5.82474, 1.3152,
  4.42297, 7.74816, 7.43062, 6.1763, 5.95075, 2.88342, 7.09444, 4.75928,
  1.4045, 5.23899, 4.33172, 7.24986, 1.36206, 5.2533, 2.20187, 4.28686,
  5.15976, 7.10084, 3.94747, 8.35342, 5.14955, 8.86991, 8.09366, 8.63887,
  7.75269, 1.89737, 7.99683, 3.647, 7.92814, 5.55242, 7.53343, 3.61857,
  5.42578, 4.23853, 6.75374, 1.22771, 3.65524, 4.67863, 2.73407, 5.60271,
  4.5889, 4.94626, 3.71972, 1.98472, 3.39027, 3.60859, 4.19673, 3.69714,
  5.71022, 3.98649, 5.3179, 9.68978, 5.47549, 6.0248, 5.18728, 3.42875,
  5.39232, 6.26611, 5.39163, 7.72863, 4.91038, 7.06875, 8.6335, 6.46053,
  2.22597, 9.88136, 4.35017, 3.9709, 3.28432, 4.5368, 5.26912, 5.91956,
  7.49933, 5.18696, 4.45942, 7.38225, 6.11881, 3.02024, 2.91134, 7.57287,
  4.13167, 4.13609, 6.59553, 6.17133, 7.10308, 6.27703, 5.77434, 4.57499,
  4.39461, 5.24939, 3.01652, 2.6434, 2.08883, 5.57534, 4.11733, 3.73639,
  4.97523, 2.71895, 7.74084, 7.72327, 2.41717, 4.66064, 3.83951, 4.50686,
  4.04301, 11.0145, 7.48091, 2.53182, 4.9258, 6.09926, 3.80715, 6.68383,
  5.54487, 6.46694, 4.91589, 1.08428, 4.35259, 7.61093, 7.20992, 3.73499,
  4.00097, 4.14892, 6.48711, 5.53195, 2.80911, 9.0876, 7.52355, 3.27326,
  3.55973, 3.84773, 7.20995, 3.72421, 6.34558, 2.74962, 3.23117, 5.68612,
  7.54516, 5.85574, 6.76082, 2.01992, 7.09861, 4.16337, 5.31565, 1.16398,
  10.6078, 5.11911, 4.13446, 3.84323, 4.59044, 4.86854, 3.83877, 7.48445,
  5.21359, 4.57376, 2.59772, 4.69119, 4.0848, 1.91101, 6.40955, 8.81098,
  6.05101, 6.68874, 9.00843, 5.91329, 2.84289, 2.4551, 6.24878, 4.3082,
  7.65247, 3.50035, 4.52246, 1.19834, 3.53355, 7.26628, 3.29647, 5.25677,
  2.58958, 1.98177, 4.56076, 5.49644, 5.65899, 4.77945, 5.93405, 8.0065,
  6.66116, 5.39909, 8.06297, 3.79751, 5.87282, 5.48306, 7.51893, 6.4666,
  6.00521, 5.76892, 4.92146, 5.54499, 4.67063, 2.94906, 4.51079, 7.21147,
  5.10072, 4.64961, 3.89251, 5.8381, 8.14643, 3.24756, 4.039, 7.78009, 7.80315,
  3.47635, 3.89256, 3.59402, 3.89006, 3.54258, 5.59295, 7.91678, 3.23508,
  5.07794, 7.65145, 5.44826, 7.83258, 5.06866, 6.21248, 6.93621, 5.3102,
  7.36867, 5.8049, 2.58475, 3.67607, 6.99889, 4.06114, 2.31516, 5.02674,
  3.46404, 6.83416, 5.57367, 4.24441, 2.97181, 6.30519, 3.26835, 4.62302,
  5.06121, 6.37343, 6.29689, 2.04075, 7.12575, 3.38397, 4.09358, 4.21725,
  3.52268, 6.24594, 3.65682, 6.12111, 4.69063, 1.24379, 6.63878, 3.54959,
  5.20072, 4.29299, 9.96669, 5.82424, 6.60838, 3.72079, 5.74915, 4.31803,
  4.84466, 5.30664, 5.94183, 0.921594, 4.99442, 4.40909, 3.09413, 6.56587,
  3.16328, 5.00768, 8.26265, 2.61079, 6.08695, 2.89205, 3.29266, 10.5321,
  5.8471, 4.24495, 5.02224, 8.94809, 3.95535, 6.33227, 6.21458, 0.093124,
  7.54325, 3.49482, 7.01269, 9.18211, 7.3379, 6.13333, 6.06164, 2.1621,
  6.91741, 5.35038, 3.43557, 6.44791, 5.7279, 3.09841, 4.95306, 2.81172,
  3.33748, 5.08459, 4.83968, 3.47002, 4.80719, 4.59163, 4.67636, 7.19356,
  8.12938, 5.47156, 4.35427, 7.22154, 2.93059, 5.09846, 5.68203, 7.15972,
  4.76615, 5.06998, 10.0939, 3.76629, 8.00249, 4.9218, 3.33045, 3.32976,
  1.87892, 6.09153, 6.89007, 6.95256, 4.25387, 4.65993, 6.13676, 2.15952,
  1.48566, 3.4015, 4.38669, 6.68338, 4.85043, 6.36901, 3.31693, 6.31974,
  5.74972, 5.63467, 7.1214, 7.39508, 3.3323, 6.22029, 9.66129, 0.802543,
  8.11605, 5.08465, 5.26762, 2.52856, 3.06856, 3.6285, 6.6531, 3.88695,
  6.35675, 3.50618, 3.30173, 6.02117, 3.37675, 3.09707, 5.9637, 4.45083,
  1.28598, 4.39541, 8.2357, 3.68832, 2.83799, 2.98512, 9.76876, 3.07625,
  7.00224, 4.24489, 4.4521, 2.61614, 2.46173, 5.42847, 2.90403, 2.48382,
  9.12332, 8.82612, 3.36588, 4.00126, 5.31517, 6.50875, 3.23623, 8.83119,
  2.37067, 9.04649, 5.0525, 8.26038, 0.810264, 6.01775, 9.75017, 4.34106,
  2.68089, 6.07367, 2.57177, 3.84325, 4.34846, 4.79402, 3.88552, 6.17927,
  4.4537, 2.90859, 7.14076, 4.01477, 4.24394, 1.89585, 2.49846, 4.77348,
  3.15655, 8.29483, 1.86465, 3.7633, 3.39983, 6.3074, 6.17767, 4.84623,
  3.04334, 4.12072, 1.3179, 6.6498, 8.08484, 7.72631, 7.44791, 5.85074,
  6.03321, 5.23505, 1.68979, 2.47749, 6.5358, 3.35196, 3.37198, 3.44557,
  3.9356, 7.39705, 7.45047, 4.54984, 4.8186, 4.56242, 4.23459, 3.34319, 2.7763,
  5.45421, 5.46698, 1.38683, 4.26866, 8.50132, 2.25758, 6.87048, 5.34012,
  6.7948, 2.87162, 4.9013, 1.0917, 6.20715, 6.70291, 5.01481, 1.77818, 5.25352,
  7.9532, 1.92988, 4.10043, 4.91485, 2.80054, 0.803101, 6.78552, 5.25747,
  4.08841, 5.60184, 2.20107, 6.36988, 6.41094, 7.54146, 4.84053, 4.58932,
  9.63284, 4.934, 7.31683, 4.1758, 4.7033, 0.208342, 6.21255, 6.3187, 4.85842,
  3.95378, 4.74772, 7.08977, 3.07363, 1.98256, 4.55334, 4.43168, 4.43954,
  3.83998, 3.10454, 6.22781, 6.78672, 1.3965, 6.10619, 7.2355, 3.60138,
  2.67304, 8.54262, 9.16108, 6.05929, 7.10588, 5.96121, 5.23752, 3.37937,
  3.78387, 4.92267, 6.90796, 7.79157, 3.70072, 3.483, 8.41343, 7.67419,
  0.143017, 6.65402, 2.88605, 8.38626, 5.39663, 3.83923, 5.64344, 6.53923,
  4.71109, 0.726223, 5.51119, 4.77106, 5.47496, 4.3157, 4.86846, 3.82698,
  5.10013, 2.91134, 7.51416, 4.19451, 3.89654, 7.54077, 4.7465, 4.79934,
  9.43866, 2.68969, 0.873492, 6.40326, 6.20478, 4.63925, 0.991927, 5.07422,
  2.84043, 4.05264, 6.5605, 6.18834, 0.713708, 1.28705, 8.12697, 6.93074,
  7.60044, 2.32946, 4.90011, 6.38252, 5.13799, 6.73639, 5.79039, 6.84869,
  8.60667, 3.49322, 3.86338, 8.13969, 3.99297, 0.0551856, 1.86429, 7.30828,
  3.07619, 4.96831, 3.8943, 3.62916, 6.02305, 5.47276, 6.28272, 8.03953,
  4.57339, 2.25639, 3.0472, 8.3095, 4.45388, 0.278131, 1.21051, 3.34577,
  3.74086, 6.79741, 8.68124, 5.35258, 3.44697, 5.53985, 5.58139, 7.95622,
  1.80747, 5.57994, 7.00715, 5.43384, 2.43144, 6.11329, 3.77768, 1.9821,
  5.58369, 6.44241, 6.92087, 3.22379, 2.14208, 0.346304, 4.52124, 8.05189,
  1.62236, 4.47784, 9.71779, 4.66612, 3.08352, 2.87019, 5.08806, 5.92438,
  7.72308, 7.12551, 3.10607, 6.04117, 5.57149, 6.13086, 8.06253, 2.57766,
  5.12586, 3.66954, 3.42067, 1.95448, 3.97676, 3.83399, 4.09653, 4.65082,
  3.06906, 2.30664, 3.74586, 2.12296, 7.28387, 7.02472, 3.34253, 7.58258,
  6.3253, 5.899, 3.77518, 5.33085, 4.35294, 5.69446, 5.98399, 4.89946, 5.43725,
  1.80676, 5.52414, 4.94847, 6.34342, 4.88507, 6.06822, 5.87011, 7.06662,
  6.54887, 3.85928, 6.11774, 6.80138, 8.7866, 4.25391, 7.37153, 6.46544,
  3.79627, 5.71578, 4.64202, 6.3559, 0.909091, 6.96291, 7.78603, 2.75511,
  3.65122, 5.26026, 5.12816, 4.56713, 1.126, 4.76378, 3.82392, 4.25959,
  6.48172, 4.82147, 4.79085, 7.97827, 3.15513, 4.02655, 1.71995, 4.03758,
  4.16855, 5.02118, 5.95592, 8.19999, 1.44462, 5.92024, 2.42147, 5.78246,
  3.18862, 1.42714, 4.86981, 1.1608, 4.78009, 6.4025, 1.90412, 4.89636,
  8.98528, 5.07207, 8.81741, 3.85985, 4.3519, 6.12325, 3.56513, 3.6306,
  3.45894, 6.67607, 4.95139, 4.85962, 7.26781, 3.17641, 7.20823, 1.75321,
  3.61935, 9.3145, 6.03384, 5.25151, 4.05679, 4.66556, 4.99487, 2.38585,
  2.85897, 2.67012, 4.85581, 4.51226, 5.26135, 6.42526, 7.72281, 5.48185,
  5.83492, 5.77206, 4.17108, 6.8773, 6.90839, 5.88687, 2.96613, 6.03332,
  6.01513, 4.94006, 4.69552, 6.67148, 3.94443, 1.08132, 3.90367, 8.29404,
  4.29582, 5.42567, 3.74238, 4.77814, 5.38079, 7.37971, 2.88408, 6.4979, 5.058,
  5.21381, 4.27176, 7.51771, 3.24064, 1.22591, 2.62171, 4.89897, 5.11959,
  5.91258, 4.19904, 5.28448, 5.1833, 3.86794, 7.4493, 7.55916, 5.68542,
  2.86703, 4.99172, 3.62994, 4.82153, 1.16215, 4.4384, 7.11614, 3.73095,
  4.68934, 9.69927, 3.60176, 5.43339, 3.55031, 5.32626, 7.67218, 2.22062,
  4.20587, 7.41475, 8.13571, 5.21346, 4.9153, 8.4023, 5.46949, 2.9654, 4.08305,
  5.88751, 7.1568, 2.38191, 5.61571, 6.57463, 1.8196, 2.75941, 10.8273, 6.0202,
  8.93665, 6.76372, 6.275, 3.62124, 2.7878, 6.34831, 4.92911, 4.62893, 5.96346,
  4.78456, 2.09408, 4.92274, 2.2004, 3.60217, 1.64152, 5.12567, 6.09728,
  8.25475, 7.99176, 3.5801, 5.3019, 2.81017, -0.445333, 8.34636, 6.4397,
  3.19121, 2.55437, 8.16071, 5.42485, 6.16041, 5.58789, 5.83703, 5.75437,
  5.00183, 3.79326, 1.41123, 6.53929, 3.21609, 2.18022, 6.71686, 5.36366,
  7.85834, 4.33297, 1.82796, 5.73187, 5.76618, 7.92049, 4.39686, 5.15859,
  3.40743, 6.41617, 4.67901, 5.17676, 4.9113, 6.24156, 6.01664, 6.78081,
  4.82784, 5.99017, 6.011, 3.23124, 4.85615, 3.86273, 3.69635, 2.97176,
  5.78565, 1.95215, 6.03626, 5.91634, 6.78584, 5.78605, 6.32216, 6.3106,
  5.2717, 3.73715, 4.79054, 4.40883, 3.32682, 6.57779, 3.30348, 2.04514,
  4.33635, 7.09442, 6.04845, 5.65288, 3.43606, 7.43226, 2.54776, 5.66292,
  3.07044, 4.71291, 6.27182, 1.90299, 2.56218, 3.96351, 9.31057, 3.92164,
  5.18077, 7.79336, 3.74768, 3.22096, 1.39938, 5.34856, 6.34052, 4.9262,
  6.01436, 3.32645, 6.19496, 6.81863, 3.9292, 8.54096, 5.15211, 4.8135,
  3.63176, 8.16376, 4.49998, 4.7547, 6.27753, 5.10494, 3.24901, 6.79549,
  9.51402, 7.78845, 4.74599, 9.16753, 1.95244, 5.57476, 4.24193, 3.95621,
  5.78266, 5.4635, 6.01696, 5.5979, 4.42573, 9.32796, 4.98407, 5.69107,
  4.59914, 5.93221, 7.64702, 7.01471, 4.58952, 3.08004, 4.39938, 6.58914,
  4.85447, 8.21781, 5.20683, 5.48144, 7.54911, 2.82116, 2.51943, 5.37899,
  2.84291, 7.00253, 6.96979, 8.1717, 4.24438, 3.27158, 2.89727, 6.48249,
  6.08011, 3.96174, 6.19533, 7.29317, 9.36674, 6.40207, 4.67599, 3.81292,
  4.91306, 0.28328, 6.64659, 4.39004, 4.91389, 4.04409, 6.44134, 0.32946,
  6.19378, 4.34197, 4.03677, 6.51482, 5.40044, 3.38599, 4.15079, 0.0950117,
  6.55107, 3.57716, 4.04645, 6.00203, 5.66595, 4.41996, 7.95501, 3.00887,
  7.45164, 6.12777, 3.94173, 1.77427, 4.85457, 6.34761, 4.44131, 0.32568,
  4.59414, 6.97057, 3.55854, 4.32896, 5.04338, 2.28737, 5.34934, 5.35341,
  5.03356, 8.43384, 5.78962, 5.56085, 4.91199, 6.26473, 3.14232, 2.53913,
  5.21811, 3.82254, 1.27994, 5.11591, 5.61093, 4.04521, 6.80784, 4.13488,
  4.60965, 4.09255, 4.0766, 5.50076, 6.4632, 5.74567, 4.40475, 4.01085,
  7.64715, 7.03475, 6.44345, 4.342, 3.70197, 4.70164, 2.44812, 5.60016,
  3.51424, 2.4946, 4.97866, 4.34664, 5.22063, 2.46917, 8.03119, 5.07767,
  5.11377, 4.32988, 4.49211, 4.14266, 2.73573, 5.11574, 4.77702, 5.79601,
  4.94066, 4.30523, 9.37537, 5.2747, 6.9992, 4.14963, 5.63143, 5.52394,
  1.72389, 3.61583, 2.94497, 4.12341, 3.67378, 3.31173, 3.91122, 5.52625,
  6.01953, 6.32136, 8.53632, 5.66473, 7.46751, 0.250103, 3.32354, 4.83793,
  3.49482, 6.12644, 5.77436, 7.36842, 5.10919, 5.62112, 2.82712, 5.1851,
  2.41435, 4.78288, 6.2138, 6.8019, 6.35311, 4.98811, 1.19491, 7.72455,
  4.14836, 7.49259, 4.89136, 4.16171, 5.52463, 8.06704, 2.31405, 8.405,
  3.65719, 5.11593, 1.51268, 4.47819, 1.84876, 5.141, 6.0089, 1.80582, 4.35851,
  4.67314, 2.58094, 1.98456, 5.3914, 5.20932, 7.09548, 2.03095, 4.466, 8.35908,
  4.55317, 8.54534, 6.07513, 6.6616, 8.65318, 1.46264, 7.10535, 5.38341,
  2.24631, 4.26311, 5.15568, 3.31734, 5.31636, 4.38404, 4.09087, 1.10691,
  5.71244, 5.78152, 6.9864, 6.71674, -0.0962443, 5.09355, 7.82747, 2.26544,
  2.11678, 5.04787, 6.23645, 5.63394, 7.61503, 9.2668, 7.73077, 6.13407,
  1.79309, 6.01969, 6.27681, 6.33326, 2.63452, 5.04482, 8.09185, 3.19829,
  7.21321, 5.67472, 1.64252, 7.03477, 5.59738, 7.30405, 0.871957, 6.37748,
  6.20861, 3.48136, 7.35017, 7.25573, 5.03812, 4.72556, 4.35434, 2.04436,
  4.40308, 7.31564, 4.68853, 6.40906, 6.18755, 1.89987, 5.01074, 5.84798,
  6.2833, 8.13314, 5.95462, 5.08984, 4.52957, 6.20165, -0.0912769, 4.34249,
  8.7835, 3.3574, 5.02986, 3.89009, 4.93371, 6.19667, 5.32764, 4.884, 5.84214,
  5.79276, 6.86467, 4.69764, 4.23952, 5.29655, 5.99214, 3.54507, 4.48542,
  5.19771, 2.71014, 6.01681, 4.63336, 3.83212, 5.09326, 4.59779, 5.65044,
  3.69708, 7.49578, 4.55864, 3.62616, 7.465, 4.80171, 8.92855, 6.77231,
  8.37062, 3.13632, 5.77142, 2.99463, 3.38876, 6.06396, 5.34917, 7.14175,
  5.02205, 5.35278, 5.84463, 2.71554, 4.7328, 3.35526, 3.58672, 5.60095,
  3.37759, 8.92285, 2.29002, 4.84675, 8.11158, 5.0312, 4.52302, 4.18496,
  7.27296, 4.14379, 4.39727, 2.73833, 4.52588, 2.12512, 5.02897, 5.44221,
  4.77971, 2.22773, -0.724006, 5.68467, 4.33303, 1.99064, 2.27616, 0.768654,
  6.57134, 6.26118, 2.59212, 4.01904, 3.72639, 4.45964, 1.11602, 3.15925,
  4.91734, 3.21068, 5.41891, 5.20463, 0.829772, 5.64902, 1.44667, 3.88481,
  5.28323, 9.12644, 3.87373, 4.82988, 5.10225, 5.04382, 3.87691, 4.72456,
  5.47339, 9.10188, 4.91053, 8.30659, 3.56662, 4.35286, 6.50226, 5.90669,
  7.30085, 6.76536, 7.45, 4.88163, 4.99476, 4.37031, 4.85539, 4.04563, 4.2115,
  6.66141, 2.58253, 8.0716, 5.35939, 6.0417, 0.943797, 5.00969, 4.34013,
  2.61138, 6.81654, 4.39162, 4.24062, 3.02856, 5.53358, 4.99001, 2.67177,
  4.53113, 4.40043, 6.78999, 6.75244, 6.11493, 5.56569, 4.84363, 9.61416,
  1.71875, 5.03891, 1.91332, 3.24843, 5.57856, 6.16002, 5.44736, 3.39938,
  6.18465, 3.15994, 2.98172, 4.41363, 5.01468, 5.74865, 4.682, 6.97564,
  5.35939, 3.55609, 7.4103, 5.97475, 4.43601, 0.958276, 4.17341, 5.99322,
  3.53916, 1.01477, 3.0659, 4.96741, 3.91912, 4.57884, 4.99408, 6.11101,
  3.40979, 3.84287, 1.57586, 6.56948, 4.11133, 5.85789, 3.35584, 7.00787,
  9.38967, 1.33471, 3.99193, 5.12162, 3.82647, 5.04435, 7.42288, 6.66568,
  3.38113, 5.0103, 6.55437, 7.2716, 5.57745, 4.13538, 9.574, 5.38008, 4.06266,
  1.44281, 6.78299, 4.94966, 6.79257, 4.21786, 2.70053, 6.36376, 4.70361,
  4.33781, 2.48839, 5.08079, 5.38507, 4.82762, 6.55314, 6.87118, 4.2207,
  4.94756, 4.21976, 7.3338, 5.62528, 8.24457, 1.14479, 1.56091, 3.89058,
  5.74393, 5.27253, 5.36402, -0.0246991, 7.3378, 5.46285, 4.94172, 5.02843,
  5.00094, 5.78275, 5.941, 6.73248, 3.85768, 8.92951, 6.29436, 8.84056,
  5.70385, 6.52041, 6.99664, 6.51043, 4.63235, 9.333, 4.58166, 3.7638, 5.77146,
  5.91497, 4.62955, 6.61613, 5.0552, 5.77957, 3.98556, 5.76976, 4.98856,
  8.75951, 6.82312, 8.68462, 4.80455, 5.23055, 7.32671, 5.09542, 3.94949,
  3.49144, 4.68515, 3.70115, 3.38352, 3.17251, 0.273903, 4.71887, 1.83296,
  4.66312, 7.87277, 4.90258, 5.33407, 5.19987, 5.71639, 5.89436, -0.0725608,
  2.394, 4.79957, 8.32821, 3.02955, 4.25231, 4.83524, 6.24165, 10.3697,
  4.79723, 6.2515, 1.42442, 4.60442, 7.10076, 1.68772, 2.44678, 2.26011,
  7.56247, 8.2122, 4.62442, 3.35596, 3.18341, 3.90573, 8.21178, 2.81789,
  4.50061, 4.40506, 5.82643, 1.19871, 8.14573, 7.93369, 3.07551, 6.34079,
  5.68689, 7.31604, 3.93807, 4.80322, 0.228979, 7.24539, 6.73546, 5.18198,
  4.98533, 4.29523, 4.0157, 6.74256, 5.63536, 3.16594, 5.46906, 7.76003,
  4.98036, 4.27087, 3.92632, 2.48152, 4.27974, 6.0958, 9.24451, 5.69125,
  5.30679, 4.04365, 4.5673, 6.80192, 3.15583, 4.30414, 7.77537, 4.57266,
  3.36795, 5.80245, 6.26686, 1.17412, 5.68254, 5.3849, 3.66286, 4.95819,
  3.91536, 5.83367, 2.14973, 4.73496, 7.73153, 1.48292, 4.87239, 3.92985,
  4.47446, 7.51094, 10.6794, 5.22878, 4.58847, 7.05685, 2.57699, 7.15822,
  5.49566, 6.44334, 4.48969, 5.42846, 6.31435, 4.23242, 6.56247, 5.90801,
  5.92221, 3.92272, 3.15174, 5.20783, 4.45017, 3.96758, 6.58043, 4.86716,
  4.53794, 5.00854, 1.95185, 5.57766, 4.70367, 1.79471, 7.45854, 0.55061,
  4.2604, 6.46399, 5.67441, 5.8277, 7.37607, 3.57722, 7.34987, 4.06636,
  5.05779, 6.50571, 5.49727, 4.43581, 6.56463, 5.08681, 7.78056, 5.14148,
  4.34274, 5.79612, 7.467, 6.4122, 4.61128, 5.41178, 1.97302, 8.30769, 5.84508,
  4.36983, 4.38866, 4.92445, 5.85667, 2.87666, 3.0969, 5.34379, 4.09991,
  5.72842, 5.94446, 3.30273, 4.02361, 5.97951, 3.14697, 1.43138, 8.43879,
  4.84342, 4.48041, 5.00544, 2.06167, 6.62759, 4.81213, 8.75554, 6.26893,
  2.53774, 5.97678, 5.8121, 4.2341, 4.91126, 5.51162, 4.94485, 5.05455, 9.4881,
  7.59165, 8.93544, 6.16801, 3.83569, 5.70668, 6.97473, 5.92499, 7.56713,
  6.9526, 6.47529, 5.71113, -0.167751, 5.04871, 8.84113, 8.18138, 5.81901,
  3.8285, 0.885419, 3.30712, 7.01546, 3.09568, 7.02003, 6.32265, 6.1993,
  1.74211, 2.55494, 5.01863, 6.10781, 4.76631, 6.88654, 6.76637, 4.20461,
  4.97314, 6.85499, 6.22343, 3.94839, 5.20577, 4.34656, 2.21795, 4.03234,
  7.73698, 4.2604, 5.62983, 4.95525, 2.08839, 2.9996, 5.00563, 3.91701,
  3.87224, 2.85897, 2.35329, 5.86246, 1.70775, 5.99432, 3.6401, 3.07133,
  0.571156, 7.10579, 4.95173, 6.54055, 5.8046, 6.39142, 5.26296, 8.14433,
  4.29956, 5.15963, 2.37912, 6.44859, 5.42907, 3.46438, 7.62127, 5.21749,
  4.59664, 3.76846, 8.00986, 8.33375, 4.59427, 3.93084, 7.19372, 6.12539,
  4.72539, 3.3366, 6.15844, 5.15997, 9.10168, 7.86103, 4.92389, 5.16026,
  6.48198, 3.77718, 2.80943, 6.85968, 3.0429, 6.5701, 2.93565, 3.87503,
  5.97626, 5.04195, 4.98108, 7.14425, 6.72607, 3.25808, 1.48382, 8.84063,
  5.90187, 7.34406, 4.20272, 4.02484, 5.58074, 8.41222, 7.84697, 6.08786,
  5.7526, 6.85457, 5.90359, 6.18944, 5.94421, 5.30596, 5.18218, 5.56125,
  3.99491, 3.64818, 1.11387, 5.83075, 7.65457, 9.25109, 5.51181, 3.48123,
  10.0196, 5.89722, 3.8719, 5.5285, 5.44772, 5.75265, 7.65641, 6.01356,
  4.11655, 3.95844, 8.24485, 4.17756, 1.00631, 7.18268, 1.14116, 3.29298,
  3.63421, 7.49833, 3.25536, 4.95134, 7.01358, 3.54683, 6.96742, 5.52749,
  4.5285, 6.74964, 3.22319, 5.44272, 3.68948, 4.36553, 6.96815, 7.40486,
  7.18675, 3.99971, 7.74479, 4.85296, 6.75292, 5.3486, 6.2903, 6.0626, 8.02584,
  5.22723, 1.58964, 8.03563, 6.90578, 5.89153, 4.85439, 2.63505, 7.47967,
  4.91346, 1.29617, 4.28597, 7.92894, 7.75195, 3.91382, 3.12053, 7.07537,
  0.892509, 1.77475, 6.55883, 5.79862, 7.59966, 5.52176, 6.09847, 11.1532,
  7.39136, 3.67202, 4.26942, 2.45242, 8.13827, 5.18136, 7.61323, 6.11921,
  0.394242, 8.27647, 3.11126, 6.47512, 3.37151, 5.44903, 6.8563, 5.97861,
  1.97081, -0.89268, 5.87171, 4.07374, 5.82228, 6.43032, 6.17989, 6.94095,
  7.52617, 5.12574, 6.53761, 3.04694, 4.92544, 5.46832, 4.36368, 3.81898,
  2.43435, 7.53869, 5.65066, 0.722467, 3.2746, 5.92739, 3.07723, 10.132,
  5.14809, 5.71471, 1.64338, 5.91954, 5.21919, 4.87043, 8.27355, 9.23678,
  3.91331, 4.74741, 5.50817, 2.44289, 5.39552, 0.366979, 5.50793, 4.64854,
  5.74136, 3.43318, 8.02934, 4.04152, 6.99089, 7.84266, 10.0667, 3.21876,
  5.32639, 7.54402, 6.55163, 8.15726, 1.10949, 5.79704, 5.53143, 5.32519,
  6.19531, 2.3725, 9.66198, 3.4125, 4.8124, 4.12494, 4.64426, 3.71132, 6.60183,
  4.56, 7.09152, 2.72265, 6.77964, 5.7207, 2.62824, 5.18451, 4.27648, 4.94463,
  6.39609, 5.89824, 1.69854, 3.76832, 3.1017, 7.55644, 5.70232, 3.80509,
  6.29151, 7.17414, 4.46833, 9.20768, 6.40154, 3.71988, 5.5584, 5.83449,
  8.96432, 1.5366, 3.78357, 6.79206, 7.6992, 7.06521, 4.14123, 6.29877,
  4.19054, 5.24491, 4.55046, 2.88818, 4.52276, 6.3971, 5.48837, 6.11675,
  7.03737, 4.52301, 6.19314, 1.87898, 4.719, 1.88657, 4.72515, 5.31001,
  4.79035, 3.35268, 2.92701, 6.05221, 4.12264, 6.68298, 7.24517, 4.47602,
  5.05612, 5.34284, 2.22399, 4.02062, 2.28887, 1.76453, 6.83636, 6.37016,
  5.56094, 5.95923, 2.61262, 3.00058, 3.85433, 1.55179, 5.90365, 5.94011,
  1.44228, 6.27284, 6.06341, 1.38349, 1.35024, 6.58804, 2.81204, 8.04853,
  5.13519, 8.57719, 5.89782, 3.39148, 7.31977, 5.86232, 3.53635, 7.53823,
  4.6018, 4.65753, 1.11099, 6.28223, 5.80326, 3.96348, 3.59682, 4.90516,
  7.14701, 6.03101, 3.86326, 6.32068, 7.8991, 6.79059, 3.95603, 7.16407,
  5.65351, 5.35069, 7.16177, 7.65138, 5.21334, 4.93886, 6.00902, 6.57826,
  4.69908, 7.87701, 5.0577, 6.46935, 4.34265, 2.74617, 1.21546, 6.1245,
  7.78991, 2.64862, 2.94703, 6.50865, 1.53688, 4.91612, 3.63144, 4.01813,
  3.02116, 3.58415, 2.4555, 2.50368, 5.73656, 7.09798, 4.77533, 5.85677,
  4.43772, 8.22647, 3.29884, 4.54146, 8.02269, 6.15567, 4.4109, 4.1133,
  7.11411, 5.38732, 5.78609, 7.20663, 7.07961, 5.84398, 7.25506, 4.71497,
  7.2447, 6.64163, 9.28097, 2.05522, 6.48798, 5.44609, 6.15833, 5.45834,
  4.3417, 1.81443, 2.72026, 4.60044, 5.53086, 7.82885, 3.01374, 6.79074,
  2.97571, 3.62067, 7.10517, 6.05063, 4.70835, 4.03834, 3.40926, 8.44981,
  6.64344, 5.18654, 8.89032, 3.71847, 4.88368, 4.73274, 5.78363, 7.32357,
  4.91598, 4.46398, 3.04229, 8.61135, 7.47678, 4.23244, 4.4723, 2.59107,
  5.5128, 2.24182, 5.01747, 1.82223, 3.54453, 6.27647, 8.38491, 4.30537,
  1.93217, 4.11903, 3.94742, 4.28775, 5.49313, 7.77053, 3.59545, 8.35981,
  4.4961, 6.73089, 5.56122, 4.25243, 4.43417, 3.69941, 4.69318, 5.74566,
  1.0842, 3.15023, 4.45864, 4.46046, 3.63593, 6.29814, 7.52096, 3.12316,
  1.32394, 1.73617, 2.70269, 4.84453, 5.93376, 4.82801, 4.9342, 2.63783,
  6.39188, 3.44236, 4.17555, 5.27835, 5.17167, 2.4226, 5.64337, 7.10597,
  6.18424, 7.13226, 3.94927, 7.41167, 4.27004, 7.37848, 4.93709, 6.06954,
  4.98494, 4.60433, 5.20105, 3.06169, 8.86104, 6.21171, 7.65418, 7.84167,
  3.81352, 5.44793, 6.90535, 4.07576, 9.0029, 5.37819, 1.10657, 5.07796,
  6.22261, 5.57075, 5.85389, 9.93893, 5.68872, 3.27937, 7.70957, 5.87657,
  2.05615, 4.67979, 5.92765, 2.63676, 2.17229, 1.06248, 5.88911, 7.66574,
  10.0577, 5.85857, 3.35215, 5.72933, -0.539317, 4.48258, 9.41321, 1.72438,
  3.42396, 4.50109, 2.03232, 7.02414, 4.97025, 3.43899, 2.14375, 5.58337,
  4.99508, 6.10483, 4.16022, 4.48571, 3.13541, 7.08354, 5.13534, 7.35646,
  5.0204, 6.82935, 5.58327, 5.83764, 7.07168, 4.74466, 5.17199, 5.85153,
  3.37399, 5.63742, 6.2143, 6.6443, 3.18845, 6.78548, 3.50968, 5.24131,
  5.88544, 4.87555, 2.57616, 4.52616, 6.25785, 6.35231, 2.7508, 3.55758,
  3.09684, 1.93197, 3.76199, 9.82211, 7.77554, 4.94051, 4.83704, 5.50132,
  6.71067, 6.47504, 4.02863, 5.67316, 4.68856, 4.96759, 6.66168, 4.5711,
  2.80276, 4.76491, 5.30565, 7.30984, 7.68234, 2.89956, 6.49129, 3.89235,
  3.38763, 5.77565, 5.20569, 7.04662, 2.81439, 4.66951, 4.98079, 2.96166,
  2.49033, 5.24922, 5.29581, 5.24791, 5.12425, 4.53036, 4.88238, 4.90298,
  2.77123, 3.62579, 4.7044, 6.54186, 6.36447, 3.0214, 6.73522, 4.82247,
  8.01091, 5.73834, 1.39115, 3.79936, 4.42679, 6.27078, 7.21718, 3.49943,
  1.57874, 4.93336, 7.93149, 6.46315, 5.21875, 1.85368, 6.13226, 4.07815,
  6.70702, 3.05088, 6.25708, 3.38404, 6.03878, 6.01346, 6.16121, 5.90362,
  7.41413, 2.22012, 3.11635, 5.40637, 6.15018, 2.0998, 6.37272, 3.70375,
  4.47949, 3.98195, 5.47157, 5.69937, 4.19477, 6.57436, 6.28801, 6.44989,
  4.32168, 6.47, 5.94731, 4.71357, 3.62506, 3.24837, 6.78335, 5.85277, 5.54621,
  6.78999, 5.26941, 4.87242, 4.36558, 5.84505, 3.59157, 2.87549, 7.53707,
  5.93287, 4.29436, 6.39084, 3.86173, 4.09454, 3.48407, 6.78128, 5.10255,
  3.62986, 4.0445, 3.47146, 5.70515, 3.43935, 4.91934, 2.84573, 5.1928, 4.9535,
  5.50311, 4.50086, 5.90436, 4.04018, 6.94497, 3.70806, 6.95925, 6.3904,
  3.78074, 8.30912, 5.81268, 5.19924, 6.56437, 6.18314, 5.38018, 7.64278,
  8.23851, 4.18985, 5.48248, 5.79874, 6.31137, 5.64365, 4.06618, 3.1568,
  5.61577, 6.65508, 8.209, 7.04369, 6.73821, 7.17099, 3.04074, 4.98741,
  3.34733, 7.02613, 6.07177, 3.60866, 2.23547, 8.56349, 8.34575, 6.07122,
  3.56102, 6.50718, 5.82782, 4.77263, 3.92344, 4.04443, 2.76277, 3.29661,
  6.74782, 5.60243, 3.08284, 7.6145, 3.64103, 2.06407, 4.57488, 6.36709,
  5.89587, 5.1313, 4.72321, 2.64072, 3.45406, 6.12908, -0.00611626, 3.94674,
  4.92328, 6.36224, 6.88566, 6.21488, 8.03852, 5.02064, 4.76666, 3.21105,
  4.26853, 6.42786, 6.55552, 6.20539, 3.55478, 6.84469, 3.61097, 4.66961,
  7.48593, 4.49803, 2.9819, 5.32639, 4.44777, 4.12385, 3.15206, 3.1978,
  6.49863, 6.2535, 5.5132, 8.16704, 1.21616, 5.13345, 3.46533, 7.68617,
  5.62844, 6.00409, 6.30926, 9.60422, 0.769202, 4.20689, 5.43262, 3.44991,
  4.03118, 5.61427, 4.76256, 7.49161, 0.539265, 5.92342, 6.57561, 2.70059,
  4.11979, 4.43832, 6.49192, 2.62957, 3.13954, 5.60899, 4.14957, 2.98014,
  8.06135, 4.56145, 5.24373, 6.10866, 2.58432, 4.72634, 2.41191, 6.01479,
  5.11973, 6.74634, 6.48312, 5.83711, 6.1593, 6.12677, 1.94822, 3.45472,
  3.57311, 1.71531, 6.32235, 3.61514, 3.1375, 2.64593, 8.73059, 4.75078,
  6.78579, 5.40549, 8.03214, 7.09353, 7.81597, 5.01447, 7.40464, 5.7909,
  7.67921, 4.71864, 2.04216, 7.49784, 8.88033, 8.38889, 4.15365, 7.14188,
  5.82192, 1.92725, 4.20735, 3.32456, 3.39035, 3.41169, 4.50749, 8.484,
  7.22724, 8.38792, 6.71337, 4.52634, 3.8036, 6.12727, 3.63153, 4.47157,
  5.31527, 6.39461, 4.26412, 8.67902, 6.63649, 7.3117, 6.62086, 6.24108,
  7.26329, 5.39086, 6.87832, 7.13664, 4.46217, 4.46863, 5.82226, 1.84116,
  1.10976, 9.55586, 6.55365, 4.62735, 4.96034, 0.403931, 4.30588, 7.29335,
  2.96311, 4.8138, 6.69792, 6.16822, 4.71998, 5.70629, 2.42799, 9.25578,
  5.17029, 5.35069, 7.22382, 5.42732, 7.31335, 5.2364, 5.40183, 3.10503,
  6.02821, 4.66922, 1.56849, 6.44236, 6.07804, 5.76258, 2.33246, 8.66817,
  5.20741, 2.5001, 4.58049, 4.41286, 4.72582, 4.43776, 5.75013, 4.60335,
  8.00864, 4.22316, 4.98014, 3.49319, 7.32116, 1.73005, 6.21888, 5.8273,
  5.11384, 5.94721, 4.42737, 6.06786, 5.53823, 3.30176, 6.13387, 4.52314,
  5.65424, 7.11806, 4.62369, 1.25308, 7.96742, 6.08435, 6.08618, 5.03491,
  5.23369, 5.71686, 2.68913, 5.43203, 1.96616, 4.16766, 6.61934, 6.33065,
  5.18686, 3.72741, 3.88909, 2.52648, 4.81031, 3.80068, 1.65545, 6.42915,
  2.56448, 9.07409, -0.124919, 7.26093, 1.08755, 5.7571, 4.24512, 4.2547,
  6.65528, 8.05178, 1.90325, 6.49439, 5.45716, 0.525495, 1.86703, 4.67262,
  3.13776, 3.56394, 0.0111102, 6.17731, 6.662, 6.68103, 1.13277, 4.99763,
  3.41157, 4.72937, 5.40083, 4.4725, 2.14406, 6.30808, 7.90338, 2.18431,
  3.2355, 10.2537, 1.42567, 5.88586, 6.86358, 4.36713, 1.31344, 3.89606,
  0.459294, 2.59371, 9.82777, 7.46596, 4.50585, 4.10122, 4.86659, 4.00199,
  2.91319, 6.68225, 4.8591, 3.18124, 6.29423, 4.36255, 4.20868, 5.71709,
  8.0249, 4.15752, 4.72875, 7.00949, 4.77223, 5.32703, 5.37256, 3.12741,
  3.93729, 6.41169, 7.57056, 9.09769, 2.32108, 2.72283, 5.63024, 5.07048,
  5.62341, 3.29035, 2.20456, 4.6144, 4.68067, 2.63154, 3.37773, 5.51561,
  5.90742, 1.91401, 9.92111, 7.0141, 6.40078, 6.02753, 3.36194, 7.06094,
  3.55078, 5.05602, 7.94237, 5.33723, 9.62044, 4.25928, 3.11591, 3.43737,
  3.44899, 9.0669, 3.03531, 4.67975, 3.44214, 2.95309, 2.61954, 6.26545,
  2.60856, 1.10092, 10.1488, 5.01116, 5.3048, 4.65596, 2.63419, 4.21127,
  3.86477, 5.76174, 3.352, 6.71725, 3.6628, 6.36349, 5.04039, 9.49935, 10.9012,
  4.71908, 7.1987, 3.30976, 5.46754, 3.84017, 7.9793, 2.86487, 3.89984,
  4.52025, 5.3863, 3.53137, 5.63964, 7.73445, 1.72572, 6.11017, 4.52196,
  3.47244, 6.29847, 3.34359, 10.662, 4.55612, 7.86466, 3.63835, 3.19122,
  3.73848, 7.15421, 5.93842, 3.22906, 3.62794, 3.56499, 7.90433, 2.7197,
  6.46196, 6.23903, 3.43287, 3.9468, 7.1827, 2.43594, 7.29165, 2.19237,
  3.59075, 8.35948, 2.89586, 2.48558, 7.72749, 3.20383, 3.12063, 8.01061,
  3.07678, 5.31699, 3.07562, 1.622, 2.83869, 3.59936, 2.87812, 1.58108, 4.1938,
  5.43118, 5.15721, 3.01099, 4.8594, 6.09087, 4.09553, 5.43568, 3.9189,
  5.71295, 5.46187, 6.69766, 3.96882, 1.94598, 3.97241, 5.78342, 5.02547,
  7.57661, 8.72381, 5.96474, 4.92254, 4.53541, 3.60071, 5.42536, 6.9945,
  7.94287, 2.76474, 3.68853, 6.54173, 1.16887, 1.38076, 6.5021, 2.60447,
  4.4077, 5.73208, 5.42884, 3.54185, 1.35843, 4.06883, 7.16516, 4.63623,
  7.13502, 3.59584, 4.75249, 8.66852, 6.01915, 6.24013, 3.38239, 6.48651,
  7.68347, 5.97514, 4.47642, 3.79319, 7.77987, 6.60787, 3.49549, 7.39058,
  2.53939, 3.01248, 4.11654, 5.70536, 4.05478, 6.1005, 5.36294, 1.11532,
  4.02034, 7.01693, 3.03738, 7.2373, 4.70032, 5.1157, 5.62488, 5.39281,
  3.56298, 4.56707, 6.90159, 2.10655, 6.7958, 4.87719, 3.37707, 5.2683,
  5.16647, 2.76541, 4.42268, 5.34326, 1.66553, 4.73084, 3.3783, 6.97722,
  4.51522, 6.14869, 3.45661, 1.0764, 3.88468, 4.99641, 5.01211, 6.73775,
  6.25383, 5.2107, 5.03403, 3.74506, 5.09409, 6.66812, 6.40719, 3.99628,
  5.51572, 6.66814, 7.29551, 6.46251, 4.67545, 1.59635, 4.7656, 3.14673,
  3.63437, 5.5887, 4.31045, 6.07261, 1.04755, 5.4185, 7.94216, 1.10918,
  1.89531, 4.54519, 5.34602, 3.83026, 5.56748, 5.79972, 6.47701, 6.83219,
  6.55239, 3.43026, 6.39098, 5.68555, 4.87914, 3.85775, 6.88215, 4.03825,
  4.7832, 9.13661, 7.53273, 8.37345, 8.21991, 4.46711, 4.60821, 2.94653,
  6.28439, 3.31684, 5.70488, 8.8665, 4.67852, 5.04718, 3.50851, 4.68931,
  4.75045, 5.78615, 4.32451, 4.18752, 4.26647, 6.02812, 6.445, 7.25222,
  4.66111, 5.80198, 5.78455, 9.94328, 5.51944, 6.79937, 5.12126, 4.2194,
  5.06653, 7.3237, 5.70219, 0.314251, 7.61824, 6.27475, 4.86847, 2.87059,
  7.70704, 2.19757, 4.69713, 3.89236, 4.62132, 4.12976, 2.99646, 5.889,
  5.69091, 3.86409, 1.96571, 8.36435, 2.88642, 2.25078, 4.90023, 4.39022,
  2.6942, 3.58654, 4.35358, 6.01427, 5.3651, 2.10704, 4.43219, 6.62873,
  4.71527, 6.71844, 7.60029, 7.12061, 3.10049, 3.88577, 7.01007, 6.44799,
  4.28927, 1.08303, 4.50535, 4.15458, 5.97313, 4.2579, 6.05134, 5.14599,
  0.482241, 0.765745, 2.43152, 6.73714, 4.91614, 6.56314, 2.42325, 4.6177,
  2.7871, 7.82748, 0.955289, 5.10207, 3.67567, 5.21609, 4.20963, 4.47578,
  7.72302, 4.45917, 5.62278, 2.32563, 2.47934, 4.3298, 5.56838, 2.69498,
  4.41482, 6.11849, 6.63637, 4.46314, 6.70601, 4.53381, 5.85429, 5.41415,
  10.4784, 2.87641, 5.11271, 7.43555, 7.00549, 7.82443, 9.1126, 3.45496,
  5.71178, 10.6973, 2.88013, 5.73425, 9.1556, 0.63391, 4.50243, 5.77811,
  6.46682, 5.68307, 7.56761, 4.39106, 4.37998, 3.8192, 7.42299, 6.2173, 3.5624,
  5.46784, 5.54343, 7.20433, 5.60152, 7.4301, 3.52954, 5.72141, 0.358383,
  9.68421, 6.62102, 5.86311, 5.05837, 4.33403, 5.74624, 6.48975, 1.46643,
  5.81043, 4.53512, 9.74918, 5.14074, 5.26304, 7.3414, 5.82375, 5.2664,
  4.61198, 6.08937, 1.24642, 5.56225, 7.40388, 8.38658, 6.27795, 5.6425,
  5.87252, 6.29086, 3.13848, 6.54697, 3.72485, 6.09656, 5.20299, 6.04178,
  3.57947, 5.46147, 4.57589, 5.89056, 3.65916, 4.9071, 8.0165, 5.45731,
  7.55026, 4.47823, 7.34286, 4.16664, 3.57177, 3.71491, 5.10637, 6.03213,
  10.3632, 7.04773, 7.81964, 4.45801, 8.05145, 8.16601, 4.07259, 5.987,
  3.91017, 4.94203, 2.3507, 2.62036, 2.80866, 6.47016, 7.49851, 4.90099,
  8.05086, 1.84202, 3.80482, 7.35523, 7.18891, 5.29897, 5.85105, 6.00741,
  4.3631, 7.622, 1.09021, 4.73623, 6.28574, 5.21681, 7.29206, 0.766538,
  4.36427, 5.57452, 7.05878, 7.25602, 5.26401, 9.86678, 5.44248, 3.92436,
  3.57555, 4.44004, 4.70752, 5.96683, 3.35322, 4.73783, 6.87501, 7.74429,
  2.35932, 6.87911, 8.37387, 6.62835, 2.11293, 6.49212, 4.49109, 6.04444,
  5.24888, 1.51332, 3.08009, 3.36744, 3.70575, 4.26756, 2.38098, 6.83837,
  7.64185, 6.76491, 1.22809, 2.47398, 8.46211, 3.23938, 6.40942, 4.85451,
  7.22729, 3.59315, 6.21673, 2.82649, 7.67953, 6.19829, 5.78288, 6.29438,
  2.38798, 2.46392, 5.33662, 6.54512, 7.0208, 5.36753, 6.05201, 6.16517,
  3.18899, 3.94271, 7.59458, 3.11451, 8.14626, 7.59764, 4.51707, 4.69663,
  4.88002, 5.84183, 6.92652, 5.4611, 3.86944, 3.83183, 5.33923, 8.06783,
  5.21674, 2.74383, 4.83745, 6.17589, 2.61192, 6.97427, 5.88192, 7.5792,
  4.72992, 2.88027, 5.37879, 7.36124, 4.97614, 4.09111, 8.1172, 4.67038,
  5.97524, 4.87531, 6.17958, 5.25471, 6.62768, 6.40249, 4.5145, 5.40642,
  2.84726, 3.26518, 6.74467, 4.18785, 5.91896, 6.24806, 5.72286, 5.84195,
  1.2192, 3.7721, 6.49035, 4.47952, 6.25199, 9.07945, 2.58075, 3.82808,
  0.80211, 8.36188, 6.6552, 3.85052, 5.22345, 4.939, 4.06123, 4.40438, 3.92286,
  7.33183, 4.90722, 7.93182, 0.927118, 7.52319, 4.11019, 4.5644, 3.99472,
  3.57183, 8.23011, 4.11972, 5.86793, 7.26589, 7.23756, 6.80606, 2.9345,
  4.64744, 1.23924, 3.3683, 7.82092, -0.512768, 3.03523, 1.81471, 4.91122,
  4.56787, 8.14311, 3.73404, 7.96959, 6.61157, 3.6123, 7.17288, 2.83834,
  10.0725, 0.994574, 3.4867, 4.03801, 2.83354, 4.56551, 7.27031, 6.41569,
  4.37064, 2.01641, 6.98363, 4.75605, 2.99698, 4.07198, 4.28477, 3.68533,
  3.37446, 5.40971, 6.15364, 5.21851, 6.88104, 9.35911, 3.39076, 3.84212,
  3.62928, 2.70987, 5.16405, 6.80033, 3.70855, 3.16651, 2.89323, 5.24551,
  3.57659, 4.56211, 3.90242, 7.33747, 7.8484, 8.08851, 2.72728, 6.65368,
  6.94298, 5.75877, 0.327813, 5.10476, 4.33659, 7.86497, 2.64948, 2.8595,
  4.51204, 3.96297, 4.40441, 8.23126, 9.0407, 3.73313, 5.15299, 6.3115,
  6.17681, 3.61823, 5.78856, 2.77492, 8.55077, 5.19783, 0.999647, 4.1396,
  4.41086, 4.29732, 9.22984, 6.20281, 7.79661, 5.02758, 7.3587, 5.28071,
  7.35016, 7.12686, 3.05801, 5.57911, 6.54376, 2.39046, 3.03428, 8.24335,
  3.89793, 4.21297, 0.964724, 2.15636, 6.34999, 1.14306, 3.68325, 4.77935,
  5.40557, 5.11935, 3.45336, 10.5985, 5.49027, 6.95143, 2.5995, 5.47407,
  4.26129, 5.74207, 4.1607, 2.99647, 5.21797, 3.03975, 3.20002, 4.98965,
  6.79847, 5.8146, 6.30152, 7.87603, 5.33204, 6.76692, 6.44493, 5.59135,
  3.54847, 7.61165, 5.48395, 5.95953, 6.35168, 2.955, 3.97537, 3.51251, 4.1858,
  4.25301, 8.45574, 7.08222, 3.49783, 7.01067, 5.77572, 8.65097, 6.6901,
  6.45781, 4.56838, 7.37418, 4.67689, 8.00877, 3.6299, 6.17937, 1.73565,
  2.71471, 5.23014, 5.07651, 1.17026, 6.99098, 4.55492, 4.6879, 2.73115,
  6.35692, 7.70944, 5.67799, 5.35907, 5.37368, 6.66233, 8.94174, 4.07541,
  4.47003, 4.14672, 6.15439, 7.23216, 1.57932, 4.50615, 6.06137, 7.92414,
  4.95263, 1.52836, 2.16368, 4.53723, 3.96684, 4.21357, 3.42091, 3.89524,
  3.91981, 3.99441, 6.30172, 4.66599, 5.58706, 4.16015, 4.38364, 3.37383,
  7.07682, 3.2594, 3.89029, 5.34605, 3.63847, 2.65379, 7.76676, 4.61183,
  6.73616, 1.82217, 6.21841, 6.11664, 5.08583, 0.560872, 5.47681, 2.30336,
  3.8472, 5.31596, 4.83931, 0.381977, 6.75813, 7.86265, 5.55449, 0.813303,
  4.67114, 6.16287, 3.52351, 4.67238, 5.29323, 5.89485, 2.3856, 6.51969,
  5.25965, 5.9603, 9.92328, 2.80611, 6.00696, 4.67019, 2.77056, 3.18142,
  5.71531, 6.22992, 5.82071, 7.42205, 1.77219, 4.46684, 6.31718, 2.43316,
  7.65348, 2.51199, 4.22925, 5.94956, 2.29347, 2.96261, 11.6922, 4.80927,
  5.32236, 8.17681, 4.26058, 6.1835, 3.84473, 4.35916, 3.86092, 6.69234,
  4.38549, 4.77638, 4.6601, 7.95575, 5.1358, 2.99024, 9.61706, 6.75721,
  4.13767, 7.63777, 2.18029, 6.32821, 5.61082, 4.63049, 7.43484, 3.78442,
  4.08076, 6.44838, 3.99243, 5.53104, 5.43132, 3.33438, 6.98345, 3.22925,
  4.47998, 5.49508, 5.48555, 7.68152, 5.0627, 6.37498, 5.13347, 5.33225,
  4.97275, 3.2124, 5.02374, 5.20541, 3.20713, 6.52873, 9.27445, 4.48124,
  4.87639, -0.30083, 4.83961, 4.41766, 6.13242, 5.5935, 4.7154, 5.51767,
  1.8235, 5.98259, 5.06916, 4.49304, 4.552, 6.88522, 10.0501, 3.63206, 2.76747,
  0.727984, 7.62081, 3.62505, 2.14804, 6.09414, 5.40159, 7.2918, 4.78239,
  5.04932, 4.40022, 5.60144, 4.91088, 5.2788, 6.17052, 2.55224, 4.50102,
  6.20839, 7.59507, 6.79449, 3.38294, 5.09318, 3.45692, 4.77839, 5.49548,
  3.19367, 6.944, 4.00609, 7.47348, 7.89804, 5.79774, 2.67779, 4.21858,
  6.02327, 5.49502, 4.15702, 1.02561, 5.36291, 4.09277, 4.14785, 6.11421,
  6.74298, 6.97397, 2.76618, 2.31764, 3.3384, 6.2087, 2.20462, 7.23554,
  4.88997, 7.44157, 1.30583, 6.18151, 2.81961, 7.00244, 5.94738, 2.6267,
  6.99049, 6.03554, 6.71634, 2.31801, 4.1234, 3.38612, 7.44211, 5.31086,
  6.37772, 7.63828, 4.2789, 4.99501, 4.62078, 5.54641, 5.99415, 2.81383,
  3.40585, 5.22863, 4.49274, 3.55715, 5.58207, 3.73085, 2.38999, 5.23019,
  4.69816, 6.37625, 5.9652, 2.12613, 4.29932, 3.15791, 2.84353, 1.93149,
  2.03241, 2.77078, 5.99784, 4.82535, 0.258155, 4.99869, 7.80368, 6.71815,
  5.67462, 4.28908, 3.27958, 1.78456, 6.5561, 6.03794, 5.68622, 5.60332,
  -2.22232, 7.0606, 6.10874, 6.08512, 6.21849, 4.72125, 6.4378, 6.4386,
  5.12223, 5.13025, 1.13669, 8.64864, 4.87134, 3.29431, 6.85898, 7.64315,
  4.71766, 2.74105, 2.35899, 4.49524, 4.59417, 8.90975, 3.01977, 7.55051,
  4.062, 4.71973, 7.0529, 7.52642, 7.31515, 4.03729, 3.74362, 4.97658, 3.20612,
  2.98926, 2.62266, 8.08207, 9.4238, 2.28835, 4.08578, 1.68455, 4.73431,
  8.56657, 4.55599, 5.0496, 5.51175, 6.67788, 5.20763, 5.91177, 2.97698,
  4.01327, 2.48639, 1.61364, 3.49235, 4.86593, 6.50674, 8.37943, 9.93716,
  3.35168, 7.19583, 5.23065, 4.22181, 1.20201, 6.5206, 6.83819, 6.21638,
  5.12872, 5.97241, 5.80261, 5.63686, 4.93733, 2.93035, 6.36079, 3.57241,
  5.90745, 7.90938, 2.59942, 5.47098, 2.60318, 4.14021, 9.05276, 7.28427,
  2.85827, 3.28788, 3.60647, 4.49065, 7.81168, 9.30509, 5.93604, 5.35889,
  3.12755, 5.79291, 2.95782, 6.64728, 7.846, 3.77129, 6.6529, 4.80429, 4.90868,
  2.36617, 2.91022, 5.21172, 4.94958, 4.85088, 1.35577, 5.14823, 4.03121,
  6.75885, 3.5576, 3.26022, 5.79379, 2.25691, 7.49284, -0.00576072, 3.27878,
  1.89007, 2.95557, 3.11952, 4.79203, 4.79052, 8.04729, 4.43845, 6.4011,
  4.55648, 5.67787, 4.55217, 3.97464, 6.0691, 6.82102, 4.70584, 5.86104,
  5.52352, 3.96271, 5.5548, 3.92426, 5.35425, 6.70742, 6.18667, 2.37762,
  2.83393, 6.0435, 4.10714, 3.03188, 4.49561, 7.15741, 5.95587, 7.44241,
  4.99105, 2.58813, 3.41202, 4.58767, 8.38021, 3.68588, 6.50858, 2.66641,
  2.69378, 2.81996, 7.17905, 6.10603, 4.00797, 5.00233, 3.09232, 5.46924,
  6.15896, 2.67698, 9.5502, 4.94648, 2.04085, 4.79528, 5.47049, 3.80378,
  5.44144, 2.14369, 3.72561, 8.17828, 5.9706, 4.62838, 3.85886, 0.891503,
  3.82566, 4.45889, 6.90516, 6.35997, 5.44199, 6.49223, 4.10952, 7.57732,
  6.04725, 7.37566, 5.82675, 4.69652, 7.7498, 3.8832, 8.1867, 3.17231, 4.44906,
  7.12581, 2.18903, 4.26394, 9.3719, 3.78601, 5.18484, 4.10536, 3.49906,
  2.62728, 5.10667, 2.26865, 2.46619, 5.68081, 2.73509, 4.11106, 8.51679,
  1.75435, 9.09393, 5.59445, 3.61797, 4.20435, 3.55526, 4.46973, 2.11604,
  6.59029, 7.1277, 1.65102, 6.98852, 5.58985, 6.317, 6.65696, 3.39519, 7.02251,
  7.10016, 4.76962, 5.10703, 4.9454, 3.23162, 1.61041, 5.88992, 5.55148,
  3.51125, 6.50618, 7.82046, 4.6042, 7.66186, 6.11058, 2.04255, 3.26722,
  5.49827, 4.9354, 4.13397, 6.03184, 4.45003, 3.77237, 4.79674, 6.60707,
  1.67699, 6.28556, 5.76103, 0.962684, 5.75425, 6.97752, 4.00314, 4.28418,
  3.87252, 5.16349, 3.07288, 6.39683, 4.56837, -1.03402, 5.23162, 1.9483,
  9.46964, 3.42841, 2.05441, 0.177304, 6.19155, 6.63233, 6.9668, 2.85434,
  6.29747, 5.5001, 7.1698, 6.00681, 3.71751, 6.02488, 8.74592, 8.91389,
  2.13634, 8.75978, 3.50419, 6.25839, 7.66357, 5.75048, 5.60513, 1.15153,
  5.00847, -0.540252, 1.35409, 4.21081, 4.87128, 5.34476, 6.5042, 6.79312,
  6.00169, 9.19996, 7.62468, 5.11579, 8.7856, 3.43207, 6.50989, 4.56343,
  6.07972, 5.17231, 6.36288, 3.91826, 5.04577, 5.30204, 7.57117, 2.66015,
  4.6183, 1.32127, 7.66326, 10.8477, 2.04361, 6.42205, 1.74833, 4.83868,
  0.407901, 6.79488, 4.0934, 5.33989, 1.23017, 5.10216, 3.65629, 6.40624,
  10.8136, 4.41835, 2.82762, 3.82148, 8.03306, 5.53872, 6.77221, 3.75332,
  4.63715, 7.16748, 6.47864, 3.49203, 4.48288, 3.57061, 8.19722, 4.34841,
  8.63659, 6.88709, 4.67096, 4.36509, 6.2604, 4.62456, 4.58703, 3.12779,
  6.67195, 6.57008, 4.07668, 4.21798, 6.58862, 2.84942, 6.72928, 4.19232,
  1.74521, 7.54448, 5.37154, 7.20368, 3.50678, 1.85157, 4.78818, 3.19464,
  4.51658, 3.70344, 1.16801, 6.66402, 5.57354, 2.62713, 5.28395, 5.78305,
  5.43077, 5.62439, 3.35117, 6.17747, 4.10951, 3.53548, 4.43493, 7.42754,
  8.55841, 8.79625, 6.68195, 2.97901, 5.8077, 6.74248, 5.13688, 7.61341,
  7.69795, 3.0381, 6.66243, 4.85822, 3.15645, 7.76386, 3.7183, 4.49881,
  2.32603, 6.94615, 4.63247, 3.26188, 4.78529, 5.00539, 2.30363, 10.1894,
  3.8653, 4.66295, 5.53252, -0.47746, 5.28083, 4.27968, 8.13997, 2.88531,
  7.21534, 3.4808, 3.18836, 4.46684, 2.81071, 5.19504, 10.3925, 4.47783,
  7.43548, 9.19193, 5.64229, 4.91579, 7.60405, 7.35332, 5.85487, 4.95028,
  6.24438, 2.3073, 6.53475, 4.72074, 2.65169, 5.58742, 6.79568, 2.68354,
  8.39627, 5.11574, 1.07366, 4.48103, 2.20854, 5.9751, 5.06167, 4.83165,
  3.69586, 7.80654, 6.80354, 7.15795, 5.72912, 6.56603, 3.23564, 4.77795,
  5.61365, 6.44069, 6.39457, 9.65851, 6.32384, 3.46212, 2.32262, 7.81521,
  6.1664, 5.61053, 0.440491, 6.47485, 6.46361, 4.58764, 3.28563, 6.111,
  4.12627, 6.85887, 4.26829, 3.35206, 6.02359, 5.37899, 5.50401, 3.76343,
  1.78103, 4.26696, 8.16345, 2.97918, 5.86254, 4.63366, 8.008, 2.92913, 7.4635,
  2.6503, 8.39421, 4.55481, 7.17833, 6.27671, 7.82661, 4.58241, 1.86492,
  5.38541, 8.47065, 5.93667, 1.26315, 8.4975, 6.31475, 0.89089, 5.18652,
  1.46432, 7.4211, 3.35225, 3.74343, 1.88443, 7.42323, 4.35272, 4.09861,
  6.07059, 2.81601, 3.25615, 12.4025, 3.5876, 1.41055, 6.5217, 4.72475,
  6.58239, 3.86275, 4.54331, 7.05073, 8.511, 2.84825, 4.05288, 8.66259, 6.4658,
  3.5116, 5.72882, 2.53839, 2.58603, 3.10022, 4.76414, 5.43115, 7.71897,
  4.58115, 4.88687, 5.62016, 4.37864, 0.457368, 3.97454, 3.61568, 4.449,
  3.06568, 6.22552, 5.66656, 5.39195, 2.7631, 2.61192, 3.5347, 5.83107,
  2.26457, 5.92129, 7.1482, 5.14424, 7.10402, 5.10056, 6.0696, 3.41425,
  4.52387, 3.33609, 4.64766, 2.73009, 6.24603, 2.88184, 4.15692, 2.03905,
  5.94705, 9.17429, 3.95521, 1.6124, 8.60505, 3.86549, 3.17925, 6.1738,
  5.37623, 5.16839, 4.97426, 2.36287, 4.65842, 7.35851, 5.59218, 3.3534,
  5.38266, 8.7766, 5.22238, 4.37883, 8.51897, 11.5959, 5.57327, 7.9247,
  6.88666, 7.76499, 4.34316, 4.35386, 8.05233, 4.49564, 3.47256, 7.75529,
  2.2626, 7.24628, 2.83975, 6.04926, 1.66742, 2.50242, 5.25996, 4.23411,
  5.90313, 7.36964, 3.57849, 0.451924, 6.31662, 5.95323, 5.01247, 3.27992,
  1.00763, 5.17044, 8.35017, 1.79036, 5.21544, 6.28277, 2.96472, 4.32627,
  3.04987, 9.78079, 6.8943, 2.99894, 3.01145, 5.9868, 5.68706, 3.21355,
  3.46271, 4.72889, 1.41614, 0.405699, 3.10806, -0.498348, 2.83091, 5.10952,
  7.20587, 7.61635, 2.97933, 7.54597, 5.91542, 3.97113, 8.71603, 3.49264,
  7.12144, 8.09773, 1.75289, 1.74011, 10.4472, 6.33534, 4.03944, 2.95464,
  4.24714, 7.71377, 5.57027, 2.96867, 6.69552, 4.3913, 6.09502, 6.69219,
  6.67947, 6.40076, 5.56396, 1.68354, 5.91449, 3.73182, 0.647248, 6.56085,
  5.40277, 5.10946, 4.23135, 5.00706, 6.5452, 6.07113, 0.953849, 5.80697,
  5.60235, 2.79277, 5.36666, 6.49078, 8.48539, 5.97296, 7.07424, 7.00466,
  3.77448, 6.89626, 5.29945, 5.32253, 8.27374, 4.78709, 3.29729, 3.59467,
  4.20385, 0.444276, 6.39074, 5.06983, 3.01822, 4.30674, 5.68504, 5.31234,
  7.46241, 6.24281, 5.171, 5.23888, 1.79312, 4.15836, 3.2265, 6.52739, 4.66633,
  8.52723, 8.85433, 3.6609, 4.28018, 7.64906, 5.76791, 5.44391, 5.54544,
  6.38028, 1.90244, 7.00822, 3.40479, 4.52961, 2.54214, 7.14939, 7.27834,
  4.44236, 7.20794, 5.63507, 8.25822, 7.51888, 6.11974, 4.95116, 9.30124,
  4.19947, 5.15832, 1.9623, 2.25956, 4.36764, 4.59078, 3.4914, 5.94962,
  5.76732, 6.29925, 7.86493, 4.46679, 2.79398, 4.37275, 3.928, 5.93072,
  6.19239, 8.99023, 6.03855, 6.28208, 6.87349, 5.0555, 3.56822, 2.0481,
  5.29003, 4.56742, 8.3062, 3.87542, 8.73514, 5.00089, 2.45876, 3.12111,
  6.01912, 4.55676, 6.26066, 9.78674, 4.81315, 8.2011, 4.08268, 4.45423,
  8.01672, 5.16869, 2.50508, 5.18123, 6.73991, 1.68394, 5.67652, 6.91543,
  8.8342, 3.40057, 4.36878, 4.3295, 7.38828, 6.71329, 7.74248, 3.56976,
  5.49016, 5.94651, 5.5299, 6.46185, 6.13591, 5.4775, 6.45234, 6.51342, 1.9686,
  7.20409, 6.00169, 4.59118, 11.822, 3.6549, 2.85916, 3.09927, 2.04007,
  7.53491, 7.40733, 5.10931, 3.25485, 7.02593, 5.24568, 3.80888, 5.3109,
  2.67051, 0.879269, 4.52484, 3.21152, 4.71241, 4.977, 6.91431, 4.37127,
  4.0184, 8.59241, 2.68644, 5.09072, 4.69175, 6.33216, 7.57764, 5.76837,
  5.8324, 5.80382, 2.92216, 3.36796, 4.53038, 2.5331, 4.70855, 9.03864,
  2.66708, 6.51073, 6.15957, 5.01313, 5.4201, 3.04331, 2.96834, 2.39016,
  4.54304, 7.32121, 6.64537, 4.75425, 4.10383, 4.47383, 4.81863, 3.19823,
  3.45902, 3.36491, 6.13265, 9.26533, 5.50999, 5.92294, 6.66316, 6.71004,
  7.02371, 7.55449, 6.45212, 5.34723, 4.21618, 3.20812, 6.5926, 2.42839,
  3.02085, 8.92788, 8.62373, 4.8154, 3.11284, 3.65549, 7.59549, 5.86016,
  4.38685, 4.04667, 6.0042, 2.47922, 6.76612, 5.97663, 8.65474, 6.68379,
  6.13514, 6.64407, 3.92745, 5.75588, 6.72476, 8.5098, 4.4804, 3.76546,
  2.46971, 5.01773, 8.01706, 5.41313, 5.53383, 5.04494, 7.39805, 4.18333,
  3.9728, 4.14484, 2.99484, 7.71211, 8.04073, 4.22135, 5.72475, 2.73584,
  6.37224, 5.16056, 5.84854, 4.76692, 4.35883, 8.27479, 4.66066, 8.07617,
  6.84804, 7.47794, 6.59312, 6.77843, 4.14247, 5.96346, 7.53503, 4.82317,
  1.65486, 2.67102, 4.30866, 6.26498, 4.20215, 4.70382, 1.81928, 4.49023,
  3.84648, 7.70383, 5.76509, 4.01845, 4.51867, 5.4954, 5.18091, 4.15802,
  7.54375, 6.97347, 3.24422, 3.66053, 5.94665, 2.08546, 8.90829, 6.77737,
  7.13256, 3.80996, 3.54875, 3.86289, 6.03314, 6.45634, 0.882964, 3.99303,
  1.94484, 6.03654, 5.56539, 3.79652, 5.97638, 4.87035, 4.80526, 6.4391,
  1.76404, 2.14781, 5.60012, 3.79183, 3.64469, 8.10351, 4.38742, 4.01062,
  5.19146, 4.12658, 6.67447, 6.71504, 6.67492, 3.35823, 1.70468, 5.96602,
  5.69713, 2.57387, 7.96559, 8.25612, 2.67473, 2.78101, 3.91655, 4.17659,
  5.78738, 7.86728, 5.45005, 2.43056, 8.9723, 3.45089, 5.45913, 5.70461,
  5.61268, 7.71782, 3.8677, 3.40877, 3.43238, 2.97405, 6.04599, 3.44117,
  4.60448, 5.10142, 6.01809, 3.13791, 6.30667, 7.38395, 7.45984, 4.72318,
  6.41437, 7.51012, 5.13963, 6.95466, 5.23072, 4.44615, 4.67988, 6.40699,
  6.72352, 4.80887, 3.32531, 4.82333, 5.8826, 6.26467, 4.97064, 7.32339,
  5.19685, 5.58942, 0.954072, 5.55048, 3.05093, 7.32227, 7.07419, 6.22735,
  1.84544, 6.47994, 4.8165, 3.44697, 4.88468, 3.1095, 8.96904, 6.37754,
  5.23036, 4.7368, 3.7648, 4.55317, 6.4014, 5.56271, 6.79987, 2.79056, 5.70965,
  6.2955, 3.70341, 6.99628, 5.43448, 2.27865, 5.06683, 8.05477, 6.11033,
  4.18997, 6.41011, 10.2001, 3.75381, 3.41736, 7.92221, 3.00289, 5.23813,
  4.82537, 7.0588, 5.63824, 7.36228, 6.44877, 5.47706, 7.9666, 4.27635,
  4.27355, 3.54984, 6.41757, 4.99647, 4.23979, 5.04868, 6.21485, 5.04072,
  5.91958, 7.51305, 5.95656, 4.83546, 6.72258, 5.40716, 6.8248, 5.05898,
  3.21051, 5.07121, 6.16424, 3.95778, 4.17979, 5.02484, 6.26021, 6.97151,
  2.95871, 6.57827, 5.22625, 5.83528, 10.3685, 3.91155, 5.30634, 1.58076,
  6.33866, 2.05276, 5.38012, 2.60854, 3.63472, 3.40758, 4.08708, 3.52396,
  6.35358, 5.49864, 2.61907, 5.26636, 2.75954, 5.15494, 6.05814, 5.46498,
  2.14079, 2.7291, 1.78573, 7.34261, 6.33551, 4.7336, 4.95624, 2.77648,
  3.25304, 3.4064, 5.21665, 6.10629, 6.89371, 4.15125, 3.79847, 5.18802,
  3.34013, 5.40487, 6.97708, 6.39621, 4.71021, 7.27268, 7.14298, 4.47132,
  7.22673, 5.73955, 2.07431, 5.28033, 4.64987, 7.26098, 2.32209, 2.42837,
  4.88508, 5.20585, 5.18161, 6.3906, 5.01353, 4.30426, 6.00627, 4.01256,
  5.95908, 2.22188, 4.95252, 7.0397, 6.44478, 5.82411, 5.11843, 6.48126,
  5.30517, 3.58989, 3.88191, 7.20352, 5.87092, 3.6665, 5.36752, 5.18492,
  1.62777, 6.08892, 6.05683, 5.43791, 4.09889, 4.95399, 5.70385, 7.17481,
  7.40824, 5.32755, 6.19774, 2.45696, 5.32444, 6.54949, 4.2101, 3.11413,
  3.90028, 1.13038, 0.287137, 6.29798, 3.87846, 4.63718, 7.50082, 8.42036,
  4.98972, 6.67984, 8.73172, 5.10202, 7.49823, 5.18582, 7.12449, 4.05443,
  3.50767, 4.38505, 8.01644, 4.44006, 2.99561, 4.21709, 6.50372, 3.50017,
  5.28147, 4.78341, 3.61185, 3.25538, 6.39773, 3.30708, 3.89415, 4.82296,
  4.50305, 3.89326, 4.83599, 5.25916, 4.85154, 3.65258, 5.1511, 2.01742,
  6.71197, 3.59701, 3.1724, 1.67097, 4.00981, 3.19613, 6.86156, 6.95359,
  9.60213, 0.844471, 6.29556, 6.07853, 5.76423, 8.32176, 5.62158, 5.92687,
  3.243, 9.37627, 4.23388, 7.39837, 6.22972, 4.6924, 2.68526, 5.54399, 4.07378,
  3.39579, 3.73768, 4.9825, 4.43274, 5.10201, 4.52081, 5.2234, 6.66542,
  4.48807, 4.23642, 6.74993, 5.42052, 1.54418, 4.1737, 6.58103, 4.82828,
  5.32847, 6.07235, 7.2047, 3.98773, 4.40188, 2.84301, 0.236788, 3.22326,
  3.84569, 4.7071, 5.15135, 3.11062, 5.9386, 5.63794, 4.65816, 7.58441,
  8.06266, 5.95075, 3.74874, 2.73483, 2.36113, 5.37236, 2.09585, 0.508978,
  6.80538, 6.62286, 7.70545, 7.4582, 1.94947, 4.68611, 3.51575, 8.12867,
  5.54771, 8.19083, 5.50383, 5.26862, 7.55895, 4.28053, 2.29088, 3.3332,
  5.95481, 5.28412, 3.93517, 5.37052, 4.57992, 7.45614, 5.59121, 6.02009,
  4.4418, 7.60805, 6.0609, 2.92094, 5.89248, 3.53504, 5.89799, 9.77334,
  7.68321, 5.37062, 2.04556, 5.15725, 5.36938, 7.83075, 2.23678, 4.02121,
  1.15034, 5.13054, 7.44419, 3.77752, 5.1204, 2.22968, 6.5509, 7.29867,
  6.18985, 5.93272, 4.47361, 6.09199, 4.39804, 5.3835, 6.23449, 6.42367,
  4.96299, 2.92928, 4.0518, 6.88705, 4.1709, 3.3432, 0.511851, 4.68451,
  4.75061, 2.83663, 6.24837, 4.29946, 7.09687, 7.66594, 7.56694, 4.57843,
  4.64643, 7.82482, 6.356, 4.69265, 8.162, 7.81649, 8.65872, 5.86373, 4.70921,
  4.24693, 4.66342, 2.49141, 4.17653, 7.34735, 4.38008, 5.94521, 5.56468,
  4.08431, 4.74258, 5.86022, -1.85978, 3.51743, 5.93235, 5.35807, 3.85488,
  2.35012, 3.86691, 6.89308, 8.16151, 5.09674, 3.37398, 4.19727, 7.26709,
  5.13527, 5.08773, 5.38178, 3.69415, 5.66293, 9.65606, 6.22009, 4.52432,
  3.2538, 4.54497, 4.06816, 3.73261, 6.96567, 7.01588, 6.04046, 2.57347,
  1.32821, 6.00168, 3.81765, 5.86788, 4.9523, 5.28409, 2.26119, 5.0963,
  5.09032, 5.29316, 1.71913, 6.37788, 5.94068, 3.23926, 8.53526, 7.02429,
  4.59062, 2.48875, 5.14832, 7.43138, 3.16851, 3.70853, 3.01713, 2.90889,
  6.53748, 6.61702, 2.58623, 3.56623, 5.53505, 1.88823, 4.91293, 4.02189,
  7.64895, 4.9909, 5.69149, 8.5972, 2.43604, 2.15875, 2.29051, 3.309, 3.60981,
  1.78534, 6.11575, 5.95129, 4.10121, 4.20649, 4.63208, 4.07465, 7.63439,
  7.27775, 4.83269, 3.80429, 2.0175, 1.45599, 7.91738, 3.02743, 5.96941,
  4.72971, 1.95565, 4.39284, 4.69533, 6.09335, 3.7603, 4.36958, 6.39686,
  3.74329, -0.719257, 5.80517, 6.58481, 5.34782, 4.61166, 8.17074, 4.91272,
  7.70741, 4.31262, 6.28935, 6.70485, 6.23791, 3.97396, 3.28389, 7.40242,
  5.18049, 2.90527, 4.01887, 5.57625, 7.73972, 3.94739, 3.70191, 5.34826,
  4.43919, 7.80994, 0.858753, 5.65614, 4.28755, 4.30649, 3.24434, 4.65504,
  4.15359, 5.16435, 3.09593, 2.86485, 1.55218, 7.86139, 2.12655, 7.79252,
  5.62707, 8.56011, 3.4041, 2.95879, 6.87438, 7.31714, 3.92956, 9.11474,
  5.8063, 6.07547, 7.70827, 2.96519, 5.90602, 6.77375, 4.90065, 9.42394,
  3.87292, 3.84383, 3.29207, 4.62121, 5.41019, 5.25179, 6.11872, 6.81866,
  5.52121, 3.65978, 3.25143, 6.22549, 1.86287, 5.84074, 3.98324, 4.80783,
  6.19939, 2.63007, 5.34683, 8.20582, 2.46091, 3.9495, 2.13597, 7.334, 7.23518,
  3.35709, 4.61941, -0.206344, 8.85055, 4.21407, -0.855359, 3.05435, 5.93842,
  5.88788, 2.92766, 3.19058, 5.09454, 4.77409, 3.73474, 4.72688, 3.73385,
  5.16121, 5.15253, -1.36112, 6.74335, 4.45012, 6.78723, 6.37631, 4.76747,
  5.04455, 5.26515, 6.60507, 5.45432, 3.46093, 9.40385, 2.42567, 7.60504,
  2.16268, 8.31174, 4.01556, 5.68965, 7.38373, 5.80868, 5.95082, 5.72224,
  7.49245, 2.95061, 6.23165, 4.47794, 3.47019, 7.28119, 5.94857, 7.43539,
  5.69596, 2.77792, 1.5481, 8.57192, 3.65657, 0.0893697, 2.21467, 8.0728,
  3.53764, 4.77911, 5.20918, 1.48407, 6.68237, 6.43226, 6.4561, 4.73533,
  3.6608, 9.37771, 7.02898, 7.22242, 5.61878, 9.64819, 6.88005, 5.8659,
  5.49964, 2.70673, 4.58835, 9.20703, 3.70766, 5.88163, 2.59, 4.94922, 4.10779,
  6.80733, 3.02813, 4.39404, 4.95296, 7.54018, 7.61272, 3.80833, 5.58334,
  0.894547, 7.01171, 4.95722, 8.01009, 4.83523, 6.92613, 6.41735, 11.6799,
  6.98063, 1.70556, 4.07687, 2.95885, 5.55873, 6.28336, 7.93785, 6.19346,
  5.10574, 4.2168, 1.6319, 3.94133, 3.94378, 5.47871, 5.47194, 3.22415,
  2.62554, 4.86922, 4.721, 7.95853, 5.52369, 8.9492, 5.2741, 5.19134, 5.1768,
  5.53281, 3.46336, 4.24523, 8.70913, 3.85496, 7.13942, 3.00775, 6.4758,
  6.0574, 6.52024, 4.48418, 2.57934, 1.44039, 4.61727, 5.27785, 6.29256,
  8.29256, 5.74279, 7.82765, 4.83024, 7.37482, 6.844, 3.70298, 4.91544,
  4.74993, 3.87689, 6.0759, 4.61479, 4.07145, 0.821766, 4.83651, 1.61745,
  5.30477, 7.50121, 7.32977, 3.30584, 2.08721, 3.95561, 5.20417, 8.40374,
  5.45106, 4.84826, 1.53021, 4.09114, 7.12686, 3.02517, 7.41955, 3.87651,
  3.80455, 3.72479, 3.48949, 3.51769, 3.19402, 4.19563, 6.57734, 3.57764,
  3.89668, 4.22719, 4.36763, 1.69137, 8.6942, 8.20503, 8.56977, 6.09322,
  -0.637552, 6.08215, 0.97352, 5.57641, 4.50738, 5.7928, 6.74727, 3.85033,
  4.45885, 5.7171, 4.64423, 7.43129, 7.75685, 3.38167, 4.19489, 5.53595,
  3.78393, 5.63757, 5.02101, 6.61677, 9.87312, 5.23646, 7.12758, 1.63643,
  2.37183, 4.43367, 4.80517, 4.27198, 6.94916, 3.04285, 3.08749, 4.16994,
  4.36065, 6.53302, 6.71542, 5.26838, 3.51968, 6.8015, 5.08447, 8.6321, 4.395,
  4.92849, 6.50976, 6.11842, 3.72793, 4.64469, 7.34852, 4.19952, 7.86234,
  5.45915, 2.82666, 6.76278, 1.30191, 6.99128, 6.09676, 3.67075, 6.03423,
  2.978, 2.388, 4.36074, 3.75105, 4.9566, 8.47526, 5.96903, 5.78599, 1.59978,
  2.29726, 7.22459, 5.7841, 1.74739, 2.09656, 5.81457, 4.97645, 1.7102,
  7.56544, 2.13258, 6.8806, 5.4227, 5.71605, 4.65652, 3.62235, 5.00275,
  5.58401, 3.37045, 5.90185, 8.32092, 8.58779, 4.47971, 2.17899, 2.25592,
  1.72227, 2.41532, 5.9435, 5.63111, 3.04814, 1.02997, 7.29492, 4.78337,
  7.19591, 5.8598, 6.28245, 8.06428, 5.62791, 5.34644, 4.28204, 4.15513,
  2.82355, 5.47852, 6.58911, 7.58789, 5.41164, 5.14192, 4.81702, 6.66208,
  4.20404, 5.69093, 6.86509, 5.53008, 5.39996, 4.56057, 4.56721, 2.85369,
  6.61719, 4.89398, 4.6822, 6.13788, 7.68084, 2.91986, 4.49653, 3.20342,
  4.93089, 4.55225, 5.41249, 6.42857, 5.51682, 3.29427, 7.97546, 7.62771,
  2.32264, 4.56895, 4.96061, 5.43891, 6.44532, 4.03885, 5.84576, 4.16904,
  5.98563, 4.03915, 0.793471, 4.80523, 0.92322, 7.89861, 5.49705, 4.80187,
  3.87224, 7.37531, 4.49513, 9.74382, 5.63602, 3.24546, 3.29004, 6.1002,
  4.65807, 2.69044, 4.86215, 3.27282, 5.60158, 4.03005, 4.60813, 5.86604,
  3.74994, 3.13856, 6.19709, 5.68133, 3.30718, 2.42523, 5.44097, 5.96415,
  2.48758, 7.60317, 1.92689, 3.92355, 1.87917, 4.95979, 4.6911, 4.56508,
  2.62219, -2.87777, 7.86606, 4.818, 3.14513, 2.97841, 7.59629, 7.66463,
  4.32179, 4.05165, 3.90666, 7.53286, 2.10711, 3.77619, 2.39828, 4.19864,
  1.86742, 4.31685, 4.78505, 3.84629, 4.8424, 4.56138, 6.7049, 5.02507,
  3.95096, 5.7112, 4.73016, 5.27046, 4.90314, 6.16746, 3.44775, 5.09592,
  3.3435, 8.02736, 5.46715, 8.53743, 6.71337, 5.87085, 7.03983, 8.54071,
  3.75082, 7.34855, 3.95262, 6.82673, 2.49217, 6.53543, 4.98617, 5.62474,
  2.80112, 6.01723, 0.0694797, 5.63711, 2.39577, 3.04181, 4.24012, 5.61344,
  4.13018, 4.28689, 3.7095, 4.67208, 2.62393, 5.16096, 6.41888, 7.89494,
  1.37915, 5.85736, 5.16857, 3.45744, 2.07276, 5.5177, 8.8515, 6.13051,
  5.30615, 5.06284, 1.5989, 5.06823, 3.77233, 2.99172, 4.00895, 3.67266,
  6.4341, 2.77406, 8.68717, 3.6669, 5.68894, 1.08086, 2.41837, 4.26576, 5.6719,
  0.492854, 7.52934, 4.30026, 3.57599, 7.38991, 5.23278, 7.52467, 4.74095,
  6.11842, 2.12583, 4.20263, 4.74757, 5.59501, 4.67303, 6.95785, 4.05997,
  3.64842, 2.92589, 6.47512, 4.61382, 5.50026, 6.78236, 6.27505, 5.97951,
  -1.82885, 5.26204, 4.49941, 5.04193, 7.02138, 4.65382, 3.1788, 2.94788,
  5.3131, 5.45827, 6.07477, 4.26374, 6.14667, 2.32702, 1.17271, 5.94781,
  5.4391, 2.73155, 2.55975, 2.93167, 3.33396, 4.61173, 2.81963, 3.87476,
  2.23537, 7.38507, 3.84536, 5.67642, 7.2499, 6.83749, 2.40383, 1.27264,
  6.68727, 8.10375, 3.85793, 5.17612, 8.56567, 1.4189, 4.5354, 6.06433,
  8.96602, 6.87336, 5.07189, -0.118159, 6.98368, 5.30381, 4.87085, 5.2852,
  6.42691, 1.76195, 1.45738, 4.89108, 5.60742, 6.82132, 4.40388, 7.42837,
  5.94613, 6.02781, 4.46535, 5.44678, 6.71309, 4.61042, 3.29414, 6.4776, 4.078,
  7.02426, 4.11593, 5.00664, 3.4688, 2.72931, 8.6061, 4.7597, 4.4962, 4.44859,
  3.57059, 4.54498, 3.26229, 3.3446, 7.55553, 8.75224, 3.88607, 5.87432,
  8.19818, 1.2357, 1.86652, 4.4988, 7.63985, 4.78692, 7.17055, 5.12897,
  3.73265, 7.32454, 0.278767, 5.74536, 1.65048, 6.38701, 3.34414, 4.28991,
  4.89895, 3.57942, 5.31444, 5.94905, 4.60492, 5.98302, 6.10979, 2.89463,
  6.56843, 4.88483, 4.42739, 6.81221, 7.81451, 4.69063, 4.10333, 3.06819,
  4.88068, 5.81462, 1.58257, 3.9121, 6.12635, 8.78667, 6.69784, 3.53529,
  6.19581, 3.92409, 6.29664, 7.16557, 6.29377, 2.00642, 4.4705, 3.39437,
  5.32856, 2.8457, 7.75769, 7.08185, 5.63919, 6.8111, 6.26974, 6.78038,
  2.88262, 2.9859, 2.28388, 8.32226, 9.46957, 7.73052, 3.60337, 5.168, 4.3623,
  3.3197, 2.79538, 4.81986, 3.2358, 7.46349, 6.31862, 6.44804, 5.05225,
  6.81572, 5.4812, 7.04616, 4.10224, 4.0161, 3.35, 4.42866, 5.63674, 4.42226,
  1.78676, 6.11244, 6.11524, 8.22294, 4.49981, 4.67067, 5.95926, 6.70434,
  11.0039, 0.335186, 2.73683, 4.93103, 6.30083, 2.59042, 6.08144, 5.9186,
  8.69565, 4.59627, 4.36767, 9.54454, 2.20168, 3.52513, 2.31048, 1.6813,
  3.89515, 6.04146, 4.91264, 1.94291, 3.02306, 6.30125, 0.83211, 6.13503,
  4.53398, 4.99662, 4.13176, 6.03908, 6.28578, 5.00415, 3.59983, 5.3254,
  4.70687, 3.41629, 6.01489, 4.75176, 3.53099, 6.51798, 4.62635, 7.21345,
  6.89908, 7.83249, 1.67481, 3.73662, 3.33939, 3.70995, 4.91657, 3.27805,
  6.63659, 3.88691, 1.99101, 3.62826, 7.00586, 6.41175, 3.0806, 4.37796,
  4.85521, 8.07828, 3.86559, 6.61604, 7.82744, 8.37837, 1.5633, 0.621153,
  6.29814, 4.10442, 8.35646, 4.70844, 3.77818, 2.83656, 4.89627, 4.05077,
  5.00674, 7.29834, 6.55015, 7.93492, 5.72633, 3.43446, 4.62446, 6.20418,
  6.26935, 2.9365, 4.03655, 1.69872, 4.22273, 8.4557, 6.48872, 3.90171,
  3.95554, 5.6754, 3.97698, 6.92943, 2.7661, 2.8725, 3.68737, 3.67129, 6.69593,
  6.81706, 0.942727, 2.94317, 6.31659, 6.29217, 2.43746, 4.95822, 7.22134,
  3.4103, 3.55855, 4.10681, 7.32929, 4.58522, 4.79207, 5.1911, 3.88779,
  4.14307, 6.66488, 4.04003, 5.92645, 4.93404, 3.6853, 3.98096, 8.1296,
  5.66805, 8.25337, 3.44379, 7.12589, 1.54415, 3.94696, 5.37794, 5.33825,
  2.57136, 6.50037, 6.05919, 4.89663, 4.61857, 7.55123, 4.98478, 5.4194,
  3.10985, 7.79868, 6.34877, 2.52221, 5.71573, 5.21283, 9.89345, 3.92674,
  4.37004, 4.44763, 3.49115, 9.55607, 4.58821, 5.90829, 4.67137, 3.75901,
  7.26334, 1.95873, 4.54719, 4.54397, 4.69103, 2.34471, 4.45822, 5.46916,
  3.81144, 0.665101, 5.56835, 6.54021, 3.22443, 3.58911, 2.03106, 3.47693,
  3.61788, 2.22159, 3.29258, 3.67115, 11.1248, 7.57798, 6.42064, 3.869,
  1.29855, 1.62834, 5.5778, 2.619, 3.15966, 7.64469, 4.8359, 3.98527, 3.43425,
  4.13805, 0.877373, 9.95864, 7.43193, 4.57666, 2.8057, 4.73555, 4.53267,
  9.34185, 4.55719, 3.39077, 4.5737, 4.6596, 9.42609, 5.11162, 4.88977,
  7.33743, 6.10679, 8.289, 5.08361, 4.25441, 3.32868, 5.98685, 8.59356,
  2.93206, 5.73726, 4.26488, 3.47597, 6.14645, 4.23291, 7.1672, 5.83981,
  4.59251, 5.19758, 6.86332, 4.01371, 1.11021, 3.90573, 3.50124, 4.5392,
  1.70502, 5.3224, 8.97542, 5.29589, 1.19649, 4.80147, 1.7406, 6.33021,
  6.74532, 5.39953, 2.46091, 8.58345, 4.71101, 4.9776, 6.62053, 1.75881,
  5.11408, 3.69121, 4.03547, 5.34517, 4.35105, 7.4269, 1.41258, 6.61226,
  6.06037, 2.24485, 3.30192, 6.20633, 4.50723, 8.70982, 7.39544, 6.66121,
  7.61361, 5.02429, 4.10531, 4.7366, 5.2554, 6.20236};


