﻿/* Distributed under the Apache License, Version 2.0.
   See accompanying NOTICE file for details.*/

using UnityEngine;

// Abstract class for any pulse data consumer that holds a reference
// to the pulse data source and the data field to consume
public abstract class PulseDataConsumer : MonoBehaviour
{
  [HideInInspector]
  public PulseDataSource source;  // Source of the data to consume
  [HideInInspector]
  public int dataFieldIndex;      // Index of the data field to consume


  // MARK: Monobehavior methods

  // We need to consume the data after it has been generated by
  // a PulseDataSource, which occurs in an Update() call
  protected virtual void Update()
  {
    // Ensure we only read data when the application is running
    if (!Application.isPlaying)
      return;

    // Ensure there is data to consume
    if (source == null ||
        source.data == null ||
        source.data.timeStampList == null ||
        source.data.timeStampList.IsEmpty() ||
        dataFieldIndex >= source.data.valuesTable.Count)
      return;

    // Forward time points and values for the data field being consumed
    var dataFieldValues = source.data.valuesTable[dataFieldIndex];
    UpdateFromPulse(source.data.timeStampList, dataFieldValues);
  }


  // MARK: Custom methods

  // Abstract method exposing time points and data field values
  // to be consumed by the implemented subclass
  abstract internal void UpdateFromPulse(DoubleList times, DoubleList values);
}
