/* Distributed under the Apache License, Version 2.0.
See accompanying NOTICE file for details.*/
#include "actions/SubstanceInfusionWidget.h"
#include "controls/ScalarQuantityWidget.h"
#include "controls/LabeledComboBox.h"

#include "PulsePhysiologyEngine.h"
#include "cdm/CommonDataModel.h"
#include "PhysiologyEngine.h"
#include "cdm/patient/actions/SESubstanceInfusion.h"
#include "cdm/properties/SEScalarVolumePerTime.h"
#include "cdm/properties/SEScalarMassPerVolume.h"
#include "cdm/substance/SESubstance.h"
#include "cdm/substance/SESubstanceManager.h"

class QSubstanceInfusionWidget::Controls
{
public:
  Controls() {}
  SESubstanceInfusion* Action = nullptr;
  std::map<std::string, SESubstance*>       Substances;
  QScalarQuantityWidget<VolumePerTimeUnit>* Rate;
  QScalarQuantityWidget<MassPerVolumeUnit>* Concentration;
  QLabeledComboBox*                         Drugs;
};

QSubstanceInfusionWidget::QSubstanceInfusionWidget(QPulse& qp, QWidget *parent, Qt::WindowFlags flags) : QActionWidget(qp, parent, flags)
{
  m_Controls = new Controls();
  m_Controls->Rate = new QScalarQuantityWidget<VolumePerTimeUnit>("Rate", 0, 100, 1, VolumePerTimeUnit::mL_Per_s, ScalarOptionWidget::None, this); //check this
  m_Controls->Concentration = new QScalarQuantityWidget<MassPerVolumeUnit>("Concentration", 0, 1000, 1, MassPerVolumeUnit::mg_Per_mL, ScalarOptionWidget::None, this);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::mg_Per_L);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::g_Per_mL);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::g_Per_L);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::kg_Per_L);

  std::vector<QString> drugs = { "Epinephrine", "Fentanyl","Furosemide",
                                  "Ketamine", "Midazolam","Morphine",
                                  "Naloxone", "Pralidoxime","Prednisone",
                                  "Propofol", "Rocuronium","Succinylcholine" };
  m_Controls->Drugs = new QLabeledComboBox(this, "Drugs", drugs, 120);
  for (QString d : drugs)
  {
    SESubstance* drug;
    drug = qp.GetEngine().GetSubstanceManager().GetSubstance(d.toStdString());
    m_Controls->Substances[drug->GetName()] = drug;
  }

  Properties()->layout()->addWidget(m_Controls->Drugs);
  Properties()->layout()->addWidget(m_Controls->Rate);
  Properties()->layout()->addWidget(m_Controls->Concentration);
  Reset();
}

QSubstanceInfusionWidget::~QSubstanceInfusionWidget()
{
  SAFE_DELETE(m_Controls->Action);
  delete m_Controls;
}

void QSubstanceInfusionWidget::Reset()
{
  QActionWidget::Reset();
  SAFE_DELETE(m_Controls->Action);
}

void QSubstanceInfusionWidget::SetEnabled(bool b)
{
  QActionWidget::SetEnabled(b);
  m_Controls->Rate->setEnabled(b);
  m_Controls->Rate->enableInput(b);
  m_Controls->Concentration->setEnabled(b);
  m_Controls->Concentration->enableInput(b);
  m_Controls->Drugs->SetEnabled(b);
}

void QSubstanceInfusionWidget::ControlsToAction()
{
  QActionWidget::ControlsToAction();
  m_Controls->Action = new SESubstanceInfusion(*m_Controls->Substances[m_Controls->Drugs->GetText()]);
  m_Controls->Rate->getValue(m_Controls->Action->GetRate());
  m_Controls->Concentration->getValue(m_Controls->Action->GetConcentration());
}

void QSubstanceInfusionWidget::AtSteadyState(PhysiologyEngine& pulse)
{
  // Nothing expected here
}
void QSubstanceInfusionWidget::AtSteadyStateUpdateUI()
{
  // Nothing expected here
}

void QSubstanceInfusionWidget::ProcessPhysiology(PhysiologyEngine& pulse)
{// This is called from a thread, you should NOT update UI here
  // This is where we pull data from pulse, and push any actions to it
  if (m_Controls->Action != nullptr)
  {
    if (m_Controls->Action->IsValid())
      pulse.ProcessAction(*m_Controls->Action);
    SAFE_DELETE(m_Controls->Action);
  }
}
void QSubstanceInfusionWidget::PhysiologyUpdateUI()
{// This is called from a slot, you can update UI here

  // Nothing is expected to be in the environment
}