/* Distributed under the Apache License, Version 2.0.
See accompanying NOTICE file for details.*/
#include "actions/SubstanceBolusWidget.h"
#include "controls/ScalarQuantityWidget.h"
#include "controls/LabeledComboBox.h"

#include "PhysiologyEngine.h"
#include "cdm/CommonDataModel.h"
#include "cdm/patient/actions/SESubstanceBolus.h"
#include "cdm/properties/SEScalarVolume.h"
#include "cdm/properties/SEScalarMassPerVolume.h"
#include "cdm/substance/SESubstance.h"
#include "cdm/substance/SESubstanceManager.h"

#include <QString>

class QSubstanceBolusWidget::Controls // based off of chronic anemia widget
{
public:
  Controls() {}
  SESubstanceBolus* Action = nullptr;
  std::map<std::string, SESubstance*>       Substances;
  QScalarQuantityWidget<VolumeUnit>*        Dose;
  QScalarQuantityWidget<MassPerVolumeUnit>* Concentration;
  QLabeledComboBox*                         AdministrationRoute;
  QLabeledComboBox*                         Drugs;
};

QSubstanceBolusWidget::QSubstanceBolusWidget(QPulse& qp, QWidget *parent, Qt::WindowFlags flags) : QActionWidget(qp, parent, flags)
{
  m_Controls = new Controls();
  m_Controls->Dose = new QScalarQuantityWidget<VolumeUnit>("Dose", 0, 1000, 1, VolumeUnit::mL, ScalarOptionWidget::None, this);
  m_Controls->Concentration = new QScalarQuantityWidget<MassPerVolumeUnit>("Concentration", 0, 10000, 1, MassPerVolumeUnit::ug_Per_mL, ScalarOptionWidget::None, this);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::mg_Per_mL);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::mg_Per_L);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::g_Per_mL);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::g_Per_L);
  m_Controls->Concentration->addUnit(MassPerVolumeUnit::kg_Per_L);

  std::vector<QString> adminOptions = { "Intravenous", "Intraarterial","Intramuscular" };
  m_Controls->AdministrationRoute = new QLabeledComboBox(this, "AdministrationRoute", adminOptions, 85);


  std::vector<QString> drugs = { "Epinephrine", "Fentanyl","Furosemide",
                                 "Ketamine", "Midazolam","Morphine",
                                 "Naloxone", "Pralidoxime","Prednisone", 
                                 "Propofol", "Rocuronium","Succinylcholine" };
  m_Controls->Drugs = new QLabeledComboBox(this, "Drugs", drugs, 120);
  for (QString d : drugs)
  {
    SESubstance* drug;
    drug = qp.GetEngine().GetSubstanceManager().GetSubstance(d.toStdString());
    m_Controls->Substances[drug->GetName()] = drug;
  }
 
  SetEnabled(false);

  Properties()->layout()->addWidget(m_Controls->Drugs);
  Properties()->layout()->addWidget(m_Controls->Dose);
  Properties()->layout()->addWidget(m_Controls->AdministrationRoute);
  Properties()->layout()->addWidget(m_Controls->Concentration);
  Reset();
}

QSubstanceBolusWidget::~QSubstanceBolusWidget()
{
  SAFE_DELETE(m_Controls->Action);
  delete m_Controls;
}

void QSubstanceBolusWidget::Reset()
{
  QActionWidget::Reset();
  SAFE_DELETE(m_Controls->Action);
}

void QSubstanceBolusWidget::SetEnabled(bool b)
{
  QActionWidget::SetEnabled(b);
  m_Controls->Dose->setEnabled(b);
  m_Controls->Dose->enableInput(b);
  m_Controls->Concentration->setEnabled(b);
  m_Controls->Concentration->enableInput(b);
  m_Controls->AdministrationRoute->SetEnabled(b);
  m_Controls->Drugs->SetEnabled(b);
}

void QSubstanceBolusWidget::ControlsToAction()
{
  QActionWidget::ControlsToAction();
  m_Controls->Action = new SESubstanceBolus(*m_Controls->Substances[m_Controls->Drugs->GetText()]);
  switch (m_Controls->AdministrationRoute->GetIndex())
  {
  case 0:
    m_Controls->Action->SetAdminRoute(eSubstanceAdministration_Route::Intravenous);
    break;
  case 1:
    m_Controls->Action->SetAdminRoute(eSubstanceAdministration_Route::Intraarterial);
    break;
  case 2:
    m_Controls->Action->SetAdminRoute(eSubstanceAdministration_Route::Intramuscular);
    break;
  }
  m_Controls->Dose->getValue(m_Controls->Action->GetDose());
  m_Controls->Concentration->getValue(m_Controls->Action->GetConcentration());
}

void QSubstanceBolusWidget::AtSteadyState(PhysiologyEngine& pulse)
{
  // Nothing expected here
}
void QSubstanceBolusWidget::AtSteadyStateUpdateUI()
{
  // Nothing expected here
}

void QSubstanceBolusWidget::ProcessPhysiology(PhysiologyEngine& pulse)
{// This is called from a thread, you should NOT update UI here
  // This is where we pull data from pulse, and push any actions to it
  if (m_Controls->Action != nullptr)
  {
    if (m_Controls->Action->IsValid())
      pulse.ProcessAction(*m_Controls->Action);
    SAFE_DELETE(m_Controls->Action);
  }
}
void QSubstanceBolusWidget::PhysiologyUpdateUI()
{// This is called from a slot, you can update UI here

  // Nothing is expected to be in the environment
}