/* Distributed under the Apache License, Version 2.0.
See accompanying NOTICE file for details.*/
#include "actions/HemorrhageWidget.h"
#include "controls/ScalarQuantityWidget.h"
#include "controls/LabeledComboBox.h"

#include "CommonDataModel.h"
#include "PhysiologyEngine.h"
#include "properties/SEScalarVolumePerTime.h"


class QHemorrhageWidget::Controls // based off of chronic anemia widget
{
public:
  Controls(QPulse& qp) : Pulse(qp) {}
  QPulse&                                   Pulse;
  SEHemorrhage                              Action;

  QLabeledComboBox*                         Type;
  QScalarQuantityWidget<VolumePerTimeUnit>* Rate;
  QLabeledComboBox*                         ExternalCmpts;
  QLabeledComboBox*                         InternalCmpts;
};

QHemorrhageWidget::QHemorrhageWidget(QPulse& qp, QWidget *parent, Qt::WindowFlags flags) : QActionWidget(qp, parent, flags)
{
  m_Controls = new Controls(qp);
  
  std::vector<QString> typeOptions = {"External", "Internal"};
  m_Controls->Type = new QLabeledComboBox(this, "Type", typeOptions);
  Properties()->layout()->addWidget(m_Controls->Type);

  m_Controls->Rate = new QScalarQuantityWidget<VolumePerTimeUnit>("Rate", 0, 1000, 1, VolumePerTimeUnit::mL_Per_min, ScalarOptionWidget::None, this);
  m_Controls->Rate->AddUnit(VolumePerTimeUnit::mL_Per_min);
  m_Controls->Rate->AddUnit(VolumePerTimeUnit::L_Per_s);
  m_Controls->Rate->AddUnit(VolumePerTimeUnit::L_Per_min);
  Properties()->layout()->addWidget(m_Controls->Rate);

  std::vector<QString> externalCmpts = { "Brain", "RightLeg", "LeftLeg", "RightLeg", "LeftArm", 
                                         "LeftKidney", "RightKidney", "Liver", "Spleen", "Splanchnic",
                                         "SmallIntestine", "LargeIntestine", "Aorta", "VenaCava" };
  m_Controls->ExternalCmpts = new QLabeledComboBox(this, "Compartments", externalCmpts, 120);
  Properties()->layout()->addWidget(m_Controls->ExternalCmpts);
  m_Controls->ExternalCmpts->show();

  std::vector<QString> internalCmpts = { "LeftKidney", "RightKidney", "Liver", "Spleen", "Splanchnic", "SmallIntestine", "LargeIntestine", "Aorta", "VenaCava" };
  m_Controls->InternalCmpts = new QLabeledComboBox(this, "Compartments", internalCmpts, 120);
  Properties()->layout()->addWidget(m_Controls->InternalCmpts);
  m_Controls->InternalCmpts->hide();

  Properties()->layout()->addWidget(GetProcessTimeCtrl());

  connect(m_Controls->Type->GetComboBox(), SIGNAL(currentIndexChanged(int)), this, SLOT(ChangeType()));
  Reset();
}

QHemorrhageWidget::~QHemorrhageWidget()
{
  delete m_Controls;
}

void QHemorrhageWidget::Reset()
{
  QActionWidget::Reset();
  m_Controls->Action.Clear();
}

SEAction& QHemorrhageWidget::GetAction()
{
  return m_Controls->Action;
}
const SEAction& QHemorrhageWidget::GetAction() const
{
  return m_Controls->Action;
}

void QHemorrhageWidget::SetEnabled(bool b)
{
  QActionWidget::SetEnabled(b);
  m_Controls->Type->SetEnabled(b);
  m_Controls->Rate->EnableInput(b);;
  m_Controls->Rate->setEnabled(b);
  if (!b)
  {
    m_Controls->ExternalCmpts->SetEnabled(b);
    m_Controls->InternalCmpts->SetEnabled(b);
  }
  else
    ChangeType();
}

void QHemorrhageWidget::ChangeType()
{
  if (m_Controls->Type->GetText() == "External")
  {
    m_Controls->ExternalCmpts->show();
    m_Controls->ExternalCmpts->SetEnabled(true);
    m_Controls->InternalCmpts->hide();
    m_Controls->InternalCmpts->SetEnabled(false);
  }
  else
  {
    m_Controls->ExternalCmpts->hide();
    m_Controls->ExternalCmpts->SetEnabled(false);
    m_Controls->InternalCmpts->show();
    m_Controls->InternalCmpts->SetEnabled(true);
  }
}

void QHemorrhageWidget::ControlsToAction()
{
  QActionWidget::ControlsToAction();

  std::string cmpt;
  if (m_Controls->Type->GetText() == "External")
  {
    cmpt = m_Controls->ExternalCmpts->GetText();
    m_Controls->Action.SetType(eHemorrhage_Type::External);
  }
  else
  {
    cmpt = m_Controls->InternalCmpts->GetText();
    m_Controls->Action.SetType(eHemorrhage_Type::Internal);
  }
  m_Controls->Action.SetCompartment(cmpt);
  m_Controls->Rate->GetValue(m_Controls->Action.GetRate());
  emit UpateAction(m_Controls->Action, GetProcessTime());
}


void QHemorrhageWidget::ActionToControls(const SEHemorrhage& action)
{
  QComboBox* typeBox = m_Controls->Type->GetComboBox();
  switch (action.GetType())
  {
  case eHemorrhage_Type::External:
    typeBox->setCurrentIndex(typeBox->findText("External"));
    m_Controls->ExternalCmpts->GetComboBox()->setCurrentIndex(
      m_Controls->ExternalCmpts->GetComboBox()->findText(QString::fromStdString(action.GetCompartment()))
    );
    break;
  default:
    typeBox->setCurrentIndex(typeBox->findText("Internal"));
    m_Controls->InternalCmpts->GetComboBox()->setCurrentIndex(
      m_Controls->InternalCmpts->GetComboBox()->findText(QString::fromStdString(action.GetCompartment()))
    );
    break;
  }
  SEScalarQuantity< VolumePerTimeUnit>(rateQuantity);
  rateQuantity.SetValue(action.GetRate(VolumePerTimeUnit::mL_Per_min), VolumePerTimeUnit::mL_Per_min);
  m_Controls->Rate->SetValue(rateQuantity);
}
