/* Distributed under the Apache License, Version 2.0.
See accompanying NOTICE file for details.*/
#include "ConditionsEditorWidget.h"
#include "ui_ConditionsEditor.h"
#include <QMessageBox>

#include "CollapsableWidget.h"

#include "conditions/ConditionWidget.h"
#include "conditions/ChronicAnemiaWidget.h"
#include "conditions/ChronicObstructivePulmonaryDiseaseWidget.h"
#include "conditions/ChronicPericardialEffusionWidget.h"
#include "conditions/ChronicRenalStenosisWidget.h"
#include "conditions/ChronicVentricularSystolicDysfunctionWidget.h"
#include "conditions/ImpairedAlveolarExchangeWidget.h"
#include "conditions/LobarPneumoniaWidget.h"

#include "cdm/CommonDataModel.h"
#include "cdm/PhysiologyEngine.h"
#include "cdm/engine/SEPatientConfiguration.h"
#include "cdm/engine/SEConditionManager.h"
#include "cdm/engine/SECondition.h"
#include "cdm/patient/conditions/SEChronicAnemia.h"
#include "cdm/patient/conditions/SEChronicObstructivePulmonaryDisease.h"
#include "cdm/patient/conditions/SEChronicPericardialEffusion.h"
#include "cdm/patient/conditions/SEChronicRenalStenosis.h"
#include "cdm/patient/conditions/SEChronicVentricularSystolicDysfunction.h"
#include "cdm/patient/conditions/SEImpairedAlveolarExchange.h"
#include "cdm/patient/conditions/SELobarPneumonia.h"

class QConditionsEditorWidget::Controls : public Ui::ConditionsEditorWidget
{
public:
  Controls(QPulse& qp) : Pulse(qp) {}
  QPulse&                Pulse;

  QChronicAnemiaWidget*                          ChronicAnemia;
  QCollapsableWidget*                            CWChronicAnemia;

  QChronicObstructivePulmonaryDiseaseWidget*     ChronicObstructivePulmonaryDisease;
  QCollapsableWidget*                            CWChronicObstructivePulmonaryDisease;

  QChronicPericardialEffusionWidget*             ChronicPericardialEffusion;
  QCollapsableWidget*                            CWChronicPericardialEffusion;

  QChronicRenalStenosisWidget*                   ChronicRenalStenosis;
  QCollapsableWidget*                            CWChronicRenalStenosis;

  QChronicVentricularSystolicDysfunctionWidget*  ChronicVentricularSystolicDysfunction;
  QCollapsableWidget*                            CWChronicVentricularSystolicDysfunction;

  QImpairedAlveolarExchangeWidget*               ImpairedAlveolarExchange;
  QCollapsableWidget*                            CWImpairedAlveolarExchange;

  QLobarPneumoniaWidget*                         LobarPneumonia;
  QCollapsableWidget*                            CWLobarPneumonia;
};

QConditionsEditorWidget::QConditionsEditorWidget(QPulse& qp, QWidget *parent, Qt::WindowFlags flags) : QDockWidget(parent,flags)
{
  m_Controls = new Controls(qp);
  m_Controls->setupUi(this);

  m_Controls->CWChronicAnemia = new QCollapsableWidget("Anemia", 120, this);
  m_Controls->ChronicAnemia = new QChronicAnemiaWidget(this);
  m_Controls->CWChronicAnemia->setVisible(true);
  m_Controls->CWChronicAnemia->layout()->addWidget(m_Controls->ChronicAnemia);
  m_Controls->CWChronicAnemia->setContentLayout(*m_Controls->ChronicAnemia->layout());
  m_Controls->ConditionWidgets->layout()->addWidget(m_Controls->CWChronicAnemia);

  m_Controls->CWChronicObstructivePulmonaryDisease = new QCollapsableWidget("COPD", 120, this);
  m_Controls->ChronicObstructivePulmonaryDisease = new QChronicObstructivePulmonaryDiseaseWidget(this);
  m_Controls->CWChronicObstructivePulmonaryDisease->setVisible(true);
  m_Controls->CWChronicObstructivePulmonaryDisease->layout()->addWidget(m_Controls->ChronicObstructivePulmonaryDisease);
  m_Controls->CWChronicObstructivePulmonaryDisease->setContentLayout(*m_Controls->ChronicObstructivePulmonaryDisease->layout());
  m_Controls->ConditionWidgets->layout()->addWidget(m_Controls->CWChronicObstructivePulmonaryDisease);

  m_Controls->CWChronicPericardialEffusion = new QCollapsableWidget("Pericardial Effusion", 120, this);
  m_Controls->ChronicPericardialEffusion = new QChronicPericardialEffusionWidget(this);
  m_Controls->CWChronicPericardialEffusion->setVisible(true);
  m_Controls->CWChronicPericardialEffusion->layout()->addWidget(m_Controls->ChronicPericardialEffusion);
  m_Controls->CWChronicPericardialEffusion->setContentLayout(*m_Controls->ChronicPericardialEffusion->layout());
  m_Controls->ConditionWidgets->layout()->addWidget(m_Controls->CWChronicPericardialEffusion);

  m_Controls->CWChronicRenalStenosis = new QCollapsableWidget("Renal Stenosis", 120, this);
  m_Controls->ChronicRenalStenosis = new QChronicRenalStenosisWidget(this);
  m_Controls->CWChronicRenalStenosis->setVisible(true);
  m_Controls->CWChronicRenalStenosis->layout()->addWidget(m_Controls->ChronicRenalStenosis);
  m_Controls->CWChronicRenalStenosis->setContentLayout(*m_Controls->ChronicRenalStenosis->layout());
  m_Controls->ConditionWidgets->layout()->addWidget(m_Controls->CWChronicRenalStenosis);
  
  m_Controls->CWChronicVentricularSystolicDysfunction = new QCollapsableWidget("Ventricular Systolic Dysfunction", 120, this);
  m_Controls->ChronicVentricularSystolicDysfunction = new QChronicVentricularSystolicDysfunctionWidget(this);
  m_Controls->CWChronicVentricularSystolicDysfunction->setVisible(true);
  m_Controls->CWChronicVentricularSystolicDysfunction->layout()->addWidget(m_Controls->ChronicVentricularSystolicDysfunction);
  m_Controls->CWChronicVentricularSystolicDysfunction->setContentLayout(*m_Controls->ChronicVentricularSystolicDysfunction->layout());
  m_Controls->ConditionWidgets->layout()->addWidget(m_Controls->CWChronicVentricularSystolicDysfunction);

  m_Controls->CWImpairedAlveolarExchange = new QCollapsableWidget("Impaired Alveolar Exchange", 120, this);
  m_Controls->ImpairedAlveolarExchange = new QImpairedAlveolarExchangeWidget(this);
  m_Controls->CWImpairedAlveolarExchange->setVisible(true);
  m_Controls->CWImpairedAlveolarExchange->layout()->addWidget(m_Controls->ImpairedAlveolarExchange);
  m_Controls->CWImpairedAlveolarExchange->setContentLayout(*m_Controls->ImpairedAlveolarExchange->layout());
  m_Controls->ConditionWidgets->layout()->addWidget(m_Controls->CWImpairedAlveolarExchange);

  m_Controls->CWLobarPneumonia = new QCollapsableWidget("Lobar Pneumonia", 120, this);
  m_Controls->LobarPneumonia = new QLobarPneumoniaWidget(this);
  m_Controls->CWLobarPneumonia->setVisible(true);
  m_Controls->CWLobarPneumonia->layout()->addWidget(m_Controls->LobarPneumonia);
  m_Controls->CWLobarPneumonia->setContentLayout(*m_Controls->LobarPneumonia->layout());
  m_Controls->ConditionWidgets->layout()->addWidget(m_Controls->CWLobarPneumonia);

  QSpacerItem *spacer = new QSpacerItem(40, 1000, QSizePolicy::Expanding, QSizePolicy::Expanding);
  m_Controls->ConditionWidgets->layout()->addItem(spacer);
}

QConditionsEditorWidget::~QConditionsEditorWidget()
{
  delete m_Controls;
}

void QConditionsEditorWidget::Reset()
{
  m_Controls->Pulse.GetPatientConfiguration().GetConditions().Clear();

  m_Controls->ChronicAnemia->reset();
  m_Controls->ChronicObstructivePulmonaryDisease->reset();
  m_Controls->ChronicPericardialEffusion->reset();
  m_Controls->ChronicRenalStenosis->reset();
  m_Controls->ChronicVentricularSystolicDysfunction->reset();
  m_Controls->ImpairedAlveolarExchange->reset();
  m_Controls->LobarPneumonia->reset();

  m_Controls->CWChronicAnemia->expand(false);
  m_Controls->CWChronicObstructivePulmonaryDisease->expand(false);
  m_Controls->CWChronicPericardialEffusion->expand(false);
  m_Controls->CWChronicRenalStenosis->expand(false);
  m_Controls->CWChronicVentricularSystolicDysfunction->expand(false);
  m_Controls->CWImpairedAlveolarExchange->expand(false);
  m_Controls->CWLobarPneumonia->expand(false);

  EnableInput(true);
}

void QConditionsEditorWidget::EnableInput(bool b)
{
  EnableConverter(!b);
  m_Controls->ChronicAnemia->setEnabled(b);
  m_Controls->ChronicObstructivePulmonaryDisease->setEnabled(b);
  m_Controls->ChronicPericardialEffusion->setEnabled(b);
  m_Controls->ChronicRenalStenosis->setEnabled(b);
  m_Controls->ChronicVentricularSystolicDysfunction->setEnabled(b);
  m_Controls->ImpairedAlveolarExchange->setEnabled(b);
  m_Controls->LobarPneumonia->setEnabled(b);
}

void QConditionsEditorWidget::EnableConverter(bool b)
{
  m_Controls->ChronicAnemia->enableConverter(b);
  m_Controls->ChronicObstructivePulmonaryDisease->enableConverter(b);
  m_Controls->ChronicPericardialEffusion->enableConverter(b);
  m_Controls->ChronicRenalStenosis->enableConverter(b);
  m_Controls->ChronicVentricularSystolicDysfunction->enableConverter(b);
  m_Controls->ImpairedAlveolarExchange->enableConverter(b);
  m_Controls->LobarPneumonia->enableConverter(b);
}

void QConditionsEditorWidget::ControlsToConditions()
{
  const SECondition* c;
  SEConditionManager& Conditions = m_Controls->Pulse.GetPatientConfiguration().GetConditions();
  Conditions.Clear();
  // Pull Each Condition if active
  if (m_Controls->ChronicAnemia->isSelected())
  {
    c = &m_Controls->ChronicAnemia->getCondition();
    if (c->IsActive())
      Conditions.ProcessCondition(*c);
  }
  if (m_Controls->ChronicObstructivePulmonaryDisease->isSelected())
  {
    c = &m_Controls->ChronicObstructivePulmonaryDisease->getCondition();
    if (c->IsActive())
      Conditions.ProcessCondition(*c);
  }
  if (m_Controls->ChronicPericardialEffusion->isSelected())
  {
    c = &m_Controls->ChronicPericardialEffusion->getCondition();
    if (c->IsActive())
      Conditions.ProcessCondition(*c);
  }
  if (m_Controls->ChronicRenalStenosis->isSelected())
  {
    c = &m_Controls->ChronicRenalStenosis->getCondition();
    if (c->IsActive())
      Conditions.ProcessCondition(*c);
  }
  if (m_Controls->ChronicVentricularSystolicDysfunction->isSelected())
  {
    c = &m_Controls->ChronicVentricularSystolicDysfunction->getCondition();
    if (c->IsActive())
      Conditions.ProcessCondition(*c);
  }
  if (m_Controls->ImpairedAlveolarExchange->isSelected())
  {
    c = &m_Controls->ImpairedAlveolarExchange->getCondition();
    if (c->IsActive())
      Conditions.ProcessCondition(*c);
  }
  if (m_Controls->LobarPneumonia->isSelected())
  {
    c = &m_Controls->LobarPneumonia->getCondition();
    if (c->IsActive())
      Conditions.ProcessCondition(*c);
  }
}


void QConditionsEditorWidget::ConditionsToControls()
{
  SEConditionManager& mgr = m_Controls->Pulse.GetPatientConfiguration().GetConditions();

  if (mgr.HasChronicAnemia())
  {
    m_Controls->ChronicAnemia->setCondition(*mgr.GetChronicAnemia());
    m_Controls->CWChronicAnemia->expand(true);
  }
  if (mgr.HasChronicObstructivePulmonaryDisease())
  {
    m_Controls->ChronicObstructivePulmonaryDisease->setCondition(*mgr.GetChronicObstructivePulmonaryDisease());
    m_Controls->CWChronicObstructivePulmonaryDisease->expand(true);
  }
  if (mgr.HasChronicPericardialEffusion())
  {
    m_Controls->ChronicPericardialEffusion->setCondition(*mgr.GetChronicPericardialEffusion());
    m_Controls->CWChronicPericardialEffusion->expand(true);
  }
  if (mgr.HasChronicRenalStenosis())
  {
    m_Controls->ChronicRenalStenosis->setCondition(*mgr.GetChronicRenalStenosis());
    m_Controls->CWChronicRenalStenosis->expand(true);
  }
  if (mgr.HasChronicVentricularSystolicDysfunction())
  {
    m_Controls->ChronicVentricularSystolicDysfunction->setCondition(*mgr.GetChronicVentricularSystolicDysfunction());
    m_Controls->CWChronicVentricularSystolicDysfunction->expand(true);
  }
  if (mgr.HasImpairedAlveolarExchange())
  {
    m_Controls->ImpairedAlveolarExchange->setCondition(*mgr.GetImpairedAlveolarExchange());
    m_Controls->CWImpairedAlveolarExchange->expand(true);
  }
  if (mgr.HasLobarPneumonia())
  {
    m_Controls->LobarPneumonia->setCondition(*mgr.GetLobarPneumonia());
    m_Controls->CWLobarPneumonia->expand(true);
  }
}

void QConditionsEditorWidget::AtSteadyState(PhysiologyEngine& pulse)
{

}
void QConditionsEditorWidget::AtSteadyStateUpdateUI()
{
  EnableInput(false);
  // Pull out conditions and set the data in the GUI
}

