/* Distributed under the Apache License, Version 2.0.
See accompanying NOTICE file for details.*/
#include "actions/SubstanceCompoundInfusionWidget.h"
#include "controls/ScalarQuantityWidget.h"
#include "controls/LabeledComboBox.h"

#include "cdm/CommonDataModel.h"
#include "PhysiologyEngine.h"
#include "cdm/patient/actions/SESubstanceCompoundInfusion.h"
#include "cdm/substance/SESubstanceCompound.h"
#include "cdm/substance/SESubstanceManager.h"
#include "cdm/properties/SEScalarVolume.h"
#include "cdm/properties/SEScalarVolumePerTime.h"

class QSubstanceCompoundInfusionWidget::Controls
{
public:
  Controls() {}
  SESubstanceCompoundInfusion* Action = nullptr;
  std::map<std::string, SESubstanceCompound*> SubstanceCompounds;
  QScalarQuantityWidget<VolumeUnit>*          BagVolume;
  QScalarQuantityWidget<VolumePerTimeUnit>*   Rate;
  QLabeledComboBox*                           Compounds;
};

QSubstanceCompoundInfusionWidget::QSubstanceCompoundInfusionWidget(QPulse& qp, QWidget *parent, Qt::WindowFlags flags) : QActionWidget(qp, parent, flags)
{
  m_Controls = new Controls();
  m_Controls->BagVolume = new QScalarQuantityWidget<VolumeUnit>("Bag Volume", 0, 1000, 1, VolumeUnit::mL, ScalarOptionWidget::None, this); //check this
  m_Controls->BagVolume->addUnit(VolumeUnit::L);

  m_Controls->Rate = new QScalarQuantityWidget<VolumePerTimeUnit>("Rate", 0, 1000, 1, VolumePerTimeUnit::mL_Per_s, ScalarOptionWidget::None, this);
  m_Controls->Rate->addUnit(VolumePerTimeUnit::L_Per_min);
  m_Controls->Rate->addUnit(VolumePerTimeUnit::mL_Per_min);
  m_Controls->Rate->addUnit(VolumePerTimeUnit::L_Per_s);


  std::vector<QString> compounds = { "Saline", "Blood","PackedRBC" };
  m_Controls->Compounds = new QLabeledComboBox(this, "Compounds", compounds, 120);
  for (QString c : compounds)
  {
    SESubstanceCompound* compound;
    compound = qp.GetEngine().GetSubstanceManager().GetCompound(c.toStdString());
    m_Controls->SubstanceCompounds[compound->GetName()] = compound;
  }
  Properties()->layout()->addWidget(m_Controls->Compounds);
  Properties()->layout()->addWidget(m_Controls->BagVolume);
  Properties()->layout()->addWidget(m_Controls->Rate);
  Reset();
}

QSubstanceCompoundInfusionWidget::~QSubstanceCompoundInfusionWidget()
{
  SAFE_DELETE(m_Controls->Action);
  delete m_Controls;
}

void QSubstanceCompoundInfusionWidget::Reset()
{
  QActionWidget::Reset();
  SAFE_DELETE(m_Controls->Action);
}

void QSubstanceCompoundInfusionWidget::SetEnabled(bool b)
{
  QActionWidget::SetEnabled(b);
  m_Controls->Rate->setEnabled(b);
  m_Controls->Rate->enableInput(b);
  m_Controls->BagVolume->setEnabled(b);
  m_Controls->BagVolume->enableInput(b);
  m_Controls->Compounds->SetEnabled(b);
}

void QSubstanceCompoundInfusionWidget::ControlsToAction()
{
  QActionWidget::ControlsToAction();
  m_Controls->Action = new SESubstanceCompoundInfusion(*m_Controls->SubstanceCompounds[m_Controls->Compounds->GetText()]);
  m_Controls->Rate->getValue(m_Controls->Action->GetRate());
  m_Controls->BagVolume->getValue(m_Controls->Action->GetBagVolume());
}

void QSubstanceCompoundInfusionWidget::AtSteadyState(PhysiologyEngine& pulse)
{
  // Nothing expected here
}
void QSubstanceCompoundInfusionWidget::AtSteadyStateUpdateUI()
{
  // Nothing expected here
}

void QSubstanceCompoundInfusionWidget::ProcessPhysiology(PhysiologyEngine& pulse)
{// This is called from a thread, you should NOT update UI here
  // This is where we pull data from pulse, and push any actions to it
  if (m_Controls->Action != nullptr)
  {
    if (m_Controls->Action->IsValid())
      pulse.ProcessAction(*m_Controls->Action);
    SAFE_DELETE(m_Controls->Action);
  }
}
void QSubstanceCompoundInfusionWidget::PhysiologyUpdateUI()
{// This is called from a slot, you can update UI here

  // Nothing is expected to be in the environment
}