from paraview.simple import *

print("Executing visualization")

# ------------------------------------------------------------------------------
# Catalyst options
from paraview import catalyst
options = catalyst.Options()
options.GlobalTrigger = 'TimeStep'
options.CatalystLiveTrigger = 'TimeStep'
options.EnableCatalystLive = 1

# ----------------------------------------------------------------
# setup views used in the visualization
# ----------------------------------------------------------------

# Create a new 'Render View'
renderView1 = CreateView('RenderView')
renderView1.Set(
    ViewSize=[2200, 1170],
    CenterOfRotation=[60.0, 35.5, 62.0],
    CameraPosition=[336.29935284394315, 185.73204450788, 238.147135622826],
    CameraFocalPoint=[60.00000000000001, 35.5, 62.0],
    CameraViewUp=[-0.18433324259790532, 0.8716043417369134, -0.4542324593627186],
)

SetActiveView(None)

# ----------------------------------------------------------------
# setup view layouts
# ----------------------------------------------------------------

# create new layout object 'Layout #1'
layout1 = CreateLayout(name='Layout #1')
layout1.AssignView(0, renderView1)
layout1.SetSize(2200, 1170)

# ----------------------------------------------------------------
# restore active view
SetActiveView(renderView1)

# registrationName must match the channel name used in the
# 'CatalystAdaptor'.
producer = TrivialProducer(registrationName="steerchannel")

# show data from slice1
slice1Display = Show(producer, renderView1, 'GeometryRepresentation')

# get color transfer function/color map for 'pressure'
rTDataLUT = GetColorTransferFunction('pressure')
rTDataLUT.Set(
    RGBPoints=GenerateRGBPoints(
        range_min=-14.125329971313477,
        range_max=276.68310546875,
    ),
    ScalarRangeInitialized=1.0,
)

# trace defaults for the display properties.
slice1Display.Set(
    Representation='Surface',
    ColorArrayName=['CELLS', 'pressure'],
    LookupTable=rTDataLUT,
)

# init the 'Piecewise Function' selected for 'ScaleTransferFunction'
slice1Display.ScaleTransferFunction.Points = [-7.056811332702637, 0.0, 0.5, 0.0, 180.25868225097656, 1.0, 0.5, 0.0]

# init the 'Piecewise Function' selected for 'OpacityTransferFunction'
slice1Display.OpacityTransferFunction.Points = [-7.056811332702637, 0.0, 0.5, 0.0, 180.25868225097656, 1.0, 0.5, 0.0]

# setup the color legend parameters for each legend in this view

# get color legend/bar for rTDataLUT in view renderView1
rTDataLUTColorBar = GetScalarBar(rTDataLUT, renderView1)
rTDataLUTColorBar.Set(
    Title='pressure',
    ComponentTitle='',
)

# set color bar visibility
rTDataLUTColorBar.Visibility = 1

# show color legend
slice1Display.SetScalarBarVisibility(renderView1, True)

# ----------------------------------------------------------------
# setup color maps and opacity maps used in the visualization
# note: the Get..() functions create a new object, if needed
# ----------------------------------------------------------------

# get opacity transfer function/opacity map for 'pressure'
rTDataPWF = GetOpacityTransferFunction('pressure')
rTDataPWF.Set(
    Points=[-14.125329971313477, 0.0, 0.5, 0.0, 276.68310546875, 1.0, 0.5, 0.0],
    ScalarRangeInitialized=1,
)

# ----------------------------------------------------------------
# setup extractors
# ----------------------------------------------------------------

# create extractor
pNG1 = CreateExtractor('PNG', renderView1, registrationName='PNG1')
# trace defaults for the extractor.
# init the 'PNG' selected for 'Writer'
pNG1.Writer.Set(
    FileName='RenderView1_{timestep:06d}{camera}.png',
    ImageResolution=[2200, 1170],
    Format='PNG',
)

# ----------------------------------------------------------------
# restore active source
SetActiveSource(pNG1)


# Special catalyst method called at each timestep, it is optional, it's use here
# only for print purpose
def catalyst_execute(info):
    global producer
    producer.UpdatePipeline()
    print("-----------------------------------")
    print("executing (cycle={}, time={})".format(info.cycle, info.time))
    print("bounds:", producer.GetDataInformation().GetBounds())
    print("pressure-range:", producer.CellData["pressure"].GetRange(0))

# ------------------------------------------------------------------------------
if __name__ == '__main__':
    from paraview.simple import SaveExtractsUsingCatalystOptions
    # Code for non in-situ environments; if executing in post-processing
    # i.e. non-Catalyst mode, let's generate extracts using Catalyst options
    SaveExtractsUsingCatalystOptions(options)
