#ifndef CatalystAdaptor_h
#define CatalystAdaptor_h

#include "DataStructures.h"
#include <catalyst.hpp>

#include <array>
#include <catalyst_conduit.hpp>
#include <catalyst_conduit_blueprint.hpp>
#include <catalyst_conduit_blueprint_mangle.h>
#include <catalyst_conduit_error.hpp>
#include <iostream>
#include <string>

#include <mpi.h>

namespace CatalystAdaptor
{
inline void Initialize(int argc, char* argv[])
{
  conduit_cpp::Node node;
  for (int cc = 1; cc < argc; ++cc)
  {
    if (strstr(argv[cc], "xml"))
    {
      node["adios/config_filepath"].set_string(argv[cc]);
    }
    else
    {
      node["catalyst/scripts/script" + std::to_string(cc - 1)].set_string(argv[cc]);
    }
  }
  node["catalyst_load/implementation"] = "adios";
  catalyst_status err = catalyst_initialize(conduit_cpp::c_node(&node));
  if (err != catalyst_status_ok)
  {
    std::cerr << "Failed to initialize Catalyst: " << err << std::endl;
  }
}

inline void Execute(int cycle, double time, Grid& grid, Attributes& attribs)
{
  conduit_cpp::Node exec_params;

  // add time/cycle information
  auto state = exec_params["catalyst/state"];
  state["timestep"].set(cycle);
  state["time"].set(time);
  state["multiblock"].set(1);

  auto channel = exec_params["catalyst/channels/grid"];

  channel["type"].set("mesh");
  auto mesh = channel["data"];

  int mpiRank = 0;
  MPI_Comm_rank(MPI_COMM_WORLD, &mpiRank);

  if (mpiRank == 0)
  {
    std::array<double, 4> pointsX{ 0, 1, 2, 3 };
    std::array<double, 4> pointsY{ 4, 5, 6, 7 };
    std::array<double, 4> pointsZ{ 8, 9, 10, 11 };
    mesh["coordsets/coords/type"] = "explicit";
    mesh["coordsets/coords/values/x"].set_float64_ptr(pointsX.data(), pointsX.size());
    mesh["coordsets/coords/values/y"].set_float64_ptr(pointsY.data(), pointsY.size());
    mesh["coordsets/coords/values/z"].set_float64_ptr(pointsZ.data(), pointsZ.size());

    mesh["topologies/mesh/type"] = "unstructured";
    mesh["topologies/mesh/coordset"] = "coords";
    mesh["topologies/mesh/elements/shape"] = "point";
    std::array<int, 4> connectivity{ 0, 1, 2, 3 };
    mesh["topologies/mesh/elements/connectivity"].set_int32_ptr(
      connectivity.data(), connectivity.size());
  }
  else
  {
    // Create an unbalanced simulation: one node has all data, the other one has none.
    // A null Conduit mesh should still work all the way to the end. 
    std::array<double, 0> nullPts{};
    mesh["coordsets/coords/type"] = "explicit";
    mesh["coordsets/coords/values/x"].set_float64_ptr(nullPts.data(), nullPts.size());
    mesh["coordsets/coords/values/y"].set_float64_ptr(nullPts.data(), nullPts.size());
    mesh["coordsets/coords/values/z"].set_float64_ptr(nullPts.data(), nullPts.size());

    mesh["topologies/mesh/type"] = "unstructured";
    mesh["topologies/mesh/coordset"] = "coords";
    mesh["topologies/mesh/elements/shape"] = "point";
    std::array<int, 0> nullConn{};
    mesh["topologies/mesh/elements/connectivity"].set_int32_ptr(nullConn.data(), nullConn.size());

  }

  catalyst_status err = catalyst_execute(conduit_cpp::c_node(&exec_params));
  if (err != catalyst_status_ok)
  {
    std::cerr << "Failed to execute Catalyst: " << err << std::endl;
  }
}

inline void Finalize()
{
  conduit_cpp::Node node;
  catalyst_status err = catalyst_finalize(conduit_cpp::c_node(&node));
  if (err != catalyst_status_ok)
  {
    std::cerr << "Failed to finalize Catalyst: " << err << std::endl;
  }
}
}

#endif
