#ifndef CatalystAdaptor_h
#define CatalystAdaptor_h

#include "DataStructures.h"
#include <catalyst.hpp>

#include <iostream>
#include <string>

#include <mpi.h>

namespace CatalystAdaptor
{

/**
 * In this example, we show how we can use Catalysts's C++
 * wrapper around conduit's C API to create Conduit nodes.
 * This is not required. A C++ adaptor can just as
 * conveniently use the Conduit C API to setup the
 * `conduit_node`. However, this example shows that one can
 * indeed use Catalyst's C++ API, if the developer so chooses.
 */
inline void Initialize(int argc, char* argv[])
{
  conduit_cpp::Node node;
  for (int cc = 1; cc < argc; ++cc)
  {
    if (strstr(argv[cc], "xml"))
    {
      node["adios/config_filepath"].set_string(argv[cc]);
    }
  }
  node["catalyst_load/implementation"] = "adios";

  // Set hybrid pipeline file
  node["hybrid/pipeline"].set_string(argv[argc - 1]);

  catalyst_status err = catalyst_initialize(conduit_cpp::c_node(&node));
  if (err != catalyst_status_ok)
  {
    std::cerr << "Failed to initialize Catalyst: " << err << std::endl;
  }
}

inline void Execute(int cycle, double time, Grid& grid, Attributes& attribs)
{
  conduit_cpp::Node exec_params;

  // add time/cycle information
  auto state = exec_params["catalyst/state"];
  state["timestep"].set(cycle);
  state["time"].set(time);
  state["sstFileName"].set(attribs.GetSSTFileName());

  // Add channels.
  // We only have 1 channel here. Let's name it 'grid'.
  auto channel = exec_params["catalyst/channels/grid"];

  // Since this example is using Conduit Mesh Blueprint to define the mesh,
  // we set the channel's type to "mesh".
  channel["type"].set("multimesh");

  // now create the mesh.
  auto root = channel["data"];

  channel["assembly/UGBlock"] = "MixedBlock";

  auto mixedBlock = root["MixedBlock"];

  auto coords_node = mixedBlock["coordsets/coords"];
  auto topologies_node = mixedBlock["topologies/mesh"];
  auto field = mixedBlock["fields/myField"];

  int mpiRank = 0;
  MPI_Comm_rank(MPI_COMM_WORLD, &mpiRank);

  if (mpiRank == 0)
  {
    // Fill coordinates
    coords_node["type"] = "explicit";
    coords_node["values/x"] = std::vector<float>{ 0, 1, 2, 0, 1, 2, 0, 1, 2, 0, 1, 2, 0, 1, 2, 0, 1,
      2, 0, 1, 2, 0, 1, 2, 0, 1, 2 };
    coords_node["values/y"] = std::vector<float>{ 0, 0, 0, 1, 1, 1, 0, 0, 0, 1, 1, 1, 1, 1, 1, 1, 1,
      1, 1, 1, 1, 1, 1, 1, 1, 1, 1 };
    coords_node["values/z"] = std::vector<float>{ 0, 0, 0, 0, 0, 0, 1, 1, 1, 1, 1, 1, 2, 2, 2, 3, 3,
      3, 4, 4, 4, 5, 5, 5, 6, 6, 6 };

    // Fill topology
    topologies_node["type"] = "unstructured";
    topologies_node["coordset"] = "coords";
    topologies_node["elements/shape"] = "mixed";
    topologies_node["elements/shape_map/point"] = 1;
    topologies_node["elements/shape_map/line"] = 3;
    topologies_node["elements/shape_map/tri"] = 5;
    topologies_node["elements/shape_map/polygonal"] = 7;
    topologies_node["elements/shapes"] =
      std::vector<conduit_uint8>{ 1, 1, 1, 1, 1, 1, 3, 3, 3, 5, 5, 7, 7 };
    topologies_node["elements/offsets"] =
      std::vector<conduit_int64>{ 0, 1, 2, 3, 4, 5, 6, 8, 10, 12, 15, 18, 21 };
    topologies_node["elements/sizes"] =
      std::vector<conduit_int64>{ 1, 1, 1, 1, 1, 1, 2, 2, 2, 3, 3, 3, 6 };
    topologies_node["elements/connectivity"] = std::vector<conduit_int64>{ 0, 1, 17, 18, 19, 20, 2,
      3, 13, 14, 15, 16, 21, 22, 23, 24, 25, 26, 4, 5, 6, 7, 8, 9, 10, 11, 12 };

    // Fill fields
    field["association"] = "element";
    field["topology"] = "mesh";
    field["volume_dependent"] = "false";
    field["values"] = std::vector<double>{ 0.0, 1.0, 2.0, 3.0, 4.0, 5.0, 6.0, 7.0, 8.0, 9.0, 10.0, 11.0, 12.0 };
  }
  else
  {
    // Fill coordinates
    coords_node["type"] = "explicit";
    coords_node["values/x"] = std::vector<float>{};
    coords_node["values/y"] = std::vector<float>{};
    coords_node["values/z"] = std::vector<float>{};

    // Fill topology
    topologies_node["type"] = "unstructured";
    topologies_node["coordset"] = "coords";
    topologies_node["elements/shape"] = "mixed";
    topologies_node["elements/shape_map/point"] = 1;
    topologies_node["elements/shape_map/line"] = 3;
    topologies_node["elements/shape_map/tri"] = 5;
    topologies_node["elements/shape_map/polygonal"] = 7;
    topologies_node["elements/shapes"] = std::vector<conduit_uint8>{};
    topologies_node["elements/offsets"] = std::vector<conduit_int64>{};
    topologies_node["elements/sizes"] = std::vector<conduit_int64>{};
    topologies_node["elements/connectivity"] = std::vector<conduit_int64>{};

    // Fill fields
    field["association"] = "element";
    field["topology"] = "mesh";
    field["volume_dependent"] = "false";
    field["values"] = std::vector<double>{};
  }

  catalyst_status err = catalyst_execute(conduit_cpp::c_node(&exec_params));
  if (err != catalyst_status_ok)
  {
    std::cerr << "Failed to execute Catalyst: " << err << std::endl;
  }
}

inline void Finalize()
{
  conduit_cpp::Node node;
  catalyst_status err = catalyst_finalize(conduit_cpp::c_node(&node));
  if (err != catalyst_status_ok)
  {
    std::cerr << "Failed to finalize Catalyst: " << err << std::endl;
  }
}
}

#endif
