/*=========================================================================

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#ifndef AdiosPipeline_h
#define AdiosPipeline_h

#include <adios2.h>

#include <catalyst_conduit.hpp>

#include <map>    // For std::map
#include <string> // For std::string
#include <vector> // For std::vector

/**
 * @class AdiosPipeline
 * @brief inTransit pipeline
 *
 * AdiosPipeline is a pipeline that can be used to save out data using writers
 * supported by Adios.
 *
 * A pipeline has three stages: Initialize, Execute and Finalize.
 *
 * `Initialize` is called exactly once before the first call to `Execute`. If
 * `Initialize` returns `false`, the initialization is deemed failed and
 * pipeline is skipped for rest of the execution i.e. either Execute nor
 * Finalize will be called.
 *
 * `Execute` is called on each cycle. If the method returns false, then the
 * execution is deemed failed and the `Execute` method will not be called in
 * subsequent cycles.
 *
 * If `Initialize` succeeded, then `Finalize` is called as the end of the
 * simulation execution. `Finalize` is called even if `Execute` returned
 * failure. However, it will not be called if `Initialize` returned failure too.
 */
class AdiosPipeline
{
public:
  AdiosPipeline() = default;
  ~AdiosPipeline() = default;

  ///@{
  /**
   * AdiosPipeline API implementaton.
   */
  bool Initialize(const std::string& adiosFileName, const std::string& CatalystInitializeParametersAsString);
  bool Execute(int timestep, conduit_cpp::Node channels);
  bool Finalize(const std::string& catalystFinalizeParametersAsString);
  ///@}

protected:
  /**
   * Convenient method used to extract 'dims' from a conduit_node.
   */
  std::vector<int> GetDimensions(const conduit_cpp::Node& data);

  /**
   * Setup MPI and initialize all variables at the beginning of the first execute call because we
   * need to know each variables dimensions that is only know during the Catalyst::Execute.
   */
  bool InitializeVariables(const conduit_cpp::Node& root);

  /**
   * Define variable with the adios engine and store it inside the correct map.
   */
  bool FillVariables(const conduit_cpp::Node& values, const std::string& name);

  /**
   * Start the adios writer and find for each values in the 'fields' the PutVariables() method.
   */
  bool Put(const conduit_cpp::Node& fields, int timestep);

  /**
   * Apply the PutVariables() by the adios engine depending on their types
   */
  bool PutVariables(const conduit_cpp::Node& values, const std::string& name);

private:
  AdiosPipeline(const AdiosPipeline&) = delete;
  void operator=(const AdiosPipeline&) = delete;

  // Path to a specific xml file used to setup adios engine
  std::string AdiosFileName;

  // Content of "catalyst" conduit node in a json formated string
  std::string CatalystInitializeParametersAsString = "";

  adios2::ADIOS Adios;
  adios2::IO IO;
  adios2::Engine Writer;

  adios2::Dims Shape;
  adios2::Dims Start;
  adios2::Dims Count;

  // Data type supported :
  // https://adios2.readthedocs.io/en/latest/components/components.html#data-types
  std::map<std::string, adios2::Variable<short>> VarsShort;
  std::map<std::string, adios2::Variable<int>> VarsInt;
  std::map<std::string, adios2::Variable<long int>> VarsLongInt;

  std::map<std::string, adios2::Variable<char>> VarsChar;

  std::map<std::string, adios2::Variable<unsigned short>> VarsUShort;
  std::map<std::string, adios2::Variable<unsigned int>> VarsUInt;
  std::map<std::string, adios2::Variable<unsigned long int>> VarsULongInt;

  std::map<std::string, adios2::Variable<float>> VarsFloat;
  std::map<std::string, adios2::Variable<double>> VarsDouble;

  adios2::Variable<int> Timestep;

  // Used to initialize at the first execute called, all variables
  bool FirstExecute = true;
};

#endif
