//==============================================================================
// Copyright 2018-2020 Kitware, Inc., Kitware SAS
// Author: Guilbert Pierre (Kitware SAS)
//         Cadart Nicolas (Kitware SAS)
// Creation date: 2018-03-27
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
//==============================================================================

// This slam algorithm is inspired by the LOAM algorithm:
// J. Zhang and S. Singh. LOAM: Lidar Odometry and Mapping in Real-time.
// Robotics: Science and Systems Conference (RSS). Berkeley, CA, July 2014.

// The algorithm is composed of three sequential steps:
//
// - Keypoints extraction: this step consists of extracting keypoints over
// the points clouds. To do that, the laser lines / scans are treated independently.
// The laser lines are projected onto the XY plane and are rescaled depending on
// their vertical angle. Then we compute their curvature and create two classes of
// keypoints. The edges keypoints which correspond to points with a high curvature
// and planar points which correspond to points with a low curvature.
//
// - Ego-Motion: this step consists of recovering the motion of the lidar
// sensor between two frames (two sweeps). The motion is modelized by a constant
// velocity and angular velocity between two frames (i.e null acceleration).
// Hence, we can parameterize the motion by a rotation and translation per sweep / frame
// and interpolate the transformation inside a frame using the timestamp of the points.
// Since the points clouds generated by a lidar are sparse we can't design a
// pairwise match between keypoints of two successive frames. Hence, we decided to use
// a closest-point matching between the keypoints of the current frame
// and the geometric features derived from the keypoints of the previous frame.
// The geometric features are lines or planes and are computed using the edges
// and planar keypoints of the previous frame. Once the matching is done, a keypoint
// of the current frame is matched with a plane / line (depending of the
// nature of the keypoint) from the previous frame. Then, we recover R and T by
// minimizing the function f(R, T) = sum(d(point, line)^2) + sum(d(point, plane)^2).
// Which can be writen f(R, T) = sum((R*X+T-P).t*A*(R*X+T-P)) where:
// - X is a keypoint of the current frame
// - P is a point of the corresponding line / plane
// - A = (n*n.t) with n being the normal of the plane
// - A = (I - n*n.t).t * (I - n*n.t) with n being a director vector of the line
// Since the function f(R, T) is a non-linear mean square error function
// we decided to use the Levenberg-Marquardt algorithm to recover its argmin.
//
// - Localization: This step consists of refining the motion recovered in the Ego-Motion
// step and to add the new frame in the environment map. Thanks to the ego-motion
// recovered at the previous step it is now possible to estimate the new position of
// the sensor in the map. We use this estimation as an initial point (R0, T0) and we
// perform an optimization again using the keypoints of the current frame and the matched
// keypoints of the map (and not only the previous frame this time!). Once the position in the
// map has been refined from the first estimation it is then possible to update the map by
// adding the keypoints of the current frame into the map.
//
// In the following programs, three 3D coordinates system are used :
// - LIDAR {L} : attached to the geometric center of the LiDAR sensor. The
//   coordinates of the received pointclouds are expressed in this system.
//   LIDAR is rigidly linked (static transform) to BASE.
// - BASE  {B} : attached to the origin of the moving body (e.g. vehicle). We
//   are generally interested in tracking an other point of the moving body than
//   the LiDAR's (for example, we prefer to track the GPS antenna pose).
// - WORLD {W} : The world coordinate system {W} coincides with BASE at the
//   initial position. The output trajectory describes BASE origin in WORLD.

// LOCAL
#include "LidarSlam/Slam.h"
#include "LidarSlam/Utilities.h"
#include "LidarSlam/CeresCostFunctions.h"
#ifdef USE_G2O
#include "LidarSlam/PoseGraphOptimization.h"
#endif  // USE_G2O
// CERES
#include <ceres/ceres.h>
// PCL
#include <pcl/common/common.h>
#include <pcl/common/transforms.h>
#include <pcl/io/pcd_io.h>
// EIGEN
#include <Eigen/Dense>

#define PRINT_VERBOSE(minVerbosityLevel, stream) if (this->Verbosity >= (minVerbosityLevel)) {std::cout << stream << std::endl;}
#define IF_VERBOSE(minVerbosityLevel, command) if (this->Verbosity >= (minVerbosityLevel)) { command; }

namespace
{
//-----------------------------------------------------------------------------
std::array<double, 36> FlipAndConvertCovariance(const Eigen::Matrix6d& covar)
{
  // Reshape covariance from DoF order (rX, rY, rZ, X, Y, Z) to (X, Y, Z, rX, rY, rZ)
  const double* c = covar.data();
  std::array<double, 36> cov = {c[21], c[22], c[23],   c[18], c[19], c[20],
                                c[27], c[28], c[29],   c[24], c[25], c[26],
                                c[33], c[34], c[35],   c[30], c[31], c[32],

                                c[ 3], c[ 4], c[ 5],   c[ 0], c[ 1], c[ 2],
                                c[ 9], c[10], c[11],   c[ 6], c[ 7], c[ 8],
                                c[15], c[16], c[17],   c[12], c[13], c[14]};
  return cov;
}

//-----------------------------------------------------------------------------
//! Approximate pointcloud memory size
inline size_t PointCloudMemorySize(const Slam::PointCloud& cloud)
{
  return (sizeof(cloud) + (sizeof(Slam::PointCloud::PointType) * cloud.size()));
}

//-----------------------------------------------------------------------------
//! Approximate logged keypoints size
void LoggedKeypointsSize(const std::deque<PointCloudStorage<Slam::Point>>& log, size_t& totalMemory, size_t& totalPoints)
{
  totalMemory = 0;
  totalPoints = 0;
  for (auto const& storage: log)
  {
    totalPoints += storage.PointsSize();
    totalMemory += storage.MemorySize();
  }
}
}

//==============================================================================
//   Main SLAM use
//==============================================================================

//-----------------------------------------------------------------------------
Slam::Slam()
{
  // Allocate Keypoints Extractor
  this->KeyPointsExtractor = std::make_shared<SpinningSensorKeypointExtractor>();

  // Allocate maps
  this->EdgesPointsLocalMap = std::make_shared<RollingGrid>();
  this->PlanarPointsLocalMap = std::make_shared<RollingGrid>();
  this->BlobsPointsLocalMap = std::make_shared<RollingGrid>();

  // Set default maps parameters
  this->SetVoxelGridResolution(10.);
  this->SetVoxelGridSize(50);
  this->SetVoxelGridLeafSizeEdges(0.30);
  this->SetVoxelGridLeafSizePlanes(0.60);
  this->SetVoxelGridLeafSizeBlobs(0.30);

  // Reset SLAM internal state
  this->Reset();
}

//-----------------------------------------------------------------------------
void Slam::Reset(bool resetLog)
{
  // Reset keypoints maps
  this->ClearMaps();

  // n-DoF parameters
  this->Tworld = Eigen::Isometry3d::Identity();
  this->PreviousTworld = Eigen::Isometry3d::Identity();
  this->Trelative = Eigen::Isometry3d::Identity();
  this->TworldFrameStart = Eigen::Isometry3d::Identity();
  this->WithinFrameMotion.SetTransforms(this->TworldFrameStart, this->Tworld);
  this->TworldCovariance = Eigen::Matrix6d::Identity();

  // Reset point clouds
  this->CurrentFrame.reset(new PointCloud);
  this->CurrentEdgesPoints.reset(new PointCloud);
  this->CurrentPlanarsPoints.reset(new PointCloud);
  this->CurrentBlobsPoints.reset(new PointCloud);
  this->CurrentWorldEdgesPoints.reset(new PointCloud);
  this->CurrentWorldPlanarsPoints.reset(new PointCloud);
  this->CurrentWorldBlobsPoints.reset(new PointCloud);

  // Reset debug variables
  this->EgoMotionEdgesPointsUsed = 0;
  this->EgoMotionPlanesPointsUsed = 0;
  this->LocalizationEdgesPointsUsed = 0;
  this->LocalizationPlanesPointsUsed = 0;
  this->LocalizationBlobsPointsUsed = 0;
  this->LocalizationVarianceError = 0.;

  // Reset log history
  if (resetLog)
  {
    this->NbrFrameProcessed = 0;
    this->LogTrajectory.clear();
    this->LogEdgesPoints.clear();
    this->LogPlanarsPoints.clear();
    this->LogBlobsPoints.clear();
  }
}

//-----------------------------------------------------------------------------
void Slam::AddFrame(const PointCloud::Ptr& pc, const std::vector<size_t>& laserIdMapping)
{
  InitTime("SLAM frame processing");

  // Skip frame if empty
  if (pc->empty())
  {
    std::cerr << "[ERROR] SLAM entry is an empty pointcloud : frame ignored.\n";
    return;
  }

  // Skip frame if it has the same timestamp as previous one (will induce problems in extrapolation)
  if (pc->header.stamp == this->CurrentFrame->header.stamp)
  {
    std::cerr << "[ERROR] SLAM entry has the same timestamp (" << pc->header.stamp 
              << ") as previous pointcloud : frame ignored." << std::endl;
    return;
  }

  PRINT_VERBOSE(2, "\n#########################################################");
  PRINT_VERBOSE(1, "Processing frame " << this->NbrFrameProcessed);
  PRINT_VERBOSE(2, "#########################################################\n");

  // Update current frame (check frame dropping, correct time field) and
  // estimate new state (extrapolate new pose with a constant velocity model)
  IF_VERBOSE(3, InitTime("Update frame and state"));
  this->UpdateFrameAndState(pc);
  IF_VERBOSE(3, StopTimeAndDisplay("Update frame and state"));

  // Compute the edges and planars keypoints
  IF_VERBOSE(3, InitTime("Keypoints extraction"));
  this->ExtractKeypoints(laserIdMapping);
  IF_VERBOSE(3, StopTimeAndDisplay("Keypoints extraction"));

  // If the new frame is the first one we just add the extracted keypoints into
  // the map without running odometry and localization steps
  if (this->NbrFrameProcessed > 0)
  {
    // Compute Trelative by registering current frame on previous one
    if (this->EgoMotion == EgoMotionMode::REGISTRATION ||
        this->EgoMotion == EgoMotionMode::MOTION_EXTRAPOLATION_AND_REGISTRATION)
    {
      IF_VERBOSE(3, InitTime("Ego-Motion"));
      this->ComputeEgoMotion();
      IF_VERBOSE(3, StopTimeAndDisplay("Ego-Motion"));
    }

    // Integrate the relative motion to the world transformation
    this->Tworld = this->PreviousTworld * this->Trelative;
    if (this->Undistortion)
    {
      // Use extrapolated motion to initialize keypoints undistortion.
      // (All keypoints will be transformed to the BASE referential at the end
      // of frame acquisition)
      this->TworldFrameStart = this->InterpolateBeginScanPose();
      this->WithinFrameMotion.SetTransforms(this->TworldFrameStart, this->Tworld);
    }

    // Perform Localization : compute Tworld from map and current frame keypoints
    IF_VERBOSE(3, InitTime("Localization"));
    this->Localization();
    IF_VERBOSE(3, StopTimeAndDisplay("Localization"));
  }

  // Update keypoints maps : add current keypoints to map using Tworld
  if (this->UpdateMap)
  {
    IF_VERBOSE(3, InitTime("Maps update"));
    this->UpdateMapsUsingTworld();
    IF_VERBOSE(3, StopTimeAndDisplay("Maps update"));
  }

  // Log current frame processing results : pose, covariance and keypoints.
  IF_VERBOSE(3, InitTime("Logging"));
  this->LogCurrentFrameState(PclStampToSec(this->CurrentFrame->header.stamp), this->WorldFrameId);
  IF_VERBOSE(3, StopTimeAndDisplay("Logging"));

  // Motion and localization parameters estimation information display
  if (this->Verbosity >= 2)
  {
    SET_COUT_FIXED_PRECISION(3);
    std::cout << "========== SLAM results ==========\n"
                 "Ego-Motion:\n"
                 " translation = [" << this->Trelative.translation().transpose()                          << "]\n"
                 " rotation    = [" << Rad2Deg(RotationMatrixToRPY(this->Trelative.linear())).transpose() << "]\n";
    if (this->Undistortion)
    {
      Eigen::Isometry3d motion = this->TworldFrameStart.inverse() * this->Tworld;
      std::cout << "Within frame motion:\n"
                   " translation = [" << motion.translation().transpose()                          << "]\n"
                   " rotation    = [" << Rad2Deg(RotationMatrixToRPY(motion.linear())).transpose() << "]\n";
    }
    std::cout << "Localization:\n"
                 " position    = [" << this->Tworld.translation().transpose()                          << "]\n"
                 " orientation = [" << Rad2Deg(RotationMatrixToRPY(this->Tworld.linear())).transpose() << "]" << std::endl;
    RESET_COUT_FIXED_PRECISION;
  }

  if (this->Verbosity >= 5)
  {
    SET_COUT_FIXED_PRECISION(3);
    std::cout << "========== Memory usage ==========\n";
    // SLAM maps
    PointCloud::Ptr edgesMap   = this->GetEdgesMap(),
                    planarsMap = this->GetPlanarsMap(),
                    blobsMap   = this->GetBlobsMap();
    std::cout << "Edges map   : " << edgesMap->size()   << " points, " << PointCloudMemorySize(*edgesMap)   * 1e-6 << " MB\n";
    std::cout << "Planars map : " << planarsMap->size() << " points, " << PointCloudMemorySize(*planarsMap) * 1e-6 << " MB\n";
    std::cout << "Blobs map   : " << blobsMap->size()   << " points, " << PointCloudMemorySize(*blobsMap)   * 1e-6 << " MB\n";

    // Logged keypoints
    size_t memory, points;
    LoggedKeypointsSize(this->LogEdgesPoints, memory, points);
    std::cout << "Edges log   : " << this->LogEdgesPoints.size()   << " frames, " << points << " points, " << memory * 1e-6 << " MB\n";
    LoggedKeypointsSize(this->LogPlanarsPoints, memory, points);
    std::cout << "Planars log : " << this->LogPlanarsPoints.size() << " frames, " << points << " points, " << memory * 1e-6 << " MB\n";
    LoggedKeypointsSize(this->LogBlobsPoints, memory, points);
    std::cout << "Blobs log   : " << this->LogBlobsPoints.size()   << " frames, " << points << " points, " << memory * 1e-6 << " MB" << std::endl;
    RESET_COUT_FIXED_PRECISION;
  }

  // Frame processing duration
  this->Latency = GetTime("SLAM frame processing");
  this->NbrFrameProcessed++;
  IF_VERBOSE(1, StopTimeAndDisplay("SLAM frame processing"));
}

//-----------------------------------------------------------------------------
void Slam::RunPoseGraphOptimization(const std::vector<Transform>& gpsPositions,
                                    const std::vector<std::array<double, 9>>& gpsCovariances,
                                    Eigen::Isometry3d& gpsToSensorOffset,
                                    const std::string& g2oFileName)
{
  #ifdef USE_G2O
  IF_VERBOSE(1, InitTime("Pose graph optimization"));
  IF_VERBOSE(3, InitTime("PGO : optimization"));

  // Transform to modifiable vectors
  std::vector<Transform> slamPoses(this->LogTrajectory.begin(), this->LogTrajectory.end());
  std::vector<std::array<double, 36>> slamCovariances(this->LogCovariances.begin(), this->LogCovariances.end());

  if (this->LoggingTimeout == 0.)
  {
    std::cerr << "[WARNING] SLAM logging is not enabled : covariances will be "
                 "arbitrarly set and maps will not be optimized during pose "
                 "graph optimization.\n";

    // Set all poses covariances equal to twice the last one if we did not log it
    std::array<double, 36> fakeSlamCovariance = FlipAndConvertCovariance(this->TworldCovariance * 2);
    for (unsigned int i = 0; i < slamPoses.size(); i++)
      slamCovariances.emplace_back(fakeSlamCovariance);
  }

  // Init pose graph optimizer
  PoseGraphOptimization poseGraphOptimization;
  poseGraphOptimization.SetNbIteration(500);
  poseGraphOptimization.SetVerbose(this->Verbosity >= 2);
  poseGraphOptimization.SetSaveG2OFile(!g2oFileName.empty());
  poseGraphOptimization.SetG2OFileName(g2oFileName);
  poseGraphOptimization.SetGpsToSensorCalibration(gpsToSensorOffset);

  // Run pose graph optimization
  // TODO : templatize poseGraphOptimization to accept any STL container and avoid deque <-> vector copies
  std::vector<Transform> optimizedSlamPoses;
  if (!poseGraphOptimization.Process(slamPoses, gpsPositions,
                                     slamCovariances, gpsCovariances,
                                     optimizedSlamPoses))
  {
    std::cerr << "[ERROR] Pose graph optimization failed.\n";
    return;
  }

  IF_VERBOSE(3, StopTimeAndDisplay("PGO : optimization"));

  // Update GPS/LiDAR calibration
  gpsToSensorOffset = optimizedSlamPoses.front().GetIsometry();

  // Update SLAM trajectory and maps
  IF_VERBOSE(3, InitTime("PGO : SLAM reset"));
  this->Reset(false);
  IF_VERBOSE(3, StopTimeAndDisplay("PGO : SLAM reset"));
  IF_VERBOSE(3, InitTime("PGO : frames keypoints aggregation"));
  PointCloud edgesKeypoints, planarsKeypoints, blobsKeypoints;
  PointCloud::Ptr aggregatedEdgesMap(new PointCloud),
                  aggregatedPlanarsMap(new PointCloud),
                  aggregatedBlobsMap(new PointCloud);
  for (unsigned int i = 0; i < optimizedSlamPoses.size(); i++)
  {
    // Update SLAM pose
    this->LogTrajectory[i].SetIsometry(gpsToSensorOffset.inverse() * optimizedSlamPoses[i].GetIsometry());

    // Transform frame keypoints to world coordinates
    Eigen::Matrix4d currentTransform = this->LogTrajectory[i].GetMatrix();

    pcl::transformPointCloud(*this->LogEdgesPoints[i].GetCloud(), edgesKeypoints, currentTransform);
    pcl::transformPointCloud(*this->LogPlanarsPoints[i].GetCloud(), planarsKeypoints, currentTransform);
    if (!this->FastSlam)
      pcl::transformPointCloud(*this->LogBlobsPoints[i].GetCloud(), blobsKeypoints, currentTransform);

    // TODO: Deal with undistortion case (properly transform pointclouds before aggreagtion)

    // Aggregate new keypoints to maps
    *aggregatedEdgesMap += edgesKeypoints;
    *aggregatedPlanarsMap += planarsKeypoints;
    if (!this->FastSlam)
      *aggregatedBlobsMap += blobsKeypoints;
  }

  IF_VERBOSE(3, StopTimeAndDisplay("PGO : frames keypoints aggregation"));
  IF_VERBOSE(3, InitTime("PGO : final SLAM map update"));

  // Set final pose
  Transform& finalPose = this->LogTrajectory.back();
  this->Tworld = finalPose.GetIsometry();

  // TODO : Deal with undistortion case (update motionParameters)

  // Update SLAM maps
  this->EdgesPointsLocalMap->Roll(this->Tworld.translation());
  this->EdgesPointsLocalMap->Add(aggregatedEdgesMap);
  this->PlanarPointsLocalMap->Roll(this->Tworld.translation());
  this->PlanarPointsLocalMap->Add(aggregatedPlanarsMap);
  if (!this->FastSlam)
  {
    this->BlobsPointsLocalMap->Roll(this->Tworld.translation());
    this->BlobsPointsLocalMap->Add(aggregatedBlobsMap);
  }

  // Processing duration
  IF_VERBOSE(3, StopTimeAndDisplay("PGO : final SLAM map update"));
  IF_VERBOSE(1, StopTimeAndDisplay("Pose graph optimization"));
  #else
  #define UNUSED(var) (void)(var)
  UNUSED(gpsPositions); UNUSED(gpsCovariances); UNUSED(gpsToSensorOffset); UNUSED(g2oFileName);
  std::cerr << "[ERROR] SLAM PoseGraphOptimization requires G2O, but it was not found.\n";
  #endif  // USE_G2O
}

//-----------------------------------------------------------------------------
void Slam::SetWorldTransformFromGuess(const Transform& poseGuess)
{
  // Reset previous frame keypoints because Ego-Motion is not valid since we imposed a discontinuity
  this->PreviousEdgesPoints->clear();
  this->PreviousPlanarsPoints->clear();

  // Set current pose
  this->Tworld = poseGuess.GetIsometry();
  // TODO update motionParameters
}

//-----------------------------------------------------------------------------
void Slam::SaveMapsToPCD(const std::string& filePrefix, PCDFormat pcdFormat) const
{
  IF_VERBOSE(3, InitTime("Keypoints maps saving to PCD"));

  // Save keypoints maps
  savePointCloudToPCD(filePrefix + "edges.pcd",   *this->GetEdgesMap(), pcdFormat, true);
  savePointCloudToPCD(filePrefix + "planars.pcd", *this->GetPlanarsMap(), pcdFormat, true);
  savePointCloudToPCD(filePrefix + "blobs.pcd",   *this->GetBlobsMap(), pcdFormat, true);

  // TODO : save map origin (in which coordinates?) in title or VIEWPOINT field

  IF_VERBOSE(3, StopTimeAndDisplay("Keypoints maps saving to PCD"));
}

//-----------------------------------------------------------------------------
void Slam::LoadMapsFromPCD(const std::string& filePrefix, bool resetMaps)
{
  IF_VERBOSE(3, InitTime("Keypoints maps loading from PCD"));

  // In most of the cases, we would like to reset SLAM internal maps before
  // loading new maps to avoid conflicts.
  if (resetMaps)
    this->ClearMaps();

  auto loadMapFromPCD = [](const std::string& path, std::shared_ptr<RollingGrid>& map)
  {
    PointCloud::Ptr keypoints(new PointCloud);
    if (pcl::io::loadPCDFile(path, *keypoints) == 0)
    {
      std::cout << "SLAM keypoints map successfully loaded from " << path << std::endl;
      map->Add(keypoints);
    }
  };

  loadMapFromPCD(filePrefix + "edges.pcd",   this->EdgesPointsLocalMap);
  loadMapFromPCD(filePrefix + "planars.pcd", this->PlanarPointsLocalMap);
  loadMapFromPCD(filePrefix + "blobs.pcd",   this->BlobsPointsLocalMap);

  // TODO : load/use map origin (in which coordinates?) in title or VIEWPOINT field

  IF_VERBOSE(3, StopTimeAndDisplay("Keypoints maps loading from PCD"));
}

//==============================================================================
//   SLAM results getters
//==============================================================================

//-----------------------------------------------------------------------------
Transform Slam::GetWorldTransform() const
{
  return this->LogTrajectory.empty() ? Transform::Identity() : this->LogTrajectory.back();
}

//-----------------------------------------------------------------------------
Transform Slam::GetLatencyCompensatedWorldTransform() const
{
  constexpr double MAX_EXTRAPOLATION_RATIO = 3.;

  // Get 2 last transforms
  unsigned int trajectorySize = this->LogTrajectory.size();
  if (trajectorySize == 0)
    return Transform::Identity();
  else if (trajectorySize == 1)
    return this->LogTrajectory.back();
  const Transform& previous = this->LogTrajectory[trajectorySize - 2];
  const Transform& current = this->LogTrajectory[trajectorySize - 1];
  const Eigen::Isometry3d& H0 = previous.GetIsometry();
  const Eigen::Isometry3d& H1 = current.GetIsometry();

  // Linearly compute normalized timestamp of Hpred.
  // We expect H0 and H1 to match with time 0 and 1.
  // If timestamps are not defined or too close, extrapolation is impossible.
  if (std::abs(current.time - previous.time) < 1e-6)
  {
    std::cerr << "[WARNING] Unable to compute latency-compensated transform : timestamps undefined or too close.\n";
    return current;
  }
  // If requested extrapolation timestamp is too far from previous frames timestamps, extrapolation is impossible.
  if (std::abs(this->Latency / (current.time - previous.time)) > MAX_EXTRAPOLATION_RATIO)
  {
    std::cerr << "[WARNING] Unable to compute latency-compensated transform : extrapolation time is too far.\n";
    return current;
  }

  // Extrapolate H0 and H1 to get expected Hpred at current time
  Eigen::Isometry3d Hpred = LinearInterpolation(H0, H1, current.time + this->Latency, previous.time, current.time);

  return Transform(Hpred, current.time, current.frameid);
}

//-----------------------------------------------------------------------------
std::array<double, 36> Slam::GetTransformCovariance() const
{
  // Reshape covariance from DoF order (rX, rY, rZ, X, Y, Z) to (X, Y, Z, rX, rY, rZ)
  return FlipAndConvertCovariance(this->TworldCovariance);
}

//-----------------------------------------------------------------------------
std::vector<Transform> Slam::GetTrajectory() const
{
  std::vector<Transform> slamPoses(this->LogTrajectory.begin(), this->LogTrajectory.end());
  return slamPoses;
}

//-----------------------------------------------------------------------------
std::vector<std::array<double, 36>> Slam::GetCovariances() const
{
  std::vector<std::array<double, 36>> slamCovariances(this->LogCovariances.begin(), this->LogCovariances.end());
  return slamCovariances;
}

//-----------------------------------------------------------------------------
std::unordered_map<std::string, double> Slam::GetDebugInformation() const
{
  std::unordered_map<std::string, double> map;
  map["EgoMotion: edges used"]   = this->EgoMotionEdgesPointsUsed;
  map["EgoMotion: planes used"]  = this->EgoMotionPlanesPointsUsed;
  map["Localization: edges used"]     = this->LocalizationEdgesPointsUsed;
  map["Localization: planes used"]    = this->LocalizationPlanesPointsUsed;
  map["Localization: blobs used"]     = this->LocalizationBlobsPointsUsed;
  map["Localization: variance error"] = this->LocalizationVarianceError;
  return map;
}

//-----------------------------------------------------------------------------
std::unordered_map<std::string, std::vector<double>> Slam::GetDebugArray() const
{
  auto toDoubleVector = [](auto const& scalars) { return std::vector<double>(scalars.begin(), scalars.end()); };

  std::unordered_map<std::string, std::vector<double>> map;
  map["EgoMotion: edges matches"]  = toDoubleVector(this->EdgePointRejectionEgoMotion);
  map["EgoMotion: planes matches"] = toDoubleVector(this->PlanarPointRejectionEgoMotion);
  map["Localization: edges matches"]    = toDoubleVector(this->EdgePointRejectionLocalization);
  map["Localization: planes matches"]   = toDoubleVector(this->PlanarPointRejectionLocalization);
  return map;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetOutputFrame()
{
  PointCloud::Ptr output(new PointCloud);

  // Transform from LiDAR sensor to BASE coordinate system,
  // followed by rigid transform or undistortion
  if (this->Undistortion)
  {
    const Eigen::Isometry3d beginPose = this->TworldFrameStart * this->BaseToLidarOffset;
    const Eigen::Isometry3d endPose = this->Tworld * this->BaseToLidarOffset;
    LinearTransformInterpolator<double> transformInterpolator(beginPose, endPose);

    output->header = this->CurrentFrame->header;
    output->points.reserve(this->CurrentFrame->size());
    for (const Slam::Point& p : *this->CurrentFrame)
      output->push_back(TransformPoint(p, transformInterpolator(p.time)));
  }
  else
  {
    const Eigen::Isometry3d endPose = this->Tworld * this->BaseToLidarOffset;
    pcl::transformPointCloud(*this->CurrentFrame, *output, endPose.matrix());
  }

  output->header.frame_id = this->WorldFrameId;
  return output;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetEdgesMap() const
{
  PointCloud::Ptr map = this->EdgesPointsLocalMap->Get();
  map->header.frame_id = this->WorldFrameId;
  map->header.stamp = SecToPclStamp(this->GetWorldTransform().time);
  return map;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetPlanarsMap() const
{
  PointCloud::Ptr map = this->PlanarPointsLocalMap->Get();
  map->header.frame_id = this->WorldFrameId;
  map->header.stamp = SecToPclStamp(this->GetWorldTransform().time);
  return map;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetBlobsMap() const
{
  PointCloud::Ptr map = this->BlobsPointsLocalMap->Get();
  map->header.frame_id = this->WorldFrameId;
  map->header.stamp = SecToPclStamp(this->GetWorldTransform().time);
  return map;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetEdgesKeypoints(bool worldCoordinates) const
{
  return worldCoordinates ? this->CurrentWorldEdgesPoints : this->CurrentEdgesPoints;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetPlanarsKeypoints(bool worldCoordinates) const
{
  return worldCoordinates ? this->CurrentWorldPlanarsPoints : this->CurrentPlanarsPoints;
}

//-----------------------------------------------------------------------------
Slam::PointCloud::Ptr Slam::GetBlobsKeypoints(bool worldCoordinates) const
{
  return worldCoordinates ? this->CurrentWorldBlobsPoints : this->CurrentBlobsPoints;
}

//==============================================================================
//   Main SLAM steps
//==============================================================================

//-----------------------------------------------------------------------------
void Slam::UpdateFrameAndState(const PointCloud::Ptr& inputPc)
{
  // Check frame dropping
  unsigned int droppedFrames = inputPc->header.seq - this->PreviousFrameSeq - 1;
  if ((this->PreviousFrameSeq > 0) && (droppedFrames > 0))
    std::cerr << "[WARNING] SLAM dropped " << droppedFrames << " frame" << (droppedFrames > 1 ? "s" : "") << ".\n\n";
  this->PreviousFrameSeq = inputPc->header.seq;

  // Estimate world pose at current time
  // Use previous pose as new pose estimation
  Eigen::Isometry3d TworldEstimation = this->Tworld;
  // Or linearly extrapolate previous motion to estimate new pose
  if (this->NbrFrameProcessed >= 2 &&
      (this->EgoMotion == EgoMotionMode::MOTION_EXTRAPOLATION ||
       this->EgoMotion == EgoMotionMode::MOTION_EXTRAPOLATION_AND_REGISTRATION))
  {
    // Estimate new Tworld with a constant velocity model
    const double t = PclStampToSec(inputPc->header.stamp);
    const double t1 = this->LogTrajectory[this->LogTrajectory.size() - 1].time;
    const double t0 = this->LogTrajectory[this->LogTrajectory.size() - 2].time;
    TworldEstimation = LinearInterpolation(this->PreviousTworld, this->Tworld, t, t0, t1);
  }
  this->PreviousTworld = this->Tworld;
  this->Tworld = TworldEstimation;
  this->Trelative = this->PreviousTworld.inverse() * this->Tworld;

  SET_COUT_FIXED_PRECISION(3);
  PRINT_VERBOSE(2, "========== Update SLAM State ==========\n"
                   "Estimated Ego-Motion:\n"
                   " translation = [" << this->Trelative.translation().transpose()                          << "]\n"
                   " rotation    = [" << Rad2Deg(RotationMatrixToRPY(this->Trelative.linear())).transpose() << "]");
  RESET_COUT_FIXED_PRECISION;

  // Current keypoints become previous ones
  this->PreviousEdgesPoints = this->CurrentEdgesPoints;
  this->PreviousPlanarsPoints = this->CurrentPlanarsPoints;

  // Copy the input cloud to avoid modifying input
  // CHECK : no other way to avoid this copy? Could be heavy with 64 or 128...
  this->CurrentFrame = inputPc->makeShared();

  // Get frame duration
  double frameStartTime = std::numeric_limits<double>::max(),
         frameEndTime   = std::numeric_limits<double>::min();
  for (const Point& point: *this->CurrentFrame)
  {
    frameStartTime = std::min(frameStartTime, point.time);
    frameEndTime = std::max(frameEndTime, point.time);
  }
  this->FrameDuration = frameEndTime - frameStartTime;

  // Modify the points so that time becomes a relative advancement (between 0 and 1)
  for (Point& point: *this->CurrentFrame)
    point.time = (point.time - frameStartTime) / this->FrameDuration;
}

//-----------------------------------------------------------------------------
void Slam::ExtractKeypoints(const std::vector<size_t>& laserIdMapping)
{
  auto transformToBase = [this](const Slam::PointCloud::Ptr& inputPc)
  {
    PointCloud::Ptr baseCloud;
    // If transform to apply is identity, avoid much work and just change frame id if it is defined
    if (this->BaseToLidarOffset.isApprox(Eigen::Isometry3d::Identity()))
    {
      baseCloud = inputPc;
      baseCloud->header.frame_id = this->BaseFrameId.empty() ? inputPc->header.frame_id : this->BaseFrameId;
    }
    // If transform is set and non trivial, run transformation and notify it by changing frame id
    else
    {
      baseCloud.reset(new PointCloud);
      pcl::transformPointCloud(*inputPc, *baseCloud, this->BaseToLidarOffset.matrix());
      baseCloud->header.frame_id = this->BaseFrameId.empty() ? this->BaseFrameIdDefault : this->BaseFrameId;
    }
    return baseCloud;
  };

  // Extract keypoints from input cloud,
  this->KeyPointsExtractor->ComputeKeyPoints(this->CurrentFrame, laserIdMapping);

  // Get keypoints and transform them from LIDAR to BASE coordinates if needed.
  this->CurrentEdgesPoints   = transformToBase(this->KeyPointsExtractor->GetEdgePoints());
  this->CurrentPlanarsPoints = transformToBase(this->KeyPointsExtractor->GetPlanarPoints());
  this->CurrentBlobsPoints   = transformToBase(this->KeyPointsExtractor->GetBlobPoints());

  // Set keypoints bounds in rolling grids to reduce map searching radius during localization step
  this->SetFrameMinMaxKeypoints();

  PRINT_VERBOSE(2, "========== Keypoints extraction ==========\n"
                   "Extracted features : " << this->CurrentEdgesPoints->size()   << " edges, "
                                           << this->CurrentPlanarsPoints->size() << " planes, "
                                           << this->CurrentBlobsPoints->size()   << " blobs.");
}

//-----------------------------------------------------------------------------
void Slam::ComputeEgoMotion()
{
  // kd-tree to process fast nearest neighbor
  // among the keypoints of the previous pointcloud
  // CHECK : This step behaves strangely much slower when using OpenMP.
  // This section processing duration (arbitrary unit) :
  //  1. without OpenMP included nor used in any code (nor in Slam or SSKE) : time = 1.
  //  2. with OpenMP, globally used with only 1 thread : time = 1.
  //  3. with OpenMP, globally used with 2 threads : time = 2.
  //  4. with OpenMP used in other parts but removing here parallel section : time = 2. ???
  // => Even if we don't use OpenMP, it is slower ! We expect (4) to behaves at similarly as (1) or (2)...
  IF_VERBOSE(3, InitTime("EgoMotion : build KD tree"));
  KDTreePCLAdaptor kdtreePreviousEdges, kdtreePreviousPlanes;
  #pragma omp parallel sections num_threads(std::min(this->NbThreads, 2))
  {
    #pragma omp section
    kdtreePreviousEdges.Reset(this->PreviousEdgesPoints);
    #pragma omp section
    kdtreePreviousPlanes.Reset(this->PreviousPlanarsPoints);
  }

  PRINT_VERBOSE(2, "========== Ego-Motion ==========\n"
                   << "Keypoints extracted from previous frame : "
                   << this->PreviousEdgesPoints->size() << " edges, "
                   << this->PreviousPlanarsPoints->size() << " planes");

  IF_VERBOSE(3, StopTimeAndDisplay("EgoMotion : build KD tree"));

  // Reset ICP results
  const unsigned int nbKeypoints =   this->CurrentEdgesPoints->size()
                                   + this->CurrentPlanarsPoints->size();
  this->Xvalues.reserve(nbKeypoints);
  this->Avalues.reserve(nbKeypoints);
  this->Pvalues.reserve(nbKeypoints);
  this->TimeValues.reserve(nbKeypoints);
  this->residualCoefficient.reserve(nbKeypoints);
  this->EdgePointRejectionEgoMotion.assign(this->CurrentEdgesPoints->size(), MatchingResult::UNKOWN);
  this->PlanarPointRejectionEgoMotion.assign(this->CurrentPlanarsPoints->size(), MatchingResult::UNKOWN);

  IF_VERBOSE(3, InitTime("Ego-Motion : whole ICP-LM loop"));

  // ICP - Levenberg-Marquardt loop
  // At each step of this loop an ICP matching is performed. Once the keypoints
  // are matched, we estimate the the 6-DOF parameters by minimizing the
  // non-linear least square cost function using Levenberg-Marquardt algorithm.
  for (unsigned int icpIter = 0; icpIter < this->EgoMotionICPMaxIter; ++icpIter)
  {
    IF_VERBOSE(3, InitTime("  Ego-Motion : ICP"));

    // clear all keypoints matching data
    this->ResetDistanceParameters();

    // loop over edges if there is enough previous edge keypoints
    if (!this->CurrentEdgesPoints->empty() && this->PreviousEdgesPoints->size() > this->EgoMotionLineDistanceNbrNeighbors)
    {
      #pragma omp parallel for num_threads(this->NbThreads) schedule(guided, 8)
      for (int edgeIndex = 0; edgeIndex < static_cast<int>(this->CurrentEdgesPoints->size()); ++edgeIndex)
      {
        // Find the closest correspondence edge line of the current edge point
        // Compute the parameters of the point - line distance
        // i.e A = (I - n*n.t)^2 with n being the director vector
        // and P a point of the line
        const Point& currentPoint = this->CurrentEdgesPoints->points[edgeIndex];
        MatchingResult rejectionIndex = this->ComputeLineDistanceParameters(kdtreePreviousEdges, currentPoint, MatchingMode::EGO_MOTION);
        this->EdgePointRejectionEgoMotion[edgeIndex] = rejectionIndex;
        #pragma omp atomic
        this->MatchRejectionHistogramLine[rejectionIndex]++;
      }
    }

    // loop over planars if there is enough previous planar keypoints
    if (!this->CurrentPlanarsPoints->empty() && this->PreviousPlanarsPoints->size() > this->EgoMotionPlaneDistanceNbrNeighbors)
    {
      #pragma omp parallel for num_threads(this->NbThreads) schedule(guided, 8)
      for (int planarIndex = 0; planarIndex < static_cast<int>(this->CurrentPlanarsPoints->size()); ++planarIndex)
      {
        // Find the closest correspondence plane of the current planar point
        // Compute the parameters of the point - plane distance
        // i.e A = n * n.t with n being a normal of the plane
        // and is a point of the plane
        const Point& currentPoint = this->CurrentPlanarsPoints->points[planarIndex];
        MatchingResult rejectionIndex = this->ComputePlaneDistanceParameters(kdtreePreviousPlanes, currentPoint, MatchingMode::EGO_MOTION);
        this->PlanarPointRejectionEgoMotion[planarIndex] = rejectionIndex;
        #pragma omp atomic
        this->MatchRejectionHistogramPlane[rejectionIndex]++;
      }
    }

    // ICP matching summary
    this->EgoMotionEdgesPointsUsed = this->MatchRejectionHistogramLine[MatchingResult::SUCCESS];
    this->EgoMotionPlanesPointsUsed = this->MatchRejectionHistogramPlane[MatchingResult::SUCCESS];

    // Skip this frame if there are too few geometric keypoints matched
    if ((this->EgoMotionEdgesPointsUsed + this->EgoMotionPlanesPointsUsed) < this->MinNbrMatchedKeypoints)
    {
      std::cerr << "[WARNING] Not enough keypoints, EgoMotion skipped for this frame.\n";
      break;
    }

    IF_VERBOSE(3, StopTimeAndDisplay("  Ego-Motion : ICP"));
    IF_VERBOSE(3, InitTime("  Ego-Motion : build ceres problem"));

    // Arctan loss scale factor to saturate costs according to their distance
    double lossScale = this->EgoMotionInitLossScale + icpIter * (this->EgoMotionFinalLossScale - this->EgoMotionInitLossScale) / this->EgoMotionICPMaxIter;

    // Convert Isometry to 6 DoF state vector (rX, rY, rZ, X, Y, Z)
    Eigen::Vector6d TrelativeArray = IsometryToRPYXYZ(this->Trelative);

    // We want to estimate our 6-DOF parameters using a non linear least square
    // minimization. The non linear part comes from the parametrization of the
    // rotation endomorphism SO(3). To minimize it, we use CERES to perform
    // Levenberg-Marquardt algorithm.
    ceres::Problem problem;
    for (unsigned int k = 0; k < Xvalues.size(); ++k)
    {
      ceres::CostFunction* cost_function = new ceres::AutoDiffCostFunction<CostFunctions::MahalanobisDistanceAffineIsometryResidual, 1, 6>(
                                            new CostFunctions::MahalanobisDistanceAffineIsometryResidual(this->Avalues[k], this->Pvalues[k],
                                                                                                         this->Xvalues[k], this->residualCoefficient[k]));
      problem.AddResidualBlock(cost_function,
                               new ceres::ScaledLoss(new ceres::ArctanLoss(lossScale), this->residualCoefficient[k], ceres::TAKE_OWNERSHIP),
                               TrelativeArray.data());
    }

    IF_VERBOSE(3, StopTimeAndDisplay("  Ego-Motion : build ceres problem"));
    IF_VERBOSE(3, InitTime("  Ego-Motion : LM optim"));

    ceres::Solver::Options options;
    options.max_num_iterations = this->EgoMotionLMMaxIter;
    options.linear_solver_type = ceres::DENSE_QR;  // TODO : try also DENSE_NORMAL_CHOLESKY or SPARSE_NORMAL_CHOLESKY
    options.minimizer_progress_to_stdout = false;
    options.num_threads = this->NbThreads;

    ceres::Solver::Summary summary;
    ceres::Solve(options, &problem, &summary);
    PRINT_VERBOSE(4, summary.BriefReport());

    // Unpack Trelative back to isometry
    this->Trelative = RPYXYZtoIsometry(TrelativeArray);

    IF_VERBOSE(3, StopTimeAndDisplay("  Ego-Motion : LM optim"));

    // If no L-M iteration has been made since the last ICP matching, it means
    // that we reached a local minimum for the ICP-LM algorithm.
    if (summary.num_successful_steps == 1)
    {
      break;
    }
  }

  IF_VERBOSE(3, StopTimeAndDisplay("Ego-Motion : whole ICP-LM loop"));

  PRINT_VERBOSE(2, "Matched keypoints: " << this->Xvalues.size() << " ("
                    << this->EgoMotionEdgesPointsUsed << " edges, "
                    << this->EgoMotionPlanesPointsUsed << " planes).");
}

//-----------------------------------------------------------------------------
void Slam::Localization()
{
  // Get keypoints from maps and build kd-trees for fast nearest neighbors search
  IF_VERBOSE(3, InitTime("Localization : keypoints extraction"));
  PointCloud::Ptr subEdgesPointsLocalMap, subPlanarPointsLocalMap, subBlobPointsLocalMap(new PointCloud);
  KDTreePCLAdaptor kdtreeEdges, kdtreePlanes, kdtreeBlobs;

  auto extractMapKeypointsAndBuildKdTree = [this](const RollingGrid& map, PointCloud::Ptr& keypoints, KDTreePCLAdaptor& kdTree)
  {
    keypoints = map.Get(this->Tworld.translation());
    kdTree.Reset(keypoints);
  };

  // CHECK : This step behaves strangely much slower when using OpenMP.
  // This section processing duration (arbitrary unit) :
  //  1. without OpenMP included nor used in any code (nor in Slam or SSKE) : time = 1.
  //  2. with OpenMP, globally used with only 1 thread                      : time ~ 1.
  //  3. with OpenMP, globally used with 3 threads                          : time ~ 2.
  //  4. with OpenMP used in other parts but removing here parallel section : time ~ 2.2 ?!
  // => Even if we don't use OpenMP, it is slower ! We expect (4) to behaves at similarly as (1) or (2)...
  #pragma omp parallel sections num_threads(std::min(this->NbThreads, 3))
  {
    #pragma omp section
    extractMapKeypointsAndBuildKdTree(*this->EdgesPointsLocalMap, subEdgesPointsLocalMap, kdtreeEdges);
    #pragma omp section
    extractMapKeypointsAndBuildKdTree(*this->PlanarPointsLocalMap, subPlanarPointsLocalMap, kdtreePlanes);
    #pragma omp section
    if (!this->FastSlam)
      extractMapKeypointsAndBuildKdTree(*this->BlobsPointsLocalMap, subBlobPointsLocalMap, kdtreeBlobs);
  }

  PRINT_VERBOSE(2, "========== Localization ==========\n"
                   << "Keypoints extracted from map : "
                   << subEdgesPointsLocalMap->size() << " edges, "
                   << subPlanarPointsLocalMap->size() << " planes, "
                   << subBlobPointsLocalMap->size() << " blobs");

  IF_VERBOSE(3, StopTimeAndDisplay("Localization : keypoints extraction"));

  // Reset ICP results
  const unsigned int nbKeypoints =   this->CurrentEdgesPoints->size()
                                   + this->CurrentPlanarsPoints->size()
                                   + this->CurrentBlobsPoints->size();
  this->Xvalues.reserve(nbKeypoints);
  this->Avalues.reserve(nbKeypoints);
  this->Pvalues.reserve(nbKeypoints);
  this->TimeValues.reserve(nbKeypoints);
  this->residualCoefficient.reserve(nbKeypoints);
  this->EdgePointRejectionLocalization.assign(this->CurrentEdgesPoints->size(), MatchingResult::UNKOWN);
  this->PlanarPointRejectionLocalization.assign(this->CurrentPlanarsPoints->size(), MatchingResult::UNKOWN);
  this->BlobPointRejectionLocalization.assign(this->CurrentBlobsPoints->size(), MatchingResult::UNKOWN);

  IF_VERBOSE(3, InitTime("Localization : whole ICP-LM loop"));

  // ICP - Levenberg-Marquardt loop
  // At each step of this loop an ICP matching is performed. Once the keypoints
  // are matched, we estimate the the 6-DOF parameters by minimizing the
  // non-linear least square cost function using Levenberg-Marquardt algorithm.
  for (unsigned int icpIter = 0; icpIter < this->LocalizationICPMaxIter; ++icpIter)
  {
    IF_VERBOSE(3, InitTime("  Localization : ICP"));

    // clear all keypoints matching data
    this->ResetDistanceParameters();

    // loop over edges
    if (!this->CurrentEdgesPoints->empty() && subEdgesPointsLocalMap->size() > this->LocalizationLineDistanceNbrNeighbors)
    {
      #pragma omp parallel for num_threads(this->NbThreads) schedule(guided, 8)
      for (int edgeIndex = 0; edgeIndex < static_cast<int>(this->CurrentEdgesPoints->size()); ++edgeIndex)
      {
        // Find the closest correspondence edge line of the current edge point
        const Point& currentPoint = this->CurrentEdgesPoints->points[edgeIndex];
        MatchingResult rejectionIndex = this->ComputeLineDistanceParameters(kdtreeEdges, currentPoint, MatchingMode::LOCALIZATION);
        this->EdgePointRejectionLocalization[edgeIndex] = rejectionIndex;
        #pragma omp atomic
        this->MatchRejectionHistogramLine[rejectionIndex]++;
      }
    }

    // loop over surfaces
    if (!this->CurrentPlanarsPoints->empty() && subPlanarPointsLocalMap->size() > this->LocalizationPlaneDistanceNbrNeighbors)
    {
      #pragma omp parallel for num_threads(this->NbThreads) schedule(guided, 8)
      for (int planarIndex = 0; planarIndex < static_cast<int>(this->CurrentPlanarsPoints->size()); ++planarIndex)
      {
        // Find the closest correspondence plane of the current planar point
        const Point& currentPoint = this->CurrentPlanarsPoints->points[planarIndex];
        MatchingResult rejectionIndex = this->ComputePlaneDistanceParameters(kdtreePlanes, currentPoint, MatchingMode::LOCALIZATION);
        this->PlanarPointRejectionLocalization[planarIndex] = rejectionIndex;
        #pragma omp atomic
        this->MatchRejectionHistogramPlane[rejectionIndex]++;
      }
    }

    // loop over blobs
    if (!this->FastSlam && !this->CurrentBlobsPoints->empty()  && subBlobPointsLocalMap->size() > this->LocalizationBlobDistanceNbrNeighbors)
    {
      #pragma omp parallel for num_threads(this->NbThreads) schedule(guided, 8)
      for (int blobIndex = 0; blobIndex < static_cast<int>(this->CurrentBlobsPoints->size()); ++blobIndex)
      {
        // Find the closest correspondence plane of the current blob point
        const Point& currentPoint = this->CurrentBlobsPoints->points[blobIndex];
        MatchingResult rejectionIndex = this->ComputeBlobsDistanceParameters(kdtreeBlobs, currentPoint, MatchingMode::LOCALIZATION);
        this->BlobPointRejectionLocalization[blobIndex] = rejectionIndex;
        #pragma omp atomic
        this->MatchRejectionHistogramBlob[rejectionIndex]++;
      }
    }

    // ICP matching summary
    this->LocalizationEdgesPointsUsed  = this->MatchRejectionHistogramLine[MatchingResult::SUCCESS];
    this->LocalizationPlanesPointsUsed = this->MatchRejectionHistogramPlane[MatchingResult::SUCCESS];
    this->LocalizationBlobsPointsUsed  = this->MatchRejectionHistogramBlob[MatchingResult::SUCCESS];

    // Skip this frame if there is too few geometric keypoints matched
    if ((this->LocalizationEdgesPointsUsed + this->LocalizationPlanesPointsUsed + this->LocalizationBlobsPointsUsed) < this->MinNbrMatchedKeypoints)
    {
      std::cerr << "[WARNING] Not enough keypoints, Localization skipped for this frame.\n";
      break;
    }

    IF_VERBOSE(3, StopTimeAndDisplay("  Localization : ICP"));
    IF_VERBOSE(3, InitTime("  Localization : build ceres problem"));

    // Arctan loss scale factor to saturate costs according to their distance
    double lossScale = this->LocalizationInitLossScale + icpIter * (this->LocalizationFinalLossScale - this->LocalizationInitLossScale) / this->LocalizationICPMaxIter;

    // Convert isometries to 6D state vectors : rX, rY, rZ, X, Y, Z
    Eigen::Vector6d TworldArray = IsometryToRPYXYZ(this->Tworld);  // pose at the end of frame
    Eigen::Vector6d TworldStartArray = IsometryToRPYXYZ(this->TworldFrameStart);  // pose at the beginning of frame

    // We want to estimate our 6-DOF parameters using a non linear least square
    // minimization. The non linear part comes from the parametrization of the
    // rotation endomorphism SO(3). To minimize it, we use CERES to perform
    // Levenberg-Marquardt algorithm.
    ceres::Problem problem;
    if (this->Undistortion == UndistortionMode::OPTIMIZED)
    {
      for (unsigned int k = 0; k < Xvalues.size(); ++k)
      {
        ceres::CostFunction* cost_function = new ceres::AutoDiffCostFunction<CostFunctions::MahalanobisDistanceInterpolatedMotionResidual, 1, 6, 6>(
                                                new CostFunctions::MahalanobisDistanceInterpolatedMotionResidual(
                                                  this->Avalues[k], this->Pvalues[k], this->Xvalues[k], this->TimeValues[k], this->residualCoefficient[k]));
        problem.AddResidualBlock(cost_function, 
                                 new ceres::ScaledLoss(new ceres::ArctanLoss(lossScale), this->residualCoefficient[k], ceres::TAKE_OWNERSHIP),
                                 TworldStartArray.data(), TworldArray.data());
      }
    }
    else
    {
      for (unsigned int k = 0; k < Xvalues.size(); ++k)
      {
        ceres::CostFunction* cost_function = new ceres::AutoDiffCostFunction<CostFunctions::MahalanobisDistanceAffineIsometryResidual, 1, 6>(
                                                new CostFunctions::MahalanobisDistanceAffineIsometryResidual(
                                                  this->Avalues[k], this->Pvalues[k], this->Xvalues[k], this->residualCoefficient[k]));
        problem.AddResidualBlock(cost_function,
                                 new ceres::ScaledLoss(new ceres::ArctanLoss(lossScale), this->residualCoefficient[k], ceres::TAKE_OWNERSHIP),
                                 TworldArray.data());
      }
    }

    IF_VERBOSE(3, StopTimeAndDisplay("  Localization : build ceres problem"));
    IF_VERBOSE(3, InitTime("  Localization : LM optim"));

    ceres::Solver::Options options;
    options.max_num_iterations = this->LocalizationLMMaxIter;
    options.linear_solver_type = ceres::DENSE_QR;  // TODO test other optimizer
    options.minimizer_progress_to_stdout = false;
    options.num_threads = this->NbThreads;

    ceres::Solver::Summary summary;
    ceres::Solve(options, &problem, &summary);
    PRINT_VERBOSE(4, summary.BriefReport());

    // Unpack Tworld and TworldFrameStart
    this->Tworld = RPYXYZtoIsometry(TworldArray);
    this->Trelative = this->PreviousTworld.inverse() * this->Tworld;
    if (this->Undistortion == UndistortionMode::OPTIMIZED)
    {
      this->TworldFrameStart = RPYXYZtoIsometry(TworldStartArray);
      this->WithinFrameMotion.SetTransforms(this->TworldFrameStart, this->Tworld);
    }
    else if (this->Undistortion == UndistortionMode::APPROXIMATED)
    {
      this->TworldFrameStart = this->InterpolateBeginScanPose();
      this->WithinFrameMotion.SetTransforms(this->TworldFrameStart, this->Tworld);
    }

    IF_VERBOSE(3, StopTimeAndDisplay("  Localization : LM optim"));

    // If no L-M iteration has been made since the last ICP matching, it means
    // that we reached a local minimum for the ICP-LM algorithm.
    // We evaluate the quality of the Tworld optimization using an approximate
    // computation of the variance covariance matrix.
    if ((summary.num_successful_steps == 1) || (icpIter == this->LocalizationICPMaxIter - 1))
    {
      // Covariance computation options
      ceres::Covariance::Options covOptions;
      covOptions.apply_loss_function = true;
      covOptions.algorithm_type = ceres::CovarianceAlgorithmType::DENSE_SVD;

      // Computation of the variance-covariance matrix
      ceres::Covariance covariance(covOptions);
      std::vector<std::pair<const double*, const double*>> covariance_blocks;
      const double* paramBlock = TworldArray.data();
      covariance_blocks.push_back(std::make_pair(paramBlock, paramBlock));
      covariance.Compute(covariance_blocks, &problem);
      covariance.GetCovarianceBlock(paramBlock, paramBlock, this->TworldCovariance.data());
      break;
    }
  }

  IF_VERBOSE(3, StopTimeAndDisplay("Localization : whole ICP-LM loop"));

  Eigen::SelfAdjointEigenSolver<Eigen::Matrix6d> eig(this->TworldCovariance);
  Eigen::Vector6d D = eig.eigenvalues();
  this->LocalizationVarianceError = D(5);

  PRINT_VERBOSE(2, "Matched keypoints: " << this->Xvalues.size() << " ("
                   << this->LocalizationEdgesPointsUsed  << " edges, "
                   << this->LocalizationPlanesPointsUsed << " planes, "
                   << this->LocalizationBlobsPointsUsed  << " blobs)."
                   "\nCovariance eigen values: " << D.transpose() <<
                   "\nMaximum variance eigen vector: " << eig.eigenvectors().col(5).transpose() <<
                   "\nMaximum variance: " << D(5));
}

//-----------------------------------------------------------------------------
void Slam::UpdateMapsUsingTworld()
{
  // it would be nice to add the point from the frame directly to the map
  auto updateMap = [this](std::shared_ptr<RollingGrid> map, PointCloud::Ptr baseFrame, PointCloud::Ptr worldFrame)
  {
    // Transform keypoints to WORLD coordinates
    worldFrame->clear();
    worldFrame->points.reserve(baseFrame->size());
    worldFrame->header = baseFrame->header;
    worldFrame->header.frame_id = this->WorldFrameId;
    if (this->Undistortion)
      for (const Point& p : *baseFrame)
        worldFrame->push_back(TransformPoint(p, this->WithinFrameMotion(p.time)));
    else
      for (const Point& p : *baseFrame)
        worldFrame->push_back(TransformPoint(p, this->Tworld));
    // Roll grid to current position, and add new keypoints
    map->Roll(this->Tworld.translation());
    map->Add(worldFrame);
  };

  // run maps update
  #pragma omp parallel sections num_threads(std::min(this->NbThreads, 3))
  {
    #pragma omp section
    updateMap(this->EdgesPointsLocalMap, this->CurrentEdgesPoints, this->CurrentWorldEdgesPoints);
    #pragma omp section
    updateMap(this->PlanarPointsLocalMap, this->CurrentPlanarsPoints, this->CurrentWorldPlanarsPoints);
    #pragma omp section
    if (!this->FastSlam)
      updateMap(this->BlobsPointsLocalMap, this->CurrentBlobsPoints, this->CurrentWorldBlobsPoints);
  }
}

//-----------------------------------------------------------------------------
void Slam::LogCurrentFrameState(double time, const std::string& frameId)
{
  // If logging is enabled
  if (this->LoggingTimeout)
  {
    // Save current frame data to buffer
    this->LogTrajectory.emplace_back(this->Tworld, time, frameId);
    this->LogCovariances.emplace_back(FlipAndConvertCovariance(this->TworldCovariance));
    this->LogEdgesPoints.emplace_back(this->CurrentEdgesPoints, this->LoggingStorage);
    this->LogPlanarsPoints.emplace_back(this->CurrentPlanarsPoints, this->LoggingStorage);
    if (!this->FastSlam)
      this->LogBlobsPoints.emplace_back(this->CurrentBlobsPoints, this->LoggingStorage);

    // If a timeout is defined, forget too old data
    if (this->LoggingTimeout > 0)
    {
      // Forget all previous data older than LoggingTimeout, but keep at least 2 last transforms
      while (time - this->LogTrajectory.front().time > this->LoggingTimeout
             && this->LogTrajectory.size() > 2)
      {
        this->LogTrajectory.pop_front();
        this->LogCovariances.pop_front();
        this->LogEdgesPoints.pop_front();
        this->LogPlanarsPoints.pop_front();
        if (!this->FastSlam)
          this->LogBlobsPoints.pop_front();
      }
    }
  }

  // If logging is disabled, only keep last 2 transforms for latency compensation
  else
  {
    this->LogTrajectory.emplace_back(this->Tworld, time, frameId);
    while (this->LogTrajectory.size() > 2)
      this->LogTrajectory.pop_front();
  }
}

//==============================================================================
//   Features associations and optimization
//==============================================================================

//-----------------------------------------------------------------------------
void Slam::ResetDistanceParameters()
{
  this->Xvalues.clear();
  this->Avalues.clear();
  this->Pvalues.clear();
  this->TimeValues.clear();
  this->residualCoefficient.clear();
  this->MatchRejectionHistogramLine.fill(0);
  this->MatchRejectionHistogramPlane.fill(0);
  this->MatchRejectionHistogramBlob.fill(0);
}

//-----------------------------------------------------------------------------
void Slam::ComputePointInitAndFinalPose(MatchingMode matchingMode, const Point& p, Eigen::Vector3d& pInit, Eigen::Vector3d& pFinal)
{
  // Undistortion can only be done during Localization step
  const bool isLocalizationStep = matchingMode == MatchingMode::LOCALIZATION;
  const Eigen::Vector3d pos = p.getVector3fMap().cast<double>();

  if (this->Undistortion == UndistortionMode::OPTIMIZED && isLocalizationStep)
  {
    pInit = pos;
    pFinal = this->WithinFrameMotion(p.time) * pos;
  }
  else if (this->Undistortion == UndistortionMode::APPROXIMATED && isLocalizationStep)
  {
    pFinal = this->WithinFrameMotion(p.time) * pos;
    pInit = this->Tworld.inverse() * pFinal;
  }
  else
  {
    const Eigen::Isometry3d& transform = isLocalizationStep ? this->Tworld : this->Trelative;
    pInit = pos;
    pFinal = transform * pos;
  }
}

//-----------------------------------------------------------------------------
Slam::MatchingResult Slam::ComputeLineDistanceParameters(KDTreePCLAdaptor& kdtreePreviousEdges, const Point& p, MatchingMode matchingMode)
{
  // =====================================================
  // Transform the point using the current pose estimation

  // Rigid transform or time continuous motion model to take into account the
  // rolling shutter distortion.
  Eigen::Vector3d pInit, pFinal;
  this->ComputePointInitAndFinalPose(matchingMode, p, pInit, pFinal);

  // ===================================================
  // Get neighboring points in previous set of keypoints

  unsigned int minNeighbors;     //< minimum numbers of neighbors below which neighborhood is rejected
  unsigned int requiredNearest;  //< number of neighbors required to approximate the corresponding edge line
  double eigenValuesRatio;       //< min eigen values ratio to conbsider a neighborhood as flat
  double squaredMaxDist;         //< maximum distance between keypoints and their computed line
  std::vector<int> nearestIndex;
  std::vector<double> nearestDist;
  if (matchingMode == MatchingMode::EGO_MOTION)
  {
    requiredNearest = this->EgoMotionLineDistanceNbrNeighbors;
    eigenValuesRatio = this->EgoMotionLineDistancefactor;
    squaredMaxDist = this->EgoMotionMaxLineDistance * this->EgoMotionMaxLineDistance;
    minNeighbors = this->EgoMotionMinimumLineNeighborRejection;
    GetEgoMotionLineSpecificNeighbor(nearestIndex, nearestDist, requiredNearest, kdtreePreviousEdges, pFinal.data());
  }
  else if (matchingMode == MatchingMode::LOCALIZATION)
  {
    requiredNearest = this->LocalizationLineDistanceNbrNeighbors;
    eigenValuesRatio = this->LocalizationLineDistancefactor;
    squaredMaxDist = this->LocalizationMaxLineDistance * this->LocalizationMaxLineDistance;
    minNeighbors = this->LocalizationMinimumLineNeighborRejection;
    GetLocalizationLineSpecificNeighbor(nearestIndex, nearestDist, this->LocalizationLineMaxDistInlier, requiredNearest, kdtreePreviousEdges, pFinal.data());
  }
  else
  {
    throw "ComputeLineDistanceParameters function got invalid step parameter";
  }

  // If not enough neighbors, abort
  if (nearestIndex.size() < minNeighbors)
  {
    return MatchingResult::NOT_ENOUGH_NEIGHBORS;
  }
  requiredNearest = nearestIndex.size();

  // If the nearest edges are too far from the current edge keypoint,
  // we skip this point.
  if (nearestDist.back() > this->MaxDistanceForICPMatching)
  {
    return MatchingResult::NEIGHBORS_TOO_FAR;
  }

  // Shortcut to keypoints cloud
  const PointCloud& previousEdgesPoints = *kdtreePreviousEdges.getInputCloud();

  // =======================================================
  // Check if neighborhood is a good line candidate with PCA

  // Compute PCA to determine best line approximation
  // of the requiredNearest nearest edges points extracted.
  // Thanks to the PCA we will check the shape of the neighborhood
  // and keep it if it is well distributed along a line.
  Eigen::MatrixXd data(requiredNearest, 3);
  for (unsigned int k = 0; k < requiredNearest; k++)
  {
    const Point& pt = previousEdgesPoints[nearestIndex[k]];
    data.row(k) << pt.x, pt.y, pt.z;
  }
  Eigen::Vector3d mean;
  Eigen::SelfAdjointEigenSolver<Eigen::Matrix3d> eig = ComputePCA(data, mean);

  // PCA eigenvalues
  Eigen::Vector3d D = eig.eigenvalues();

  // If the first eigen value is significantly higher than the second one,
  // it means that the sourrounding points are distributed on an edge line.
  // Otherwise, discard this bad unstructured neighborhood.
  if (D(2) < eigenValuesRatio * D(1))
  {
    return MatchingResult::BAD_PCA_STRUCTURE;
  }

  // =============================================
  // Compute point-to-line optimization parameters

  // n is the director vector of the line
  Eigen::Vector3d n = eig.eigenvectors().col(2);

  // A = (I-n*n.t).t * (I-n*n.t) = (I - n*n.t)^2
  // since (I-n*n.t) is a symmetric matrix
  // Then it comes A (I-n*n.t)^2 = (I-n*n.t) since
  // A is the matrix of a projection endomorphism
  Eigen::Matrix3d A = Eigen::Matrix3d::Identity() - n * n.transpose();

  // =========================
  // Check parameters validity

  // It would be the case if P1 = P2, for instance if the sensor has some dual
  // returns that hit the same point.
  if (!std::isfinite(A(0, 0)))
  {
    return MatchingResult::INVALID_NUMERICAL;
  }

  // Evaluate the distance from the fitted line distribution of the neighborhood
  double meanSquaredDist = 0.;
  for (unsigned int nearestPointIndex: nearestIndex)
  {
    const Point& pt = previousEdgesPoints[nearestPointIndex];
    Eigen::Vector3d Xtemp(pt.x, pt.y, pt.z);
    double squaredDist = (Xtemp - mean).transpose() * A * (Xtemp - mean);
    // CHECK invalidate all neighborhood even if only one point is bad?
    if (squaredDist > squaredMaxDist)
    {
      return MatchingResult::MSE_TOO_LARGE;
    }
    meanSquaredDist += squaredDist;
  }
  meanSquaredDist /= static_cast<double>(requiredNearest);

  // ===========================================
  // Add valid parameters for later optimization

  // Quality score of the point-to-line match
  double fitQualityCoeff = 1.0 - std::sqrt(meanSquaredDist / squaredMaxDist);

  // Store the distance parameters values
  #pragma omp critical(addIcpMatch)
  {
    this->Avalues.emplace_back(A);
    this->Pvalues.emplace_back(mean);
    this->Xvalues.emplace_back(pInit);
    this->TimeValues.emplace_back(p.time);
    this->residualCoefficient.emplace_back(fitQualityCoeff);
  }
  return MatchingResult::SUCCESS;
}

//-----------------------------------------------------------------------------
Slam::MatchingResult Slam::ComputePlaneDistanceParameters(KDTreePCLAdaptor& kdtreePreviousPlanes, const Point& p, MatchingMode matchingMode)
{
  // =====================================================
  // Transform the point using the current pose estimation

  // Rigid transform or time continuous motion model to take into account the
  // rolling shutter distortion.
  Eigen::Vector3d pInit, pFinal;
  this->ComputePointInitAndFinalPose(matchingMode, p, pInit, pFinal);

  // ===================================================
  // Get neighboring points in previous set of keypoints

  unsigned int requiredNearest;  //< number of neighbors planar points required to approximate the corresponding plane
  double significantlyFactor1;   //< PCA eigenvalues ratio to consider a neighborhood fits a plane model :
  double significantlyFactor2;   //<     V2 < factor2 * V1  and  V1 > factor1 * V0
  double squaredMaxDist;         //< maximum distance between keypoints and their computed plane
  if (matchingMode == MatchingMode::EGO_MOTION)
  {
    significantlyFactor1 = this->EgoMotionPlaneDistancefactor1;
    significantlyFactor2 = this->EgoMotionPlaneDistancefactor2;
    requiredNearest = this->EgoMotionPlaneDistanceNbrNeighbors;
    squaredMaxDist = this->EgoMotionMaxPlaneDistance * this->EgoMotionMaxPlaneDistance;
  }
  else if (matchingMode == MatchingMode::LOCALIZATION)
  {
    significantlyFactor1 = this->LocalizationPlaneDistancefactor1;
    significantlyFactor2 = this->LocalizationPlaneDistancefactor2;
    requiredNearest = this->LocalizationPlaneDistanceNbrNeighbors;
    squaredMaxDist = this->LocalizationMaxPlaneDistance * this->LocalizationMaxPlaneDistance;
  }
  else
  {
    throw "ComputeLineDistanceParameters function got invalide step parameter";
  }

  std::vector<int> nearestIndex(requiredNearest, -1);
  std::vector<double> nearestDist(requiredNearest, -1.0);
  kdtreePreviousPlanes.query(pFinal.data(), requiredNearest, nearestIndex.data(), nearestDist.data());

  // It means that there is not enough keypoints in the neighborhood
  if (nearestIndex.back() == -1)
  {
    return MatchingResult::NOT_ENOUGH_NEIGHBORS;
  }

  // If the nearest planar points are too far from the current keypoint,
  // we skip this point.
  if (nearestDist.back() > this->MaxDistanceForICPMatching)
  {
    return MatchingResult::NEIGHBORS_TOO_FAR;
  }

  // Shortcut to keypoints cloud
  const PointCloud& previousPlanesPoints = *kdtreePreviousPlanes.getInputCloud();

  // ========================================================
  // Check if neighborhood is a good plane candidate with PCA

  // Compute PCA to determine best plane approximation
  // of the requiredNearest nearest edges points extracted.
  // Thanks to the PCA we will check the shape of the neighborhood
  // and keep it if it is well distributed along a plane.
  Eigen::MatrixXd data(requiredNearest, 3);
  for (unsigned int k = 0; k < requiredNearest; k++)
  {
    const Point& pt = previousPlanesPoints[nearestIndex[k]];
    data.row(k) << pt.x, pt.y, pt.z;
  }
  Eigen::Vector3d mean;
  Eigen::SelfAdjointEigenSolver<Eigen::Matrix3d> eig = ComputePCA(data, mean);

  // PCA eigenvalues
  Eigen::Vector3d D = eig.eigenvalues();

  // If the second eigen value is close to the highest one and bigger than the
  // smallest one, it means that the points are distributed along a plane.
  // Otherwise, discard this bad unstructured neighborhood.
  if ( significantlyFactor2 * D(1) < D(2) || D(1) < significantlyFactor1 * D(0) )
  {
    return MatchingResult::BAD_PCA_STRUCTURE;
  }

  // ==============================================
  // Compute point-to-plane optimization parameters

  // n is the normal vector of the plane
  Eigen::Vector3d n = eig.eigenvectors().col(0);
  Eigen::Matrix3d A = n * n.transpose();

  // It would be the case if P1 = P2, P1 = P3 or P3 = P2, for instance if the
  // sensor has some dual returns that hit the same point.
  if (!std::isfinite(A(0, 0)))
  {
    return MatchingResult::INVALID_NUMERICAL;
  }

  // Evaluate the distance from the fitted plane distribution of the neighborhood
  double meanSquaredDist = 0.;
  for (unsigned int nearestPointIndex: nearestIndex)
  {
    const Point& pt = previousPlanesPoints[nearestPointIndex];
    Eigen::Vector3d Xtemp(pt.x, pt.y, pt.z);
    double squaredDist = (Xtemp - mean).transpose() * A * (Xtemp - mean);
    // CHECK invalidate all neighborhood even if only one point is bad?
    if (squaredDist > squaredMaxDist)
    {
      return MatchingResult::MSE_TOO_LARGE;
    }
    meanSquaredDist += squaredDist;
  }
  meanSquaredDist /= static_cast<double>(requiredNearest);

  // ===========================================
  // Add valid parameters for later optimization

  // Quality score of the point-to-plane match
  double fitQualityCoeff = 1.0 - std::sqrt(meanSquaredDist / squaredMaxDist);

  // Store the distance parameters values
  #pragma omp critical(addIcpMatch)
  {
    this->Avalues.emplace_back(A);
    this->Pvalues.emplace_back(mean);
    this->Xvalues.emplace_back(pInit);
    this->TimeValues.emplace_back(p.time);
    this->residualCoefficient.emplace_back(fitQualityCoeff);
  }
  return MatchingResult::SUCCESS;
}

//-----------------------------------------------------------------------------
Slam::MatchingResult Slam::ComputeBlobsDistanceParameters(KDTreePCLAdaptor& kdtreePreviousBlobs, const Point& p, MatchingMode matchingMode)
{
  // =====================================================
  // Transform the point using the current pose estimation

  // Rigid transform or time continuous motion model to take into account the
  // rolling shutter distortion.
  Eigen::Vector3d pInit, pFinal;
  this->ComputePointInitAndFinalPose(matchingMode, p, pInit, pFinal);

  // ===================================================
  // Get neighboring points in previous set of keypoints

  unsigned int requiredNearest = 25;  //< number of blob neighbors required to approximate the corresponding ellipsoid
  double maxDist = this->MaxDistanceForICPMatching;  //< maximum distance between keypoints and its neighbors
  float maxDiameter = 4.;

  std::vector<int> nearestIndex(requiredNearest, -1);
  std::vector<double> nearestDist(requiredNearest, -1.0);
  kdtreePreviousBlobs.query(pFinal.data(), requiredNearest, nearestIndex.data(), nearestDist.data());

  // It means that there is not enough keypoints in the neighborhood
  if (nearestIndex.back() == -1)
  {
    return MatchingResult::NOT_ENOUGH_NEIGHBORS;
  }

  // If the nearest blob points are too far from the current keypoint,
  // we skip this point.
  if (nearestDist.back() > maxDist)
  {
    return MatchingResult::NEIGHBORS_TOO_FAR;
  }

  // Shortcut to keypoints cloud
  const PointCloud& previousBlobsPoints = *kdtreePreviousBlobs.getInputCloud();

  // ======================================
  // Check the diameter of the neighborhood

  // If the diameter is too big, we don't want to keep this blob.
  // We must do that since the fitted ellipsoid assumes to encode the local
  // shape of the neighborhood.
  float squaredDiameter = 0.;
  for (unsigned int nearestPointIndexI: nearestIndex)
  {
    const Point& ptI = previousBlobsPoints[nearestPointIndexI];
    for (unsigned int nearestPointIndexJ: nearestIndex)
    {
      const Point& ptJ = previousBlobsPoints[nearestPointIndexJ];
      float squaredDistanceIJ = (ptI.getVector3fMap() - ptJ.getVector3fMap()).squaredNorm();
      squaredDiameter = std::max(squaredDiameter, squaredDistanceIJ);
    }
  }
  if (squaredDiameter > maxDiameter * maxDiameter)
  {
    return MatchingResult::MSE_TOO_LARGE;
  }

  // ======================================================
  // Compute point-to-blob optimization parameters with PCA

  // Compute PCA to determine best ellipsoid approximation
  // of the requiredNearest nearest blobs points extracted.
  // Thanks to the PCA we will check the shape of the neighborhood and
  // tune a distance function adapted to the distribution (Mahalanobis distance)
  Eigen::MatrixXd data(requiredNearest, 3);
  for (unsigned int k = 0; k < requiredNearest; k++)
  {
    const Point& pt = previousBlobsPoints[nearestIndex[k]];
    data.row(k) << pt.x, pt.y, pt.z;
  }
  Eigen::Vector3d mean = data.colwise().mean();
  Eigen::MatrixXd centered = data.rowwise() - mean.transpose();
  Eigen::Matrix3d varianceCovariance = centered.transpose() * centered;

  // Check that the covariance matrix is inversible
  if (std::abs(varianceCovariance.determinant()) < 1e-6)
  {
    return MatchingResult::BAD_PCA_STRUCTURE;
  }

  // Sigma is the inverse of the covariance matrix encoding the mahalanobis distance
  Eigen::Matrix3d sigma = varianceCovariance.inverse();
  Eigen::SelfAdjointEigenSolver<Eigen::Matrix3d> eig(sigma);

  // Rescale the variance covariance matrix to preserve the shape of the
  // mahalanobis distance, but removing the variance values scaling.
  Eigen::Vector3d D = eig.eigenvalues(); D /= D(2);
  Eigen::Matrix3d U = eig.eigenvectors();
  Eigen::Matrix3d A = U * D.asDiagonal() * U.transpose();

  if (!std::isfinite(A.determinant()))
  {
    return MatchingResult::INVALID_NUMERICAL;
  }

  // ===========================================
  // Add valid parameters for later optimization

  // Weigh using the distance between the point and its matching blob.
  // The aim is to prevent wrong matching pulling the pointcloud in a bad direction.
  double fitQualityCoeff = 1.0;//1.0 - nearestDist.back() / maxDist;

  // store the distance parameters values
  #pragma omp critical(addIcpMatch)
  {
    this->Avalues.emplace_back(A);
    this->Pvalues.emplace_back(mean);
    this->Xvalues.emplace_back(pInit);
    this->TimeValues.emplace_back(p.time);
    this->residualCoefficient.emplace_back(fitQualityCoeff);
  }
  return MatchingResult::SUCCESS;
}

//-----------------------------------------------------------------------------
void Slam::GetEgoMotionLineSpecificNeighbor(std::vector<int>& nearestValid, std::vector<double>& nearestValidDist,
                                            unsigned int nearestSearch, KDTreePCLAdaptor& kdtreePreviousEdges, const double pos[3]) const
{
  // Clear vector
  nearestValid.clear();
  nearestValidDist.clear();

  // Get nearest neighbors of the query point
  std::vector<int> nearestIndex(nearestSearch, -1);
  std::vector<double> nearestDist(nearestSearch, -1.0);
  kdtreePreviousEdges.query(pos, nearestSearch, nearestIndex.data(), nearestDist.data());

  // Check neighborhood validity
  unsigned int neighborhoodSize = nearestIndex.size();
  while (nearestIndex[neighborhoodSize - 1] == -1 && neighborhoodSize > 1)
  {
    --neighborhoodSize;
  }

  // Shortcut to keypoints cloud
  const PointCloud& previousEdgesPoints = *kdtreePreviousEdges.getInputCloud();

  // Take the closest point
  const Point& closest = previousEdgesPoints[nearestIndex[0]];

  // Invalid all points that are on the same scan line than the closest one
  std::vector<uint8_t> idAlreadyTook(this->KeyPointsExtractor->GetNLasers(), 0);
  idAlreadyTook[static_cast<int>(closest.laserId)] = 1;

  // Invalid all points from scan lines that are too far from the closest one
  const int maxScanLineDiff = 4;  // TODO : add parameter to discard too far laser rings
  for (int scanLine = 0; scanLine < this->KeyPointsExtractor->GetNLasers(); ++scanLine)
  {
    if (std::abs(static_cast<int>(closest.laserId) - scanLine) > maxScanLineDiff)
      idAlreadyTook[scanLine] = 1;
  }

  // Make a selection among the neighborhood of the query point.
  // We can only take one edge per scan line.
  for (unsigned int k = 0; k < neighborhoodSize; ++k)
  {
    unsigned int scanLine = previousEdgesPoints[nearestIndex[k]].laserId;
    if (!idAlreadyTook[scanLine] && nearestDist[k] < this->MaxDistanceForICPMatching)
    {
      idAlreadyTook[scanLine] = 1;
      nearestValid.push_back(nearestIndex[k]);
      nearestValidDist.push_back(nearestDist[k]);
    }
  }
}

//-----------------------------------------------------------------------------
void Slam::GetLocalizationLineSpecificNeighbor(std::vector<int>& nearestValid, std::vector<double>& nearestValidDist, double maxDistInlier,
                                               unsigned int nearestSearch, KDTreePCLAdaptor& kdtreePreviousEdges, const double pos[3]) const
{
  // reset vectors
  nearestValid.clear();
  nearestValidDist.clear();

  // Get nearest neighbors of the query point
  std::vector<int> nearestIndex(nearestSearch, -1);
  std::vector<double> nearestDist(nearestSearch, -1.0);
  kdtreePreviousEdges.query(pos, nearestSearch, nearestIndex.data(), nearestDist.data());

  // Check neighborhood validity
  unsigned int neighborhoodSize = nearestIndex.size();
  while (nearestIndex[neighborhoodSize - 1] == -1 && neighborhoodSize > 1)
  {
    --neighborhoodSize;
  }

  // Shortcut to keypoints cloud
  const PointCloud& previousEdgesPoints = *kdtreePreviousEdges.getInputCloud();

  // to avoid square root when performing comparison
  const float squaredMaxDistInlier = maxDistInlier * maxDistInlier;

  // take the closest point
  const Point& closest = previousEdgesPoints[nearestIndex[0]];
  const auto P1 = closest.getVector3fMap();

  // Loop over neighbors of the neighborhood. For each of them, compute the line
  // between closest point and current point and compute the number of inliers
  // that fit this line.
  std::vector<std::vector<unsigned int>> inliersList;
  for (unsigned int ptIndex = 1; ptIndex < neighborhoodSize; ++ptIndex)
  {
    // Fit line that links P1 and P2
    const auto P2 = previousEdgesPoints[nearestIndex[ptIndex]].getVector3fMap();
    Eigen::Vector3f dir = (P2 - P1).normalized();

    // Compute number of inliers of this model
    std::vector<unsigned int> inlierIndex;
    for (unsigned int candidateIndex = 1; candidateIndex < neighborhoodSize; ++candidateIndex)
    {
      if (candidateIndex == ptIndex)
        inlierIndex.push_back(candidateIndex);
      else
      {
        const auto Pcdt = previousEdgesPoints[nearestIndex[candidateIndex]].getVector3fMap();
        if (((Pcdt - P1).cross(dir)).squaredNorm() < squaredMaxDistInlier)
          inlierIndex.push_back(candidateIndex);
      }
    }
    inliersList.push_back(inlierIndex);
  }

  // Keep the line and its inliers with the most inliers.
  std::size_t maxInliers = 0;
  int indexMaxInliers = -1;
  for (unsigned int k = 0; k < inliersList.size(); ++k)
  {
    if (inliersList[k].size() > maxInliers)
    {
      maxInliers = inliersList[k].size();
      indexMaxInliers = k;
    }
  }

  // fill
  nearestValid.push_back(nearestIndex[0]);
  nearestValidDist.push_back(nearestDist[0]);
  for (unsigned int inlier: inliersList[indexMaxInliers])
  {
    nearestValid.push_back(nearestIndex[inlier]);
    nearestValidDist.push_back(nearestDist[inlier]);
  }
}

//==============================================================================
//   Geometrical transformations
//==============================================================================

//-----------------------------------------------------------------------------
Eigen::Isometry3d Slam::InterpolateBeginScanPose()
{
  if (this->NbrFrameProcessed > 0)
  {
    const double prevFrameEnd = this->LogTrajectory.back().time;
    const double currFrameEnd = PclStampToSec(this->CurrentFrame->header.stamp);
    const double currFrameStart = currFrameEnd - this->FrameDuration;
    return LinearInterpolation(this->PreviousTworld, this->Tworld, currFrameStart, prevFrameEnd, currFrameEnd);
  }
  else
    return Eigen::Isometry3d::Identity();
}

//==============================================================================
//   Rolling grids parameters setting
//==============================================================================

//-----------------------------------------------------------------------------
void Slam::ClearMaps()
{
  this->EdgesPointsLocalMap->Reset();
  this->PlanarPointsLocalMap->Reset();
  this->BlobsPointsLocalMap->Reset();
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridLeafSizeEdges(double size)
{
  this->EdgesPointsLocalMap->SetLeafSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridLeafSizePlanes(double size)
{
  this->PlanarPointsLocalMap->SetLeafSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridLeafSizeBlobs(double size)
{
  this->BlobsPointsLocalMap->SetLeafSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridSize(int size)
{
  this->EdgesPointsLocalMap->SetGridSize(size);
  this->PlanarPointsLocalMap->SetGridSize(size);
  this->BlobsPointsLocalMap->SetGridSize(size);
}

//-----------------------------------------------------------------------------
void Slam::SetVoxelGridResolution(double resolution)
{
  this->EdgesPointsLocalMap->SetVoxelResolution(resolution);
  this->PlanarPointsLocalMap->SetVoxelResolution(resolution);
  this->BlobsPointsLocalMap->SetVoxelResolution(resolution);
}

//-----------------------------------------------------------------------------
void Slam::SetFrameMinMaxKeypoints()
{
  // Get pointcloud bounds and set map search bounds accordingly
  auto SetMinMax = [](const PointCloud::Ptr& pc, std::shared_ptr<RollingGrid>& map)
  {
    Eigen::Vector4f minPoint, maxPoint;
    pcl::getMinMax3D(*pc, minPoint, maxPoint);
    map->SetMinMaxPoints(minPoint.head<3>().cast<double>().array(), maxPoint.head<3>().cast<double>().array());
  };

  SetMinMax(this->CurrentEdgesPoints, this->EdgesPointsLocalMap);
  SetMinMax(this->CurrentPlanarsPoints, this->PlanarPointsLocalMap);
  SetMinMax(this->CurrentBlobsPoints, this->BlobsPointsLocalMap);
}