//============================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//============================================================================

#ifndef vtk_m_filter_image_processing_ImageDifference_h
#define vtk_m_filter_image_processing_ImageDifference_h

#include <vtkm/filter/NewFilterField.h>
#include <vtkm/filter/image_processing/vtkm_filter_image_processing_export.h>

namespace vtkm
{
namespace filter
{
namespace image_processing
{
/// \brief Construct an ImageDifference of a given DataSet
///
/// The dataset generated by executing this filter is a Dataset with two Fields:
///   - "image-diff": Uniform Structured Dataset, difference values of image A - B
///   - "threshold-output": Uniform Structured Dataset, the magnitudes of the pixel differences
///
/// The threshold-output is calculated for each pixel using the `vtkm::Magnitude` vector function
/// on the individual pixel difference.
///
class VTKM_FILTER_IMAGE_PROCESSING_EXPORT ImageDifference : public vtkm::filter::NewFilterField
{
public:
  VTKM_CONT ImageDifference();

  VTKM_CONT vtkm::IdComponent GetAverageRadius() const { return this->AverageRadius; }
  VTKM_CONT void SetAverageRadius(const vtkm::IdComponent& averageRadius)
  {
    this->AverageRadius = averageRadius;
  }

  VTKM_CONT vtkm::IdComponent GetPixelShiftRadius() const { return this->PixelShiftRadius; }
  VTKM_CONT void SetPixelShiftRadius(const vtkm::IdComponent& pixelShiftRadius)
  {
    this->PixelShiftRadius = pixelShiftRadius;
  }

  VTKM_CONT vtkm::FloatDefault GetAllowedPixelErrorRatio() const
  {
    return this->AllowedPixelErrorRatio;
  }
  VTKM_CONT void SetAllowedPixelErrorRatio(const vtkm::FloatDefault& pixelErrorRatio)
  {
    this->AllowedPixelErrorRatio = pixelErrorRatio;
  }

  VTKM_CONT vtkm::FloatDefault GetPixelDiffThreshold() const { return this->PixelDiffThreshold; }
  VTKM_CONT void SetPixelDiffThreshold(const vtkm::FloatDefault& threshold)
  {
    this->PixelDiffThreshold = threshold;
  }

  VTKM_CONT bool GetImageDiffWithinThreshold() const { return this->ImageDiffWithinThreshold; }
  VTKM_CONT void SetThresholdFieldName(const std::string& name) { this->ThresholdFieldName = name; }
  VTKM_CONT std::string GetThresholdFieldName() const { return this->ThresholdFieldName; }

  /// Choose the primary field to operate on.  For Image difference A - B, A is the
  /// primary field.
  VTKM_CONT
  void SetPrimaryField(
    const std::string& name,
    vtkm::cont::Field::Association association = vtkm::cont::Field::Association::Any)
  {
    this->SetActiveField(name, association);
  }

  VTKM_CONT std::string GetPrimaryFieldName() const { return this->GetActiveFieldName(); }
  VTKM_CONT vtkm::cont::Field::Association GetPrimaryFieldAssociation() const
  {
    return this->GetActiveFieldAssociation();
  }

  /// Choose the secondary field to operate on.  For Image difference A - B, B is the
  /// secondary field.
  VTKM_CONT
  void SetSecondaryField(
    const std::string& name,
    vtkm::cont::Field::Association association = vtkm::cont::Field::Association::Any)
  {
    this->SetActiveField(1, name, association);
  }

  VTKM_CONT std::string GetSecondaryFieldName() const { return this->GetActiveFieldName(1); }
  VTKM_CONT vtkm::cont::Field::Association GetSecondaryFieldAssociation() const
  {
    return this->GetActiveFieldAssociation(1);
  }

private:
  VTKM_CONT vtkm::cont::DataSet DoExecute(const vtkm::cont::DataSet& primaryArray) override;

  vtkm::IdComponent AverageRadius = 0;
  vtkm::IdComponent PixelShiftRadius = 0;
  vtkm::FloatDefault AllowedPixelErrorRatio = 0.00025f;
  vtkm::FloatDefault PixelDiffThreshold = 0.05f;
  bool ImageDiffWithinThreshold = true;
  std::string ThresholdFieldName = "threshold-output";
};
} // namespace image_processing
class VTKM_DEPRECATED(1.8, "Use vtkm::filter::image_processing::ImageDifference.") ImageDifference
  : public vtkm::filter::image_processing::ImageDifference
{
  using image_processing::ImageDifference::ImageDifference;
};
} // namespace filter
} // namespace vtkm

#endif // vtk_m_filter_image_processing_ImageDifference_h
