//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#include "qtSharedPropertiesItem.h"

#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/GroupItem.h"
#include "smtk/extension/qt/qtAttributeItemInfo.h"
#include "smtk/extension/qt/qtItem.h"
#include "smtk/extension/qt/qtUIManager.h"
#include "smtk/view/Configuration.h"

#include "smtk/simulation/truchas/qt/ctkCollapsibleButton.h"
#include "smtk/simulation/truchas/qt/qtMaterialAttribute.h"

#include <QDebug>
#include <QFrame>
#include <QHBoxLayout>
#include <QIcon>
#include <QLabel>
#include <QVBoxLayout>

#define ALERT_ICON_PATH ":/icons/attribute/errorAlert.png"

class qtSharedPropertiesItemInternals
{
public:
  qtMaterialAttribute* m_qAttribute;
  smtk::attribute::GroupItemPtr m_phasesItem;
  smtk::attribute::GroupItemPtr m_sharedItem;
  QLabel* m_alertLabel;

  qtSharedPropertiesItemInternals()
    : m_qAttribute(nullptr)
    , m_alertLabel(nullptr)
  {
  }
};

static smtk::extension::qtItem* createItemWidget(
  const qtItemInfo& info, qtMaterialAttribute* attribute)
{
  return new qtSharedPropertiesItem(info, attribute);
}

qtSharedPropertiesItem::qtSharedPropertiesItem(
  const qtItemInfo& sharedInfo, qtMaterialAttribute* qAttribute)
  : smtk::extension::qtItem(sharedInfo)
{
  this->Internals = new qtSharedPropertiesItemInternals;
  this->Internals->m_qAttribute = qAttribute;
  auto att = qAttribute->attribute();
  this->Internals->m_phasesItem = att->findGroup("phases");
  this->Internals->m_sharedItem = att->findGroup("shared-properties");
  this->createWidget();
}

qtSharedPropertiesItem::~qtSharedPropertiesItem()
{
  delete this->Internals;
}

void qtSharedPropertiesItem::onItemModified()
{
  // qDebug() << "qtSharedPropertiesItem::onItemModified()";
  auto qItem = qobject_cast<smtk::extension::qtItem*>(this->sender());
  if (qItem == nullptr)
  {
    return;
  }
  emit this->modified();

  // Check validity
  bool isValid = this->Internals->m_sharedItem->isValid();
  this->Internals->m_alertLabel->setVisible(!isValid);

  // Check enabled state
  smtk::attribute::ItemPtr item = qItem->item();
  bool isEnabled = item->isEnabled();
  bool isInSet = m_enabledItemNames.count(item->name());
  // qDebug() << "isEnabled" << isEnabled << "isInSet" << isInSet;
  if (isEnabled == isInSet)
  {
    return;
  }
  if (isEnabled)
  {
    m_enabledItemNames.insert(item->name());
  }
  else
  {
    m_enabledItemNames.erase(item->name());
  }

  emit this->itemEnabledStateChanged(QString::fromStdString(item->name()), isEnabled);
}

void qtSharedPropertiesItem::onPhaseCountChanged()
{
  bool makeVisible = this->Internals->m_phasesItem->numberOfGroups() > 1;
  m_widget->setVisible(makeVisible);
}

void qtSharedPropertiesItem::createWidget()
{
  if (this->Internals->m_sharedItem == nullptr)
  {
    return;
  }
  m_enabledItemNames.clear();

  auto parentWidget = this->Internals->m_qAttribute->widget();
  auto* topFrame = new QFrame(parentWidget);
  auto* topLayout = new QHBoxLayout(topFrame);

  auto* label = new QLabel(topFrame);
  auto icon = QIcon(ALERT_ICON_PATH);
  label->setPixmap(icon.pixmap(16));
  label->setMinimumWidth(24);
  topLayout->addWidget(label);

  bool isValid = this->Internals->m_sharedItem->isValid();
  label->setVisible(!isValid);
  this->Internals->m_alertLabel = label;

  m_widget = topFrame;
  m_widget->setLayout(topLayout);

  auto itemFrame = new ctkCollapsibleButton(parentWidget);
  // itemFrame->setVerticalPadding(verticalPadding); // set this *before* text
  itemFrame->setText("Shared Material Properties");
  itemFrame->setContentsFrameShape(QFrame::Box);
  itemFrame->setCollapsedHeight(0);
  itemFrame->setMinimumWidth(400); // cosmetic hack
  topLayout->addWidget(itemFrame, 1);

  auto itemLayout = new QVBoxLayout();
  itemFrame->setLayout(itemLayout);

  for (std::size_t i = 0; i < this->Internals->m_sharedItem->numberOfItemsPerGroup(); ++i)
  {
    auto item = this->Internals->m_sharedItem->item(i);
    smtk::view::Configuration::Component comp; // create a default style (empty component)
    qtItemInfo info(item, comp, m_widget, m_itemInfo.baseView());
    qtItem* childItem = m_itemInfo.uiManager()->createItem(info);
    if (childItem)
    {
      if (item->isEnabled())
      {
        m_enabledItemNames.insert(item->name());
      }
      itemLayout->addWidget(childItem->widget());
      QObject::connect(childItem, &qtItem::modified, this, &qtSharedPropertiesItem::onItemModified);
    }
  } // for (i)

  this->onPhaseCountChanged();
}
