//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "smtk/simulation/truchas/qt/qtMaterialAttribute.h"

// Truchas Extension headers
#include "smtk/simulation/truchas/qt/qtMaterialItem.h"
#include "smtk/simulation/truchas/qt/qtPhasePropertyItem.h"
#include "smtk/simulation/truchas/qt/qtSharedPropertiesItem.h"

// SMTK headers
#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/GroupItem.h"
#include "smtk/attribute/IntItem.h"
#include "smtk/attribute/SearchStyle.h"
#include "smtk/extension/qt/qtBaseAttributeView.h"
#include "smtk/extension/qt/qtBaseView.h"
#include "smtk/extension/qt/qtGroupItem.h"
#include "smtk/extension/qt/qtUIManager.h"
#include "smtk/io/Logger.h"
#include "smtk/operation/Manager.h"
#include "smtk/simulation/UserData.h"
#include "smtk/view/Configuration.h"

#include <QDebug>
#include <QFrame>
#include <QLabel>
#include <QList>
#include <QPointer>
#include <QVBoxLayout>

#include <map>
#include <set>
#include <string>

using namespace smtk::attribute;
using namespace smtk::extension;

class qtMaterialAttributeInternals
{
public:
  // Member data
  smtk::attribute::WeakAttributePtr m_attribute;
  QPointer<QWidget> m_parentWidget;
  QList<smtk::extension::qtItem*> m_items;
  qtBaseView* m_view;
  smtk::view::Configuration::Component m_attComp;
  std::map<std::string, qtAttributeItemInfo> m_itemViewMap;

  qtMaterialItem* m_materialItem;
  qtSharedPropertiesItem* m_sharedPropertiesItem;

  // Methods
  qtMaterialAttributeInternals(smtk::attribute::AttributePtr myAttribute,
    const smtk::view::Configuration::Component& comp, QWidget* p, qtBaseView* myView)
    : m_parentWidget(p)
    , m_attribute(myAttribute)
    , m_view(myView)
    , m_attComp(comp)
    , m_materialItem(nullptr)
    , m_sharedPropertiesItem(nullptr)
  {
    // Check for ItemViews
    int sindex = comp.findChild("ItemViews");
    if (sindex == -1)
    {
      return;
    }
    auto iviews = m_attComp.child(sindex);
    qtAttributeItemInfo::buildFromComponent(iviews, m_view, m_itemViewMap);
  }

  ~qtMaterialAttributeInternals() {}
};

qtMaterialAttribute::qtMaterialAttribute(smtk::attribute::AttributePtr myAttribute,
  const smtk::view::Configuration::Component& comp, QWidget* p, qtBaseView* myView)
{
  m_internals = new qtMaterialAttributeInternals(myAttribute, comp, p, myView);
  m_widget = NULL;
  m_useSelectionManager = false;
  this->createWidget();
  m_isEmpty = true;

  // Initialize shared-state for phase items
  // This logic *might* belong in qtMaterialItem, but for now...
  std::set<std::string> sharedNames;
  auto sharedItem = myAttribute->findGroup("shared-properties");
  assert(sharedItem);
  for (std::size_t i = 0; i < sharedItem->numberOfItemsPerGroup(); ++i)
  {
    auto item = sharedItem->item(i);
    if (item->isEnabled())
    {
      sharedNames.insert(item->name());
    }
  } // for (i)

  if (sharedNames.empty())
  {
    return;
  }

  auto key = qtPhasePropertyItem::userDataKey();
  int isShared = 1;
  auto phasesItem = myAttribute->findGroup("phases");
  for (std::size_t element = 0; element < phasesItem->numberOfGroups(); ++element)
  {
    for (auto iter = sharedNames.begin(); iter != sharedNames.end(); ++iter)
    {
      auto name = *iter;
      auto item = phasesItem->find(element, name, smtk::attribute::RECURSIVE);

      auto data = smtk::simulation::UserDataInt::New();
      smtk::dynamic_pointer_cast<smtk::simulation::UserDataInt>(data)->setValue(isShared);
      item->setUserData(key, data);
    }
  } // for (i)
}

qtMaterialAttribute::~qtMaterialAttribute()
{
  // First Clear all the items
  for (int i = 0; i < m_internals->m_items.count(); i++)
  {
    delete m_internals->m_items.value(i);
  }

  m_internals->m_items.clear();
  if (m_widget)
  {
    delete m_widget;
  }

  delete m_internals;
}

void qtMaterialAttribute::createWidget()
{
  // Initially there are no items being displayed
  m_isEmpty = true;
  auto att = this->attribute();
  if ((att == nullptr) || ((att->numberOfItems() == 0) && (att->associations() == nullptr)))
  {
    return;
  }

  int numShowItems = 0;
  std::size_t i, n = att->numberOfItems();
  if (m_internals->m_view)
  {
    for (i = 0; i < n; i++)
    {
      if (static_cast<qtBaseAttributeView*>(m_internals->m_view)
            ->displayItem(att->item(static_cast<int>(i))))
      {
        numShowItems++;
      }
    }
    // also check associations
    if (static_cast<qtBaseAttributeView*>(m_internals->m_view)->displayItem(att->associations()))
    {
      numShowItems++;
    }
  }
  else // show everything
  {
    numShowItems = static_cast<int>(att->associations() ? n + 1 : n);
  }
  if (numShowItems == 0)
  {
    return;
  }
  m_isEmpty = false;
  QFrame* attFrame = new QFrame(this->parentWidget());
  attFrame->setFrameShape(QFrame::Box);
  m_widget = attFrame;

  QVBoxLayout* layout = new QVBoxLayout(m_widget);
  layout->setMargin(3);
  m_widget->setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Expanding);
}

void qtMaterialAttribute::addItem(qtItem* child)
{
  if (!m_internals->m_items.contains(child))
  {
    m_internals->m_items.append(child);
    QObject::connect(child, &qtItem::modified, this, &qtMaterialAttribute::onItemModified);
  }
}

QList<qtItem*>& qtMaterialAttribute::items() const
{
  return m_internals->m_items;
}

void qtMaterialAttribute::showAdvanceLevelOverlay(bool show)
{
  for (int i = 0; i < m_internals->m_items.count(); i++)
  {
    m_internals->m_items.value(i)->showAdvanceLevelOverlay(show);
  }
}

void qtMaterialAttribute::createBasicLayout(bool includeAssociations)
{
  // Initially we have not displayed any items
  m_isEmpty = true;
  // If there is no main widget there is nothing to show
  if (!m_widget)
  {
    return;
  }
  QLayout* layout = m_widget->layout();
  smtk::attribute::AttributePtr att = this->attribute();
  auto uiManager = m_internals->m_view->uiManager();

  std::size_t phaseCount = 0;
  // Get "phases" and "transitions" group items,
  // which are displayed after everything else
  smtk::attribute::GroupItemPtr phasesItem = att->findGroup("phases");
  if (phasesItem == nullptr)
  {
    QString msg("Cannot display phase/transition table; missing \"phases\" item");
    qWarning() << msg;
    auto* label = new QLabel(msg, m_widget);
    layout->addWidget(label);
    return;
  }
  phaseCount = phasesItem->numberOfGroups();

  smtk::attribute::GroupItemPtr transitionsItem = att->findGroup("transitions");
  if (transitionsItem == nullptr)
  {
    QString msg("Cannot display phase/transition table; missing \"transitions\" item");
    qWarning() << msg;
    auto label = new QLabel(msg, m_widget);
    layout->addWidget(label);
    return;
  }

  qtSharedPropertiesItem* qSharedProperties = nullptr;
  qtMaterialItem* qMaterial = nullptr;

  // Handle shared-properties group item
  smtk::attribute::GroupItemPtr sharedPropertiesItem = att->findGroup("shared-properties");
  if (sharedPropertiesItem == nullptr)
  {
    QString msg("Missing \"shared-properties\" item");
    qWarning() << msg;
    auto* label = new QLabel(msg, m_widget);
    layout->addWidget(label);
  }
  else
  {
    smtk::view::Configuration::Component comp;
    qtAttributeItemInfo sharedInfo(sharedPropertiesItem, comp, m_widget, m_internals->m_view);
    qSharedProperties = new qtSharedPropertiesItem(sharedInfo, this);
    if (qSharedProperties && qSharedProperties->widget())
    {
      m_isEmpty = false;
      layout->addWidget(qSharedProperties->widget());
      this->addItem(qSharedProperties);
      QObject::connect(
        qSharedProperties, &qtItem::modified, this, &qtMaterialAttribute::onItemModified);
    }
  }

  // Handle phase/transition sequence separately
  auto it = m_internals->m_itemViewMap.find(phasesItem->name());
  if (it != m_internals->m_itemViewMap.end())
  {
    auto info = it->second;
    info.setParentWidget(m_widget);
    // Cast to base item before calling setItem (otherwise compiler error)
    auto item = std::static_pointer_cast<smtk::attribute::Item>(phasesItem);
    info.setItem(item);
    qMaterial = new qtMaterialItem(info, this);
  }
  else
  {
    smtk::view::Configuration::Component comp;
    qtAttributeItemInfo phasesInfo(phasesItem, comp, m_widget, m_internals->m_view);
    qMaterial = new qtMaterialItem(phasesInfo, this);
  }

  if (qMaterial && qMaterial->widget())
  {
    m_isEmpty = false;
    layout->addWidget(qMaterial->widget());
    this->addItem(qMaterial);
    QObject::connect(qMaterial, &qtMaterialItem::phaseCountChanged, this,
      &qtMaterialAttribute::onPhaseCountChanged);
  }

  if (qMaterial && qSharedProperties)
  {
    // Notify qtMaterialItem when shared property is selected/unselected
    QObject::connect(qSharedProperties, &qtSharedPropertiesItem::itemEnabledStateChanged, qMaterial,
      &qtMaterialItem::onSharedStateChanged);
  }
  m_internals->m_materialItem = qMaterial;
  m_internals->m_sharedPropertiesItem = qSharedProperties;

  m_isEmpty = false;
}

smtk::attribute::AttributePtr qtMaterialAttribute::attribute()
{
  return m_internals->m_attribute.lock();
}

QWidget* qtMaterialAttribute::parentWidget()
{
  return m_internals->m_parentWidget;
}

/* Slot for properly emitting signals when an attribute's item is modified */
void qtMaterialAttribute::onItemModified()
{
  // are we here due to a signal?
  QObject* sobject = this->sender();
  if (sobject == nullptr)
  {
    return;
  }
  auto iobject = qobject_cast<smtk::extension::qtItem*>(sobject);
  if (iobject == nullptr)
  {
    return;
  }

  emit this->itemModified(iobject);
}

void qtMaterialAttribute::onPhaseCountChanged()
{
  // Notify shared properties item
  m_internals->m_sharedPropertiesItem->onPhaseCountChanged();

  // For notifying view(s)
  emit this->modified();
}

bool qtMaterialAttribute::isEmpty() const
{
  return m_isEmpty;
}
