"""
Mixin class for writing materials
"""
INDUCTION_HEATING = 'Induction Heating'


from .cardformat import CardFormat

PHASE_FORMAT_LIST = [
    # Note that item_path is actually the item name under the extensible "phases" group item
    CardFormat('density', item_path='density', as_property=True),
    CardFormat('viscosity', item_path='fluid/viscosity', as_property=True, is_custom=True),
    CardFormat('density deviation', item_path='density-deviation', as_property=True),
    CardFormat('specific heat', item_path='specific-heat', as_property=True),
    CardFormat('conductivity', item_path='conductivity', as_property=True),
    CardFormat('electrical_conductivity', item_path='electrical-properties/electrical-conductivity', as_property=True, is_custom=True),
    CardFormat('electrical_susceptibility', item_path='electrical-properties/electrical-susceptibility', as_property=True, is_custom=True),
    CardFormat('magnetic_susceptibility', item_path='electrical-properties/magnetic-susceptibility', as_property=True, is_custom=True)
]

VOID_MATERIAL_FORMAT_LIST = [
    CardFormat('void_temperature', item_path='void-temperature'),
    CardFormat('sound_speed', item_path='sound-speed')
]


class MaterialWriter:
# ---------------------------------------------------------------------
    def _write_physical_materials(self, namelist, format_list):
        '''Writes interleaved MATERIAL, PHASE, MATERIAL_SET namelists

        '''
        print('Writing MATERIAL/PHASE/MATERIAL_SYSTEM namelists')

        # Write physical materials
        separator = namelist.separator
        att_list = self.sim_atts.findAttributes('material.real')
        att_list.sort(key=lambda att: att.name())
        for att in att_list:
            self._check_attribute(att)

            if separator is not None:
                self.out.write('\n')
                label = '### {} '.format(namelist.separator)
                line = label.ljust(80, '#')
                self.out.write(line)
                self.out.write('\n')
                separator = None

            phases_group = att.findGroup('phases')
            transitions_group = att.findGroup('transitions')
            self._write_material_system_namelist(phases_group, transitions_group)

            # Create map of <keyword, smtk item> for the shared properties that have
            # been enabled.
            shared_props = self._get_shared_properties(att)

            num_phases = phases_group.numberOfGroups()
            for element in range(num_phases):
                if num_phases == 1:
                    # For single-phase, use attribute name
                    name = phases_group.attribute().name()
                else:
                    name = phases_group.find(element, 'name').value()
                self._write_phase_namelist(phases_group, name, element, shared_props)
                self._write_material_namelist(phases_group, name, element)

        self._write_void_material()

    def _get_shared_properties(self, att):
        """Returns dictionary of <keyword, item> for the shared properties that are enabled"""
        shared_dict = dict()
        shared_group = att.findGroup('shared-properties')
        if shared_group is None:
            print('WARNING shared-properties item is missing')
            return shared_dict
        for card in PHASE_FORMAT_LIST:
            split_path = card.item_path.split('/')
            item_name = split_path[-1]
            item = shared_group.find(item_name)
            if item.isEnabled():
                shared_dict[card.keyword] = item
        return shared_dict

    def _write_material_system_namelist(self, phases_group, transitions_group):
        """Write MATERIAL_SYSTEM namelist for one element in phases group item"""
        self._start_namelist('MATERIAL_SYSTEM')

        att = phases_group.attribute()
        name = att.name()
        CardFormat.write_value(self.out, 'name', name[:31], tab=4)

        num_phases = phases_group.numberOfGroups()
        if num_phases == 1:
            CardFormat.write_value(self.out, 'phases', name[:31], tab=4)
            self._finish_namelist()
            return

        num_transitions = transitions_group.numberOfGroups()

        # Traverse subgroups to collect data
        phase_list = [None] * num_phases
        lo_temps = [None] * num_transitions
        hi_temps = [None] * num_transitions
        latent_heats = [None] * num_transitions
        smoothing_radius = None
        for i in range(num_phases):
            name = phases_group.find(i, 'name').value()
            phase_list[i] = '\"{}\"'.format(name[:31])

        for i in range(num_transitions):
            lo_temps[i] = transitions_group.find(i, 'lower-transition-temperature').value()
            hi_temps[i] = transitions_group.find(i, 'upper-transition-temperature').value()
            latent_heats = transitions_group.find(i, 'latent-heat').value()

            # Smoothing radius currently part of the transition group, but only a single value
            # applies to the material system. This code takes the smoothing radius
            # from the *first* transition that has its item enabled.
            if smoothing_radius is None:
                smoothing_radius_item = transitions_group.find(i, 'smoothing-radius')
                if smoothing_radius_item and smoothing_radius_item.isEnabled():
                    smoothing_radius = smoothing_radius_item.value()

        # Write the lists
        CardFormat.write_value(self.out, 'phases', phase_list)
        CardFormat.write_value(self.out, 'transition_temps_low', lo_temps)
        CardFormat.write_value(self.out, 'transition_temps_high', hi_temps)
        CardFormat.write_value(self.out, 'latent_heat', latent_heats)
        if smoothing_radius is not None:
            CardFormat.write_value(self.out, 'smoothing_radius', smoothing_radius)

        self._finish_namelist()

    def _write_material_namelist(self, phases_group, name, element):
        """Write MATERIAL namelist for one element in phases group item"""
        self._start_namelist('MATERIAL')
        att = phases_group.attribute()

        # Assign next material number to this att+element
        key = self._material_key(att, element)
        number = len(self.material_number_dict)
        self.material_number_dict[key] = number

        CardFormat.write_value(self.out, 'material_name', name[:31])
        CardFormat.write_value(self.out, 'material_number', number)

        # Write material density (1 for physical material, 0 for void)
        # We *should* only be passing in physical materials,
        # but check just in case...
        density = 0.0 if att.type() == 'material.void' else 1.0
        CardFormat.write_value(self.out, 'density', density)

        # Write immobile property
        fluid_item = phases_group.find(element, 'fluid')
        immobile = not fluid_item.isEnabled()
        CardFormat.write_value(self.out, 'immobile', immobile, as_boolean=True)

        # If background material not assigned, use this material
        if self.background_material_id is None:
            self.background_material_id = att.id()

        # Check if this material is the background material
        if att.id() == self.background_material_id:
            CardFormat.write_value(self.out, 'material_feature', 'background')

        self._finish_namelist()

    def _write_phase_namelist(self, phases_group, name, element, shared_props):
        """Write PHASE namelist for one element in phases group item"""
        self._start_namelist('PHASE')

        CardFormat.write_value(self.out, 'name', name[:31], tab=4)

        CardFormat.PropertyIndex = 0
        for card in PHASE_FORMAT_LIST:
            card_item = None

            # Check for shared item first
            shared_item = shared_props.get(card.keyword)
            if shared_item is not None:
                card_item = shared_item
            elif card.is_custom:
                item_name = card.item_path.split('/')[-1]
                if card.keyword == 'viscosity':
                    fluid_item = phases_group.find(element, 'fluid')
                    if fluid_item.isEnabled():
                        card_item = fluid_item.find(0, 'viscosity')
                elif card.item_path.startswith('electrical-properties') and \
                        INDUCTION_HEATING in CardFormat.Conditions:
                    em_item = phases_group.find(element, 'electrical-properties')
                    card_item = em_item.find(0, item_name)
            else:
                card_item = phases_group.find(element, card.item_path)

            if card_item is not None:
                card.write_item(card_item, self.out)

        self._finish_namelist()

    def _write_void_material(self):
        """Write void attribute if required"""
        if self.skip_void_material:
            return

        void_list = self.sim_atts.findAttributes('material.void')
        if not void_list:
            return

        void_att = void_list[0]
        self._check_attribute(void_att)

        self._start_namelist('MATERIAL')

        # Assign next material number
        key = self._material_key(void_att, 0)
        number = len(self.material_number_dict)
        self.material_number_dict[key] = number

        CardFormat.write_value(self.out, 'material_name', 'void')
        CardFormat.write_value(self.out, 'material_number', number)
        CardFormat.write_value(self.out, 'density', 0.0)

        for card in VOID_MATERIAL_FORMAT_LIST:
            card.write(self.out, void_att)

        # If background material not assigned, use this material
        if self.background_material_id is None:
            self.background_material_id = void_att.id()

        # Check if this material is the background material
        if void_att.id() == self.background_material_id:
            CardFormat.write_value(self.out, 'material_feature', 'background')

        self._finish_namelist()
